/** @file
  USB policy

 @copyright
  INTEL CONFIDENTIAL
  Copyright 2016 - 2018 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#ifndef _USB_CONFIG_H_
#define _USB_CONFIG_H_

#define USB_CONFIG_REVISION 3
extern EFI_GUID gUsbConfigGuid;

#pragma pack (push,1)

//
// Overcurrent pins, the values match the setting of SC EDS, please refer to SC EDS for more details
//
typedef enum {
  ScUsbOverCurrentPin0 = 0,
  ScUsbOverCurrentPin1,
  ScUsbOverCurrentPinSkip,
  ScUsbOverCurrentPinMax
} SC_USB_OVERCURRENT_PIN;

#define XHCI_MODE_OFF 0
#define XHCI_MODE_ON  1


/**
This structure configures per USB2 AFE settings.
It allows to setup the port parameters.
**/
typedef struct {
  /** Per Port HS Preemphasis Bias (PERPORTPETXISET)
  000b - 0mV
  001b - 40.5mV
  010b - 60.5mV
  011b - 102mV
  100b - 102mV
  101b - 142mV
  110b - 162.5mV
  111b - 202.5mV
  **/
  UINT8   Petxiset;
  /** Per Port HS Transmitter Bias (PERPORTTXISET)
  000b - 0mV
  001b - 40.5mV
  010b - 60.5mV
  011b - 102mV
  100b - 102mV
  101b - 142mV
  110b - 162.5mV
  111b - 202.5mV
  **/
  UINT8   Txiset;
  /**
  Per Port HS Transmitter Emphasis (IUSBTXEMPHASISEN)
  00b - Emphasis OFF
  01b - De-emphasis ON
  10b - Pre-emphasis ON
  11b - Pre-emphasis & De-emphasis ON
  **/
  UINT8   Predeemp;
  /**
  Per Port Half Bit Pre-emphasis (PERPORTTXPEHALF)
  1b - half-bit pre-emphasis
  0b - full-bit pre-emphasis
  **/
  UINT8   Pehalfbit;
} USB20_AFE;

typedef struct {
  UINT8 Mode     : 2;    /// 0: Disable; 1: Enable, 2: Auto, 3: Smart Auto
  UINT8 Rsvdbits : 6;
} SC_USB30_CONTROLLER_SETTINGS;

/**
  This structure configures per USB2 port physical settings.
  It allows to setup the port location and port length, and configures the port strength accordingly.
**/
typedef struct {
  UINT32 Enable    :  1;   ///< 0: Disable; <b>1: Enable</b>.
  UINT32 RsvdBits0 : 31;   ///< Reserved bits
  /**
    These members describe the specific over current pin number of USB 2.0 Port N.
    It is SW's responsibility to ensure that a given port's bit map is set only for
    one OC pin Description. USB2 and USB3 on the same combo Port must use the same
    OC pin (see: SC_USB_OVERCURRENT_PIN).
  **/
  UINT8  OverCurrentPin;
  UINT8  Rsvd0[3];         ///< Reserved bytes, align to multiple 4.
  USB20_AFE  Afe;
  UINT32 Rsvd1[1];         ///< Reserved bytes
} SC_USB20_PORT_CONFIG;

/**
  This structure describes whether the USB3 Port is enabled by platform modules.
**/
typedef struct {
  UINT32 Enable    :  1;   ///< 0: Disable; <b>1: Enable</b>.
  UINT32 RsvdBits0 : 31;   ///< Reserved bits
  /**
    These members describe the specific over current pin number of USB 3.0 Port N.
    It is SW's responsibility to ensure that a given port's bit map is set only for
    one OC pin Description. USB2 and USB3 on the same combo Port must use the same
    OC pin (see: SC_USB_OVERCURRENT_PIN).
  **/
  UINT8  OverCurrentPin;
  UINT8  Rsvd0[3];         ///< Reserved bytes, align to multiple 4
  UINT32 Rsvd1[2];         ///< Reserved bytes
} SC_USB30_PORT_CONFIG;

typedef struct {
  /**
    This member describes whether or not the xDCI controller should be enabled.
    0: Disabled; <b>1: PCI Mode</b> 2: ACPI Mode
  **/
  UINT32 Enable   : 2;
  UINT32 RsvdBits : 30;  ///< Reserved bits
} SC_XDCI_CONFIG;

/**
  These members describe some settings which are related to the HSIC ports.
**/
typedef struct {
  /**
    <b>0: Disable</b>; 1: Enable HSIC support.
  **/
  UINT32  Enable          : 1;
  UINT32  RsvdBits        : 31;
} SC_XHCI_HSIC_PORT;

/**
  These members describe some settings which are related to the SSIC ports.
**/
typedef struct {
  SC_XHCI_HSIC_PORT  HsicPort[XHCI_MAX_HSIC_PORTS];
} SC_HSIC_CONFIG;

typedef enum {
  XhciSsicRateA = 1,
  XhciSsicRateB
} SC_XHCI_SSIC_PORT_RATE;

/**
  These members describe some settings which are related to the SSIC ports.
**/
typedef struct {
  /**
    <b>0: Disable</b>; 1: Enable SSIC support.
  **/
  UINT32  Enable          : 1;
  /**
    <b>(Test)</b> Default is <b>XhciSsicRateA</b>
  **/
  UINT32  Rate            : 2;
  UINT32  RsvdBits        : 29;
} SC_XHCI_SSIC_PORT;
/**
  These members describe some settings which are related to the SSIC ports.
**/
typedef struct {
  SC_XHCI_SSIC_PORT  SsicPort[XHCI_MAX_SSIC_PORTS];
  /**
    0: Disable <b>1: Enable</b> SSIC Data lane Power Gating
  **/
  UINT32  DlanePwrGating : 1;
  UINT32  RsvdBits       : 31;
} SC_SSIC_CONFIG;

/**
  This member describes the expected configuration of the SC USB controllers,
  Platform modules may need to refer Setup options, schematic, BIOS specification
  to update this field.
  The Usb20OverCurrentPins and Usb30OverCurrentPins field must be updated by referring
  the schematic.
**/
typedef struct {
  /**
    Revision 1: Init version
    Revision 2: Deprecate UsbPerPortCtl
  **/
  CONFIG_BLOCK_HEADER   Header;     ///< Config Block Header
  /**
    This policy will disable XHCI compliance mode on all ports. Complicance Mode should be default enabled.
    For the platform that support USB Type-C, it can disable Compliance Mode, and enable Compliance Mode when testing.
    <b>0:Disable</b> , 1: Enable
  **/
  UINT32 DisableComplianceMode : 1;
  UINT32 UsbPerPortCtl         : 1; ///< @deprecated since revision 2
  /**
  This policy option when set will make BIOS program Port Disable Override register during PEI phase.
  When disabled BIOS will not program the PDO during PEI phase and leave PDO register unlocked for later programming.
  If this is disabled, platform code MUST set it before booting into OS.
  <b>1: Enable</b>
  0: Disable
  **/
  UINT32 PdoProgramming        : 1;
  UINT32 ModPhyIfValue         : 8;
  UINT32 RsvdBits              : 21;
  SC_USB30_CONTROLLER_SETTINGS  Usb30Settings;
  UINT8  RsvdBytes[3];
  SC_USB20_PORT_CONFIG          PortUsb20[SC_MAX_USB2_PORTS];
  /**
    These members describe whether the USB3 Port N of SC is enabled by platform modules.
  **/
  SC_USB30_PORT_CONFIG          PortUsb30[SC_MAX_USB3_PORTS];
  /**
    This member describes whether or not the xDCI controller should be enabled.
  **/
  SC_XDCI_CONFIG                XdciConfig;
  /**
    These members describe some settings which are related to the HSIC ports.
  **/
  SC_HSIC_CONFIG                HsicConfig;
  /**
    These members describe some settings which are related to the SSIC ports.
  **/
  SC_SSIC_CONFIG                SsicConfig;
} SC_USB_CONFIG;

#pragma pack (pop)

#endif // _USB_CONFIG_H_
