/** @file
  Register names for USB devices.

  Conventions:

  - Prefixes:
    Definitions beginning with "R_" are registers
    Definitions beginning with "B_" are bits within registers
    Definitions beginning with "V_" are meaningful values of bits within the registers
    Definitions beginning with "S_" are register sizes
    Definitions beginning with "N_" are the bit position
  - In general, SC registers are denoted by "_SC_" in register names
  - Registers / bits that are different between SC generations are denoted by
    "_SC_<generation_name>_" in register/bit names.
  - Registers / bits that are different between SKUs are denoted by "_<SKU_name>"
    at the end of the register/bit names
  - Registers / bits of new devices introduced in a SC generation will be just named
    as "_SC_" without <generation_name> inserted.

 @copyright
  INTEL CONFIDENTIAL
  Copyright 1999 - 2016 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#ifndef _REGS_USB_H_
#define _REGS_USB_H_

///
/// USB Definitions
///

typedef enum {
  ScEhci1 = 0,
  ScEhciControllerMax
} SC_USB20_CONTROLLER_TYPE;

#define PCI_DEVICE_NUMBER_USB           29    ///< When you refer to this macro, this mean your code is wrong since no EHCI is available in broxton
#define PCI_FUNCTION_NUMBER_EHCI        0

#define R_SC_USB_CFG_VENDOR_ID                 0x00  ///< Vendor ID
#define V_SC_USB_CFG_VENDOR_ID                 V_INTEL_VENDOR_ID

#define R_SC_USB_CFG_DEVICE_ID                 0x02  ///< Device ID
#define V_SC_DEVICE_ID_0                   0x0F34  ///< EHCI#1

#define R_SC_EHCI_CFG_COMMAND_REGISTER         0x04  ///< Command
#define B_SC_EHCI_CFG_COMMAND_INTR_DIS         BIT10 ///< Interrupt Disable
#define B_SC_EHCI_CFG_COMMAND_FBE              BIT9  ///< Fast Back to Back Enable: Reserved as '0'
#define B_SC_EHCI_CFG_COMMAND_SERR_EN          BIT8  ///< SERR# Enable
#define B_SC_EHCI_CFG_COMMAND_WCC              BIT7  ///< Wait Cycle Enable: Reserved as '0'
#define B_SC_EHCI_CFG_COMMAND_PER              BIT6  ///< Parity Error Response
#define B_SC_EHCI_CFG_COMMAND_VPS              BIT5  ///< VGA Pallete Snoop: Reserved as '0'
#define B_SC_EHCI_CFG_COMMAND_PMWE             BIT4  ///< Post Memory Write Enable: Reserved as '0'
#define B_SC_EHCI_CFG_COMMAND_SCE              BIT3  ///< Special Cycle Enable: Reserved as '0'
#define B_SC_EHCI_CFG_COMMAND_BME              BIT2  ///< Bus Master Enable
#define B_SC_EHCI_CFG_COMMAND_MSE              BIT1  ///< Memory Space Enable
#define B_SC_EHCI_CFG_COMMAND_IOSE             BIT0  ///< I/O Space Enable: Reserved as '0', R/O

#define R_SC_EHCI_CFG_PCISTS                   0x06  ///< Device Status
#define B_SC_EHCI_CFG_PCISTS_DPE               BIT15 ///< Detected Parity Error
#define B_SC_EHCI_CFG_PCISTS_SSE               BIT14 ///< Signaled System Error
#define B_SC_EHCI_CFG_PCISTS_RMA               BIT13 ///< Received Master-Abort Status
#define B_SC_EHCI_CFG_PCISTS_RTA               BIT12 ///< Received Target Abort Status
#define B_SC_EHCI_CFG_PCISTS_STA               BIT11 ///< Signaled Target-Abort Status
#define B_SC_EHCI_CFG_PCISTS_DEV_STS           (BIT10 | BIT9) ///< DEVSEL# Timing Status
#define B_SC_EHCI_CFG_PCISTS_DPED              BIT8  ///< Master Data Parity Error Detected
#define B_SC_EHCI_CFG_PCISTS_FB2BC             BIT7  ///< Fast Back-to-Back Capable: Reserved as '1'
#define B_SC_EHCI_CFG_PCISTS_UDF               BIT6  ///< User Definable Features: Reserved as '0'
#define B_SC_EHCI_CFG_PCISTS_66MHZ_CAP         BIT5  ///< 66 MHz Capable: Reserved as '0'
#define B_SC_EHCI_CFG_PCISTS_CAP_LST           BIT4  ///< Capabilities List
#define B_SC_EHCI_CFG_PCISTS_INTR_STS          BIT3  ///< Interrupt Status

#define R_SC_EHCI_CFG_RID                      0x08  ///< Revision ID
#define B_SC_EHCI_CFG_RID                      0xFF  ///< Revision ID Mask

#define R_SC_EHCI_CFG_PI                       0x09  ///< Programming Interface
#define B_SC_EHCI_CFG_PI                       0xFF  ///< Programming Interface Mask

#define R_SC_EHCI_CFG_SCC                      0x0A  ///< Sub Class Code
#define B_SC_EHCI_CFG_SCC                      0xFF  ///< Sub Class Code Mask

#define R_SC_EHCI_CFG_BCC                      0x0B  ///< Base Class Code
#define B_SC_EHCI_CFG_BCC                      0xFF  ///< Base Class Code Mask

#define R_SC_EHCI_CFG_MLT                      0x0D  ///< Master Latency Timer
#define B_SC_EHCI_CFG_MLT                      0xFF  ///< Master Latency Timer Mask

#define R_SC_EHCI_CFG_HEADTYPE                 0x0E  ///< Header Type
#define B_SC_EHCI_CFG_HEADTYPE                 0xFF  ///< Header Type Mask
#define B_SC_EHCI_CFG_HEADTYPE_MFB             BIT7  ///< Multi-Function Bit
#define B_SC_EHCI_CFG_HEADTYPE_CL              0x7F  ///< Configuration Layout

#define R_SC_EHCI_CFG_MEM_BASE                 0x10  ///< Memory Base Address
#define B_SC_EHCI_CFG_MEM_BASE_BAR             0xFFFFFC00 ///< Base Address
#define V_SC_EHCI_CFG_MEM_LENGTH               0x400 ///< Memory Space
#define N_SC_EHCI_CFG_MEM_ALIGN                10    ///< Memory Space Alignment
#define B_SC_EHCI_CFG_MEM_BASE_PREF            BIT3  ///< Prefetchable
#define B_SC_EHCI_CFG_MEM_BASE_TYPE            (BIT2 | BIT1) ///< Type
#define B_SC_EHCI_CFG_MEM_BASE_RTE             BIT0  ///< Resource Type Indicator

#define R_SC_EHCI_CFG_SVID                     0x2C  ///< USB2 Subsystem Vendor ID
#define B_SC_EHCI_CFG_SVID                     0xFFFF ///< USB2 Subsystem Vendor ID Mask

#define R_SC_EHCI_CFG_SID                      0x2E  ///< USB2 Subsystem ID
#define B_SC_EHCI_CFG_SID                      0xFFFF ///< USB2 Subsystem ID Mask

#define R_SC_EHCI_CFG_CAP_PTR                  0x34  ///< Capabilities Pointer
#define B_SC_EHCI_CFG_CAP_PTR                  0xFF  ///< Capabilities Pointer Mask

#define R_SC_EHCI_CFG_INT_LN                   0x3C  ///< Interrupt Line
#define B_SC_EHCI_CFG_INT_LN                   0xFF  ///< Interrupt Line Mask

#define R_SC_EHCI_CFG_INT_PN                   0x3D  ///< Interrupt Pin
#define B_SC_EHCI_CFG_INT_PN                   0xFF  ///< Interrupt Pin Mask

#define R_SC_EHCI_CFG_IHFCLK                   0x44  ///< Intel-Specific High Precision Frame Clock
#define B_SC_EHCI_CFG_IHFCLK_MFI               0x3FFF0000 ///< Frame List Current Index / Frame Number
#define B_SC_EHCI_CFG_IHFCLK_MFB               0x1FFF ///< Micro-frame Blif (MFB)

#define R_SC_EHCI_CFG_IHFCLKC                  0x48  ///< Intel-Specific High Precision Frame Clock Capture
#define B_SC_EHCI_CFG_IHFCLKC_CMFI             0x3FFF0000 ///< Captured Frame List Current Index / Frame Number
#define B_SC_EHCI_CFG_IHFCLKC_CMFB             0x1FFF ///< Captured Micro-frame Blif (MFB)

#define R_SC_EHCI_CFG_PWR_CAPID                0x50  ///< PCI Power management Capability ID
#define B_SC_EHCI_CFG_PWR_CAPID                0xFF  ///< PCI Power management Capability ID Mask

#define R_SC_EHCI_CFG_NXT_PTR1                 0x51  ///< Next Item Pointer #1
#define B_SC_EHCI_CFG_NXT_PTR1                 0xFF  ///< Next Item Pointer #1 Mask

#define R_SC_EHCI_CFG_PWR_CAP                  0x52  ///< Power Management Capabilities
#define B_SC_EHCI_CFG_PWR_CAP_PME_SUP          0xF800  ///< PME Support
#define B_SC_EHCI_CFG_PWR_CAP_D2_SUP           BIT10 ///< D2 State Support: not supported
#define B_SC_EHCI_CFG_PWR_CAP_D1_SUP           BIT9  ///< D1 State Support: not supported
#define B_SC_EHCI_CFG_PWR_CAP_AUX_CUR          (BIT8 | BIT7 | BIT6) ///< AUX Current
#define B_SC_EHCI_CFG_PWR_CAP_DSI              BIT5  ///< DSI: Reports 0
#define B_SC_EHCI_CFG_PWR_CAP_PME_CLK          BIT3  ///< PME Clock: Reports 0
#define B_SC_EHCI_CFG_PWR_CAP_VER              (BIT2 | BIT1 | BIT0) ///< Version: Reports "010"

#define R_SC_EHCI_CFG_PWR_CNTL_STS             0x54  ///< Power Management Control / Status
#define B_SC_EHCI_CFG_PWR_CNTL_STS_PME_STS     BIT15 ///< PME Status
#define B_SC_EHCI_CFG_PWR_CNTL_STS_DATASCL     (BIT14 | BIT13) ///< Data Scale
#define B_SC_EHCI_CFG_PWR_CNTL_STS_DATASEL     (BIT12 | BIT11 | BIT10 | BIT9) ///< Data Select
#define B_SC_EHCI_CFG_PWR_CNTL_STS_PME_EN      BIT8  ///< Power Enable
#define B_SC_EHCI_CFG_PWR_CNTL_STS_PWR_STS     (BIT1 | BIT0) ///< Power State
#define V_SC_EHCI_CFG_PWR_CNTL_STS_PWR_STS_D0  0     ///< D0 State
#define V_SC_EHCI_CFG_PWR_CNTL_STS_PWR_STS_D3  (BIT1 | BIT0) ///< D3 Hot State

#define R_SC_EHCI_CFG_DBG_CAPID                0x58  ///< Debug Port Capability ID
#define B_SC_EHCI_CFG_DBG_CAPID                0xFF  ///< Debug Port Capability ID Mask

#define R_SC_EHCI_CFG_NXT_PTR2                 0x59  ///< Next Item Pointer #2
#define B_SC_EHCI_CFG_NXT_PTR2                 0xFF  ///< Next Item Pointer #2 Mask

#define R_SC_EHCI_CFG_DBG_BASE                 0x5A  ///< Debug Port Base Offset
#define B_SC_EHCI_CFG_DBG_BASE_BAR_NUM         0xE000 ///< BAR Number
#define B_SC_EHCI_CFG_DBG_BASE_PORT_OFFSET     0x1FFF ///< Debug Port Offset

#define R_SC_EHCI_CFG_USB_RELNUM               0x60  ///< Serial Bus Release Number
#define B_SC_EHCI_CFG_USB_RELNUM               0xFF  ///< Serial Bus Release Number Mask

#define R_SC_EHCI_CFG_FL_ADJ                   0x61  ///< Frame Length Adjustment
#define B_SC_EHCI_CFG_FL_ADJ                   0x3F  ///< Frame Length Timing Value

#define R_SC_EHCI_CFG_PWAKE_CAP                0x62  ///< Port Wake Capability
#define B_SC_EHCI_CFG_PWAKE_CAP_MASK           0x7FE ///< Port Wake Up Capability Mask
#define B_SC_EHCI_CFG_PWAKE_CAP_PWK_IMP        BIT0  ///< Port Wake Implemented

#define R_SC_EHCI_CFG_PDO                      0x64  ///< Port Disable Override
#define B_SC_EHCI_CFG_PDO_DIS_PORT7            BIT7  ///< USB Port 7 disable
#define B_SC_EHCI_CFG_PDO_DIS_PORT6            BIT6  ///< USB Port 6 disable
#define B_SC_EHCI_CFG_PDO_DIS_PORT5            BIT5  ///< USB Port 5 disable
#define B_SC_EHCI_CFG_PDO_DIS_PORT4            BIT4  ///< USB Port 4 disable
#define B_SC_EHCI_CFG_PDO_DIS_PORT3            BIT3  ///< USB Port 3 disable
#define B_SC_EHCI_CFG_PDO_DIS_PORT2            BIT2  ///< USB Port 2 disable
#define B_SC_EHCI_CFG_PDO_DIS_PORT1            BIT1  ///< USB Port 1 disable
#define B_SC_EHCI_CFG_PDO_DIS_PORT0            BIT0  ///< USB Port 0 disable

#define R_SC_EHCI_CFG_RMHDEVR                  0x66  ///< RMH Device Removable Field
#define B_SC_EHCI_CFG_RMHDEVR_DRBM             0x1FE ///< Device Removable Bit Map

#define R_SC_EHCI_CFG_LEGEXT_CAP               0x68  ///< USB2 Legacy Support Extended Capability
#define B_SC_EHCI_CFG_LEGEXT_CAP_HCOS          BIT24 ///< HC OS Owned Semaphore
#define B_SC_EHCI_CFG_LEGEXT_CAP_HCBIOS        BIT16 ///< HC BIOS Owned Semaphore
#define B_SC_EHCI_CFG_LEGEXT_CAP_NEXT          0x0000FF00 ///< Next EHCI Capability Pointer
#define B_SC_EHCI_CFG_LEGEXT_CAP_CAPID         0x000000FF ///< Capability ID

#define R_SC_EHCI_CFG_LEGEXT_CS                0x6C  ///< USB2 Legacy Support Control / Status
#define B_SC_EHCI_CFG_LEGEXT_CS_SMIBAR         BIT31 ///< SMI on BAR
#define B_SC_EHCI_CFG_LEGEXT_CS_SMIPCI         BIT30 ///< SMI on PCI Command
#define B_SC_EHCI_CFG_LEGEXT_CS_SMIOS          BIT29 ///< SMI on OS Ownership Change
#define B_SC_EHCI_CFG_LEGEXT_CS_SMIAA          BIT21 ///< SMI on Async Advance
#define B_SC_EHCI_CFG_LEGEXT_CS_SMIHSE         BIT20 ///< SMI on Host System Error
#define B_SC_EHCI_CFG_LEGEXT_CS_SMIFLR         BIT19 ///< SMI on Frame List Rollover
#define B_SC_EHCI_CFG_LEGEXT_CS_SMIPCD         BIT18 ///< SMI on Port Change Detect
#define B_SC_EHCI_CFG_LEGEXT_CS_SMIERR         BIT17 ///< SMI on USB Error
#define B_SC_EHCI_CFG_LEGEXT_CS_SMICOMP        BIT16 ///< SMI on USB Complete
#define B_SC_EHCI_CFG_LEGEXT_CS_SMIBAR_EN      BIT15 ///< SMI on BAR Enable
#define B_SC_EHCI_CFG_LEGEXT_CS_SMIPCI_EN      BIT14 ///< SMI on PCI Command Enable
#define B_SC_EHCI_CFG_LEGEXT_CS_SMIOS_EN       BIT13 ///< SMI on OS Ownership Enable
#define B_SC_EHCI_CFG_LEGEXT_CS_SMIAA_EN       BIT5  ///< SMI on Async Advance Enable
#define B_SC_EHCI_CFG_LEGEXT_CS_SMIHSE_EN      BIT4  ///< SMI on Host System Error Enable
#define B_SC_EHCI_CFG_LEGEXT_CS_SMIFLR_EN      BIT3  ///< SMI on Frame List Rollover Enable
#define B_SC_EHCI_CFG_LEGEXT_CS_SMIPCD_EN      BIT2  ///< SMI on Port Change Enable
#define B_SC_EHCI_CFG_LEGEXT_CS_SMIERR_EN      BIT1  ///< SMI on USB Error Enable
#define B_SC_EHCI_CFG_LEGEXT_CS_SMICOMP_EN     BIT0  ///< SMI on USB Complete Enable

#define R_SC_EHCI_CFG_SPCSMI                   0x70  ///< Intel-Specific USB2 SMI
#define B_SC_EHCI_CFG_SPCSMI_SMIPO             0xFFC00000 ///< SMI on Port Owner
#define B_SC_EHCI_CFG_SPCSMI_PMCSR             BIT21 ///< SMI on PMCSR
#define B_SC_EHCI_CFG_SPCSMI_ASYNC             BIT20 ///< SMI on Async
#define B_SC_EHCI_CFG_SPCSMI_PERIODIC          BIT19 ///< SMI on Periodic
#define B_SC_EHCI_CFG_SPCSMI_CF                BIT18 ///< SMI on CF
#define B_SC_EHCI_CFG_SPCSMI_HCHALT            BIT17 ///< SMI on HCHalted
#define B_SC_EHCI_CFG_SPCSMI_HCRESET           BIT16 ///< SMI on HCReset
#define B_SC_EHCI_CFG_SPCSMI_PO_EN             0x0000FFC0 ///< SMI on PortOwner Enable
#define B_SC_EHCI_CFG_SPCSMI_PMCSR_EN          BIT5  ///< SMI on PMSCR Enable
#define B_SC_EHCI_CFG_SPCSMI_ASYNC_EN          BIT4  ///< SMI on Async Enable
#define B_SC_EHCI_CFG_SPCSMI_PERIODIC_EN       BIT3  ///< SMI on Periodic Enable
#define B_SC_EHCI_CFG_SPCSMI_CF_EN             BIT2  ///< SMI on CF Enable
#define B_SC_EHCI_CFG_SPCSMI_HCHALT_EN         BIT1  ///< SMI on HCHalted Enable
#define B_SC_EHCI_CFG_SPCSMI_HCRESET_EN        BIT0  ///< SMI on HCReset Enable

#define R_SC_EHCI_CFG_OCMAP                    0x74  ///< Over-Current Mapping
#define B_SC_EHCI_CFG_OCMAP_D                  0xFF000000 ///< OC Mapping D
#define B_SC_EHCI_CFG_OCMAP_C                  0x00FF0000 ///< OC Mapping C
#define B_SC_EHCI_CFG_OCMAP_B                  0x0000FF00 ///< OC Mapping B
#define B_SC_EHCI_CFG_OCMAP_A                  0x000000FF ///< OC Mapping A

#define R_SC_EHCI_CFG_AFEMCTLTM                0x78  ///< AFE Misc. Control & Test Mode
#define B_SC_EHCI_CFG_AFEMCTLTM_MOCD           BIT12 ///< Mask Over-current Detection
#define B_SC_EHCI_CFG_AFEMCTLTM_UC480GE        BIT9  ///< USB Clock 480 Gate Enable
#define B_SC_EHCI_CFG_AFEMCTLTM_BTATS          (BIT8 | BIT7 | BIT6 | BIT5 | BIT4) ///< Bus Turn Around Time Select
#define B_SC_EHCI_CFG_AFEMCTLTM_HBPSE          BIT3  ///< HS BIAS_EN Port Staggering Enable
#define B_SC_EHCI_CFG_AFEMCTLTM_FLDPSE         BIT2  ///< FS / LS DRIVE_EN Port Staggering Enable
#define B_SC_EHCI_CFG_AFEMCTLTM_UACGEODP       BIT1  ///< USB AFE Clock Gating Enable on Disconnected Ports
#define B_SC_EHCI_CFG_AFEMCTLTM_UACGEOSP       BIT0  ///< USB AFE Clock Gating Enable on Suspended Ports

#define R_SC_EHCI_CFG_ESUBFEN                  0x7A  ///< EHCI Sub-Feature Enable
#define B_SC_EHCI_CFG_ESUBFEN_S0PSEH           (BIT31 | BIT30 | BIT29)
#define B_SC_EHCI_CFG_ESUBFEN_S0PSEN           BIT12 ///< S0 PLL Shutdown Enable
#define B_SC_EHCI_CFG_ESUBFEN_S0PSIN           BIT11 ///< Ignore Noise on S0 PLL Shutdown
#define B_SC_EHCI_CFG_ESUBFEN_S0PSSSS          BIT10 ///< Allow Serialization of S0 PLL Shutdown Flow with Sx Entry
#define B_SC_EHCI_CFG_ESUBFEN_S0PSRWEN         BIT9  ///< S0 PLL Shutdown on Remote Wakeup Enable
#define B_SC_EHCI_CFG_ESUBFEN_EUE              BIT8  ///< EHCI USBR Enable
#define B_SC_EHCI_CFG_ESUBFEN_S0PSCFS          BIT7  ///< S0 PLL Shutdown on Configure Flag State
#define B_SC_EHCI_CFG_ESUBFEN_S0PSD3S          BIT6  ///< S0 PLL Shutdown on D3 State
#define B_SC_EHCI_CFG_ESUBFEN_S0PSRDNCS        BIT5  ///< S0 PLL Shutdown on RMH DS NOT_CONFIGURE State
#define B_SC_EHCI_CFG_ESUBFEN_S0PSRDPOS        BIT4  ///< S0 PLL Shutdown on RMH DS POWER_OFF State
#define B_SC_EHCI_CFG_ESUBFEN_S0PSRDDCS        BIT3  ///< S0 PLL Shutdown on RMH DS DISCONNECTED State
#define B_SC_EHCI_CFG_ESUBFEN_S0PSRDDS         BIT2  ///< S0 PLL Shutdown on RMH DS DISABLED State
#define B_SC_EHCI_CFG_ESUBFEN_S0PSRDSS         BIT1  ///< S0 PLL Shutdown on RMH DS SUSPENDED State
#define B_SC_EHCI_CFG_ESUBFEN_RMH_DIS          BIT0  ///< RMH Disable

#define R_SC_EHCI_CFG_EHCSUSCFG                0x7C  ///< EHC Suspend Well Configuration
#define B_SC_EHCI_CFG_EHCSUSCFG_SLCGE          BIT14 ///< SB Local Clock Gating Enable
#define B_SC_EHCI_CFG_EHCSUSCFG_ISURAD         BIT13 ///< IOSF-SB USB2 Register Access Disable
#define B_SC_EHCI_CFG_EHCSUSCFG_ISPWP          BIT12 ///< IOSF-SB PLL Wake Policy
#define B_SC_EHCI_CFG_EHCSUSCFG_RPIDSF         BIT9  ///< RMH Periodic IN Data SDC Fix
#define B_SC_EHCI_CFG_EHCSUSCFG_RSCRFD         BIT8  ///< RMH Suspend Classic RXEN Fix Disable
#define B_SC_EHCI_CFG_EHCSUSCFG_RCPSE          BIT7  ///< RMH clk240_pp Phase Select Enable
#define B_SC_EHCI_CFG_EHCSUSCFG_RCPS           BIT6  ///< RMH clk240_pp Phase Select
#define B_SC_EHCI_CFG_EHCSUSCFG_HS_COL_FIX     BIT5  ///< HS Collision Fix
#define B_SC_EHCI_CFG_EHCSUSCFG_BSHF           BIT4  ///< Bas Split Handler Fix
#define B_SC_EHCI_CFG_EHCSUSCFG_RPESROED       BIT3  ///< RMH Port Enable / Suspend / Resume on EOF Disable
#define B_SC_EHCI_CFG_EHCSUSCFG_EEDFHEPFD      BIT2  ///< EHCI EOR Duration for HS Enable Ports Fix Disable
#define B_SC_EHCI_CFG_EHCSUSCFG_RSSFR          BIT1  ///< RMH Speed Sample Fix Removal
#define B_SC_EHCI_CFG_EHCSUSCFG_RDCISFR        BIT0  ///< RMH Disconnect / Connect in Sx Fix Removal

#define R_SC_EHCI_CFG_RMHWKCTL                 0x7E  ///< RMH Wake Control
#define B_SC_EHCI_CFG_RMHWKCTL_RIEWCS          BIT8  ///< RMH Inherit EHCI Wake Control Settings
#define B_SC_EHCI_CFG_RMHWKCTL_RUWODR          BIT3  ///< RMH Upstream Wake on Device Resume
#define B_SC_EHCI_CFG_RMHWKCTL_RUWOOD          BIT2  ///< RMH Upstream Wake on OC Disable
#define B_SC_EHCI_CFG_RMHWKCTL_RUWODD          BIT1  ///< RMH Upstream Wake on Disconnect Disable
#define B_SC_EHCI_CFG_RMHWKCTL_RUWOCD          BIT0  ///< RMH Upstream Wake on Connect Disable

#define R_SC_EHCI_CFG_ACCESS_CNTL              0x80  ///< Access Control
#define B_SC_EHCI_CFG_ACCESS_CNTL_VMAC         BIT15 ///< VE MMIO Access Control
#define B_SC_EHCI_CFG_ACCESS_CNTL_RMAC         BIT8  ///< RMH MMIO Access Control
#define B_SC_EHCI_CFG_ACCESS_CNTL_WRT_RDONLY   BIT0  ///< WRT RD Only
#define V_SC_EHCI_CFG_ACCESS_CNTL_WRT_RDONLY_E 0x01  ///< WRT RD Only Enable
#define V_SC_EHCI_CFG_ACCESS_CNTL_WRT_RDONLY_D 0x00  ///< WRT RD Only Disable

#define R_SC_EHCI_CFG_RMHCTL                   0x82  ///< RMH Control
#define B_SC_EHCI_CFG_RMHCTL_ERC               BIT7  ///< Enable RMH Connect
#define B_SC_EHCI_CFG_RMHCTL_ERFR              BIT5  ///< Enable RMH Fast Reset
#define B_SC_EHCI_CFG_RMHCTL_FRDPSS            (BIT4 | BIT3) ///< Fast Reset DS PORT Speed Select
#define B_SC_EHCI_CFG_RMHCTL_DRGS              BIT2  ///< Disable RMH Global Suspend
#define B_SC_EHCI_CFG_RMHCTL_ERFD              BIT1  ///< Enable RMH Force Disconnect
#define B_SC_EHCI_CFG_RMHCTL_EOD               BIT0  ///< Enable Opportunistic Disconnect

#define R_SC_EHCI_CFG_EHCIIR1                  0x84  ///< EHC Configuration 1
#define B_SC_EHCI_CFG_EHCIIR1_DCOQWPRHT1MS     BIT31 ///< Disable Caching of QH with Poll Rate Higher Than 1ms
#define B_SC_EHCI_CFG_EHCIIR1_PCQPD            BIT30 ///< Periodic Cache QTD Prefetch Disable
#define B_SC_EHCI_CFG_EHCIIR1_AFP              BIT28 ///< ADIS Flush Policy
#define B_SC_EHCI_CFG_EHCIIR1_IBSP             BIT27 ///< Invalidate Bit Set Policy
#define B_SC_EHCI_CFG_EHCIIR1_MRP              (BIT26 | BIT25) ///< MFP Retry Policy
#define B_SC_EHCI_CFG_EHCIIR1_APRP             (BIT24 | BIT23) ///< Asynch. Prefetch Retry Policy
#define B_SC_EHCI_CFG_EHCIIR1_CWT              (BIT20 | BIT19) ///< Cx Writeback Threshold [1:0]
#define V_SC_EHCI_CFG_EHCIIR1_CWT_16E          BIT19 ///< 16 Entries
#define V_SC_EHCI_CFG_EHCIIR1_CWT_8E           BIT20 ///< 8 Entries
#define V_SC_EHCI_CFG_EHCIIR1_CWT_4E           (BIT20 | BIT19) ///< 4 Entries
#define B_SC_EHCI_CFG_EHCIIR1_COWD             BIT18 ///< Cx Opportunistic Writeback Disable
#define B_SC_EHCI_CFG_EHCIIR1_PPP              (BIT17 | BIT16) ///< Periodic Prefetch Policy [1:0]
#define B_SC_EHCI_CFG_EHCIIR1                  (BIT15 | BIT14 | BIT13) ///< Periodic Caching Horizon
#define B_SC_EHCI_CFG_EHCIIR1_CP               (BIT12 | BIT11) ///< Caching Policy [1:0]
#define B_SC_EHCI_CFG_EHCIIR1_PSCC             (BIT10 | BIT9) ///< Periodic Schedule Caching Control
#define B_SC_EHCI_CFG_EHCIIR1_DASC             BIT8  ///< Disable Asynchronous Schedule Caching
#define B_SC_EHCI_CFG_EHCIIR1_PEEF             BIT6  ///< Prefetch Engine Error Flag
#define B_SC_EHCI_CFG_EHCIIR1_PBPAP            BIT5  ///< Prefetch Based Pause-Activation Policy
#define B_SC_EHCI_CFG_EHCIIR1_PBPD             BIT4  ///< Prefetch Based Pause-Scoreboard Enable
#define B_SC_EHCI_CFG_EHCIIR1_SOAE             BIT3  ///< SERR on Aborts Enable
#define B_SC_EHCI_CFG_EHCIIR1_GLSCO            (BIT2 | BIT1 | BIT0) ///< Gross Late Start Cut-Off

#define R_SC_EHCI_CFG_EHCIIR2                  0x88  ///< EHC Configuration 2
#define B_SC_EHCI_CFG_EHCIIR2_MCFP             BIT31 ///< MFP Cache Full Policy
#define B_SC_EHCI_CFG_EHCIIR2_RBBAFD           BIT30 ///< RMH Bulk Buffer Arbitration Fix Disable
#define B_SC_EHCI_CFG_EHCIIR2_RTMSFD           BIT29 ///< RMH TT Missing SOF Fix Disable
#define B_SC_EHCI_CFG_EHCIIR2_RBOIEWSL8PFD     BIT28 ///< RMH Bytes on INTR Endpoint with SKU < 8 Ports Fix Disable
#define B_SC_EHCI_CFG_EHCIIR2_RCIRFD           BIT27 ///< RMH CPT Immediate Retry Fix Disable
#define B_SC_EHCI_CFG_EHCIIR2_RIIRFD           BIT26 ///< RMH IBX Immediate Retry Fix Disable
#define B_SC_EHCI_CFG_EHCIIR2_ROCCFR           BIT25 ///< RMH OUT Contention Check Fix Removal
#define B_SC_EHCI_CFG_EHCIIR2_PCDOFD           BIT24 ///< Payload Cache Disable Ordering Fix Disable
#define B_SC_EHCI_CFG_EHCIIR2_AACTPWUIIRS      BIT23 ///< Allow Asynchronous Cache to Proceed When UE is in RETRY State
#define B_SC_EHCI_CFG_EHCIIR2_RCSFR            BIT22 ///< RMH clk240_pp Stop Fix Removal
#define B_SC_EHCI_CFG_EHCIIR2_UIAFFD           BIT21 ///< UE Invalidation Across Frame Fix Disable
#define B_SC_EHCI_CFG_EHCIIR2_EBTIFD           BIT20 ///< UE Binary Tree Invalidation Fix Disable
#define B_SC_EHCI_CFG_EHCIIR2_UCEOSF           BIT19 ///< UE Cache Exit on SRAM Full
#define B_SC_EHCI_CFG_EHCIIR2_RSAFR            BIT18 ///< RMH SOP Arbitration Fix Removal
#define B_SC_EHCI_CFG_EHCIIR2_RCCFR            BIT17 ///< RMH Contention Check Fix Removal
#define B_SC_EHCI_CFG_EHCIIR2_CFRE             BIT16 ///< Cache Fast Reply Enable
#define B_SC_EHCI_CFG_EHCIIR2_CRDO             BIT15 ///< Cache Retry DMA Optimization
#define B_SC_EHCI_CFG_EHCIIR2_MLH              BIT14 ///< MFP Late Handling
#define B_SC_EHCI_CFG_EHCIIR2_SFP              BIT13 ///< SFP Late Handling
#define B_SC_EHCI_CFG_EHCIIR2_PLH              BIT12 ///< Payload Late Handling
#define B_SC_EHCI_CFG_EHCIIR2_HFR              BIT11 ///< HCRESET Flow Removal
#define B_SC_EHCI_CFG_EHCIIR2_FFR              BIT10 ///< Fatal Flow Removal
#define B_SC_EHCI_CFG_EHCIIR2_MTSFF            BIT9  ///< MFC to SFC Flush Flow
#define B_SC_EHCI_CFG_EHCIIR2_PCRE             BIT8  ///< Periodic Cache Retry Enable
#define B_SC_EHCI_CFG_EHCIIR2_ASQPD            BIT7  ///< Async. Single QH Prefetch Disable
#define B_SC_EHCI_CFG_EHCIIR2_SOPPWFTPC        BIT6  ///< Size of Payload Posted Writes From the Payload Cache
#define B_SC_EHCI_CFG_EHCIIR2_DPC              BIT5  ///< Disable Payload Caching
#define B_SC_EHCI_CFG_EHCIIR2_DPPAMB           BIT4  ///< Disable Payload Prefetch Across Microframe Boundaries
#define B_SC_EHCI_CFG_EHCIIR2_PMPD             BIT3  ///< Periodic MicroFrame Pipeline Disable
#define B_SC_EHCI_CFG_EHCIIR2_VCS              BIT2  ///< Virtual Channel Select
#define B_SC_EHCI_CFG_EHCIIR2_DASA             BIT1  ///< Descriptor Access Snoop Attribute
#define B_SC_EHCI_CFG_EHCIIR2_DBASA            BIT0  ///< Data Buffer Access Snoop Attribute

#define R_SC_EHCI_CFG_EHCIIR3                  0x8C  ///< EHC Configuration 3
#define B_SC_EHCI_CFG_EHCIIR3_FMHSHKASMSDC     BIT27
#define B_SC_EHCI_CFG_EHCIIR3_TTPMDFP          BIT26
#define B_SC_EHCI_CFG_EHCIIR3_TTISCOMEPDP      BIT25
#define B_SC_EHCI_CFG_EHCIIR3_TTX4DAFP         BIT24
#define B_SC_EHCI_CFG_EHCIIR3_PDAHEE           BIT23 ///< PBP DMA Active Hysteresis Effect Enable
#define B_SC_EHCI_CFG_EHCIIR3_PDAHV            (BIT22 | BIT21 | BIT20) ///< PBP DMA Active Hysteresis Value
#define B_SC_EHCI_CFG_EHCIIR3_RACFS            (BIT19 | BIT18) ///< Read After Cache Flush Select
#define B_SC_EHCI_CFG_EHCIIR3_UTIROXEFD        BIT17 ///< UH TT Internal Reset on X+4 / EOF1 Fix Disable
#define B_SC_EHCI_CFG_EHCIIR3_TSBBRF           BIT16 ///< TT STALL Bulk Buffer Release Fix
#define B_SC_EHCI_CFG_EHCIIR3_PPWP             (BIT11 | BIT10 | BIT9 | BIT8) ///< PBP Pre-Wake Policy
#define B_SC_EHCI_CFG_EHCIIR3_UHHFD            BIT6  ///< UD HCReset Hung Fix Disable
#define B_SC_EHCI_CFG_EHCIIR3_UEMBEFD          BIT5  ///< UD ECC Multiple Bit Error Fix Disable
#define B_SC_EHCI_CFG_EHCIIR3_MMIFD            BIT4  ///< MFC Missing Invalidation Fix Disable
#define B_SC_EHCI_CFG_EHCIIR3_UWP              BIT2  ///< UD Writeback Policy
#define B_SC_EHCI_CFG_EHCIIR3_UDAP             BIT1  ///< UD DMA Active Policy
#define B_SC_EHCI_CFG_EHCIIR3_PCEPEDP          BIT0  ///< Periodic Cache Enable Periodic Engine Done Policy

#define R_SC_EHCI_CFG_FLR_CID                  0x98  ///< Function Level Reset Capability ID
#define B_SC_EHCI_CFG_FLR_CID                  0xFF  ///< Function Level Reset Capability ID Mask
#define V_SC_EHCI_CFG_FLR_CID_13               0x13
#define V_SC_EHCI_CFG_FLR_CID_09               0x09

#define R_SC_EHCI_CFG_FLR_NEXT                 0x99  ///< FLR Next capability Pointer
#define B_SC_EHCI_CFG_FLR_NEXT                 0xFF  ///< FLR Next capability Pointer Mask

#define R_SC_EHCI_CFG_FLR_CLV                  0x9A  ///< FLR Capability Length and Version
#define B_SC_EHCI_CFG_FLR_CLV_CAP_SSEL0        BIT9  ///< FLR Capability
#define B_SC_EHCI_CFG_FLR_CLV_TXP_SSEL0        BIT8  ///< TXP capability
#define B_SC_EHCI_CFG_FLR_CLV_LNG              0x00FF ///< Capability Length

#define R_SC_EHCI_CFG_FLR_CTRL                 0x9C  ///< FLR Control Register
#define B_SC_EHCI_CFG_FLR_CTRL_INITFLR         BIT0  ///< Initiate FLR

#define R_SC_EHCI_CFG_FLR_STS                  0x9D  ///< FLR Status Register
#define B_SC_EHCI_CFG_FLR_STS_TXP              BIT0  ///< Transactions Pending

#define R_SC_EHCI_CFG_DP_CTRLSTS               0xA0  ///< 1.2.1.21	Debug Port Control/Status Register (DP_CTRLSTS)
#define B_SC_EHCI_CFG_DP_OWNER_CNT             BIT30  ///< OWNER_CNT (OWNERCNT_0)
#define B_SC_EHCI_CFG_DP_ENABLED_CNT           BIT28  ///< ENABLED_CNT (ENABLEDCNT_0)


#define R_SC_EHCI_CFG_STM                      0xD0  ///< Simulation Test Modes
#define B_SC_EHCI_CFG_STM_SHORTFRAME_EN        BIT7  ///< Short Frame Enable
#define B_SC_EHCI_CFG_STM_SHORTFRAME_SEL       (BIT1 | BIT0) ///< Short Frame Select

#define R_SC_EHCI_CFG_LBDATA                   0xD4  ///< Loop-Back Data
#define B_SC_EHCI_CFG_LBDATA_MASK              0x00FFFFFF ///< Three Bytes of the Loop-Back Data

#define R_SC_EHCI_CFG_MISCTM                   0xD8  ///< Misc. Test Mode
#define B_SC_EHCI_CFG_MISCTM_UTC               0xFFFF0000 ///< USB TX CRC
#define B_SC_EHCI_CFG_MISCTM_UHLPS             0x0000E000 ///< USB HS Loopback Port Select
#define B_SC_EHCI_CFG_MISCTM_UTCPS             0x1C00 ///< USB TX CRC Port Select
#define B_SC_EHCI_CFG_MISCTM_DRCPORT0SEL       BIT9  ///< DRC Port0 Select
#define B_SC_EHCI_CFG_MISCTM_DRCBYPASS         BIT8  ///< DRC Bypass
#define B_SC_EHCI_CFG_MISCTM_UHLE              BIT7  ///< USB HS Loopback Enable
#define B_SC_EHCI_CFG_MISCTM_DRB               (BIT6 | BIT5 | BIT4) ///< Discard Rdata Bits
#define B_SC_EHCI_CFG_MISCTM_UTCR              BIT3  ///< USB TX CRC Reset
#define B_SC_EHCI_CFG_MISCTM_FELPBKBE          BIT1  ///< Far end Loopback Burst Enable
#define B_SC_EHCI_CFG_MISCTM_FELPBK            BIT0  ///< Far End Loopback

#define R_SC_EHCI_CFG_CLKGATEEN                0xDC  ///< Clock Gate Enable
#define B_SC_EHCI_CFG_CLKGATEEN_USB2CDCG       BIT5  ///< USB2 Cache Dynamic Clock Gating Enable
#define B_SC_EHCI_CFG_CLKGATEEN_USB2CFCG       BIT4  ///< USB2 Cache Force Clock Gating Enable
#define B_SC_EHCI_CFG_CLKGATEEN_USB2ADCG       BIT2  ///< USB2 Aggressive Dynamic Clock Gating Enable
#define B_SC_EHCI_CFG_CLKGATEEN_USB2DCG        BIT1  ///< USB2 Dynamic Clock Gating Enable
#define B_SC_EHCI_CFG_CLKGATEEN_USB2SCG        BIT0  ///< USB2 Static Clock Gating Enable

#define R_SC_EHCI_CFG_UL                       0xF4  ///< Unit Latency
#define B_SC_EHCI_CFG_UL_LATPGMEN              BIT31 ///< LATPGMEN
#define B_SC_EHCI_CFG_UL_TRBT                  (BIT23 | BIT22) ///< Transmit and Receive Byte Threshold
#define V_SC_EHCI_CFG_UL_TRBT_20               (BIT23 | BIT22) ///< 20 High-Speed Byte Times
#define V_SC_EHCI_CFG_UL_TRBT_18               BIT23  ///< 18 High-Speed Byte Times
#define V_SC_EHCI_CFG_UL_TRBT_16               BIT22  ///< 16 High-Speed Byte Times
#define V_SC_EHCI_CFG_UL_TRBT_12               0      ///< 12 High-Speed Byte Times
#define B_SC_EHCI_CFG_UL_CUISO                 (BIT21 | BIT20) ///< Classic USB Interface Signals
#define B_SC_EHCI_CFG_UL_DOE                   BIT19 ///< Disconnect Offset Enable
#define B_SC_EHCI_CFG_UL_DSO                   (BIT18 | BIT17 | BIT16) ///< Disconnect Sample Offset
#define B_SC_EHCI_CFG_UL_TDO                   (BIT15 | BIT14 | BIT13 | BIT12) ///< Timeout Detect Offset
#define B_SC_EHCI_CFG_UL_IPGA                  (BIT11 | BIT10 | BIT9 | BIT8) ///< Inter Packet Gap Address
#define B_SC_EHCI_CFG_UL_GLSOP                 (BIT7 | BIT6 | BIT5 | BIT4) ///< Granular Late Start Offset (Periodic)
#define B_SC_EHCI_CFG_UL_GLSOA                 (BIT3 | BIT2 | BIT1 | BIT0) ///< Granular Late Start Offset (Asynchronous)

#define R_SC_EHCI_CFG_MANID                    0xF8  ///< Manufacturer's ID
#define B_SC_EHCI_CFG_MAN_ID_DPID              0xF000000 ///< Dot Portion of Process ID
#define B_SC_EHCI_CFG_MAN_ID_MSID              0xFF0000 ///< Manufacturing Stepping Identifier
#define B_SC_EHCI_CFG_MAN_ID_MID               0xFF00 ///< Manufacturing Identifier
#define B_SC_EHCI_CFG_MAN_ID_PPID              0xFF  ///< Process Portion of Process ID

///
/// EHCI MMIO registers
///
#define R_SC_EHCI_MEM_CAPLENGTH                0x00  ///< Capability Registers Length

#define R_SC_EHCI_MEM_HCIVERSION               0x02  ///< Host Controller Interface Version
#define B_SC_EHCI_MEM_HCIVERSION_NUM           0xFF  ///< Host Controller Interface Version Number

#define R_SC_EHCI_MEM_HCSPARAMS                0x04  ///< Host Controller Structural
#define B_SC_EHCI_MEM_HCSPARAMS_DP_N           (BIT23 | BIT22 | BIT21 | BIT20) ///< Debug Port Number
#define N_SC_EHCI_MEM_HCSPARAMS_DP_N           20
#define B_SC_EHCI_MEM_HCSPARAMS_P_INDICATOR    BIT16 ///< Port Indicators
#define B_SC_EHCI_MEM_HCSPARAMS_N_CC           (BIT15 | BIT14 | BIT13 | BIT12) ///< Number of Companion Controllers
#define N_SC_EHCI_MEM_HCSPARAMS_N_CC           12
#define B_SC_EHCI_MEM_HCSPARAMS_N_PCC          (BIT11 | BIT10 | BIT9 | BIT8) ///< Number of Ports per Companion Controller
#define N_SC_EHCI_MEM_HCSPARAMS_N_PCC          8
#define B_SC_EHCI_MEM_HCSPARAMS_PPC            BIT4  ///< Power Port Control
#define B_SC_EHCI_MEM_HCSPARAMS_N_PORTS        (BIT3 | BIT2 | BIT1 | BIT0) ///< Number of Ports
#define N_SC_EHCI_MEM_HCSPARAMS_N_PORTS        0

#define R_SC_EHCI_MEM_USB2CMD                  0x20  ///< USB2 Command Register
#define B_SC_EHCI_MEM_USB2CMD_ITC              0xFF0000 ///< Interrupt Threshold Control
#define B_SC_EHCI_MEM_USB2CMD_ASPE             BIT13 ///< Asynch Schedule Prefetch Enable Bit
#define B_SC_EHCI_MEM_USB2CMD_PSPE             BIT12 ///< Prediodic Schedule Prefetch Enable
#define B_SC_EHCI_MEM_USB2CMD_IOAAD            BIT6  ///< Interrupt on Async. Advance Doorbell
#define B_SC_EHCI_MEM_USB2CMD_ASE              BIT5  ///< Asynchronous Schedule Enable
#define B_SC_EHCI_MEM_USB2CMD_PSE              BIT4  ///< Periodic Schedule Enable
#define B_SC_EHCI_MEM_USB2CMD_FLS              (BIT3 | BIT2) ///< Frame List Size
#define B_SC_EHCI_MEM_USB2CMD_HCRESET          BIT1  ///< Host Controller Reset
#define B_SC_EHCI_MEM_USB2CMD_RS               BIT0  ///< Run / Stop

#define R_SC_EHCI_MEM_CONFIGFLAG               0x60  ///< Configure Flag Register
#define B_SC_EHCI_MEM_CONFIGFLAG               BIT0  ///< Configure Flag Bit

#define R_SC_EHCI_MEM_PORTSC0                  0x64  ///< Port 0 Status and Control
#define B_SC_EHCI_MEM_PORTSC0_SUSPEND          BIT7  ///< Suspend
#define B_SC_EHCI_MEM_PORTSC0_PORT_EN_DIS      BIT2  ///< Port Enable / Disable

///
/// USB3 (XHCI) related definitions
///
#define PCI_DEVICE_NUMBER_XHCI          21
#define PCI_FUNCTION_NUMBER_XHCI        0

///
/// XHCI PCI Config Space registers
///
#define R_SC_XHCI_CFG_VENDOR_ID                0x00  ///< Vendor ID
#define B_SC_XHCI_CFG_VENDOR_ID                0xFFFF

#define R_SC_XHCI_CFG_DEVICE_ID                0x02  ///< Device ID
#define B_SC_XHCI_CFG_DEVICE_ID                0xFFFF

#define R_SC_XHCI_CFG_COMMAND_REGISTER         0x04  ///< Command
#define B_SC_XHCI_CFG_COMMAND_ID               BIT10 ///< Interrupt Disable
#define B_SC_XHCI_CFG_COMMAND_FBE              BIT9  ///< Fast Back to Back Enable
#define B_SC_XHCI_CFG_COMMAND_SERR             BIT8  ///< SERR# Enable
#define B_SC_XHCI_CFG_COMMAND_WCC              BIT7  ///< Wait Cycle Control
#define B_SC_XHCI_CFG_COMMAND_PER              BIT6  ///< Parity Error Response
#define B_SC_XHCI_CFG_COMMAND_VPS              BIT5  ///< VGA Palette Snoop
#define B_SC_XHCI_CFG_COMMAND_MWI              BIT4  ///< Memory Write Invalidate
#define B_SC_XHCI_CFG_COMMAND_SCE              BIT3  ///< Special Cycle Enable
#define B_SC_XHCI_CFG_COMMAND_BME              BIT2  ///< Bus Master Enable
#define B_SC_XHCI_CFG_COMMAND_MSE              BIT1  ///< Memory Space Enable

#define R_SC_XHCI_CFG_MEM_BASE                 0x10  ///< Memory Base Address
#define B_SC_XHCI_CFG_MEM_BASE_BA              0xFFFFFFFFFFFF0000 ///< Base Address
#define V_SC_XHCI_CFG_MEM_LENGTH               0x10000 ///< 64 KB of Memory Length
#define N_SC_XHCI_CFG_MEM_ALIGN                16    ///< Memory Space Alignment
#define B_SC_XHCI_CFG_MEM_BASE_PREF            BIT3  ///< Prefetchable
#define B_SC_XHCI_CFG_MEM_BASE_TYPE            (BIT2 | BIT1) ///< Type
#define B_SC_XHCI_CFG_MEM_BASE_RTE             BIT0  ///< Resource Type Indicator

#define R_SC_XHCI_CFG_SVID                     0x2C
#define B_SC_XHCI_CFG_SVID                     0xFFFF

#define R_SC_XHCI_CFG_SID                      0x2E
#define B_SC_XHCI_CFG_SID                      0xFFFF

#define R_SC_XHCI_CFG_INT_LN                   0x3C  ///< Interrupt Line
#define B_SC_XHCI_CFG_INT_LN                   0xFF  ///< Interrupt Line Mask

#define R_SC_XHCI_CFG_INT_PN                   0x3D  ///< Interrupt Pin
#define B_SC_XHCI_CFG_INT_PN                   0xFF  ///< Interrupt Line Mask
#define V_SC_XHCI_CFG_INT_PN                   BIT0  ///< Interrupt Pin Value (INTA)

#define R_SC_XHCI_CFG_XHCC1                    0x40
#define B_SC_XHCI_CFG_XHCC1_ACCTRL             BIT31
#define B_SC_XHCI_CFG_XHCC1_RMTASERR           BIT24
#define B_SC_XHCI_CFG_XHCC1_URD                BIT23
#define B_SC_XHCI_CFG_XHCC1_URRE               BIT22
#define B_SC_XHCI_CFG_XHCC1_IIL1E              (BIT21 | BIT20 | BIT19)
#define V_SC_XHCI_CFG_XHCC1_IIL1E_DIS          0
#define V_SC_XHCI_CFG_XHCC1_IIL1E_32           (BIT19)
#define V_SC_XHCI_CFG_XHCC1_IIL1E_64           (BIT20)
#define V_SC_XHCI_CFG_XHCC1_IIL1E_128          (BIT20 | BIT19)
#define V_SC_XHCI_CFG_XHCC1_IIL1E_256          (BIT21)
#define V_SC_XHCI_CFG_XHCC1_IIL1E_512          (BIT21 | BIT19)
#define V_SC_XHCI_CFG_XHCC1_IIL1E_1024         (BIT21 | BIT20)
#define V_SC_XHCI_CFG_XHCC1_IIL1E_131072       (BIT21 | BIT20 | BIT19)
#define B_SC_XHCI_CFG_XHCC1_XHCIL1E            BIT18 ///< XHC Initiated L1 Enable
#define B_SC_XHCI_CFG_XHCC1_D3IL1E             BIT17 ///< D3 Initiated L1 Enable
#define B_SC_XHCI_CFG_XHCC1_UNPPA              (BIT16 | BIT15 | BIT14 | BIT13 | BIT12) ///< Periodic Complete Pre Wake Time
#define B_SC_XHCI_CFG_XHCC1_SWAXHCI            BIT11 ///< SW Assisted xHC Idle
#define B_SC_XHCI_CFG_XHCC1_L23HRAWC           (BIT10 | BIT9 | BIT8) ///< L23 to Host Reset Acknowledge Wait Count
#define V_SC_XHCI_CFG_XHCC1_L23HRAWC_DIS       0
#define V_SC_XHCI_CFG_XHCC1_L23HRAWC_128       (BIT8)
#define V_SC_XHCI_CFG_XHCC1_L23HRAWC_256       (BIT9)
#define V_SC_XHCI_CFG_XHCC1_L23HRAWC_512       (BIT9 | BIT8)
#define V_SC_XHCI_CFG_XHCC1_L23HRAWC_1024      (BIT10)
#define V_SC_XHCI_CFG_XHCC1_L23HRAWC_2048      (BIT10 | BIT8)
#define V_SC_XHCI_CFG_XHCC1_L23HRAWC_4096      (BIT10 | BIT9)
#define V_SC_XHCI_CFG_XHCC1_L23HRAWC_131072    (BIT10 | BIT9 | BIT8)
#define B_SC_XHCI_CFG_XHCC1_UTAGCP             (BIT7 | BIT6) ///< Upstream Type Arbiter Grant Count Posted
#define B_SC_XHCI_CFG_XHCC1_UDAGCNP            (BIT5 | BIT4) ///< Upstream Type Arbiter Grant Count Non Posted
#define B_SC_XHCI_CFG_XHCC1_UDAGCCP            (BIT3 | BIT2) ///< Upstream Type Arbiter Grant Count Completion
#define B_SC_XHCI_CFG_XHCC1_UDAGC              (BIT1 | BIT0) ///< Upstream Type Arbiter Grant Count

#define R_SC_XHCI_CFG_XHCC2                    0x44  ///< XHC System Bus Configuration 2
#define B_SC_XHCI_CFG_XHCC2_OCCFDONE           BIT31 ///< OC Configuration Done
#define B_SC_XHCI_CFG_XHCC2_DREQBCC            BIT25 ///< DMA Request Boundary Crossing Control
#define B_SC_XHCI_CFG_XHCC2_IDMARRSC           (BIT24 | BIT23 | BIT22) ///< IDMA Read Request Size Control
#define B_SC_XHCI_CFG_XHCC2_XHCUPRDROE         BIT21 ///< XHC Upstream Read Relaxed Ordering Enable
#define B_SC_XHCI_CFG_XHCC2_IOSFSRAD           BIT20 ///< IOSF Sideband Register Access Disable
#define B_SC_XHCI_CFG_XHCC2_UNPPA              0xFC000 ///< Upstream Non-Posted Pre-Allocation
#define B_SC_XHCI_CFG_XHCC2_SWAXHCIP           (BIT13 | BIT12) ///< SW Assisted xHC Idle Policy
#define B_SC_XHCI_CFG_XHCC2_RAWDD              BIT11 ///< MMIO Read After MMIO Write Delay Disable
#define B_SC_XHCI_CFG_XHCC2_WAWDE              BIT10 ///< MMIO Write After MMIO Write Delay Enable
#define B_SC_XHCI_CFG_XHCC2_SWACXIHB           (BIT9 | BIT8) ///< SW Assisted Cx Inhibit
#define B_SC_XHCI_CFG_XHCC2_SWADMIL1IHB        (BIT7 | BIT6) ///< SW Assisted DMI L1 Inhibit
#define B_SC_XHCI_CFG_XHCC2_L1FP2CGWC          (BIT5 | BIT4 | BIT3) ///< L1 Force P2 clock Gating Wait Count
#define V_SC_XHCI_CFG_XHCC2_L1FP2CGWC_DIS      0
#define V_SC_XHCI_CFG_XHCC2_L1FP2CGWC_128      (BIT3)
#define V_SC_XHCI_CFG_XHCC2_L1FP2CGWC_256      (BIT4)
#define V_SC_XHCI_CFG_XHCC2_L1FP2CGWC_512      (BIT4 | BIT3)
#define V_SC_XHCI_CFG_XHCC2_L1FP2CGWC_1024     (BIT5)
#define V_SC_XHCI_CFG_XHCC2_L1FP2CGWC_2048     (BIT5 | BIT3)
#define V_SC_XHCI_CFG_XHCC2_L1FP2CGWC_4096     (BIT5 | BIT4)
#define V_SC_XHCI_CFG_XHCC2_L1FP2CGWC_131072   (BIT5 | BIT4 | BIT3)
#define B_SC_XHCI_CFG_XHCC2_RDREQSZCTRL        (BIT2 | BIT1 | BIT0) ///< Read Request Size Control
#define V_SC_XHCI_CFG_XHCC2_RDREQSZCTRL_128    0
#define V_SC_XHCI_CFG_XHCC2_RDREQSZCTRL_256    (BIT0)
#define V_SC_XHCI_CFG_XHCC2_RDREQSZCTRL_512    (BIT1)
#define V_SC_XHCI_CFG_XHCC2_RDREQSZCTRL_64     (BIT2 | BIT1 | BIT0)

#define R_SC_XHCI_CFG_XHCLKGTEN                0x50  ///< Clock Gating
#define B_SC_XHCI_CFG_XHCLKGTEN_NUEFBCGPS      BIT28 ///< Naking USB2.0 EPs for Backbone Clock Gating and PLL Shutdown
#define B_SC_XHCI_CFG_XHCLKGTEN_SRAMPGTEN      BIT27 ///< SRAM Power Gate Enable
#define B_SC_XHCI_CFG_XHCLKGTEN_SSLSE          BIT26 ///< SS Link PLL Shutdown Enable
#define B_SC_XHCI_CFG_XHCLKGTEN_USB2PLLSE      BIT25 ///< USB2 PLL Shutdown Enable
#define B_SC_XHCI_CFG_XHCLKGTEN_IOSFSTCGE      BIT24 ///< IOSF Sideband Trunk Clock Gating Enable
#define B_SC_XHCI_CFG_XHCLKGTEN_HSTCGE         (BIT23 | BIT22 | BIT21 | BIT20) ///< HS Backbone PXP Trunk Clock Gate Enable
#define N_SC_XHCI_CFG_XHCLKGTEN_HSTCGE         20                                                                        
#define B_SC_XHCI_CFG_XHCLKGTEN_SSTCGE         (BIT19 | BIT18 | BIT17 | BIT16) ///< SS Backbone PXP Trunk Clock Gate Enable
#define N_SC_XHCI_CFG_XHCLKGTEN_SSTCGE         16
#define B_SC_XHCI_CFG_XHCLKGTEN_XHCIGEU3S      BIT15 ///< XHC Ignore_EU3S
#define B_SC_XHCI_CFG_XHCLKGTEN_XHCFTCLKSE     BIT14 ///< XHC Frame Timer Clock Shutdown Enable
#define B_SC_XHCI_CFG_XHCLKGTEN_XHCBBTCGIPISO  BIT13 ///< XHC Backbone PXP Trunk Clock Gate In Presence of ISOCH EP
#define B_SC_XHCI_CFG_XHCLKGTEN_XHCHSTCGU2NRWE BIT12 ///< XHC HS Backbone PXP Trunk Clock Gate U2 non RWE
#define B_SC_XHCI_CFG_XHCLKGTEN_XHCUSB2PLLSDLE (BIT11 | BIT10) ///< XHC USB2 PLL Shutdown Lx Enable
#define B_SC_XHCI_CFG_XHCLKGTEN_HSUXDMIPLLSE   (BIT9 | BIT8) ///< HS Backbone PXP PLL Shutdown Ux Enable
#define N_SC_XHCI_CFG_XHCLKGTEN_HSUXDMIPLLSE   8
#define B_SC_XHCI_CFG_XHCLKGTEN_SSPLLSUE       (BIT7 | BIT6 | BIT5) ///< SS backbone PXP PLL Shutdown Ux Enable
#define N_SC_XHCI_CFG_XHCLKGTEN_SSPLLSUE       5
#define B_SC_XHCI_CFG_XHCLKGTEN_XHCBLCGE       BIT4  ///< XHC Backbone Local Clock Gating Enable
#define B_SC_XHCI_CFG_XHCLKGTEN_HSLTCGE        BIT3  ///< HS Link Trunk Clock Gating Enable
#define B_SC_XHCI_CFG_XHCLKGTEN_SSLTCGE        BIT2  ///< SS Link Trunk Clock Gating Enable
#define B_SC_XHCI_CFG_XHCLKGTEN_IOSFBTCGE      BIT1  ///< IOSF Backbone Trunk Clock Gating Enable
#define B_SC_XHCI_CFG_XHCLKGTEN_IOSFGBLCGE     BIT0  ///< IOSF Gasket Backbone Local Clock Gating Enable

#define R_SC_XHCI_CFG_USB_RELNUM               0x60
#define B_SC_XHCI_CFG_USB_RELNUM               0xFF

#define R_SC_XHCI_CFG_FL_ADJ                   0x61
#define B_SC_XHCI_CFG_FL_ADJ                   0x3F

#define R_SC_XHCI_CFG_PWR_CAPID                0x70
#define B_SC_XHCI_CFG_PWR_CAPID                0xFF

#define R_SC_XHCI_CFG_NXT_PTR1                 0x71
#define B_SC_XHCI_CFG_NXT_PTR1                 0xFF

#define R_SC_XHCI_CFG_PWR_CAP                  0x72
#define B_SC_XHCI_CFG_PWR_CAP_PME_SUP          0xF800
#define B_SC_XHCI_CFG_PWR_CAP_D2_SUP           BIT10
#define B_SC_XHCI_CFG_PWR_CAP_D1_SUP           BIT9
#define B_SC_XHCI_CFG_PWR_CAP_AUX_CUR          (BIT8 | BIT7 | BIT6)
#define B_SC_XHCI_CFG_PWR_CAP_DSI              BIT5
#define B_SC_XHCI_CFG_PWR_CAP_PME_CLK          BIT3
#define B_SC_XHCI_CFG_PWR_CAP_VER              (BIT2 | BIT1 | BIT0)

#define R_SC_XHCI_CFG_PWR_CNTL_STS             0x74
#define B_SC_XHCI_CFG_PWR_CNTL_STS_PME_STS     BIT15
#define B_SC_XHCI_CFG_PWR_CNTL_STS_DATASCL     (BIT14 | BIT13)
#define B_SC_XHCI_CFG_PWR_CNTL_STS_DATASEL     (BIT12 | BIT11 | BIT10 | BIT9)
#define B_SC_XHCI_CFG_PWR_CNTL_STS_PME_EN      BIT8
#define B_SC_XHCI_CFG_PWR_CNTL_STS_PWR_STS     (BIT1 | BIT0)
#define V_SC_XHCI_CFG_PWR_CNTL_STS_PWR_STS_D3  (BIT1 | BIT0)

#define R_SC_XHCI_CFG_MSI_CAPID                0x80
#define B_SC_XHCI_CFG_MSI_CAPID                0xFF

#define R_SC_XHCI_CFG_NXT_PTR2                 0x81
#define B_SC_XHCI_CFG_NXT_PTR2                 0xFF

#define R_SC_XHCI_CFG_MSI_MCTL                 0x82
#define B_SC_XHCI_CFG_MSI_MCTL_MSIENABLE       BIT0

#define V_SC_XHCI_CFG_MSI_NEXT                 0x90  ///< Pointer to DevIdle capability structure

#define R_SC_XHCI_CFG_PCE                      0xA2  ///< Power control enable
#define B_SC_XHCI_CFG_PCE_HAE                  BIT5///<HAE: Hardware Autonomous Enable
#define B_SC_XHCI_CFG_PCE_SE                   BIT3///<SE: Sleep Enable
#define N_SC_XHCI_CFG_PCE_SE                   3                                            
#define B_SC_XHCI_CFG_PCE_D3HE                 BIT2///<D3HE: D3-Hot Enable:
#define B_SC_XHCI_CFG_PCE_I3E                  BIT1///<I3E: I3 Enable
#define B_SC_XHCI_CFG_PCE_SPE                  BIT0 ///<SPE: Software PowerGate Enable

#define R_SC_XHCI_CFG_HSCFG1                   0xAC  ///<High Speed Configuration 1
#define B_SC_XHCI_CFG_HSCFG1_UTMI_SUSPEND_CG   BIT19 ///<UTMI Gasket Local Suspended Port 480/60/48MHz Clock Gating Enable
#define B_SC_XHCI_CFG_HSCFG1_UTMI_SPEED_CG     BIT18 ///<UTMI Gasket Local Speed Port 480/60/48MHz Clock Gating Enable

#define R_SC_XHCI_CFG_HSCFG2                   0xA4  ///<High Speed Configuration 2
#define B_SC_XHCI_CFG_HSCFG2_HSAAIM            BIT15 ///<HS ASYNC Active IN Mask (HSAAIM):
#define B_SC_XHCI_CFG_HSCFG2_HSOAAPEPM         BIT14 ///<HS OUT ASYNC Active Polling EP Mask (HSOAAPEPM):
#define B_SC_XHCI_CFG_HSCFG2_HSIAAPEPM         BIT13 ///<HS IN ASYNC Active Polling EP Mask (HSIAAPEPM):
#define B_SC_XHCI_CFG_HSCFG2_HSIIPAPC          (BIT12|BIT11) ///<HS INTR IN Periodic Active Policy Control (HSIIPAPC):
#define B_SC_XHCI_CFG_HSCFG2_HSIIPANEPT        (BIT10 | BIT9 | BIT8 | BIT7 | BIT6 | BIT5 | BIT4)  ///< HS INTR IN Periodic Active Num of EP Threshold(HSIIPANEPT)
#define B_SC_XHCI_CFG_HSCFG2_HSIIPASIT         (BIT3 | BIT2 | BIT1 | BIT0)                        ///< HS INTR IN Periodic Active Service Interval Threshold (HSIIPASIT)

#define R_SC_XHCI_CFG_SSCFG1                   0xA8  ///<High Speed Configuration 1
#define B_SC_XHCI_CFG_SSCFG1_LFPS              BIT17 ///<LFPS Power Management Enable
#define B_SC_XHCI_CFG_SSCFG1_PHY_U3            BIT14 ///<USB3 PHY Power Gate Enable for U2

#define R_SC_XHCI_CFG_U2OCM                    0xB0
#define R_SC_XHCI_CFG_U3OCM                    0xD0
#define V_SC_XHCI_CFG_NUMBER_OF_OC_PINS        2


//
// Extended Capability Registers
//
#define R_SC_XHCI_CFG_USB2PDO                  0x84F8
#define B_SC_XHCI_CFG_BXT_USB2PDO_MASK         0x07
#define B_SC_XHCI_CFG_BXTP_USB2PDO_MASK        0xFF
#define B_SC_XHCI_CFG_USB2PDO_DIS_PORT0        BIT0

#define R_SC_XHCI_CFG_USB3PDO                  0x84FC
#define B_SC_XHCI_CFG_BXT_USB3PDO_MASK         0x03
#define B_SC_XHCI_CFG_BXTP_USB3PDO_MASK        0x3F
#define B_SC_XHCI_CFG_USB3PDO_DIS_PORT0        BIT0

///
/// xHCI MMIO registers
///

#define R_SC_XHCI_MEM_DUAL_ROLE_CFG0       0x80D8
#define B_SC_XHCI_MEM_EN_PIPE4_SYNC            BIT30 ///< enable/disable PIPE 4.1 synchronous phystatus
#define R_SC_XHCI_MEM_DUAL_ROLE_CFG1       0x80DC

///
/// 0x00 - 0x1F - Capability Registers
///
#define R_SC_XHCI_MEM_CAPLENGTH                0x00  ///< Capability Registers Length

#define R_SC_XHCI_MEM_HCIVERSION               0x02  ///< Host Controller Interface Version Number

#define R_SC_XHCI_MEM_HCSPARAMS1               0x04  ///< Structural Parameters 1
#define B_SC_XHCI_MEM_HCSPARAMS1_MAXPORTS      0xFF000000 ///< Number of Ports
#define B_SC_XHCI_MEM_HCSPARAMS1_MAXINTRS      0x7FF00 ///< Number of Interrupters
#define B_SC_XHCI_MEM_HCSPARAMS1_MAXSLOTS      0xFF  ///< Number of Device Slots

#define R_SC_XHCI_MEM_HCSPARAMS2               0x08  ///< Structural Parameters 2
#define B_SC_XHCI_MEM_HCSPARAMS2_MSB           0xF8000000 ///< Max Scratchpad Buffers
#define B_SC_XHCI_MEM_HCSPARAMS2_ERSTMAX       0xF0  ///< Event Ring Segment Table Max
#define B_SC_XHCI_MEM_HCSPARAMS2_IST           0x0F  ///< Isochronous Scheduling Threshold

#define R_SC_XHCI_MEM_HCSPARAMS3               0x0C  ///< Structural Parameters 3
#define B_SC_XHCI_MEM_HCSPARAMS3_U2DEL         0xFFFF0000 ///< U2 Device Exit Latency
#define B_SC_XHCI_MEM_HCSPARAMS3_U1DEL         0x000000FF ///< U1 Device Exit Latency

#define R_SC_XHCI_MEM_HCCPARAMS                0x10  ///< Capability Parameters
#define B_SC_XHCI_MEM_HCCPARAMS_XECP           0xFFFF0000 ///< xHCI Extended Capabilities Pointer
#define B_SC_XHCI_MEM_HCCPARAMS_MAXPSASIZE     (BIT15 | BIT14 | BIT13 | BIT12) ///< Maximum Primary Stream Array Size
#define B_SC_XHCI_MEM_HCCPARAMS_CFC            BIT11  ///< Contiguous Frame ID Capability
#define B_SC_XHCI_MEM_HCCPARAMS_PAE            BIT8  ///< Parst All Event Data
#define B_SC_XHCI_MEM_HCCPARAMS_NSS            BIT7  ///< No Secondary SID Support
#define B_SC_XHCI_MEM_HCCPARAMS_LTC            BIT6  ///< Latency Tolerance Messaging Capability
#define B_SC_XHCI_MEM_HCCPARAMS_LHRC           BIT5  ///< Light HC Reset Capability
#define B_SC_XHCI_MEM_HCCPARAMS_PIND           BIT4  ///< Port Indicators
#define B_SC_XHCI_MEM_HCCPARAMS_PPC            BIT3  ///< Port Power Control
#define B_SC_XHCI_MEM_HCCPARAMS_CSZ            BIT2  ///< Context Size
#define B_SC_XHCI_MEM_HCCPARAMS_BNC            BIT1  ///< BW Negotiation Capability
#define B_SC_XHCI_MEM_HCCPARAMS_AC64           BIT0  ///< 64-bit Addressing Capability

#define R_SC_XHCI_MEM_DBOFF                    0x14  ///< Doorbell Offset
#define B_SC_XHCI_MEM_DBOFF_DBAO               0xFFFFFFFC ///< Doorbell Array Offset

#define R_SC_XHCI_MEM_RTSOFF                   0x18  ///< Runtime Register Space Offset
#define B_SC_XHCI_MEM_RTSOFF_RTRSO             0xFFFFFFE0 ///< Runtime Register Space Offset

///
/// 0x80 - 0xBF - Operational Registers
///
#define R_SC_XHCI_MEM_USBCMD                   0x80  ///< USB Command
#define B_SC_XHCI_MEM_USBCMD_EU3S              BIT11 ///< Enable U3 MFINDEX Stop
#define B_SC_XHCI_MEM_USBCMD_EWE               BIT10 ///< Enable Wrap Event
#define B_SC_XHCI_MEM_USBCMD_CRS               BIT9  ///< Controller Restore State
#define B_SC_XHCI_MEM_USBCMD_CSS               BIT8  ///< Controller Save State
#define B_SC_XHCI_MEM_USBCMD_LHCRST            BIT7  ///< Light Host Controller Reset
#define B_SC_XHCI_MEM_USBCMD_HSEE              BIT3  ///< Host System Error Enable
#define B_SC_XHCI_MEM_USBCMD_INTE              BIT2  ///< Interrupter Enable
#define B_SC_XHCI_MEM_USBCMD_HCRST             BIT1  ///< Host Controller Reset
#define B_SC_XHCI_MEM_USBCMD_RS                BIT0  ///< Run/Stop

#define R_SC_XHCI_MEM_USBSTS                   0x84  ///< USB Status
#define B_SC_XHCI_MEM_USBSTS_HCE               BIT12 ///< Host Controller Error
#define B_SC_XHCI_MEM_USBSTS_CNR               BIT11 ///< Controller Not Ready
#define B_SC_XHCI_MEM_USBSTS_SRE               BIT10 ///< Save / Restore Error
#define B_SC_XHCI_MEM_USBSTS_RSS               BIT9  ///< Restore State Status
#define B_SC_XHCI_MEM_USBSTS_SSS               BIT8  ///< Save State Status
#define B_SC_XHCI_MEM_USBSTS_PCD               BIT4  ///< Port Change Detect
#define B_SC_XHCI_MEM_USBSTS_EINT              BIT3  ///< Event Interrupt
#define B_SC_XHCI_MEM_USBSTS_HSE               BIT2  ///< Host System Error
#define B_SC_XHCI_MEM_USBSTS_HCH               BIT0  ///< HC Halted

///
/// 0x480 - 0x5CF - Port Status and Control Registers
///
#define R_SC_XHCI_MEM_PORTSC01USB2             0x480
#define R_SC_XHCI_MEM_PORTSC02USB2             0x490
#define R_SC_XHCI_MEM_PORTSC03USB2             0x4A0
#define R_SC_XHCI_MEM_PORTSC04USB2             0x4B0
#define R_SC_XHCI_MEM_PORTSC05USB2             0x4C0
#define R_SC_XHCI_MEM_PORTSC06USB2             0x4D0
#define R_SC_XHCI_MEM_PORTSC07USB2             0x4E0
#define R_SC_XHCI_MEM_PORTSC08USB2             0x4F0
#define B_SC_XHCI_MEM_PORTSCXUSB2_WPR          BIT31 ///< Warm Port Reset
#define B_SC_XHCI_MEM_PORTSCXUSB2_DR           BIT30 ///< Device Removable
#define B_SC_XHCI_MEM_PORTSCXUSB2_WOE          BIT27 ///< Wake on Over-Current Enable
#define B_SC_XHCI_MEM_PORTSCXUSB2_WDE          BIT26 ///< Wake on Disconnect Enable
#define B_SC_XHCI_MEM_PORTSCXUSB2_WCE          BIT25 ///< Wake on Connect Enable
#define B_SC_XHCI_MEM_PORTSCXUSB2_CAS          BIT24 ///< Cold Attach Status
#define B_SC_XHCI_MEM_PORTSCXUSB2_CEC          BIT23 ///< Port Config Error Change
#define B_SC_XHCI_MEM_PORTSCXUSB2_PLC          BIT22 ///< Port Link State Change
#define B_SC_XHCI_MEM_PORTSCXUSB2_PRC          BIT21 ///< Port Reset Change
#define B_SC_XHCI_MEM_PORTSCXUSB2_OCC          BIT20 ///< Over-current Change
#define B_SC_XHCI_MEM_PORTSCXUSB2_WRC          BIT19 ///< Warm Port Reset Change
#define B_SC_XHCI_MEM_PORTSCXUSB2_PEC          BIT18 ///< Port Enabled Disabled Change
#define B_SC_XHCI_MEM_PORTSCXUSB2_CSC          BIT17 ///< Connect Status Change
#define B_SC_XHCI_MEM_PORTSCXUSB2_LWS          BIT16 ///< Port Link State Write Strobe
#define B_SC_XHCI_MEM_PORTSCXUSB2_PIC          (BIT15 | BIT14) ///< Port Indicator Control
#define B_SC_XHCI_MEM_PORTSCXUSB2_PS           (BIT13 | BIT12 | BIT11 | BIT10) ///< Port Speed
#define B_SC_XHCI_MEM_PORTSCXUSB2_PP           BIT9  ///< Port Power
#define B_SC_XHCI_MEM_PORTSCXUSB2_PLS          (BIT8 | BIT7 | BIT6 | BIT5) ///< Port Link State
#define B_SC_XHCI_MEM_PORTSCXUSB2_PR           BIT4  ///< Port Reset
#define B_SC_XHCI_MEM_PORTSCXUSB2_OCA          BIT3  ///< Over-Current Active
#define B_SC_XHCI_MEM_PORTSCXUSB2_PED          BIT1  ///< Port Enabled Disabled
#define B_SC_XHCI_MEM_PORTSCXUSB2_CCS          BIT0  ///< Current Connect Status

#define R_SC_BXT_XHCI_MEM_PORTSC1USB3          0x4C0
#define R_SC_BXT_XHCI_MEM_PORTSC2USB3          0x4D0
#define R_SC_BXT_XHCI_MEM_PORTSC3USB3          0x4E0
#define R_SC_BXT_XHCI_MEM_PORTSC4USB3          0x4F0
#define R_SC_BXTP_XHCI_MEM_PORTSC1USB3         0x500
#define R_SC_BXTP_XHCI_MEM_PORTSC2USB3         0x510
#define R_SC_BXTP_XHCI_MEM_PORTSC3USB3         0x520
#define R_SC_BXTP_XHCI_MEM_PORTSC4USB3         0x530
#define R_SC_BXTP_XHCI_MEM_PORTSC5USB3         0x540
#define R_SC_BXTP_XHCI_MEM_PORTSC6USB3         0x550
#define R_SC_BXTP_XHCI_MEM_PORTSC7USB3         0x560
#define B_SC_XHCI_MEM_PORTSCXUSB3_WPR          BIT31  ///</ Warm Port Reset
#define B_SC_XHCI_MEM_PORTSCXUSB3_CEC          BIT23  ///</ Port Config Error Change
#define B_SC_XHCI_MEM_PORTSCXUSB3_PLC          BIT22  ///</ Port Link State Change
#define B_SC_XHCI_MEM_PORTSCXUSB3_PRC          BIT21  ///</ Port Reset Change
#define B_SC_XHCI_MEM_PORTSCXUSB3_OCC          BIT20  ///</ Over-current Chang
#define B_SC_XHCI_MEM_PORTSCXUSB3_WRC          BIT19  ///</ Warm Port Reset Change
#define B_SC_XHCI_MEM_PORTSCXUSB3_PEC          BIT18  ///</ Port Enabled Disabled Change
#define B_SC_XHCI_MEM_PORTSCXUSB3_CSC          BIT17  ///</ Connect Status Change
#define B_SC_XHCI_MEM_PORTSCXUSB3_PP           BIT9   ///</ Port Power
#define B_SC_XHCI_MEM_PORTSCXUSB3_PR           BIT4   ///</ Port Reset
#define B_SC_XHCI_MEM_PORTSCXUSB3_PED          BIT1   ///</ Port Enabled / Disabled



#define R_SC_XHCI_MEM_XECP_CMDM_CTRL_REG1      0x818C  ///< XECP_CMDM_CTRL_REG1 - Command Manager Control 1
#define B_SC_XHCI_MEM_XECP_MAX_EP              BIT20   ///<enable cause a Configure Endpoint Command to fail if the number of active EPs post configuration exceeds the maximum number of EPs available in cache
#define B_SC_XHCI_MEM_XECP_TSP                 BIT16   ///<clearing of split state if TSP=1
#define B_SC_XHCI_MEM_XECP_CLR_CTX_ENSLOT      BIT8    ///<clearing other context during an enable slot command.

#define R_SC_XHCI_MEM_XECP_CMDM_CTRL_REG2      0x8190  ///< XECP_CMDM_CTRL_REG2 - Command Manager Control 2
#define B_SC_XHCI_MEM_XECP_FORCE_BURST_SIZE    BIT14    ///<Force the default burst size when clearing context

#define R_SC_XHCI_MEM_XECP_CMDM_CTRL_REG3      0x8194  ///< XECP_CMDM_CTRL_REG3 - Command Manager Control 3
#define B_SC_XHCI_MEM_XECP_STOP_EP             BIT25   ///<Setting this field will enable the fix for a stream issue related to Stop EP recovery

#define R_XHCI_XECP_SUPP_USB2_2                0x8008  ///< Supported Protocol Extended Capability
#define B_XHCI_XECP_SUPP_USB2_2_HLC            BIT19   ///< HW LMP (Link Power Management) Capability
#define R_SC_XHCI_MEM_XECP_SUPP_USB3_2         0x8028

#define R_SC_XHCI_MEM_PMCTRL                   0x80A4  ///<Power Management Control
#define B_SC_XHCI_MEM_PMCTRL_ASYNC_PME_SRC     BIT31   ///<Async PME Source Enable
#define B_SC_XHCI_MEM_PMCTRL_LEGACY_PME_SRC    BIT30   ///<Legacy PME Source Enable
#define B_SC_XHCI_MEM_PMCTRL_RESET_PG          BIT29   ///<Reset Warn Power Gate Trigger Disable
#define B_SC_XHCI_MEM_PMCTRL_XELFPSRTC         BIT25   ///<XELFPSRTC (Enable LFPS Filtering on RTC)
#define B_SC_XHCI_MEM_PMCTRL_XMPHYSPGDD0I2     BIT24   ///<XMPHYSPGDD0I2 (ModPhy Sus Well Power Gate Disable for D0I2)
#define B_SC_XHCI_MEM_PMCTRL_XMPHYSPGDD0I3     BIT23   ///<XMPHYSPGDD0I3 (ModPhy Sus Well Power Gate Disable for D0I3)
#define B_SC_XHCI_MEM_PMCTRL_XMPHYSPGDRTD3     BIT22   ///<XMPHYSPGDRTD3 (ModPhy Sus Well Power Gate Disable for RTD3)
#define B_SC_XHCI_MEM_PMCTRL_SSALDE            BIT16   ///<SS AON LFPS Detector Enable
#define B_SC_XHCI_MEM_PMCTRL_LFPS_THRESHOLD    (0xFF00)   ///<SS U3 LFPS Detection Threshold bit15:8
#define B_SC_XHCI_MEM_PMCTRL_FPS_OFFTIME       (0x00F0)   ///<SS U3 LFPS Periodic Sampling OFF Time Control bit7:4
#define B_SC_XHCI_MEM_PMCTRL_LFPS_SRC          BIT3   ///<PS3 LFPS Source Select
#define B_SC_XHCI_MEM_PMCTRL_APGE              BIT2   ///<XHCI Engine Autonomous Power Gate Exit Reset Policy

#define R_SC_XHCI_MEM_PGCBCTRL                   0x80A8  ///<PGCB Control
#define B_SC_XHCI_MEM_PGCBCTRL_RESET_PREP_DIS     BIT24  ///<Reset Prep override disable

#define R_SC_XHCI_MEM_AUX_CTRL_REG1            0x80E0  ///< AUX_CTRL_REG1 - AUX Power Management Control
#define B_SC_XHCI_MEM_AUX_ISOLATION            BIT22   ///<masking related to isolation missing between gated and ungated domain that will cause an immediate wake out of power gating
#define B_SC_XHCI_MEM_AUX_PORT_CG              BIT16   ///<USB3 port clock gating
#define B_SC_XHCI_MEM_AUX_CG                   BIT9    ///<When set to '1' disable core clock gating based on low power state entered
#define B_SC_XHCI_MEM_AUX_P2                   BIT6    ///<enable P2 overwrite P1 when PCIe core has indicated the transition from P0 to P1.

#define R_SC_XHCI_MEM_HOST_CTRL_SCH_REG        0x8094  ///< HOST_CTRL_SCH_REG - Host Control Scheduler
#define B_SC_XHCI_MEM_HOST_CTRL_DIS_ASYNC      BIT23   ///<Disable async. scheduling while periodic active to same port
#define B_SC_XHCI_MEM_HOST_CTRL_EN_EP_PPL      BIT22   ///<Setting this bit enables pipelining of multiple OUT EPs
#define B_SC_XHCI_MEM_HOST_CTRL_DIS_SCH_PKT    BIT21   ///<Scheduler: Enable Stop serving packets to disabled port
#define B_SC_XHCI_MEM_HOST_CTRL_DIS_PKT_CHK    BIT14   ///<TTE: Disable checking of missed microframes
#define B_SC_XHCI_MEM_HOST_CTRL_EN_1P_EP_PPL   BIT6   ///<Setting this bit enables pipelining of multiple OUT EPs on the same port. This will mainly help boost the performance for 1 port multiple OUT EPs test case.

#define R_SC_XHCI_MEM_HOST_CTRL_IDMA_REG        0x809C  ///< HOST_CTRL_SCH_REG - Host Control IN DMA Register
#define B_SC_XHCI_MEM_HOST_CTRL_IDMA_HCRST_WDT   BIT31   ///<Enable the WDT for HC reset flow

#define R_SC_XHCI_MEM_HOST_CTRL_PORT_LINK_REG  0x80EC  ///< HOST_CTRL_PORT_LINK_REG - SuperSpeed Port Link Control
#define B_SC_XHCI_MEM_HOST_CTRL_EN_TS_EXIT     BIT19   ///<enable TS receive to complete U1/U2/U3 exit LFPS handshake
#define B_SC_XHCI_MEM_HOST_CTRL_PORT_INIT_TIMEOUT   BIT17///<specifies the port initialization timeout value 1:20us-21us 0:19us-20us
#define B_SC_XHCI_MEM_DIS_LINK_CM              BIT0    ///<0: Enable link compliance mode 1: Disable link compliance mode

#define R_SC_XHCI_MEM_USB2_LINK_MGR_CTRL_REG1  0x80F0  ///< USB2_LINK_MGR_CTRL_REG1 - USB2 Port Link Control 1, 2, 3, 4
#define B_SC_XHCI_MEM_USB2_LINK_L1_EXIT        BIT20   ///<Mode for extended L1 Exit recovery delay

#define R_SC_XHCI_MEM_USB2_LINK_MGR_CTRL_REG1_CONTROL4   0x80FC ///<USB2_LINK_MGR_CTRL_REG1 - USB2 Port Link Control 1, 2, 3, 4
#define B_SC_XHCI_MEM_USB2_LINK_PRV_L1_ENTRY   BIT25 ///<(bit121) Chicken bit to enable periodic_prewake fix to prevent L1 entry if in U0, or wake from L1 if already in U2.

#define R_SC_XHCI_MEM_HOST_CTRL_TRM_REG2       0x8110  ///< HOST_CTRL_TRM_REG2 - Host Controller Transfer Manager Control 2
#define B_SC_XHCI_MEM_HOST_CTRL_MAX_BURST_CHK  BIT20   ///<TRM can check the credit returned from remote device to not excced its max burst size
#define B_SC_XHCI_MEM_HOST_CTRL_TRF_PIPE       BIT11   ///<Enable the host to transfer to the prime-pipe state
#define B_SC_XHCI_MEM_HOST_CTRL_REDEEM         BIT2    ///<enable the credit redeem when a port is in NC state

#define R_SC_XHCI_MEM_AUX_CTRL_REG2            0x8154  ///< AUX_CTRL_REG2 - Aux PM Control Register 2
#define B_SC_XHCI_MEM_AUX2_L1P2_EXIT           BIT31 ///<This bit disables the dependency on Wake Enables defined in PORTSC for L1P2 exit when in D0
#define B_SC_XHCI_MEM_AUX2_P2_D3HOT            BIT21 ///<disables p2 overwrite due to the D3HOT where PCIe core enters the L1
#define B_SC_XHCI_MEM_AUX2_PHY_P3              BIT13 ///<enables PHY P3 mode in U2.

#define R_SC_XHCI_MEM_AUX_CLOCK_CTRL_REG       0x816C  ///< xHCI Aux Clock Control Register
#define B_SC_XHCI_MEM_USB3_PELCG               BIT19 ///<enables gating of the SOSC trunk to the XHCI engine and link in the PARUSB3 partition.
#define B_SC_XHCI_MEM_USB3_PFTTCG              BIT18 ///<USB3 Partition Frame Timer trunk gating Enable
#define B_SC_XHCI_MEM_USB2_LPCG                BIT17 ///<USB2 link partition clock gating enable
#define B_SC_XHCI_MEM_USB2_USBIP_12_5HZ_CG     BIT16 ///<USB2/USHIP 12.5 MHz partition clock gating enable
#define B_SC_XHCI_MEM_USB3_PORT_ACCG           BIT14 ///<USB3 Port Aux/Core clock gating enable
#define B_SC_XHCI_MEM_AUXCG_RXDET_TIMER        (BIT13 | BIT12) ///<Rx Detect Timer when port Aux Clock is Gated 0x0:      100ms;   0x1:      12ms;   Others:    Reserved;
#define B_SC_XHCI_MEM_HOST_U2_RES_BMODCG       (BIT11 | BIT8) ///<U2 Residency Before ModPHY Clock Gating
#define B_SC_XHCI_MEM_EACG                     BIT5 ///< XHCI Engine Aux clock gating enable
#define B_SC_XHCI_MEM_APBCG                    BIT4 ///<XHCI Aux PM block clock gating enable
#define B_SC_XHCI_MEM_ACTCG                    BIT3 ///<USB3 Aux Clock Trunk Gating Enable
#define B_SC_XHCI_MEM_PORT_APCG                BIT2 ///< USB3 Port Aux/Port clock gating enable
#define B_SC_XHCI_MEM_PPACG_IN_U2              BIT1 ///< USB3 PHY port Aux clock gating enable in U2
#define B_SC_XHCI_MEM_PPACG_IN_DUD             BIT0 ///<USB3 PHY port Aux clock gating enable in Disconnected, U3 or Disabled

#define R_SC_XHCI_MEM_HOST_IF_PWR_CTRL_REG0    0x8140  ///< HOST_IF_PWR_CTRL_REG0 - Power Scheduler Control 0
#define B_SC_XHCI_MEM_HOST_EIH                 (BIT31 | BIT30 | BIT29 | BIT28 | BIT27 | BIT26 | BIT25 | BIT24) ///< Engine Idle Hysteresis (EIH), This register controls the min. idle span that has to be observed from the engine idle indicators before the power state flags (xhc_*_idle) will indicate a 1 (TBD units)
#define B_SC_XHCI_MEM_HOST_AW                  (BIT23 | BIT22 | BIT21 | BIT20 | BIT19 | BIT18 | BIT17 | BIT16 | BIT15 | BIT14 | BIT13 | BIT12)///<Advance Wake (AW):This register controls the time before the next scheduled transaction where the periodic_active & periodic_active_hs_in will assert.
#define B_SC_XHCI_MEM_HOST_MID                 (BIT11 | BIT10 | BIT9 | BIT8 | BIT7 | BIT6 | BIT5 | BIT4 | BIT3 | BIT2 | BIT1 | BIT0)///<Min. Idle Duration (MID):The sum of this register plus the Advance Wake form to a Total Idle time.

#define R_SC_XHCI_MEM_HOST_IF_PWR_CTRL_REG1    0x8144  ///< HOST_IF_PWR_CTRL_REG1 - Power Scheduler Control 1
#define B_SC_XHCI_MEM_HSII                     BIT8 ///<HS Interrupt IN Alarm (HSII)
#define R_SC_XHCI_MEM_AUX_CCR                  0x816C
#define B_SC_XHCI_MEM_AUX_CCR_XHCI_AC_GE       BIT5
#define N_SC_XHCI_MEM_AUX_CCR_XHCI_AC_GE       5

#define R_SC_XHCI_MEM_LATENCY_TOLERANCE_PARAMETERS_LTV_CONTROL                 0x8174  ///< xHCI Latency Tolerance Parameters - LTV Control
#define B_SC_XHCI_MEM_XLTRE                    BIT24 ///<XHCI LTR Enable (XLTRE) This bit must be set to enable LTV messaging from XHCI to the PMC.
#define B_SC_XHCI_MEM_USB2_PORT_L0_LTV         (BIT11 | BIT10 | BIT9 | BIT8 | BIT7 | BIT6 | BIT5 | BIT4 | BIT3 | BIT2 | BIT1 | BIT0)///< USB2 Port L0 LTV

#define R_SC_XHCI_MEM_LATENCY_TOLERANCE_PARAMETERS_HIGH_IDLE_TIME_CONTROL      0x817C  ///< xHC Latency Tolerance Parameters - High Idle Time Control
#define B_SC_XHCI_MEM_MHIT                     (BIT28 | BIT27 | BIT26 | BIT25 | BIT24 | BIT23 | BIT22 | BIT21 | BIT20 | BIT19 | BIT18 | BIT17 | BIT16) ///<Minimum High Idle Time (MHIT)This is the minimum schedule idle time that must be available before a 'High' LTR value can be indicated.
#define B_SC_XHCI_MEM_HIWL                     (BIT12 | BIT11 | BIT10 | BIT9 | BIT8 | BIT7 | BIT6 | BIT5 | BIT4 | BIT3 | BIT2 | BIT1 | BIT0) ///<High Idle Wake Latency (HIWL) This is the latency to access memory from the High Idle Latency state.

#define R_SC_XHCI_MEM_LATENCY_TOLERANCE_PARAMETERS_MEDIUM_IDLE_TIME_CONTROL    0x8180  ///< xHC Latency Tolerance Parameters - Medium Idle Time Control
#define B_SC_XHCI_MEM_MMIT                     (BIT28 | BIT27 | BIT26 | BIT25 | BIT24 | BIT23 | BIT22 | BIT21 | BIT20 | BIT19 | BIT18 | BIT17 | BIT16) ///<Minimum Medium Idle Time (MMIT)This is the minimum schedule idle time that must be available before a 'Medium' LTR value can be indicated.
#define B_SC_XHCI_MEM_MIWL                     (BIT12 | BIT11 | BIT10 | BIT9 | BIT8 | BIT7 | BIT6 | BIT5 | BIT4 | BIT3 | BIT2 | BIT1 | BIT0) ///<Medium Idle Wake Latency (MIWL)This is the latency to access memory from the Medium Idle Latency state.

#define R_SC_XHCI_MEM_LATENCY_TOLERANCE_PARAMETERS_LOW_IDLE_TIME_CONTROL       0x8184  ///< xHC Latency Tolerance Parameters - Low Idle Time Control
#define B_SC_XHCI_MEM_MLIT                     (BIT28 | BIT27 | BIT26 | BIT25 | BIT24 | BIT23 | BIT22 | BIT21 | BIT20 | BIT19 | BIT18 | BIT17 | BIT16) ///<Minimum Low Idle Time (MLIT)This is the minimum schedule idle time that must be available before a 'Low' LTR value can be indicated.
#define B_SC_XHCI_MEM_LIWL                     (BIT12 | BIT11 | BIT10 | BIT9 | BIT8 | BIT7 | BIT6 | BIT5 | BIT4 | BIT3 | BIT2 | BIT1 | BIT0) ///<Low Idle Wake Latency (LIWL) This is the latency to access memory from the Low Idle Latency state.

#define R_SC_XHCI_MEM_USB2_PHY_POWER_MANAGEMENT_CONTROL                        0x8164  ///< USB2 PHY Power Management Control
#define B_SC_XHCI_MEM_CMAI                     BIT7 ///< Enable Command Manager Active indication for Tx/Rx Bias circuit HS Phy PM Policy
#define B_SC_XHCI_MEM_TTEAI                    BIT6 ///<Enable TTE  Active indication for Tx/Rx Bias circuit HS Phy PM Policy
#define B_SC_XHCI_MEM_IDMAAI                   BIT5 ///<Enable IDMA Active indication for Tx/Rx Bias circuit HS Phy PM Policy
#define B_SC_XHCI_MEM_ODMAAI                   BIT4 ///<Enable ODMA Active indication for Tx/Rx Bias circuit HS Phy PM Policy
#define B_SC_XHCI_MEM_TMAI                     BIT3 ///<Enable  Transfer Manager  Active indication for Tx/Rx Bias circuit HS Phy PM Policy
#define B_SC_XHCI_MEM_SAI                      BIT2 ///<Enable  Scheduler  Active indication for Tx/Rx Bias circuit HS Phy PM Policy
#define B_SC_XHCI_MEM_RX_BIAS_CHT_DIS          BIT1 ///<Enable Rx Bias ckt disable
#define B_SC_XHCI_MEM_TX_BIAS_CHT_DIS          BIT0 ///<Enable Tx Bias ckt disable

#define R_SC_XHCI_MEM_HOST_CONTROLLER_MISC_REG 0x80B0
#define B_SC_XHCI_MEM_LATE_FID_CHK_DIS         BIT24 ///< This register disables the Late FID Check performed when starting an ISOCH stream.
#define B_SC_XHCI_MEM_LATE_FID_TTE_DIS         BIT23 /// < Late FID TTE  Count Adjust Disable
#define B_SC_XHCI_MEM_EXTRA_UFRAME             (BIT18|BIT17|BIT16) ///< This register controls the extra number of uFrames added onto the advancing of late FID check..

#define R_SC_XHCI_MEM_HOST_CONTROLLER_MISC2REG 0x80B4
#define B_SC_XHCI_MEM_FRAME_TIM_SEL            BIT5   ///<Frame Timer Select
#define B_SC_XHCI_MEM_WARM_PORT_RESET_ON_DISC_PORT_DIS    BIT2   ///<Disable Warm Port Reset on Disconnected Port

#define R_SC_XHCI_MEM_HOST_CONTROLLER_SSPE     0x80B8
#define B_SC_XHCI_MEM_ENCLCCS                  BIT30 ///< Enables clearing of CCS on HCRESET
#define B_SC_XHCI_MEM_SSPE                     (BIT3|BIT2|BIT1|BIT0) ///<This field controls whether SuperSpeed capability is enabled  for a given USB3 port.

#define R_SC_XHCI_MEM_HOST_CTRL_BW_MAX_REG     0x8128 ///<Max BW control Reg 4
#define V_SC_XHCI_MEM_HOST_CTRL_BW_MAX_REG_TT  0xFFF  ///<TT Max BW Units

#define R_SC_XHCI_MEM_PULLDOWN_DISABLE_CONTROL 0x8198 ///<Each bit corresponds to a USB2 port indexed by the bit number.When set, allow the pulldown on D+ or D- (as appropriate) to be disabled when the port is connected and in L2.
#define B_SC_XHCI_MEM_PULLDOWN_DISABLE_CONTROL 0xFFFFFFFF

#define R_SC_XHCI_MEM_THROTTLE_CONTROL         0x819C
#define B_SC_XHCI_MEM_SSIC_TTUM                BIT20 ///<SSIC Thermal Throttle Ux Mapping,Controls if U1 or U2 is forced upon the start of thermal throttle OFF period.
#define B_SC_XHCI_MEM_USB3_TTUM                BIT16 ///<USB3 Thermal Throttle Ux Mapping,Controls if U1 or U2 is forced upon the start of thermal throttle OFF period.
#define B_SC_XHCI_MEM_FORCE_L1                 BIT14 ///<Enable  Force L1 when throttled.
#define B_SC_XHCI_MEM_INTERRUPT_THROTTLING_DIS BIT13 ///<Disable Interrupt Throttling
#define B_SC_XHCI_MEM_ISOCHRONOUS_THROT_DIS    BIT12 ///<Disable Isochronous Throttling
#define B_SC_XHCI_MEM_T1_ACTION                (BIT11 | BIT10 | BIT9 | BIT8) ///<T1 Action, bus intervals to be idle for async traffic out of the 16 interval master period; from 0 to 15.
#define B_SC_XHCI_MEM_T2_ACITON                (BIT7 | BIT6 | BIT5 | BIT4) ///<T2 Action, # bus intervals to be idle for async traffic out of the 16 interval master period; from 0 to 15.
#define B_SC_XHCI_MEM_T3_ACTION                (BIT3 | BIT2 | BIT1 | BIT0) ///<T3 Action, # bus intervals to be idle for async traffic out of the 16 interval master period; from 0 to 15.

#define R_SC_XHCI_LFPS_MEM_PM_CONTROL          0x81A0 ///< LFPS Power Management in U3 Enable, This field allows xHC to turn off LFPS Receiver when the port is in U3.
#define B_SC_XHCI_LFPS_MEM_PM_CONTROL          0xFFFFFFFF

#define R_SC_XHCI_MEM_THROTT2                  0x81B4
#define B_SC_XHCI_MEM_TTFLA                    (BIT3 | BIT2 | BIT1 | BIT0) ///< Thermal Throttle Force LPM Accept Enable

#define R_SC_XHCI_MEM_LFPS_ON_COUNT            0x81B8
#define B_SC_XHCI_MEM_XLFPSONCNTSSIC           (BIT15 | BIT14 | BIT13 | BIT12 | BIT11 | BIT10) ///< This time would describe the number of clocks SSIC LFPS will remain ON.
#define B_SC_XHCI_MEM_XLFPSONCNTSS             (BIT9 | BIT8 | BIT7 | BIT6 | BIT5 | BIT4 | BIT3 | BIT2 | BIT1 | BIT0) ///< This time would describe the number of clocks LFPS will remain ON.

#define R_SC_XHCI_MEM_D0I2_CONTROL             0x81BC
#define B_SC_XHCI_MEM_USB2_BTB_BTO_HANDLING  BIT31 ///<USB2 Back to Back BTO Handling Enable
#define B_SC_XHCI_MEM_D0I2_MIN_RESIDENCY       (BIT29|BIT28|BIT27|BIT26) ///<This field controls the minimum time that we must stay in D0i2 to ensure that the entry sequence has settled before we attempt to exit.
#define B_SC_XHCI_MEM_D0I2_ENTRY_HYSTER_TIMER  (BIT25|BIT24|BIT23|BIT22) ///<This field allows for a hysteresis timer to be implemented specifically for D0i2.  This will allow for D0i2 entry to be controlled independently from the timer used for D0i3 and D3.
#define B_SC_XHCI_MEM_D0I2_ACT_PERIODIC_EP_DIS BIT21 ///< This field allows the xHC to control how aggressive it enters D0i2 in the presence of active Periodic EP's.
#define B_SC_XHCI_MEM_MSI_D0I2_PWT             (BIT20|BIT19|BIT18|BIT17|BIT16) ///<This is the latency that is expected to be incurred to exit the D0i2 state.This wake latency is the latency to be added to the tracked D0i2 wake by the MSI module.
#define B_SC_XHCI_MEM_MSI_IDLE_THRESHOLD       (BIT15|BIT14|BIT13|BIT12|BIT11|BIT10|BIT9|BIT8|BIT7|BIT6|BIT5|BIT4) ///<his field allows the xHC to control how aggressive it enters D0i2 in the presence of pending MSI.  This field is valid only if Pending MSI Disable is "0", allowing D0i2 in the presence of pending MSI's.
#define B_SC_XHCI_MEM_PENDING_MSI_DIS          (BIT3) ///<This field allows the xHC to disable D0i2 when there are pending MSI's in the event manager.
#define B_SC_XHCI_MEM_FRAME_TIMER_RUN_DIS      (BIT2) ///<This field allows the xHC to disable D0i2 when the frame timer is running.
#define B_SC_XHCI_MEM_USB2_L1_DIS              (BIT1) ///<This field allows the xHC to disable D0i2 when USB2 ports are in L1.  This implies that D0i2 will only be triggered when ports are in L2 or deeper.
#define B_SC_XHCI_MEM_USB3_L1_DIS              (BIT0) ///<This field allows the xHC to disable D0i2 when USB3 ports are in U2.  This implies that D0i2 will only be triggered when ports are in U3 or deeper.

#define R_SC_XHCI_MEM_D0I2_SCH_ALARM_CTRL      0x81C0
#define B_SC_XHCI_MEM_D0I2_IDLE_TIME           0x1FFF0000 ///<bit28:16 This is the minimum schedule idle time that must be available before D0i2 can be allowed.
#define B_SC_XHCI_MEM_D0I2_WAKE_LATENCY        0x1FFF ///<bit12:0 This is the latency that is expected to be incurred to exit the D0i2 state.

#define R_SC_XHCI_MEM_USB2_PM_CTRL             0x81C4
#define B_SC_XHCI_MEM_U2PSPGPSCBP              BIT11 ///<USB2 PHY SUS Power Gate PORTSC Block Policy:This controls the policy for blocking PORTSC Updates while the USB2 PHY SUS Well is power gated.
#define B_SC_XHCI_MEM_U2PSPGEHC                (BIT10|BIT9|BIT8) ///<USB2 PHY SUS Well Power Gate Entry Hysteresis Count:This controls the amount of hysteresis time the controller will enforce after detecting the USB2 PHY SUS Power Gate entry condition.
#define B_SC_XHCI_MEM_U2PSUSPGP                (BIT3|BIT2) ///<USB2 PHY SUS Well Power Gate Policy: This field controls when to enable the USB2 PHY SUS Well Power Gating when the proper conditions are met.
#define B_SC_XHCI_MEM_U2PSUSPGP_Shadow     (BIT1|BIT0) ///<Shadow of USB2 PHY SUS Well Power Gate Policy: This filed is reserved but is required to shadow bits 3:2 due to a HW bug where some logic is using bit 1:0 instead of 3:2
#define R_SC_XHCI_MEM_STRAP2                   0x8420  ///< USB3 Mode Strap
#define R_SC_XHCI_MEM_SSIC_CFG_2_PORT1         0x880C
#define B_SC_XHCI_MEM_SSIC_PORT_UNUSED         BIT31
#define B_SC_XHCI_MEM_PROG_DONE                BIT30
#define B_SC_XHCI_MEM_NUM_OF_MK0               (BIT29|BIT28|BIT27|BIT26)
#define B_SC_XHCI_MEM_DISABLE_SCRAMBLING       BIT25
#define B_SC_XHCI_MEM_RETRAIN_TIME             (BIT24|BIT23|BIT22|BIT21)
#define B_SC_XHCI_MEM_PHY_RESET_TIME           (BIT20|BIT19|BIT18|BIT17|BIT16)
#define B_SC_XHCI_MEM_LRST_TIME                0xFF00  ///<bit15:8
#define B_SC_XHCI_MEM_ACTIVATE_LRST_TIME       0xFF  ///<bit7:0

#define R_SC_XHCI_MEM_SSIC_CFG_2_PORT2         0x883C

#define R_SC_XHCI_MEM_XECP_SUPP_USB2_2         0x8008

#define R_SC_XHCI_MEM_USB_LPM_PARAM            0x8170

#define R_SC_XHCI_MEM_SSIC_GLOBAL_CONFIG_CONTROL      0x8804
#define R_SC_XHCI_MEM_PORT1_SSIC_CONFIG_REG1          0x8808
#define R_SC_XHCI_MEM_PORT1_SSIC_CONFIG_REG2          0x880C
#define R_SC_XHCI_MEM_PORT1_SSIC_CONFIG_REG3          0x8810
#define R_SC_XHCI_MEM_PORT1_REGISTER_ACCESS_CONTROL   0x8904
#define R_SC_XHCI_MEM_PORT1_PROFILE_ATTRIBUTES_REG0   0x890C
#define R_SC_XHCI_MEM_PORT1_PROFILE_ATTRIBUTES_REG1   0x8910
#define R_SC_XHCI_MEM_PORT1_PROFILE_ATTRIBUTES_REG2   0x8914
#define R_SC_XHCI_MEM_PORT1_PROFILE_ATTRIBUTES_REG3   0x8918
#define R_SC_XHCI_MEM_PORT1_PROFILE_ATTRIBUTES_REG4   0x891C
#define R_SC_XHCI_MEM_PORT1_PROFILE_ATTRIBUTES_REG5   0x8920
#define R_SC_XHCI_MEM_PORT1_PROFILE_ATTRIBUTES_REG6   0x8924
#define R_SC_XHCI_MEM_PORT1_PROFILE_ATTRIBUTES_REG7   0x8928
#define R_SC_XHCI_MEM_PORT1_PROFILE_ATTRIBUTES_REG8   0x892C
#define R_SC_XHCI_MEM_PORT1_PROFILE_ATTRIBUTES_REG9   0x8930
#define R_SC_XHCI_MEM_PORT1_PROFILE_ATTRIBUTES_REG10  0x8934
#define R_SC_XHCI_MEM_PORT1_PROFILE_ATTRIBUTES_REG11  0x8938
#define R_SC_XHCI_MEM_PORT1_PROFILE_ATTRIBUTES_REG12  0x893C

#define R_SC_UTMI_MEM_MISC_REG_PER_PORT_PP0       0x4108
#define R_SC_UTMI_MEM_MISC_REG_PER_PORT_PP1       0x4208
#define R_SC_UTMI_MEM_MISC_REG_PER_PORT_PP2       0x4308
#define B_SC_HSFSLSPSEL                       (BIT5 | BIT4)
#define B_SC_REG_BUFFER_TRISTATE_EN           BIT6

///
/// USB3 OTG PCI Config Space Registers
///
#define PCI_DEVICE_NUMBER_OTG           21
#define PCI_FUNCTION_NUMBER_OTG         1

#define R_SC_OTG_CFG_DEVVENDID                 0x00  ///< Vendor ID
#define V_SC_USB_CFG_DEVVENDID_VID             V_INTEL_VENDOR_ID

#define R_SC_OTG_CFG_STSCMD                    0x04  ///< Command Status
#define B_SC_OTG_CFG_STSCMD_INTR_DIS           BIT10 ///< Interrupt Disable
#define B_SC_OTG_CFG_STSCMD_BME                BIT2  ///< Bus Master Enable
#define B_SC_OTG_CFG_STSCMD_MSE                BIT1  ///< Memory Space Enable

#define R_SC_OTG_CFG_BAR0                      0x10  ///< BAR 0
#define B_SC_OTG_CFG_BAR0_BA                   0xFFE00000 ///< Base Address
#define V_SC_OTG_CFG_BAR0_SIZE                 0x200000
#define N_SC_OTG_CFG_BAR0_ALIGNMENT            21
#define B_SC_OTG_CFG_BAR0_PREF                 BIT3  ///< Prefetchable
#define B_SC_OTG_CFG_BAR0_ADDRNG               (BIT2 | BIT1) ///< Address Range
#define B_SC_OTG_CFG_BAR0_SPTYP                BIT0  ///< Space Type (Memory)

#define R_SC_OTG_CFG_BAR1                      0x18  ///< BAR 1
#define B_SC_OTG_CFG_BAR1_BA                   0xFFFFF000 ///< Base Address
#define B_SC_OTG_CFG_BAR1_PREF                 BIT3  ///< Prefetchable
#define B_SC_OTG_CFG_BAR1_ADDRNG               (BIT2 | BIT1) ///< Address Range
#define B_SC_OTG_CFG_BAR1_SPTYP                BIT0  ///< Space Type (Memory)
#define V_SC_OTG_CFG_BAR1_SIZE                 (1 << 12)

#define R_SC_OTG_CFG_SSID                      0x2C  ///< Sub System ID
#define B_SC_OTG_CFG_SSID_SID                  0xFFFF0000 ///< Sub System ID
#define B_SC_OTG_CFG_SSID_SVID                 0x0000FFFF ///< Sub System Vendor ID

#define R_SC_OTG_CFG_PMECTLSTS                 0x84  ///< PME Control Status
#define B_SC_OTG_CFG_PMECTLSTS_POWERSTATE      (BIT1 | BIT0) ///< Power State

#define R_SC_XDCI_CFG_POW_PG_CONF              0xA0
#define B_SC_XDCI_CFG_POW_PG_CONF_D3HEN        BIT18   ///< D3-Hot Enable
#define B_SC_XDCI_CFG_POW_PG_CONF_DEVIDLEN     BIT17   ///< DEVIDLE Enable
//#define R_SC_XDCI_CFG_D0I3_MAX_POW_LAT_PG_CONFIG              0xA0
//#define B_SC_XDCI_CFG_POW_PG_CONF_D3HEN        BIT18   ///< D3-Hot Enable
//#define B_SC_XDCI_CFG_D0I3_MAX_POW_LAT_PG_CONFIG_I3_ENABLE    BIT17   ///< DEVIDLE Enable

#define R_SC_OTG_CFG_GEN_INPUT_REGRW           0xC0
#define B_SC_OTG_CFG_GEN_INPUT_REGRW_CPSU3     (BIT11 | BIT10) ///< Current Power State u3pmu
#define B_SC_OTG_CFG_GEN_INPUT_REGRW_CPSU2     (BIT9 | BIT8) ///< Current Power State u2pmu

//
// xDCI(OTG) MMIO registers
//
#define R_SC_XDCI_MEM_GCTL                     0xC110  ///< xDCI Global Ctrl
#define B_SC_XDCI_MEM_GCTL_GHIBEREN            BIT1    ///< Hibernation enable
#define R_SC_XDCI_MEM_GUSB2PHYCFG              0xC200  ///< Global USB2 PHY Configuration Register
#define B_SC_XDCI_MEM_GUSB2PHYCFG_SUSPHY       BIT6    ///< Suspend USB2.0 HS/FS/LS PHY
#define R_SC_XDCI_MEM_GUSB3PIPECTL0            0xC2C0  ///< Global USB3 PIPE Control Register 0
#define B_SC_XDCI_MEM_GUSB3PIPECTL0_SUSPENDEN  BIT17   ///< Suspend USB3 PHY
#define R_SC_XDCI_MEM_DCTL                     0xC704  ///< xDCI Device Ctrl
#define R_SC_OTG_MEM_APBFC_U3PMU_CFG0      0x10F808
#define R_SC_OTG_MEM_APBFC_U3PMU_CFG1      0x10F80C
#define R_SC_OTG_MEM_APBFC_U3PMU_CFG2      0x10F810
#define R_SC_OTG_MEM_APBFC_U3PMU_CFG3      0x10F814
#define R_SC_OTG_MEM_APBFC_U3PMU_CFG4      0x10F818
#define R_SC_OTG_MEM_APBFC_U3PMU_CFG5      0x10F81C
#define R_SC_OTG_MEM_APBFC_U3PMU_CFG6      0x10F820

//
// xDCI(OTG) Private Space
//
#define R_SC_OTG_PCR_IOSFCTL                   0x00  ///< IOSF Control
#define B_SC_OTG_PCR_IOSFCTL_NSNPDIS           BIT7  ///< Non-Snoop Disable

#define R_SC_OTG_PCR_PMCTL                     0x1D0 ///< Power Management Control
#define B_SC_OTG_PCR_PMCTL_IOSFSB_TG_EN        BIT5  ///< IOSF Sideband Trunk Gate Enable
#define B_SC_OTG_PCR_PMCTL_IOSFPRIM_TG_EN      BIT4  ///< IOSF Primary Trunk Gate Enable
#define B_SC_OTG_PCR_PMCTL_IOSFSB_CG_EN        BIT3  ///< IOSF Sideband Clock Gate Enable
#define B_SC_OTG_PCR_PMCTL_OCPCLK_TG_EN        BIT2  ///< OCP Clock Trunk Gate Enable
#define B_SC_OTG_PCR_PMCTL_OCPCLK_GATE_EN      BIT1  ///< OCP Clock Gate Enable
#define B_SC_OTG_PCR_PMCTL_IOSFPRIM_CG_EN      BIT0  ///< IOSF Clock Gate Enable

#define R_SC_OTG_PCR_PCICFGCTR1                0x200 ///< PCI Configuration Control 1
#define B_SC_OTG_PCR_PCICFGCTR1_IPIN1          (BIT11 | BIT10 | BIT9 | BIT8) ///< Interrupt Pin
#define B_SC_OTG_PCR_PCICFGCTR1_B1D1           BIT7  ///< BAR 1 Disable
#define B_SC_OTG_PCR_PCICFGCTR1_PS             0x7C  ///< PME Support
#define B_SC_OTG_PCR_PCICFGCTR1_ACPI_INT_EN1   BIT1  ///< ACPI Interrupt Enable
#define B_SC_OTG_PCR_PCICFGCTR1_PCI_CFG_DIS1   BIT0  ///< PCI Configuration Space Disable

#define R_SC_CDN_PCR_U1_POWER_STATE_DEFINITION               0x02
#define B_SC_CDN_PCR_TX_EN                                   (BIT2)

#define R_SC_CDN_PCR_VCO_START_CALIBRATION_START_POINT       0x56
#define B_SC_CDN_PCR_VCO_START_CALIBRATION_START_POINT_VALUE 0x1F

#define R_SC_CDN_PCR_PLL_CONTROL                             0x3C2
#define R_SC_CDN_PCR_PLL_VCO_CALIBRATION_TRIM_CODE           (BIT6 | BIT5 | BIT4)

///
///USB2 PHY (USB2SHIP) side band registers
///
#define USB2_PHY_PORT_ID                 0xA7  ///< USB2 PHY Private Space PortID
#define USB2_PHY_READ_OPCODE             0x06  ///< Read Opcode
#define USB2_PHY_WRITE_OPCODE            0x07  ///< Write Opcode

#define USH_PHY_PORT_ID                  0xA5  ///< USB3 Mod PHY
#define USH_PHY_READ_OPCODE              0x06  ///< Read Opcode
#define USH_PHY_WRITE_OPCODE             0x07  ///< Write Opcode

#define HSIC_PORT_ID                     0xA1  ///< HSIC PHY
#define HSIC_PORT_READ_OPCODE            0x06  ///< Read Opcode
#define HSIC_PORT_WRITE_OPCODE           0x07  ///< Write Opcode

#define SSIC_PORT_ID                     0xAB  ///< SSIC PHY  (from BXT E0 and BXT P)
#define SSIC_PORT_READ_OPCODE            0x06  ///< Read Opcode
#define SSIC_PORT_WRITE_OPCODE           0x07  ///< Write Opcode

#define OTG_PORT_ID                      0xA4  ///< OTG Private Space PortID
#define OTG_PRIVATE_READ_OPCODE          0x06  ///< CUnit to OTG Private Space Read Opcode
#define OTG_PRIVATE_WRITE_OPCODE         0x07  ///< CUnit to OTG Private Space Write Opcode
#define R_SC_USB_SB_PCR_PCICFGCTRL_XDCI         0x200

//
//-------------------------------USB2 PHY register definition------------------------------------
//
#define R_SC_USB2_PCR_PHY_LANE_BASE_ADDR 0x4000    ///< PP LANE base address
#define R_SC_USB2_PCR_GLOBAL_PORT        0x4001    ///< USB2 GLOBAL PORT
#define R_SC_USB2_PCR_GLB_ADP_VBUS_REG   0x402B    ///< GLB ADP VBUS REG
#define R_SC_USB2_PCR_GLOBAL_PORT_2      0x402C    ///< USB2 GLOBAL PORT 2
#define R_SC_USB2_PCR_CFG_COMPBG         0x7F04    ///< USB2 COMPBG
#define R_SC_USB2_PCR_CONFIG_3           0x7014    ///< USB2 CONFIG 3
#define R_SC_USB2_PCR_DFT_1              0x7024    ///< USB2 DFT_1
#define R_SC_USB2_PCR_SFRCONFIG_0        0x702C    ///< USB2 SFRCONFIG_0
#define R_SC_USB2_PCR_PPORT              0x00      ///< USB2 PER PORT
#define R_SC_UTMI_PCR_MISC_PR            0x08      ///< UTMI MISC REG PER PORT
#define R_SC_USB2_PCR_PPORT2             0x26      ///< USB2 PER PORT 2
//
//-------------------------------USB DBC Trace Register Definition
//
#define DBC_TRACE_IN_PAYLOAD_BP_LOW                   0x50
#define DBC_TRACE_IN_PAYLOAD_BP_HI                    0x54
#define DBC_TRACE_IN_PAYLOAD_QUALIFIERS               0x58
#define DBC_TRACE_IN_STATUS_QUALIFIERS                0x68
#define DBC_TRACE_IN_STATUS_BP_LOW                    0x60
#define DBC_TRACE_IN_STATUS_BP_HI                     0x64
#define DBC_TRACE_IN_QUALIFIERS_VALUE                 0x00219000
#define DBC_TRACE_IN_PAYLOAD_BP_VALUE_OFFSET          0x80000
#define DBC_TRACE_IN_STATUS_BP_VALUE_OFFSET           0xA1008
#endif
