/** @file
  This file is PeiScPolicy library.

 @copyright
  INTEL CONFIDENTIAL
  Copyright 2014 - 2019 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

#include "PeiScPolicyLibrary.h"
#include <Library/SteppingLib.h>

GLOBAL_REMOVE_IF_UNREFERENCED UINT8 mSmbusRsvdAddresses[] = {
  0xA0,
  0xA2,
  0xA4,
  0xA6
};

GLOBAL_REMOVE_IF_UNREFERENCED PRIVATE_PCICFGCTRL mDirectIrqTable[] = {
  { SB_LPSS_PORT,        27, R_SC_LPSS_PCR_PCICFGCTRL_I2C0 , V_PCICFG_CTRL_INTA, {0, 0, 0} },
  { SB_LPSS_PORT,        28, R_SC_LPSS_PCR_PCICFGCTRL_I2C1 , V_PCICFG_CTRL_INTB, {0, 0, 0} },
  { SB_LPSS_PORT,        29, R_SC_LPSS_PCR_PCICFGCTRL_I2C2 , V_PCICFG_CTRL_INTC, {0, 0, 0} },
  { SB_LPSS_PORT,        30, R_SC_LPSS_PCR_PCICFGCTRL_I2C3 , V_PCICFG_CTRL_INTD, {0, 0, 0} },
  { SB_LPSS_PORT,        31, R_SC_LPSS_PCR_PCICFGCTRL_I2C4 , V_PCICFG_CTRL_INTA, {0, 0, 0} },
  { SB_LPSS_PORT,        32, R_SC_LPSS_PCR_PCICFGCTRL_I2C5 , V_PCICFG_CTRL_INTB, {0, 0, 0} },
  { SB_LPSS_PORT,        33, R_SC_LPSS_PCR_PCICFGCTRL_I2C6 , V_PCICFG_CTRL_INTC, {0, 0, 0} },
  { SB_LPSS_PORT,        34, R_SC_LPSS_PCR_PCICFGCTRL_I2C7 , V_PCICFG_CTRL_INTD, {0, 0, 0} },
  { SB_LPSS_PORT,         4, R_SC_LPSS_PCR_PCICFGCTRL_UART0, V_PCICFG_CTRL_INTA, {0, 0, 0} },
  { SB_LPSS_PORT,         5, R_SC_LPSS_PCR_PCICFGCTRL_UART1, V_PCICFG_CTRL_INTB, {0, 0, 0} },
  { SB_LPSS_PORT,         6, R_SC_LPSS_PCR_PCICFGCTRL_UART2, V_PCICFG_CTRL_INTC, {0, 0, 0} },
  { SB_LPSS_PORT,         7, R_SC_LPSS_PCR_PCICFGCTRL_UART3, V_PCICFG_CTRL_INTD, {0, 0, 0} },
  { SB_LPSS_PORT,        35, R_SC_LPSS_PCR_PCICFGCTRL_SPI0 , V_PCICFG_CTRL_INTA, {0, 0, 0} },
  { SB_LPSS_PORT,        36, R_SC_LPSS_PCR_PCICFGCTRL_SPI1 , V_PCICFG_CTRL_INTB, {0, 0, 0} },
  { SB_LPSS_PORT,        37, R_SC_LPSS_PCR_PCICFGCTRL_SPI2 , V_PCICFG_CTRL_INTC, {0, 0, 0} },
  { SB_SCC_PORT,          3, R_SC_SCC_PCR_PCICFGCTRL_SDCARD, V_PCICFG_CTRL_INTA, {0, 0, 0} },
  { SB_SCC_PORT,         39, R_SC_SCC_PCR_PCICFGCTRL_EMMC  , V_PCICFG_CTRL_INTA, {0, 0, 0} },
  { SB_SCC_PORT,         38, R_SC_SCC_PCR_PCICFGCTRL_UFS   , V_PCICFG_CTRL_INTA, {0, 0, 0} },
  { SB_SCC_PORT,         42, R_SC_SCC_PCR_PCICFGCTRL_SDIO  , V_PCICFG_CTRL_INTA, {0, 0, 0} },
  { SB_ISH_BRIDGE_PORT,  26, R_SC_ISH_PCR_PCICFGCTRL_ISH   , V_PCICFG_CTRL_INTA, {0, 0, 0} },
  { SB_USB_DEVICE_PORT,  13, R_SC_USB_SB_PCR_PCICFGCTRL_XDCI, V_PCICFG_CTRL_INTB, {0, 0, 0} },
};

/**
  mDevIntConfig[] table contains data on INTx and IRQ for each device.


  PCI Express Root Ports mapping should be programmed only with values as in below table (D19/20)
  otherwise _PRT methods in ACPI for RootPorts would require additional patching as
  PCIe Endpoint Device Interrupt is further subjected to INTx to PIRQy Mapping

  Configured IRQ values are not used if an OS chooses to be in PIC instead of APIC mode
**/
GLOBAL_REMOVE_IF_UNREFERENCED SC_DEVICE_INTERRUPT_CONFIG mDevIntConfig[] = {
  {19, 0, ScIntA, 16}, // PCI Express Port 3, INT is default, programmed in PciCfgSpace + FCh
  {19, 1, ScIntB, 17}, // PCI Express Port 4, INT is default, programmed in PciCfgSpace + FCh
  {19, 2, ScIntC, 18}, // PCI Express Port 5, INT is default, programmed in PciCfgSpace + FCh
  {19, 3, ScIntD, 19}, // PCI Express Port 6, INT is default, programmed in PciCfgSpace + FCh
  {20, 0, ScIntA, 16}, // PCI Express Port 1 (APL Only), INT is default, programmed in PciCfgSpace + FCh
  {20, 1, ScIntB, 17}, // PCI Express Port 2 (APL Only), INT is default, programmed in PciCfgSpace + FCh
};

/**
  Load Config block default

  @param[in] ConfigBlockPointer         Pointer to config block
**/
VOID
LoadScGeneralConfigDefault (
  IN VOID         *ConfigBlockPointer
  )
{
  SC_GENERAL_CONFIG *ScGeneralConfig;

  ScGeneralConfig = ConfigBlockPointer;

  DEBUG ((DEBUG_INFO, "ScGeneralConfig->Header.GuidHob.Name = %g\n", &ScGeneralConfig->Header.GuidHob.Name));
  DEBUG ((DEBUG_INFO, "ScGeneralConfig->Header.GuidHob.Header.HobLength = 0x%x\n", ScGeneralConfig->Header.GuidHob.Header.HobLength));

  /********************************
    General Config Configuration
  ********************************/
  ScGeneralConfig->SubSystemVendorId = V_INTEL_VENDOR_ID;
  ScGeneralConfig->SubSystemId       = V_SC_DEFAULT_SID;
  ScGeneralConfig->AcpiBase          = (UINT16)PcdGet16(PcdScAcpiIoPortBaseAddress);
  ScGeneralConfig->PmcBase           = (UINT32)PcdGet32(PcdPmcIpc1BaseAddress0);
  ScGeneralConfig->P2sbBase          = (UINT32)PcdGet32(PcdP2SBBaseAddress);
}

/**
  Load Config block default

  @param[in] ConfigBlockPointer         Pointer to config block
**/
VOID
LoadSataConfigDefault (
  IN VOID         *ConfigBlockPointer
  )
{
  SC_SATA_CONFIG  *SataConfig;
  UINT8           PortIndex;

  SataConfig = ConfigBlockPointer;

  DEBUG ((DEBUG_INFO, "SataConfig->Header.GuidHob.Name = %g\n", &SataConfig->Header.GuidHob.Name));
  DEBUG ((DEBUG_INFO, "SataConfig->Header.GuidHob.Header.HobLength = 0x%x\n", SataConfig->Header.GuidHob.Header.HobLength));

  /********************************
    SATA Configuration
  ********************************/
  SataConfig->Enable      = TRUE;
  SataConfig->SalpSupport = TRUE;
  SataConfig->SataMode    = ScSataModeAhci;
  SataConfig->SpeedLimit  = ScSataSpeedDefault;

  for (PortIndex = 0; PortIndex < SC_MAX_SATA_PORTS; PortIndex++) {
    SataConfig->PortSettings[PortIndex].Enable  = TRUE;
    SataConfig->PortSettings[PortIndex].DmVal   = 15;
    SataConfig->PortSettings[PortIndex].DitoVal = 625;
  }
}

/**
  Load Config block default

  @param[in] ConfigBlockPointer         Pointer to config block
**/
VOID
LoadPcieConfigDefault (
  IN VOID          *ConfigBlockPointer
  )
{
  SC_PCIE_CONFIG  *PcieConfig;
  UINT8           PortIndex;

  PcieConfig = ConfigBlockPointer;

  DEBUG ((DEBUG_INFO, "PcieConfig->Header.GuidHob.Name = %g\n", &PcieConfig->Header.GuidHob.Name));
  DEBUG ((DEBUG_INFO, "PcieConfig->Header.GuidHob.Header.HobLength = 0x%x\n", PcieConfig->Header.GuidHob.Header.HobLength));

  /********************************
    PCIe Configuration
  ********************************/
  for (PortIndex = 0; PortIndex < GetScMaxPciePortNum (); PortIndex++) {
    PcieConfig->RootPort[PortIndex].Aspm                 = ScPcieAspmAutoConfig;
    PcieConfig->RootPort[PortIndex].Enable               = TRUE;
    PcieConfig->RootPort[PortIndex].SlotImplemented      = TRUE;
    PcieConfig->RootPort[PortIndex].PmSci                = TRUE;
    PcieConfig->RootPort[PortIndex].AcsEnabled           = TRUE;
    PcieConfig->RootPort[PortIndex].PhysicalSlotNumber   = PortIndex;
    PcieConfig->RootPort[PortIndex].L1Substates          = ScPcieL1SubstatesL1_1_2;
    PcieConfig->RootPort[PortIndex].SelectableDeemphasis = TRUE;
    //
    // PCIe LTR Configuration.
    //
    PcieConfig->RootPort[PortIndex].LtrEnable                         = TRUE;
    PcieConfig->RootPort[PortIndex].LtrMaxSnoopLatency                = 0x1003;
    PcieConfig->RootPort[PortIndex].LtrMaxNoSnoopLatency              = 0x1003;
    PcieConfig->RootPort[PortIndex].SnoopLatencyOverrideMode          = 2;
    PcieConfig->RootPort[PortIndex].SnoopLatencyOverrideMultiplier    = 2;
    PcieConfig->RootPort[PortIndex].SnoopLatencyOverrideValue         = 60;
    PcieConfig->RootPort[PortIndex].NonSnoopLatencyOverrideMode       = 2;
    PcieConfig->RootPort[PortIndex].NonSnoopLatencyOverrideMultiplier = 2;
    PcieConfig->RootPort[PortIndex].NonSnoopLatencyOverrideValue      = 60;
  }
  PcieConfig->AspmSwSmiNumber = SW_SMI_PCIE_ASPM_OVERRIDE;
}

/**
  Load Config block default

  @param[in] ConfigBlockPointer         Pointer to config block
**/
VOID
LoadUsbConfigDefault (
  IN VOID         *ConfigBlockPointer
  )
{
  SC_USB_CONFIG  *UsbConfig;
  UINT32         PortIndex;

  UsbConfig = ConfigBlockPointer;

  DEBUG ((DEBUG_INFO, "UsbConfig->Header.GuidHob.Name = %g\n", &UsbConfig->Header.GuidHob.Name));
  DEBUG ((DEBUG_INFO, "UsbConfig->Header.GuidHob.Header.HobLength = 0x%x\n", UsbConfig->Header.GuidHob.Header.HobLength));

  /********************************
    USB Configuration
  ********************************/
  UsbConfig->Usb30Settings.Mode = XHCI_MODE_ON;
  for (PortIndex = 0; PortIndex < GetScXhciMaxUsb2PortNum (); PortIndex++) {
    UsbConfig->PortUsb20[PortIndex].Enable = TRUE;
  }
  for (PortIndex = 0; PortIndex < GetScXhciMaxUsb3PortNum (); PortIndex++) {
    UsbConfig->PortUsb30[PortIndex].Enable = TRUE;
  }
  UsbConfig->XdciConfig.Enable = ScPciMode;
#if (GLK_SI_OVERRIDE_ENABLE == 0)
  UsbConfig->SsicConfig.SsicPort[0].Enable = FALSE;
  UsbConfig->SsicConfig.SsicPort[1].Enable = FALSE;
  UsbConfig->SsicConfig.SsicPort[0].Rate   = XhciSsicRateA;
  UsbConfig->SsicConfig.SsicPort[1].Rate   = XhciSsicRateA;
  UsbConfig->SsicConfig.DlanePwrGating     = TRUE;
#endif
  //
  // BIOS should program PDO in PEI phase by default
  //
  UsbConfig->PdoProgramming = TRUE;
}

/**
  Load Config block default

  @param[in] ConfigBlockPointer         Pointer to config block
**/
VOID
LoadHpetConfigDefault (
  IN VOID         *ConfigBlockPointer
  )
{
  SC_HPET_CONFIG  *HpetConfig;

  HpetConfig = ConfigBlockPointer;

  DEBUG ((DEBUG_INFO, "HpetConfig->Header.GuidHob.Name = %g\n", &HpetConfig->Header.GuidHob.Name));
  DEBUG ((DEBUG_INFO, "HpetConfig->Header.GuidHob.Header.HobLength = 0x%x\n", HpetConfig->Header.GuidHob.Header.HobLength));

  /********************************
    HPET Configuration
  ********************************/
  HpetConfig->Enable = TRUE;
  HpetConfig->Base   = SC_HPET_BASE_ADDRESS;
}

/**
  Load Config block default

  @param[in] ConfigBlockPointer         Pointer to config block
**/
VOID
LoadSmbusConfigDefault (
  IN VOID         *ConfigBlockPointer
  )
{
  SC_SMBUS_CONFIG  *SmbusConfig;

  SmbusConfig = ConfigBlockPointer;

  DEBUG ((DEBUG_INFO, "SmbusConfig->Header.GuidHob.Name = %g\n", &SmbusConfig->Header.GuidHob.Name));
  DEBUG ((DEBUG_INFO, "SmbusConfig->Header.GuidHob.Header.HobLength = 0x%x\n", SmbusConfig->Header.GuidHob.Header.HobLength));

  /********************************
    SMBus Configuration
  ********************************/
  SmbusConfig->Enable      = TRUE;
  SmbusConfig->SmbusIoBase = PcdGet16(PcdSmbusBaseAddress);
  ASSERT (sizeof (mSmbusRsvdAddresses) <= SC_MAX_SMBUS_RESERVED_ADDRESS);
  SmbusConfig->NumRsvdSmbusAddresses = sizeof (mSmbusRsvdAddresses);
  CopyMem (
    SmbusConfig->RsvdSmbusAddressTable,
    mSmbusRsvdAddresses,
    sizeof (mSmbusRsvdAddresses)
    );
}

/**
  Load Config block default

  @param[in] ConfigBlockPointer         Pointer to config block
**/
VOID
LoadIoApicConfigDefault (
  IN VOID         *ConfigBlockPointer
  )
{
  SC_IOAPIC_CONFIG  *IoApicConfig;

  IoApicConfig = ConfigBlockPointer;

  DEBUG ((DEBUG_INFO, "IoApicConfig->Header.GuidHob.Name = %g\n", &IoApicConfig->Header.GuidHob.Name));
  DEBUG ((DEBUG_INFO, "IoApicConfig->Header.GuidHob.Header.HobLength = 0x%x\n", IoApicConfig->Header.GuidHob.Header.HobLength));

  /********************************
    IOAPIC Configuration
  ********************************/
}

/**
  Load Config block default

  @param[in] ConfigBlockPointer         Pointer to config block
**/
VOID
LoadHdaConfigDefault (
  IN VOID         *ConfigBlockPointer
  )
{
  SC_HDAUDIO_CONFIG   *HdaConfig;

  HdaConfig = ConfigBlockPointer;

  DEBUG ((DEBUG_INFO, "HdaConfig->Header.GuidHob.Name = %g\n", &HdaConfig->Header.GuidHob.Name));
  DEBUG ((DEBUG_INFO, "HdaConfig->Header.GuidHob.Header.HobLength = 0x%x\n", HdaConfig->Header.GuidHob.Header.HobLength));

  /********************************
    HD-Audio Configuration
  ********************************/
  HdaConfig->Enable                = TRUE;
  HdaConfig->DspEnable             = TRUE;
  HdaConfig->Mmt                   = ScHdaVc0;
  HdaConfig->Hmt                   = ScHdaVc0;
  HdaConfig->PwrGate               = TRUE;
  HdaConfig->ClkGate               = TRUE;
  HdaConfig->Pme                   = TRUE;
  HdaConfig->IoBufferOwnership     = ScHdaIoBufOwnerI2sPort;
  HdaConfig->VcType                = ScHdaVc0;
  HdaConfig->HdAudioLinkFrequency  = ScHdaLinkFreq24MHz;
  HdaConfig->IDispLinkFrequency    = ScHdaLinkFreq96MHz;
  HdaConfig->IDispLinkTmode        = ScHdaIDispMode2T;
  HdaConfig->ResetWaitTimer        = 300;
  HdaConfig->I2sConfiguration      = 1;
}

/**
  Load Config block default

  @param[in] ConfigBlockPointer         Pointer to config block
**/
VOID
LoadGmmConfigDefault (
  IN VOID         *ConfigBlockPointer
  )
{
  SC_GMM_CONFIG  *GmmConfig;

  GmmConfig = ConfigBlockPointer;

  DEBUG ((DEBUG_INFO, "GmmConfig->Header.GuidHob.Name = %g\n", &GmmConfig->Header.GuidHob.Name));
  DEBUG ((DEBUG_INFO, "GmmConfig->Header.GuidHob.Header.HobLength = 0x%x\n", GmmConfig->Header.GuidHob.Header.HobLength));

  /********************************
    GMM Configuration
  ********************************/
  GmmConfig->ClkGatingPgcbClkTrunk   = TRUE;
  GmmConfig->ClkGatingSb             = TRUE;
  GmmConfig->ClkGatingSbClkTrunk     = TRUE;
  GmmConfig->ClkGatingSbClkPartition = TRUE;
  GmmConfig->ClkGatingCore           = TRUE;
  GmmConfig->ClkGatingDma            = TRUE;
  GmmConfig->ClkGatingRegAccess      = TRUE;
  GmmConfig->ClkGatingHost           = TRUE;
  GmmConfig->ClkGatingPartition      = TRUE;
  GmmConfig->ClkGatingTrunk          = TRUE;
}

/**
  Load Config block default

  @param[in] ConfigBlockPointer         Pointer to config block
**/
VOID
LoadPmConfigDefault (
  IN VOID         *ConfigBlockPointer
  )
{
  SC_PM_CONFIG  *PmConfig;

  PmConfig = ConfigBlockPointer;

  DEBUG ((DEBUG_INFO, "PmConfig->Header.GuidHob.Name = %g\n", &PmConfig->Header.GuidHob.Name));
  DEBUG ((DEBUG_INFO, "PmConfig->Header.GuidHob.Header.HobLength = 0x%x\n", PmConfig->Header.GuidHob.Header.HobLength));

  /********************************
    PM Configuration
  ********************************/
#if PPV_ENABLE
  PmConfig->PciClockRun        = FALSE;
#else
  PmConfig->PciClockRun        = TRUE;
#endif
  PmConfig->Timer8254ClkGateEn = TRUE;
}

/**
  Load Config block default

  @param[in] ConfigBlockPointer         Pointer to config block
**/
VOID
LoadLockDownConfigDefault (
  IN VOID         *ConfigBlockPointer
  )
{
  SC_LOCK_DOWN_CONFIG *LockDownConfig;

  LockDownConfig = ConfigBlockPointer;

  DEBUG ((DEBUG_INFO, "LockDownConfig->Header.GuidHob.Name = %g\n", &LockDownConfig->Header.GuidHob.Name));
  DEBUG ((DEBUG_INFO, "LockDownConfig->Header.GuidHob.Header.HobLength = 0x%x\n", LockDownConfig->Header.GuidHob.Header.HobLength));

  /********************************
    Lock Down Configuration
  ********************************/

  LockDownConfig->GlobalSmi           = TRUE;
  //
  // Flash Security Recommendations,
  // Intel strongly recommends that BIOS sets the BIOS Interface Lock Down bit. Enabling this bit
  // will mitigate malicious software attempts to replace the system BIOS option ROM with its own code.
  // Here we always enable this as a Policy.
  //
  LockDownConfig->BiosInterface       = TRUE;
  LockDownConfig->RtcLock             = TRUE;
  LockDownConfig->BiosLockSwSmiNumber = SW_SMI_BIOS_LOCK;
  LockDownConfig->TcoLock             = TRUE;
}

/**
  Load Config block default

  @param[in] ConfigBlockPointer         Pointer to config block
**/
VOID
LoadSerialIrqConfigDefault (
  IN VOID         *ConfigBlockPointer
  )
{
  SC_LPC_SIRQ_CONFIG *SerialIrqConfig;

  SerialIrqConfig = ConfigBlockPointer;

  DEBUG ((DEBUG_INFO, "SerialIrqConfig->Header.GuidHob.Name = %g\n", &SerialIrqConfig->Header.GuidHob.Name));
  DEBUG ((DEBUG_INFO, "SerialIrqConfig->Header.GuidHob.Header.HobLength = 0x%x\n", SerialIrqConfig->Header.GuidHob.Header.HobLength));

  /********************************
    Serial IRQ Configuration
  ********************************/
  SerialIrqConfig->SirqEnable      = TRUE;
  SerialIrqConfig->SirqMode        = ScQuietMode;
  SerialIrqConfig->StartFramePulse = ScSfpw4Clk;
}

/**
  Load Config block default

  @param[in] ConfigBlockPointer         Pointer to config block
**/
VOID
LoadLpssConfigDefault (
  IN VOID         *ConfigBlockPointer
  )
{
  SC_LPSS_CONFIG  *LpssConfig;
  UINT8           Index;

  LpssConfig = ConfigBlockPointer;

  DEBUG ((DEBUG_INFO, "LpssConfig->Header.GuidHob.Name = %g\n", &LpssConfig->Header.GuidHob.Name));
  DEBUG ((DEBUG_INFO, "LpssConfig->Header.GuidHob.Header.HobLength = 0x%x\n", LpssConfig->Header.GuidHob.Header.HobLength));

  /********************************
    LPSS Configuration
  ********************************/
  LpssConfig->I2c0Enable    = ScPciMode;
  LpssConfig->I2c1Enable    = ScPciMode;
  LpssConfig->I2c2Enable    = ScPciMode;
  LpssConfig->I2c3Enable    = ScPciMode;
  LpssConfig->I2c4Enable    = ScPciMode;
  LpssConfig->I2c5Enable    = ScPciMode;
  LpssConfig->I2c6Enable    = ScPciMode;
  LpssConfig->I2c7Enable    = ScPciMode;
  LpssConfig->Hsuart0Enable = ScPciMode;
  LpssConfig->Hsuart1Enable = ScPciMode;
  LpssConfig->Hsuart2Enable = ScPciMode;
  LpssConfig->Hsuart3Enable = ScPciMode;
  LpssConfig->Spi0Enable    = ScPciMode;
  LpssConfig->Spi1Enable    = ScPciMode;
  LpssConfig->Spi2Enable    = ScPciMode;
  for (Index = 0; Index < LPSS_I2C_DEVICE_NUM; Index++) {
    LpssConfig->I2cClkGateCfg[Index] = TRUE;
  }
  for (Index = 0; Index < LPSS_HSUART_DEVICE_NUM; Index++) {
    LpssConfig->HsuartClkGateCfg[Index] = TRUE;
  }
  for (Index = 0; Index < LPSS_SPI_DEVICE_NUM; Index++) {
    LpssConfig->SpiClkGateCfg[Index] = TRUE;
  }
  LpssConfig->HostUartHide = 0;
  LpssConfig->SkipSpiPCP = 0;
}

/**
  Load Config block default

  @param[in] ConfigBlockPointer         Pointer to config block
**/
VOID
LoadScsConfigDefault (
  IN VOID         *ConfigBlockPointer
  )
{
  SC_SCS_CONFIG  *ScsConfig;

  ScsConfig = ConfigBlockPointer;

  DEBUG ((DEBUG_INFO, "ScsConfig->Header.GuidHob.Name = %g\n", &ScsConfig->Header.GuidHob.Name));
  DEBUG ((DEBUG_INFO, "ScsConfig->Header.GuidHob.Header.HobLength = 0x%x\n", ScsConfig->Header.GuidHob.Header.HobLength));

  /********************************
    SCS Configuration
  ********************************/
  ScsConfig->SdcardEnable     = ScPciMode;
  ScsConfig->SdioEnable       = ScPciMode;
  ScsConfig->EmmcEnable       = ScPciMode;
  ScsConfig->EmmcHostMaxSpeed = SccEmmcHs400;
  ScsConfig->GppLock          = 0;
  ScsConfig->SccEmmcTraceLength = SCC_EMMC_LONG_TRACE_LEN;
  ScsConfig->SdioRegDllConfig.TxCmdCntl      = 0x505;
  ScsConfig->SdioRegDllConfig.TxDataCntl1    = 0xE;
  ScsConfig->SdioRegDllConfig.TxDataCntl2    = 0x22272828;
  ScsConfig->SdioRegDllConfig.RxCmdDataCntl1 = 0x16161616;
  ScsConfig->SdioRegDllConfig.RxCmdDataCntl2 = 0x10000;
  ScsConfig->SdcardRegDllConfig.TxCmdCntl    = 0x505;
  if (GetBxtSeries () == Bxt1) {
    ScsConfig->SdcardRegDllConfig.TxDataCntl1    = 0xA12;
    ScsConfig->SdcardRegDllConfig.TxDataCntl2    = 0x26272727;
    ScsConfig->SdcardRegDllConfig.RxCmdDataCntl1 = 0x0B483B3B;
  } else {
    ScsConfig->SdcardRegDllConfig.TxDataCntl1    = 0xA13;
    ScsConfig->SdcardRegDllConfig.TxDataCntl2    = 0x24242828;
    ScsConfig->SdcardRegDllConfig.RxCmdDataCntl1 = 0x73A3637;
  }
  ScsConfig->SdcardRegDllConfig.RxStrobeCntl   = 0x0;
  ScsConfig->SdcardRegDllConfig.RxCmdDataCntl2 = 0x10000;

  if (ScsConfig->SccEmmcTraceLength == SCC_EMMC_SHORT_TRACE_LEN) {
    // Configure DLL settings for short trace length
    ScsConfig->EmmcRegDllConfig.TxCmdCntl      = 0x502;
    ScsConfig->EmmcRegDllConfig.TxDataCntl1    = 0x815;
    ScsConfig->EmmcRegDllConfig.TxDataCntl2    = 0x1C1C1C00;
    ScsConfig->EmmcRegDllConfig.RxCmdDataCntl1 = 0x1C1C1C00;
    ScsConfig->EmmcRegDllConfig.RxStrobeCntl   = 0x0a0a;
    ScsConfig->EmmcRegDllConfig.RxCmdDataCntl2 = 0x1001C;
    ScsConfig->EmmcRegDllConfig.MasterSwCntl   = 0x001;
  } else {
    // Configure DLL settings for long trace length
    ScsConfig->EmmcRegDllConfig.TxCmdCntl      = 0x502;
    ScsConfig->EmmcRegDllConfig.TxDataCntl1    = 0x811;
    ScsConfig->EmmcRegDllConfig.TxDataCntl2    = 0x1C2A2927;
    ScsConfig->EmmcRegDllConfig.RxCmdDataCntl1 = 0x000D162F;
    ScsConfig->EmmcRegDllConfig.RxStrobeCntl   = 0x0a0a;
    ScsConfig->EmmcRegDllConfig.RxCmdDataCntl2 = 0x1003b;
    ScsConfig->EmmcRegDllConfig.MasterSwCntl   = 0x001;
  }
}

/**
  Load Config block default

  @param[in] ConfigBlockPointer         Pointer to config block
**/
VOID
LoadVtdConfigDefault (
  IN VOID         *ConfigBlockPointer
  )
{
  SC_VTD_CONFIG  *VtdConfig;

  VtdConfig = ConfigBlockPointer;

  DEBUG ((DEBUG_INFO, "VtdConfig->Header.GuidHob.Name = %g\n", &VtdConfig->Header.GuidHob.Name));
  DEBUG ((DEBUG_INFO, "VtdConfig->Header.GuidHob.Header.HobLength = 0x%x\n", VtdConfig->Header.GuidHob.Header.HobLength));

  /********************************
    VT-d Configuration
  ********************************/
  VtdConfig->RmrrUsbBaseAddr = 0;
  VtdConfig->RmrrUsbLimit = 0;
}

/**
  Load Config block default

  @param[in] ConfigBlockPointer         Pointer to config block
**/
VOID
LoadIshConfigDefault (
  IN VOID         *ConfigBlockPointer
  )
{
  SC_ISH_CONFIG  *IshConfig;

  IshConfig = ConfigBlockPointer;

  DEBUG ((DEBUG_INFO, "IshConfig->Header.GuidHob.Name = %g\n", &IshConfig->Header.GuidHob.Name));
  DEBUG ((DEBUG_INFO, "IshConfig->Header.GuidHob.Header.HobLength = 0x%x\n", IshConfig->Header.GuidHob.Header.HobLength));

  /********************************
    ISH Configuration
  ********************************/
  IshConfig->Enable = TRUE;
}


#if (CNVI_ENABLE == 1)
VOID
LoadCnviConfigDefault (
  IN VOID         *ConfigBlockPointer
  )
{
  SC_CNVI_CONFIG  *CnviConfig;
    
  CnviConfig = ConfigBlockPointer;

  DEBUG ((DEBUG_INFO, "CnviConfig->Header.GuidHob.Name = %g\n", &CnviConfig->Header.GuidHob.Name));
  DEBUG ((DEBUG_INFO, "CnviConfig->Header.GuidHob.Header.HobLength = 0x%x\n", CnviConfig->Header.GuidHob.Header.HobLength));

  /********************************
    CNVi Configuration
  ********************************/
  CnviConfig->Mode           = CnviModeAuto;
  CnviConfig->BtCore         = 1;
  CnviConfig->BtInterface    = CnviBtIfUsb;
  CnviConfig->ModifyCrfGpios = 0;
}
#endif

/**
  Load Config block default

  @param[in] ConfigBlockPointer         Pointer to config block
**/
VOID
LoadFlashProtectionConfigDefault (
  IN VOID          *ConfigBlockPointer
  )
{
  SC_FLASH_PROTECTION_CONFIG  *FlashProtectionConfig;

  FlashProtectionConfig = ConfigBlockPointer;

  DEBUG ((DEBUG_INFO, "FlashProtectionConfig->Header.GuidHob.Name = %g\n", &FlashProtectionConfig->Header.GuidHob.Name));
  DEBUG ((DEBUG_INFO, "FlashProtectionConfig->Header.GuidHob.Header.HobLength = 0x%x\n", FlashProtectionConfig->Header.GuidHob.Header.HobLength));

  /********************************
    Flash Protection Configuration
  ********************************/
}

/**
  Load Config block default

  @param[in] ConfigBlockPointer         Pointer to config block
**/
VOID
LoadDciConfigDefault (
  IN VOID         *ConfigBlockPointer
  )
{
  SC_DCI_CONFIG  *DciConfig;

  DciConfig = ConfigBlockPointer;

  DEBUG ((DEBUG_INFO, "DciConfig->Header.GuidHob.Name = %g\n", &DciConfig->Header.GuidHob.Name));
  DEBUG ((DEBUG_INFO, "DciConfig->Header.GuidHob.Header.HobLength = 0x%x\n", DciConfig->Header.GuidHob.Header.HobLength));

  /********************************
    DCI Configuration
  ********************************/
  DciConfig->DciAutoDetect = TRUE;
}

/**
  Load Config block default

  @param[in] ConfigBlockPointer         Pointer to config block
**/

VOID
LoadP2sbConfigDefault (
  IN VOID         *ConfigBlockPointer
  )
{
  SC_P2SB_CONFIG  *P2sbConfig;

  P2sbConfig = ConfigBlockPointer;

  DEBUG ((DEBUG_INFO, "P2sbConfig->Header.GuidHob.Name = %g\n", &P2sbConfig->Header.GuidHob.Name));
  DEBUG ((DEBUG_INFO, "P2sbConfig->Header.GuidHob.Header.HobLength = 0x%x\n", P2sbConfig->Header.GuidHob.Header.HobLength));

  /********************************
    P2SB Configuration
  ********************************/

  P2sbConfig->P2sbUnhide = 0;
}

/**
  Load Config block default

  @param[in] ConfigBlockPointer         Pointer to config block
**/
VOID
LoadInterruptConfigDefault (
  IN VOID         *ConfigBlockPointer
  )
{
  UINT8                IntConfigTableEntries;
  UINT8                DirectIrqTableEntries;
  SC_INTERRUPT_CONFIG  *InterruptConfig;

  InterruptConfig = ConfigBlockPointer;

  DEBUG ((DEBUG_INFO, "InterruptConfig->Header.GuidHob.Name = %g\n", &InterruptConfig->Header.GuidHob.Name));
  DEBUG ((DEBUG_INFO, "InterruptConfig->Header.GuidHob.Header.HobLength = 0x%x\n", InterruptConfig->Header.GuidHob.Header.HobLength));

  /********************************
    Interrupt Configuration
  ********************************/

  DirectIrqTableEntries = sizeof (mDirectIrqTable) / sizeof (PRIVATE_PCICFGCTRL);
  ASSERT (DirectIrqTableEntries <= SC_MAX_DIRECT_IRQ_CONFIG);
  InterruptConfig->NumOfDirectIrqTable = DirectIrqTableEntries;
  CopyMem (
    InterruptConfig->DirectIrqTable,
    mDirectIrqTable,
    sizeof (mDirectIrqTable)
    );

  IntConfigTableEntries = sizeof (mDevIntConfig) / sizeof (SC_DEVICE_INTERRUPT_CONFIG);
  ASSERT (IntConfigTableEntries <= SC_MAX_DEVICE_INTERRUPT_CONFIG);
  InterruptConfig->NumOfDevIntConfig = IntConfigTableEntries;
  CopyMem (
    InterruptConfig->DevIntConfig,
    mDevIntConfig,
    sizeof (mDevIntConfig)
    );

  InterruptConfig->PxRcRouting[0] = V_SC_ITSS_PCR_REN_ENABLE + V_SC_ITSS_PCR_IR_IRQ3;   // R_SC_ITSS_PCR_PARC    PIRQA->IRQx Routing Control
  InterruptConfig->PxRcRouting[1] = V_SC_ITSS_PCR_REN_ENABLE + V_SC_ITSS_PCR_IR_IRQ4;   // R_SC_ITSS_PCR_PBRC    PIRQB->IRQx Routing Control
  InterruptConfig->PxRcRouting[2] = V_SC_ITSS_PCR_REN_ENABLE + V_SC_ITSS_PCR_IR_IRQ5;   // R_SC_ITSS_PCR_PCRC    PIRQC->IRQx Routing Control
  InterruptConfig->PxRcRouting[3] = V_SC_ITSS_PCR_REN_ENABLE + V_SC_ITSS_PCR_IR_IRQ6;   // R_SC_ITSS_PCR_PDRC    PIRQD->IRQx Routing Control
  InterruptConfig->PxRcRouting[4] = V_SC_ITSS_PCR_REN_ENABLE + V_SC_ITSS_PCR_IR_IRQ7;   // R_SC_ITSS_PCR_PERC    PIRQE->IRQx Routing Control
  InterruptConfig->PxRcRouting[5] = V_SC_ITSS_PCR_REN_ENABLE + V_SC_ITSS_PCR_IR_IRQ9;   // R_SC_ITSS_PCR_PFRC    PIRQF->IRQx Routing Control
  InterruptConfig->PxRcRouting[6] = V_SC_ITSS_PCR_REN_ENABLE + V_SC_ITSS_PCR_IR_IRQ10;  // R_SC_ITSS_PCR_PGRC    PIRQG->IRQx Routing Control
  InterruptConfig->PxRcRouting[7] = V_SC_ITSS_PCR_REN_ENABLE + V_SC_ITSS_PCR_IR_IRQ11;  // R_SC_ITSS_PCR_PHRC    PIRQH->IRQx Routing Control

  //
  // Configure Interrupt Polarity
  //
  InterruptConfig->IPC[0] = (UINT32)
    ((V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 31) + // IRQ 31
    (V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 30) +  // IRQ 30
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 29) +  // IRQ 29
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 28) +  // IRQ 28
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 27) +  // IRQ 27
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 26) +  // IRQ 26
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 25) +  // IRQ 25
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 24) +  // IRQ 24
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 23) +  // IRQ 23
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 22) +  // IRQ 22
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 21) +  // IRQ 21
    (V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 20) +  // IRQ 20
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 19) +  // IRQ 19
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 18) +  // IRQ 18
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 17) +  // IRQ 17
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 16) +  // IRQ 16
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 15) +  // IRQ 15
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 14) +  // IRQ 14
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 13) +  // IRQ 13
	(V_SC_ITSS_PCR_IPC_ACTIVE_HIGH << 12) + // IRQ 12
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 11) +  // IRQ 11
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 10) +  // IRQ 10
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 9) +   // IRQ  9
	(V_SC_ITSS_PCR_IPC_ACTIVE_HIGH << 8) +  // IRQ  8
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 7) +   // IRQ  7
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 6) +   // IRQ  6
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 5) +   // IRQ  5
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 4) +   // IRQ  4
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 3) +   // IRQ  3
	(V_SC_ITSS_PCR_IPC_ACTIVE_HIGH << 2) +  // IRQ  2
	(V_SC_ITSS_PCR_IPC_ACTIVE_HIGH << 1) +  // IRQ  1
	(V_SC_ITSS_PCR_IPC_ACTIVE_HIGH << 0))   // IRQ  0
    ;

  InterruptConfig->IPC[1] = (UINT32)
    ((V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 31) + // IRQ 63
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 30) +  // IRQ 62
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 29) +  // IRQ 61
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 28) +  // IRQ 60
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 27) +  // IRQ 59
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 26) +  // IRQ 58
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 25) +  // IRQ 57
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 24) +  // IRQ 56
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 23) +  // IRQ 55
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 22) +  // IRQ 54
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 21) +  // IRQ 53
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 20) +  // IRQ 52
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 19) +  // IRQ 51
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 18) +  // IRQ 50
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 17) +  // IRQ 49
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 16) +  // IRQ 48
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 15) +  // IRQ 47
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 14) +  // IRQ 46
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 13) +  // IRQ 45
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 12) +  // IRQ 44
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 11) +  // IRQ 43
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 10) +  // IRQ 42
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 9) +   // IRQ 41
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 8) +   // IRQ 40
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 7) +   // IRQ 39
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 6) +   // IRQ 38
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 5) +   // IRQ 37
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 4) +   // IRQ 36
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 3) +   // IRQ 35
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 2) +   // IRQ 34
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 1) +   // IRQ 33
	(V_SC_ITSS_PCR_IPC_ACTIVE_LOW << 0))    // IRQ 32
    ;
  InterruptConfig->IPC[2] = 0xFFFFFFFF;
  InterruptConfig->IPC[3] = 0xFFFFFFFF;
}

static IP_BLOCK_ENTRY  mScIpBlocks [] = {
  {&gScGeneralConfigGuid,         sizeof (SC_GENERAL_CONFIG),      SC_GENERAL_CONFIG_REVISION,             LoadScGeneralConfigDefault},
  {&gSataConfigGuid,              sizeof (SC_SATA_CONFIG),         SATA_CONFIG_REVISION,                   LoadSataConfigDefault},
  {&gPcieRpConfigGuid,            sizeof (SC_PCIE_CONFIG),         PCIE_RP_CONFIG_REVISION,                LoadPcieConfigDefault},
  {&gUsbConfigGuid,               sizeof (SC_USB_CONFIG),          USB_CONFIG_REVISION,                    LoadUsbConfigDefault},
  {&gHpetConfigGuid,              sizeof (SC_HPET_CONFIG),         HPET_CONFIG_REVISION,                   LoadHpetConfigDefault},
  {&gSmbusConfigGuid,             sizeof (SC_SMBUS_CONFIG),        SMBUS_CONFIG_REVISION,                  LoadSmbusConfigDefault},
  {&gIoApicConfigGuid,            sizeof (SC_IOAPIC_CONFIG),       IOAPIC_CONFIG_REVISION,                 LoadIoApicConfigDefault},
  {&gHdAudioConfigGuid,           sizeof (SC_HDAUDIO_CONFIG),      HDAUDIO_CONFIG_REVISION,                LoadHdaConfigDefault},
  {&gGmmConfigGuid,               sizeof (SC_GMM_CONFIG),          GMM_CONFIG_REVISION,                    LoadGmmConfigDefault},
  {&gPmConfigGuid,                sizeof (SC_PM_CONFIG),           PM_CONFIG_REVISION,                     LoadPmConfigDefault},
  {&gLockDownConfigGuid,          sizeof (SC_LOCK_DOWN_CONFIG),    LOCK_DOWN_CONFIG_REVISION,              LoadLockDownConfigDefault},
  {&gSerialIrqConfigGuid,         sizeof (SC_LPC_SIRQ_CONFIG),     SERIAL_IRQ_CONFIG_REVISION,             LoadSerialIrqConfigDefault},
  {&gLpssConfigGuid,              sizeof (SC_LPSS_CONFIG),         LPSS_CONFIG_REVISION,                   LoadLpssConfigDefault},
  {&gScsConfigGuid,               sizeof (SC_SCS_CONFIG),          SCS_CONFIG_REVISION,                    LoadScsConfigDefault},
  {&gVtdConfigGuid,               sizeof (SC_VTD_CONFIG),          VTD_CONFIG_REVISION,                    LoadVtdConfigDefault},
  {&gIshConfigGuid,               sizeof (SC_ISH_CONFIG),          ISH_CONFIG_REVISION,                    LoadIshConfigDefault},
  {&gFlashProtectionConfigGuid,   sizeof (SC_FLASH_PROTECTION_CONFIG),  FLASH_PROTECTION_CONFIG_REVISION,  LoadFlashProtectionConfigDefault},
  {&gDciConfigGuid,               sizeof (SC_DCI_CONFIG),          DCI_CONFIG_REVISION,                    LoadDciConfigDefault},
  {&gP2sbConfigGuid,              sizeof (SC_P2SB_CONFIG),         P2SB_CONFIG_REVISION,                   LoadP2sbConfigDefault},
  {&gInterruptConfigGuid,         sizeof (SC_INTERRUPT_CONFIG),    INTERRUPT_CONFIG_REVISION,              LoadInterruptConfigDefault},

#if (CNVI_ENABLE == 1)
  {&gCnviConfigGuid,              sizeof (SC_CNVI_CONFIG),         CNVI_CONFIG_REVISION,                   LoadCnviConfigDefault},
#endif

};


/**
  Get SC config block table total size.

  @retval Size of SC config block table
**/
UINT32
EFIAPI
ScGetConfigBlockTotalSize (
  VOID
  )
{
  UINT32            TotalBlockCount;
  UINT32            TotalBlockSize;
  UINT32            ConfigBlockHdrSize;
  UINT32            BlockCount;

  TotalBlockCount = sizeof(mScIpBlocks) / sizeof (IP_BLOCK_ENTRY);
  TotalBlockSize = 0;
  for (BlockCount = 0 ; BlockCount < TotalBlockCount; BlockCount++) {
    TotalBlockSize += (UINT32)mScIpBlocks[BlockCount].Size;
    DEBUG ((DEBUG_INFO, "TotalBlockSize after adding Block[0x%x]= 0x%x\n", BlockCount, TotalBlockSize));
  }

  ConfigBlockHdrSize = sizeof (CONFIG_BLOCK_TABLE_HEADER) + (TotalBlockCount * 4);

  //
  // Because CreateConfigBlockTable has the padding for each config block,
  // we need extra size, which is TotalBlockCount * 3, to create the table
  //
  return ConfigBlockHdrSize + TotalBlockSize + (TotalBlockCount * 3);
}

/**
  CreateConfigBlocks generates the config blocks of SC Policy.
  It allocates and zero out buffer, and fills in the Intel default settings.

  @param[out] ScPolicyPpi        The pointer to get SC Policy PPI instance

  @retval EFI_SUCCESS            The policy default is initialized.
  @retval EFI_OUT_OF_RESOURCES   Insufficient resources to create buffer
**/
EFI_STATUS
EFIAPI
ScCreateConfigBlocks (
  OUT  SC_POLICY_PPI **ScPolicyPpi
  )
{
  UINT16             TotalBlockCount;
  UINT16             BlockCount;
  VOID               *ConfigBlockPointer;
  CONFIG_BLOCK       ConfigBlockBuf;
  EFI_STATUS         Status;
  SC_POLICY_PPI      *InitPolicy;
  UINT32             RequiredSize;

  DEBUG ((DEBUG_INFO, "SC CreateConfigBlocks\n"));

  InitPolicy = NULL;
  TotalBlockCount = sizeof(mScIpBlocks) / sizeof (IP_BLOCK_ENTRY);
  DEBUG ((DEBUG_INFO, "TotalBlockCount = 0x%x\n", TotalBlockCount));

  RequiredSize = ScGetConfigBlockTotalSize ();
  Status = CreateConfigBlockTable ((UINT16)RequiredSize, (VOID *)&InitPolicy);
  ASSERT_EFI_ERROR (Status);

  //
  // Initialize Policy Revision
  //
  InitPolicy->TableHeader.Header.Revision = SC_POLICY_REVISION;
  //
  // Initialize ConfigBlockPointer to NULL
  //
  ConfigBlockPointer = NULL;
  //
  // Loop to identify each config block from mScIpBlocks[] Table and add each of them
  //
  for (BlockCount = 0 ; BlockCount < TotalBlockCount; BlockCount++) {
    CopyMem (&(ConfigBlockBuf.Header.GuidHob.Name),  mScIpBlocks[BlockCount].Guid, sizeof (EFI_GUID));
    ConfigBlockBuf.Header.GuidHob.Header.HobLength = mScIpBlocks[BlockCount].Size;
    ConfigBlockBuf.Header.Revision                 = mScIpBlocks[BlockCount].Revision;
    ConfigBlockPointer                             = (VOID *)&ConfigBlockBuf;
    Status = AddConfigBlock ((VOID *)InitPolicy, (VOID *)&ConfigBlockPointer);
    ASSERT_EFI_ERROR (Status);
    mScIpBlocks[BlockCount].LoadDefault (ConfigBlockPointer);
  }
  //
  // Assignment for returning Policy config block base address
  //
  *ScPolicyPpi = InitPolicy;
  return EFI_SUCCESS;
}

/**
  ScInstallPolicyPpi installs ScPolicyPpi.
  While installed, RC assumes the Policy is ready and finalized. So please update and override
  any setting before calling this function.

  @param[in] ScPolicyPpi         The pointer to SC Policy PPI instance

  @retval EFI_SUCCESS            The policy is installed.
  @retval EFI_OUT_OF_RESOURCES   Insufficient resources to create buffer
**/
EFI_STATUS
EFIAPI
ScInstallPolicyPpi (
  IN  SC_POLICY_PPI *ScPolicyPpi
  )
{
  EFI_STATUS              Status;
  EFI_PEI_PPI_DESCRIPTOR  *ScPolicyPpiDesc;

  ScPolicyPpiDesc = (EFI_PEI_PPI_DESCRIPTOR *) AllocateZeroPool (sizeof (EFI_PEI_PPI_DESCRIPTOR));
  if (ScPolicyPpiDesc == NULL) {
    ASSERT (FALSE);
    return EFI_OUT_OF_RESOURCES;
  }

  ScPolicyPpiDesc->Flags = EFI_PEI_PPI_DESCRIPTOR_PPI | EFI_PEI_PPI_DESCRIPTOR_TERMINATE_LIST;
  ScPolicyPpiDesc->Guid  = &gScPolicyPpiGuid;
  ScPolicyPpiDesc->Ppi   = ScPolicyPpi;

  //
  // Print whole SC_POLICY_PPI and serial out.
  //
  ScPrintPolicyPpi (ScPolicyPpi);

  //
  // Install SC Policy PPI
  //
  Status = PeiServicesInstallPpi (ScPolicyPpiDesc);
  ASSERT_EFI_ERROR (Status);
  return Status;
}
