/** @file
  This file defines the Variable Storage Command Queue module.

  This module installs the EDKII_VARIABLE_STORAGE_IO_COMPLETION_PROTOCOL.

 @copyright
  INTEL CONFIDENTIAL
  Copyright 2016 - 2017 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

#include "CseVariableStorageCommandQueueRuntimeDxe.h"
#include <Library/BootMediaLib.h>
#include <Library/UefiDriverEntryPoint.h>
#include <Library/UefiRuntimeLib.h>

extern UINT8                            *mCseVariableStorageBuffer;
extern UINT8                            *mCseVariableStorageResponseVerificationBuffer;
extern UINT8                            *mCseVariableStorageDebugBuffer;
extern UINT8                            *mCseVariableStorageNoPayloadCommunicateBuffer;
extern EFI_HECI_PROTOCOL                *mHeciProtocol;
extern EFI_SMM_COMMUNICATION_PROTOCOL   *mSmmCommunicationProtocol;

STATIC EFI_EVENT  mCseVariableStorageEndOfServicesEvent  = NULL;
STATIC EFI_EVENT  mVirtualAddressChangeEvent             = NULL;

EDKII_VARIABLE_STORAGE_IO_COMPLETION_PROTOCOL mVariableStorageIoCompletionProtocol = {
  CseVariableStorageIoCompletionComplete,
  CseVariableStorageIoCompletionGetId
};

/**
  Notification function of EVT_SIGNAL_VIRTUAL_ADDRESS_CHANGE.

  This is a notification function registered on EVT_SIGNAL_VIRTUAL_ADDRESS_CHANGE event.
  It converts pointer to new virtual address.

  @param[in]  Event        Event whose notification function is being invoked.
  @param[in]  Context      Pointer to the notification function's context.

**/
VOID
EFIAPI
CseVariableStorageIoCompletionAddressChangeEvent (
  IN EFI_EVENT             Event,
  IN VOID                  *Context
  )
{
  EfiConvertPointer (0x0, (VOID **) &mCseVariableStorageBuffer);
  EfiConvertPointer (0x0, (VOID **) &mCseVariableStorageResponseVerificationBuffer);
  EfiConvertPointer (0x0, (VOID **) &mCseVariableStorageNoPayloadCommunicateBuffer);
  EfiConvertPointer (0x0, (VOID **) &mCseVariableStorageDebugBuffer);

  EfiConvertPointer (0x0, (VOID **) &mHeciProtocol);
  EfiConvertPointer (0x0, (VOID **) &mSmmCommunicationProtocol);
}

/**
  Initialize the CSE Command Queue module and install a CSE Variable Storage instance of
  the Variable IO Completion Protocol.

  @param[in]  ImageHandle  The image handle.
  @param[in]  SystemTable  The system table.

  @retval     EFI_SUCCESS  The protocol was installed successfully.
  @retval     Others       Protocol could not be installed.
**/
EFI_STATUS
EFIAPI
CseVariableStorageCommandQueueRuntimeDxeInitialize (
  IN EFI_HANDLE         ImageHandle,
  IN EFI_SYSTEM_TABLE   *SystemTable
  )
{
  EFI_STATUS  Status;
  EFI_HANDLE  Handle;
  VOID        *EventRegistrationLocal = NULL;

  DEBUG ((EFI_D_INFO, "CSE Variable Command Queue Total Size = %d bytes.\n", CMD_QUEUE_TOTAL_BUFFER_SIZE));

  //
  // In the current boot architecture, the proxy flow and this command queue
  // module are only needed in an eMMC or UFS boot scenario.
  //
  if (!BootMediaIsEmmcOrUfs ()) {
    return EFI_SUCCESS;
  }

  Status = gBS->LocateProtocol (&gEfiSmmCommunicationProtocolGuid, NULL, (VOID **) &mSmmCommunicationProtocol);
  if (EFI_ERROR (Status)) {
    //ASSERT_EFI_ERROR (Status);
    return Status;
  }

  Status = gBS->LocateProtocol (&gEfiHeciRuntimeProtocolGuid, NULL, (VOID **) &mHeciProtocol);
  if (EFI_ERROR (Status)) {
    //ASSERT_EFI_ERROR (Status);
    return Status;
  }

  gBS->CreateEventEx (
         EVT_NOTIFY_SIGNAL,
         TPL_NOTIFY,
         CseVariableStorageIoCompletionAddressChangeEvent,
         NULL,
         &gEfiEventVirtualAddressChangeGuid,
         &mVirtualAddressChangeEvent
         );
  //ASSERT_EFI_ERROR (Status);

  //
  // Register End of Services Callback
  //
  gBS->CreateEvent (
         EVT_NOTIFY_SIGNAL,
         TPL_NOTIFY,
         CseVariableStorageEndOfServicesEvent,
         NULL,
         &mCseVariableStorageEndOfServicesEvent
         );
  //ASSERT_EFI_ERROR (Status);

  Status = gBS->RegisterProtocolNotify (
                  &gEfiCseEndofServicesProtocolGuid,
                  mCseVariableStorageEndOfServicesEvent,
                  (VOID **) &EventRegistrationLocal
                  );

  //
  // Initialize module buffers used for SMM communication
  //
  Status = InitializeModuleBuffers ();
  //ASSERT_EFI_ERROR (Status);

  //
  // Install the Variable IO Completion Protocol instance for CSE Variable Storage
  //
  Status = SystemTable->BootServices->InstallProtocolInterface (
                                        &Handle,
                                        &gEdkiiVariableStorageIoCompletionProtocolGuid,
                                        EFI_NATIVE_INTERFACE,
                                        &mVariableStorageIoCompletionProtocol
                                        );
  //ASSERT_EFI_ERROR (Status);

  return Status;
}