# This file contains common definitions for Windows, Unix and Apple (32 and 64 bit)

# Determine the operating system
if(WIN32)
    message( "-- Windows environment detected" )
    set(SYSTEM_NAME "windows")
elseif(APPLE)
    message( "-- Darwin/macOS environment detected" )
    set(SYSTEM_NAME "darwin")
elseif(UNIX)
    message( "-- UNIX/Linux environment detected" )
    set(SYSTEM_NAME "linux")
endif()

# Determine platform bitness
if(${CMAKE_SIZEOF_VOID_P} EQUAL 8)
    message( "-- 64 bits architecture detected" )
    set(ARCHITECTURE_NAME "x64")
else()
    message( "-- 32 bits architecture detected" )
    set(ARCHITECTURE_NAME "x86")
endif()

if ("${CMAKE_CXX_COMPILER_ID}" MATCHES "Clang")
    #Using Clang or AppleClang
    set(COMPILER_NAME "clang")
    set(COMPILER_IS_CLANG TRUE)
elseif ("${CMAKE_CXX_COMPILER_ID}" STREQUAL "GNU")
    #Using GCC/G++
    set(COMPILER_NAME "gnu")
    set(COMPILER_IS_GNU TRUE)
elseif ("${CMAKE_CXX_COMPILER_ID}" STREQUAL "MSVC")
    #Using MSVC
    set(COMPILER_NAME "msvc")
    set(COMPILER_IS_MSVC TRUE)
#elseif ("${CMAKE_CXX_COMPILER_ID}" STREQUAL "Intel")
#    #Using the Intel C++ compiler
#    set(COMPILER_NAME "intel")
#    set(COMPILER_IS_INTEL TRUE)
else()
    message(FATAL_ERROR "The selected compiler is not supported: '${CMAKE_CXX_COMPILER_ID}'")
endif()
message ("-- ${COMPILER_NAME} compiler detected")

if(COMPILER_IS_GNU)
    if ((CMAKE_CXX_COMPILER_VERSION VERSION_GREATER_EQUAL 4.9) AND (CMAKE_CXX_COMPILER_VERSION VERSION_LESS 5.0))
        set (COMPILER_IS_GNU_7 FALSE)
    elseif ((CMAKE_CXX_COMPILER_VERSION VERSION_GREATER_EQUAL 7.2) AND (CMAKE_CXX_COMPILER_VERSION VERSION_LESS 8.0))
        set (COMPILER_IS_GNU_7 TRUE)
    else()
        message(FATAL_ERROR "Your version of g++ is not supported. This project requires version 4.9.x or 7.3.x. Found version ${CMAKE_CXX_COMPILER_VERSION}")
    endif()
elseif(COMPILER_IS_CLANG)
    if ((CMAKE_CXX_COMPILER_VERSION VERSION_LESS 7.3) OR (CMAKE_CXX_COMPILER_VERSION VERSION_GREATER_EQUAL 10.1))
        message(FATAL_ERROR "Your version of clang is not supported. This project requires version 10.0.x. Found version ${CMAKE_CXX_COMPILER_VERSION}")
    endif()
elseif(COMPILER_IS_MSVC)
    if ((CMAKE_CXX_COMPILER_VERSION VERSION_LESS 19.0.23918) OR (CMAKE_CXX_COMPILER_VERSION VERSION_GREATER_EQUAL 20))
        message(FATAL_ERROR "Your version of MSVC/Visual Studio is not supported. This project requires Microsoft Visual Studio 2015 Update 2. Found version ${CMAKE_CXX_COMPILER_VERSION}")
    endif()
endif()

if(WIN32)
    add_definitions(-DWIN32_LEAN_AND_MEAN -DNOMINMAX -D_CRT_SECURE_NO_WARNINGS -D_SCL_SECURE_NO_WARNINGS -DUNICODE -D_UNICODE -D_WIN32_WINNT=_WIN32_WINNT_WIN8)

    # Visual Studio setting for virtual folders
    set_property(GLOBAL PROPERTY USE_FOLDERS ON)

    # CMake adds the _WINDOWS macro to the compiler flags by default. This is
    # not always desired, so we remove it here
    string (REPLACE "/D_WINDOWS" "" CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS}")
    string (REPLACE "-D_WINDOWS" "" CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS}")

    add_definitions(-DHOST_WINDOWS)
elseif(APPLE)
    # turn on OSX rpath support globally, this will affect all subordinate projects
    set(CMAKE_MACOSX_RPATH TRUE)
    # tell CMake to use the installation-time RPATH at build time. If this is not set, CMake uses its own custom RPATH
    # when the binaries are built, and will change it to the installation-time RPATH when it's performing the "install" phase
    set(CMAKE_BUILD_WITH_INSTALL_RPATH 1)
    # the following will tell the loader to look in the path of the current binary (be it an executable or a shared library)
    # when looking for shared libraries to load
    set(CMAKE_INSTALL_NAME_DIR "@loader_path")
    add_definitions(-DBOOST_NO_AUTO_PTR -DBOOST_NO_CXX11_SCOPED_ENUMS -DBOOST_NO_SLIST -DOPENSSL_NO_SSL2)

    add_definitions(-DHOST_DARWIN)
elseif(UNIX)
    # the following will tell the loader to look in the path of the current binary (be it an executable or a shared library)
    # when looking for shared libraries to load
    set(CMAKE_INSTALL_RPATH "$ORIGIN")
    add_definitions(-DBOOST_NO_AUTO_PTR -DBOOST_NO_CXX11_SCOPED_ENUMS -DBOOST_NO_SLIST -DOPENSSL_NO_SSL2)

    add_definitions(-DHOST_LINUX)
endif()

if(${ARCHITECTURE_NAME} STREQUAL "x64")
    add_definitions(-DHOST_X64)
else()
    add_definitions(-DHOST_X86)
endif()

if (COMPILER_IS_CLANG)
    macro(set_general_compilation_flags)
    	set(CMAKE_CXX_STANDARD 17)

        set(CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS} -fcolor-diagnostics -fvisibility=hidden -fvisibility-inlines-hidden")

        # Disable some harmless warnings
        set(CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS} -Wno-unknown-pragmas")
    
        # _DEBUG is not set by default, at least not when using the Xcode generator...  But we need it!
        set(CMAKE_CXX_FLAGS_DEBUG   "${CMAKE_CXX_FLAGS_DEBUG} -D_DEBUG")

        set(NO_WHOLE_ARCHIVE_OPTION "")
        set(WHOLE_ARCHIVE_OPTION "-Wl,-all_load")

        # Set linker flags
        set(CMAKE_SHARED_LINKER_FLAGS "${CMAKE_SHARED_LINKER_FLAGS} -Wl,-undefined,error")
    endmacro()

    function(enable_all_warnings)
        set(CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS} -Wall" PARENT_SCOPE)
    endfunction()

    function(enable_warnings_as_errors)
        set(CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS} -Werror" PARENT_SCOPE)
    endfunction()

    function(enable_address_sanitizer)
        message(STATUS "INFO: Build will have the Clang AddressSanitizer enabled")
        set(CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS} -fsanitize=address -fno-omit-frame-pointer" PARENT_SCOPE)
        set(CMAKE_SHARED_LINKER_FLAGS "${CMAKE_SHARED_LINKER_FLAGS} -fsanitize=address" PARENT_SCOPE)
        set(CMAKE_EXE_LINKER_FLAGS "${CMAKE_EXE_LINKER_FLAGS} -fsanitize=address" PARENT_SCOPE)
    endfunction()
elseif (COMPILER_IS_GNU)
    macro(set_general_compilation_flags)
        set(COMMON_COMPILER_FLAGS "-fno-tree-vectorize -fstack-protector-strong -fPIE -fPIC -fvisibility=hidden -fvisibility-inlines-hidden -Wformat -Wformat-security")

        # Disable some warnings, these need to be fixed!
        set(COMMON_COMPILER_FLAGS "${COMMON_COMPILER_FLAGS} -Wno-narrowing -Wno-write-strings -Wno-conversion-null")

        # Enable features from C++14/C++17
        if (COMPILER_IS_GNU_7)
            set(CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS} -std=c++14 ${COMMON_COMPILER_FLAGS}")
        else()
            set(CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS} -std=c++1y ${COMMON_COMPILER_FLAGS}")
        endif()
        set(CMAKE_C_FLAGS "${CMAKE_C_FLAGS} ${COMMON_COMPILER_FLAGS}")

        set(CMAKE_CXX_FLAGS_RELEASE  "-D_FORTIFY_SOURCE=2 -O2")
        set(CMAKE_C_FLAGS_RELEASE    ${CMAKE_CXX_FLAGS_RELEASE})

        set(NO_WHOLE_ARCHIVE_OPTION "-Wl,--no-whole-archive")
        set(WHOLE_ARCHIVE_OPTION "-Wl,--whole-archive")

        # Set linker flags
        set(COMMON_LINKER_FLAGS -Wl,-z,noexecstack,-z,now,-z,relro)
        set(CMAKE_EXE_LINKER_FLAGS "-pie ${CMAKE_EXE_LINKER_FLAGS} ${COMMON_LINKER_FLAGS}")

        set(CMAKE_SHARED_LINKER_FLAGS "${CMAKE_SHARED_LINKER_FLAGS} ${COMMON_LINKER_FLAGS} -Wl,--no-undefined")
        set(CMAKE_SHARED_LINKER_FLAGS_RELEASE "${CMAKE_SHARED_LINKER_FLAGS_RELEASE} -Wl,--strip-debug")
    endmacro()

    function(enable_all_warnings)
        set(CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS} -Wall" PARENT_SCOPE)
    endfunction()

    function(enable_warnings_as_errors)
#       todo: we should not need to do this! However, there are a lot of warnings in the sdk...
#       set(CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS} -Werror" PARENT_SCOPE)
    endfunction()

    function(enable_address_sanitizer)
        message(STATUS "INFO: Build will have the GCC AddressSanitizer enabled")
        set(CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS} -fsanitize=address -fno-omit-frame-pointer" PARENT_SCOPE)
        set(CMAKE_SHARED_LINKER_FLAGS "${CMAKE_SHARED_LINKER_FLAGS} -fsanitize=address" PARENT_SCOPE)
        set(CMAKE_EXE_LINKER_FLAGS "${CMAKE_EXE_LINKER_FLAGS} -fsanitize=address" PARENT_SCOPE)
    endfunction()
elseif (COMPILER_IS_MSVC)
    macro(set_general_compilation_flags)
        # some of these flags are defined by the SDL (#SDL325)

        # Set compiler flags
        set(CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS} /utf-8 /GS /W3 /Zi /MP /Gm- /sdl /Zm120")
        set(CMAKE_CXX_FLAGS_RELEASE "${CMAKE_CXX_FLAGS_RELEASE} /GL /Ot /Oi /O2 /Gy ")
        set(CMAKE_CXX_FLAGS_DEBUG   "${CMAKE_CXX_FLAGS_DEBUG}   /Od")

        # Set linker flags
        set(COMMON_LINKER_FLAGS "/DEBUG /NXCOMPAT /DYNAMICBASE")
        set(CMAKE_STATIC_LINKER_FLAGS_RELEASE "${CMAKE_STATIC_LINKER_FLAGS_RELEASE} /LTCG")
        set(CMAKE_SHARED_LINKER_FLAGS_RELEASE "${CMAKE_SHARED_LINKER_FLAGS_RELEASE} ${COMMON_LINKER_FLAGS} /OPT:REF /OPT:ICF /LTCG")
        set(CMAKE_SHARED_LINKER_FLAGS_DEBUG   "${CMAKE_SHARED_LINKER_FLAGS_DEBUG}   ${COMMON_LINKER_FLAGS} /OPT:NOREF /OPT:NOICF")

        set(CMAKE_EXE_LINKER_FLAGS_RELEASE "${CMAKE_EXE_LINKER_FLAGS_RELEASE} ${COMMON_LINKER_FLAGS} /OPT:REF /OPT:ICF /LTCG")
        set(CMAKE_EXE_LINKER_FLAGS_DEBUG   "${CMAKE_EXE_LINKER_FLAGS_DEBUG}   ${COMMON_LINKER_FLAGS} /OPT:NOREF /OPT:NOICF")
    endmacro()

    function(enable_all_warnings)
        set(CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS} /W4" PARENT_SCOPE)
    endfunction()

    function(enable_warnings_as_errors)
        set(CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS} /WX" PARENT_SCOPE)
    endfunction()

    function(enable_address_sanitizer)
        message(FATAL_ERROR "The address sanitizer is not supported with the MSVC compiler")
    endfunction()
endif()

set_general_compilation_flags()

if(SANITIZE_MEMORY)
    enable_address_sanitizer()
endif()

if(COMPILER_IS_CLANG)
    enable_all_warnings()
endif()

if(COMPILER_IS_MSVC OR COMPILER_IS_CLANG)
    enable_warnings_as_errors()
endif()

