/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
/// @internal
///
/// @brief Implementation of the HandleCollection class, for managing a
///        a mapping between pointers and values (handles).
///
//////////////////////////////////////////////////////////////////////////////

#ifndef ___HANDLECOLLECTION_H__
#define ___HANDLECOLLECTION_H__

#include <map>
#include <mutex>
#include <memory>

namespace InternalUtilsPrivate
{

    /// Typedef to simplify the handle number type.
    typedef unsigned int HandleNumber;

    /// @brief Represents a collection of pointers mapped to handles.
    ///
    /// Provides thread-safe support for creating, removing, and locating
    /// pointers to objects based on a numerical handle value.
    ///
    /// The handle numbers returned from the HandleCollection class are
    /// always within the given range, inclusive.  A 0 is always an invalid
    /// handle value.
    ///
    /// @par Usage:
    /// Create a global instance of this class, specifying the range of
    /// values to be used for the handles.
    /// @code
    /// HandleCollection handleCollection(1000, 1999);
    /// @endcode
    template<typename T>
    class HandleCollection
    {
    private:
        // Holds a pointer so that it can be associated with a number (handle).
        struct HandleInfo
        {
            HandleNumber handleNumber;
            std::unique_ptr<T> pResource;

            // Constructor
            HandleInfo(HandleNumber number = 0)
                : handleNumber(number)
                , pResource()
            {
            }

            HandleInfo(HandleNumber number, std::unique_ptr<T> pResource)
                : handleNumber(number)
                , pResource(std::move(pResource))
            {
            }

            HandleInfo(HandleInfo const&) = delete;

            HandleInfo(HandleInfo&& rhs)
                : handleNumber(rhs.handleNumber)
                , pResource(std::move(rhs.pResource))
            {
                rhs.handleNumber = 0;
            }

            HandleInfo& operator=(HandleInfo const&) = delete;

            HandleInfo& operator=(HandleInfo&& rhs)
            {
                handleNumber = rhs.handleNumber;
                rhs.handleNumber = 0;
                pResource = std::move(rhs.pResource);

                return *this;
            }

            ~HandleInfo()
            {
                handleNumber = 0;
            }

            // Comparison operator for using this class type in a std::map.
            bool operator==(const HandleInfo& e)
            {
                return pResource.get() == e.pResource.get();
            }

            // Comparison operator for using this class type in a std::map.
            bool operator!=(const HandleInfo& e)
            {
                return !(*this == e);
            }
        };

        /// Typedef to simplify use of the std::map type.
        typedef std::map<HandleNumber, HandleInfo> HandleMap;

    private:
        HandleNumber m_minHandleNumber;      ///< The bottom of the handle number range, inclusive.
        HandleNumber m_maxHandleNumber;      ///< THe top of the handle number range, inclusive.
        HandleNumber m_nextHandleNumber;     ///< The next "handle" to return within the range.
        std::mutex   _handleMapMutex;  ///< A mutex to guard multi-thread access to the handle map.
        HandleMap    m_handleMap;            ///< Maps handle numbers to HandleInfo structures (containing pointers)


        /// Return the next available handle number to be used as the handle.
        ///
        /// 0 is not a valid handle so this method ensures that 0 cannot be returned.
        HandleNumber _NextHandleNumber()
        {
            ++m_nextHandleNumber;
            if (m_nextHandleNumber > m_maxHandleNumber)
            {
                m_nextHandleNumber = m_minHandleNumber;
                if (m_nextHandleNumber == 0)
                {
                    ++m_nextHandleNumber;
                }
            }
            return m_nextHandleNumber;
        }

        std::unique_lock<decltype(_handleMapMutex)> _Lock()
        {
            return std::unique_lock<decltype(_handleMapMutex)>(_handleMapMutex);
        }

    public:

        /// Constructor.
        ///
        /// @param minHandleValue
        ///     The minimum value a handle number can be.
        /// @param maxHandleValue
        ///    The maximum value a handle number can be.
        HandleCollection(HandleNumber minHandleValue, HandleNumber maxHandleValue)
            : m_minHandleNumber(minHandleValue)
            , m_maxHandleNumber(maxHandleValue)
            , m_nextHandleNumber(minHandleValue)
            , _handleMapMutex()
        {
        }


        HandleCollection(const HandleCollection& other) = delete;
        HandleCollection(HandleCollection&& other) noexcept = delete;
        HandleCollection& operator=(const HandleCollection& other) = delete;
        HandleCollection& operator=(HandleCollection&& other) noexcept = delete;

        /// Destructor.
        ~HandleCollection()
        {
        }


        /// Add a pointer to the collection and return a handle representing
        /// that pointer.
        HandleNumber AddHandle(std::unique_ptr<T>&& pResource)
        {
            auto lock = _Lock();
            HandleNumber handleNumber = _NextHandleNumber();
            HandleInfo handleInfo(handleNumber, std::move(pResource));
            m_handleMap[handleNumber] = std::move(handleInfo);

            return handleNumber;
        }


        /// Given a handle, find the associated pointer and return it.
        ///
        /// Returns NULL if the handle is not in the collection.
        std::unique_ptr<T> const& FindHandle(HandleNumber handleNumber)
        {
            static std::unique_ptr<T> const EmptyHandle;
            auto lock = _Lock();
            typename HandleMap::iterator foundIter = m_handleMap.find(handleNumber);
            if (foundIter != m_handleMap.end())
            {
                return foundIter->second.pResource;
            }
            else
            {
                return EmptyHandle;
            }
        }


        /// Given a handle, find and return the associated pointer and also
        /// remove the handle from the collection.
        ///
        /// The find and remove operations are combined here so that only one
        /// lock/unlock is needed and the lock/unlock can be contained entirely
        /// within this class.
        std::unique_ptr<T> FindAndRemoveHandle(HandleNumber handleNumber)
        {
            auto lock = _Lock();
            typename HandleMap::iterator foundIter = m_handleMap.find(handleNumber);
            if (foundIter != m_handleMap.end())
            {
                std::unique_ptr<T> retVal = std::move(foundIter->second.pResource);
                m_handleMap.erase(foundIter);
                return retVal;
            }
            else
            {
                return std::unique_ptr<T>();
            }
        }
    };

} // end namespace InternalUtilsPrivate

#endif // ___HANDLECOLLECTION_H__
