/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
/// @internal
///
/// @brief Implementation of functions for managing mutexes in a cross-platform
///        manner.
///
/// These are the functions that the threadmutex.cpp module uses.  They are
/// defined here so that other modules, such as _HandleCollection.h, can also
/// use mutexes.
///
/// @warning
/// These functions are to be used only inside the InternalUtils library!
///
//////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"

#ifdef _MSC_VER

// Windows only
#include <Windows.h>

#else

// Non-Windows only
#include <pthread.h>
#include <time.h>
#include "darwinutils.h"

#endif

#include "threadtypes.h"
#include "_PrivateMutex.h"

namespace InternalUtilsPrivate
{

    //////////////////////////////////////////////////////////////////////////
    // Function: _CreateMutex
    //
    //////////////////////////////////////////////////////////////////////////
    int _CreateMutex(void** ppSyncObject)
    {
        int err = EINVAL;

        if (ppSyncObject != NULL)
        {
            void* pSyncObject = NULL;
#ifdef _MSC_VER
            HANDLE hMutex = ::CreateMutex(NULL, FALSE, NULL);
            if (hMutex != NULL)
            {
                pSyncObject = hMutex;
                err = 0;
            }
#else
            pthread_mutex_t* pMutex = new(std::nothrow) pthread_mutex_t;
            if (pMutex != NULL)
            {
                pthread_mutexattr_t mutexAttribute;
                err = pthread_mutexattr_init(&mutexAttribute);
                if (err == 0)
                {
                    err = pthread_mutexattr_settype(&mutexAttribute, PTHREAD_MUTEX_RECURSIVE);
                    if (err == 0)
                    {
                        err = pthread_mutex_init(pMutex, &mutexAttribute);
                        if (err == 0)
                        {
                            pSyncObject = pMutex;
                        }
                    }
                    pthread_mutexattr_destroy(&mutexAttribute);
                }
            }
#endif
            if (pSyncObject != NULL)
            {
                *ppSyncObject = pSyncObject;
            }
        }

        return err;
    }


    //////////////////////////////////////////////////////////////////////////
    // Function: _DestroyMutex
    //
    //////////////////////////////////////////////////////////////////////////
    int _DestroyMutex(void* pSyncObject)
    {
        int err = EINVAL;

        if (pSyncObject != NULL)
        {
#ifdef _MSC_VER
            HANDLE hMutex = reinterpret_cast<HANDLE>(pSyncObject);
            if (::CloseHandle(hMutex))
            {
                err = 0;
            }
#else
            pthread_mutex_t* pMutex = reinterpret_cast<pthread_mutex_t*>(pSyncObject);
            err = pthread_mutex_destroy(pMutex);
            delete pMutex;
#endif
        }

        return err;
    }


    //////////////////////////////////////////////////////////////////////////
    // Function: _LockMutex
    //
    //////////////////////////////////////////////////////////////////////////
    int _LockMutex(void* pSyncObject, InternalUtils::TimeoutDelay milliseconds)
    {
        int err = EINVAL;
        if (pSyncObject != NULL)
        {
 #ifdef _MSC_VER
            HANDLE hMutex = reinterpret_cast<HANDLE>(pSyncObject);
            DWORD retval = ::WaitForSingleObject(hMutex, milliseconds);
            if (retval == WAIT_OBJECT_0)
            {
                err = 0;
            }
            else if (retval == WAIT_TIMEOUT)
            {
                err = ETIMEDOUT;
            }
#else
            pthread_mutex_t* pMutex = reinterpret_cast<pthread_mutex_t*>(pSyncObject);

            if (milliseconds == InternalUtils::TIMEOUT_INFINITE)
            {
                err = pthread_mutex_lock(pMutex);
            }
            else
            {
                timeval tp;
                err = gettimeofday(&tp, NULL);
                if (err == 0)
                {
                    timespec timeDelay;
                    timeDelay.tv_sec  = tp.tv_sec;
                    timeDelay.tv_nsec = tp.tv_usec * 1000;
                    long sec = milliseconds / 1000;
                    long nsec = milliseconds - (sec * 1000);
                    timeDelay.tv_sec += sec;
                    timeDelay.tv_nsec += nsec;
                    if (timeDelay.tv_nsec >= (1000 * 1000 * 1000))
                    {
                        timeDelay.tv_sec += 1;
                        timeDelay.tv_nsec -= (1000 * 1000 * 1000);
                    }
                    err = pthread_mutex_timedlock(pMutex, &timeDelay);
                }
            }
#endif
       }

        return err;
    }


    //////////////////////////////////////////////////////////////////////////
    // Function: _UnlockMutex
    //
    //////////////////////////////////////////////////////////////////////////
    int _UnlockMutex(void* pSyncObject)
    {
        int err = EINVAL;

        if (pSyncObject != NULL)
        {
#ifdef _MSC_VER
            err = 0;
            HANDLE hMutex = reinterpret_cast<HANDLE>(pSyncObject);
            if (!::ReleaseMutex(hMutex))
            {
                // There is no POSIX error code for an unowned mutex
                // and Linux returns EPERM, which results in a misleading
                // error message on Windows.  So we create our own error
                // code that is more to the point.
                err = InternalUtils::eNOTOWNED;
            }
#else
            pthread_mutex_t* pMutex = reinterpret_cast<pthread_mutex_t*>(pSyncObject);
            // This will return EPERM if the thread doesn't own the mutex
            err = pthread_mutex_unlock(pMutex);
            if (err == EPERM)
            {
                err = InternalUtils::eNOTOWNED;
            }
#endif
        }

        return err;
    }


    //////////////////////////////////////////////////////////////////////////
    // Function: _TryLockMutex
    //
    //////////////////////////////////////////////////////////////////////////
    int _TryLockMutex(void* pSyncObject)
    {
        int err = EINVAL;

        if (pSyncObject != NULL)
        {
 #ifdef _MSC_VER
            err = 0;
            HANDLE hMutex = reinterpret_cast<HANDLE>(pSyncObject);
            DWORD retval = ::WaitForSingleObject(hMutex, 0);
            if (retval != WAIT_OBJECT_0)
            {
                err = EBUSY;
            }
#else
            pthread_mutex_t* pMutex = reinterpret_cast<pthread_mutex_t*>(pSyncObject);
            err = pthread_mutex_trylock(pMutex);
#endif
       }

        return err;
    }

} // end namespace InternalUtilsPrivate
