/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
/// @internal
///
/// @brief Implementation of functions that are operating system-specific in
///        their implementation for accessing dynamic libraries.
///
//////////////////////////////////////////////////////////////////////////////
#include "stdafx.h"
#include "paths.h"
#include <iomanip>
#include <string>
#include <sstream>

#include "accessdll.h"

#ifdef _WIN32
    #include <codecvt>
#else
    #include <dlfcn.h> // For dlopen, dlclose, dlerror, dlsym, dladdr
#endif



namespace InternalUtils
{

    //////////////////////////////////////////////////////////////////////////
    //  Function: getdllerrormessage
    //
    //////////////////////////////////////////////////////////////////////////
    std::string getdllerrormessage()
    {

#ifdef _WIN32
        // order of the code below matters!  we need to call GetLastError *immediately* before the error code gets overwritten
        DWORD lastError = ::GetLastError();

        LPWSTR buff = nullptr;
        DWORD size = FormatMessageW(
            FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS,
            NULL,
            lastError,
            MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
            (LPWSTR)&buff,
            0,
            NULL);

        //FormatMessage returns the number of characters in the buffer _excluding_ the terminating null character

        if ((size != 0) && (buff[size - 1] == L'\n'))
        {
            //The error message that's returned ends in a newline character, we need to strip that out
            buff[size - 1] = L'\0';

            if ((size > 1) && (buff[size - 2] == L'\r'))
            {
                buff[size - 2] = L'\0';
            }
        }

        std::wstring_convert<std::codecvt_utf8_utf16<wchar_t>, wchar_t> converter;
        std::stringstream msg;
        msg << "(0x" << std::hex << std::setw(8) << std::setfill('0') << lastError << ") " << converter.to_bytes(buff);
        LocalFree(buff); buff = nullptr;

        return msg.str();
#else
        char* err = dlerror();
        if (err)
        {
            return std::string(err);
        }
        else
        {
            return std::string();
        }
#endif

    }

    //////////////////////////////////////////////////////////////////////////
    //  Function: loaddll
    //
    //////////////////////////////////////////////////////////////////////////
    IPC_DLL_HANDLE loaddll(
        const std::string& dllName)
    {
        IPC_DLL_HANDLE libraryHandle = NULL;

        if (!dllName.empty())
        {
#ifdef _WIN32
            std::string path = InternalUtils::removetail(dllName.c_str());
            // Tell Windows to search the DLL's path for dependencies the DLL
            // might have (by default, Windows will not search the same folder
            // as the DLL itself for dependencies).
            std::wstring_convert<std::codecvt_utf8_utf16<wchar_t>, wchar_t> converter;
            ::SetDllDirectoryW(converter.from_bytes(path).c_str());

            libraryHandle = ::LoadLibraryW(converter.from_bytes(dllName).c_str());
#else
            libraryHandle = dlopen(dllName.c_str(), RTLD_LAZY | RTLD_LOCAL);
#endif
        }

        return libraryHandle;
    }


    //////////////////////////////////////////////////////////////////////////
    //  Function: freedll
    //
    //////////////////////////////////////////////////////////////////////////
    bool freedll(
        IPC_DLL_HANDLE libraryHandle)
    {
        bool unloadedSuccessFully = true;

        if (libraryHandle != NULL)
        {
#ifdef _WIN32
            if (!::FreeLibrary(libraryHandle))
            {
                unloadedSuccessFully = false;
            }
#else
            if (dlclose(libraryHandle))
            {
                unloadedSuccessFully = false;
            }
#endif
        }

        return unloadedSuccessFully;
    }


    //////////////////////////////////////////////////////////////////////////
    //  Function: getprocedurefromdll
    //
    //////////////////////////////////////////////////////////////////////////
    void* getprocedurefromdll(
        IPC_DLL_HANDLE     libraryHandle,
        IPC_PROC_NAME_TYPE procName)
    {
        void *pProc = NULL;

        if (libraryHandle != NULL && procName != NULL)
        {
#ifdef _WIN32
            pProc = ::GetProcAddress(libraryHandle, procName);
#else
            pProc = dlsym(libraryHandle, procName);
#endif
        }

        return pProc;
    }


    //////////////////////////////////////////////////////////////////////////
    //  Function: getcurrentmodulepath
    //
    //////////////////////////////////////////////////////////////////////////
    std::string getcurrentmodulepath()
    {
        std::string path;

#ifdef _WIN32
        IPC_DLL_HANDLE libraryHandle = NULL;
        // Note: this will get the path of the DLL that links to this function.
        // This is normally correct.  If the calling code expects the path of
        // the executable, they won't get it for a NULL handle.
        BOOL status = ::GetModuleHandleEx(
            GET_MODULE_HANDLE_EX_FLAG_FROM_ADDRESS | GET_MODULE_HANDLE_EX_FLAG_UNCHANGED_REFCOUNT,
            (LPCTSTR)getcurrentmodulepath,
            &libraryHandle);
        if (status == TRUE)
        {
            DWORD bufferSize = 0;
            DWORD numCharsCopied = 0;
            std::vector<wchar_t> buffer;
            while(numCharsCopied == bufferSize)
            {
                bufferSize += 1024;
                buffer.resize(bufferSize, L'\0');
                numCharsCopied = ::GetModuleFileNameW(libraryHandle, buffer.data(), bufferSize);
                if (numCharsCopied == 0)
                {
                    //An error occured...
                    break;
                }
            }
            buffer[numCharsCopied] = L'\0';

            std::wstring_convert<std::codecvt_utf8_utf16<wchar_t>, wchar_t> converter;
            path = converter.to_bytes(buffer.data());
        }
#else
        Dl_info dlInfo = { 0 };
        dladdr((void *)getcurrentmodulepath, &dlInfo);
        if (dlInfo.dli_fname != NULL)
        {
            path = dlInfo.dli_fname;
        }
#endif

        return path;
    }

} // end InternalUtils namespace
