/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
/// @internal
///
/// @brief Declaration of functions associated with file operations.
///
/// Functions:
///  - openfile()
///  - closefile()
///  - writefile()
///  - flushfile()
///
//////////////////////////////////////////////////////////////////////////////

#ifndef __OPENFILE_H__
#define __OPENFILE_H__

#include <string>
#include <stdio.h>

namespace InternalUtils
{

    //////////////////////////////////////////////////////////////////////////
    //  Function: openfile
    //
    /// @brief This function opens a file and returns a pointer to a FILE
    ///        object representing the opened file.
    ///
    /// This function is a wrapper for the fopen() C++ standard library 
    /// function for opening a stream to disk.  It hides the differences
    /// between different C++ standard libraries.
    ///
    /// @par Assumptions:
    ///   None.
    ///
    /// @par Side Effects:
    ///   A file is opened.
    ///
    /// @param pstream
    ///   Returns a FILE pointer to the opened file.  Returns NULL if there
    ///   is an error.
    ///
    /// @param filename
    ///   The name of the file to open.
    ///
    /// @param mode
    ///   Describes how to open the file.  See documentation for the fopen()
    ///   C++ standard library function for details.
    ///
    /// @return (int): Returns 0 for success, non-zero for failure.  The error
    ///   codes are defined in errno.h.
    /// @retval EINVAL The @a pstream, @a filename, or @a mode parameter is NULL.
    //////////////////////////////////////////////////////////////////////////
    int openfile(FILE** pstream, const char* filename, const char* mode);


    //////////////////////////////////////////////////////////////////////////
    //  Function: openfile_unbuffered
    //
    /// @brief This function opens an unbuffered file and returns a pointer to
    ///        a FILE object representing the opened file.
    ///
    /// This function is a wrapper for the fopen() C++ standard library 
    /// function for opening a stream to disk.  It hides the differences
    /// between different C++ standard libraries.  In addition, this function
    /// marks the opened file as unbuffered.  This has an effect only if the
    /// file is opened for writing.
    ///
    /// @par Assumptions:
    ///   None.
    ///
    /// @par Side Effects:
    ///   A file is opened.
    ///
    /// @param pstream
    ///   Returns a FILE pointer to the opened file.  Returns NULL if there
    ///   is an error.
    ///
    /// @param filename
    ///   The name of the file to open.
    ///
    /// @param mode
    ///   Describes how to open the file.  See documentation for the fopen()
    ///   C++ standard library function for details.
    ///
    /// @return (int): Returns 0 for success, non-zero for failure.  The error
    ///   codes are defined in errno.h.
    /// @retval EINVAL The @a pstream, @a filename, or @a mode parameter is NULL.
    //////////////////////////////////////////////////////////////////////////
    int openfile_unbuffered(FILE** pstream, const char* filename, const char* mode);


    //////////////////////////////////////////////////////////////////////////
    //  Function: closefile
    //
    /// @brief Closes the specified file.
    ///
    /// This function wraps the fclose() C++ standard library function.
    ///
    /// @par Assumptions:
    ///   None.
    ///
    /// @par Side Effects:
    ///   The file is closed.
    ///
    /// @param stream
    ///   The FILE pointer representing the stream to close.
    ///
    /// @return (int): Returns 0 for success, non-zero for failure.  The error
    ///   codes are defined in errno.h.
    /// @retval EINVAL The @a stream parameter is NULL.
    //////////////////////////////////////////////////////////////////////////
    int closefile(FILE* stream);

    //////////////////////////////////////////////////////////////////////////
    //  Function: getfilesize
    //
    /// @brief Retrieves the size of the specified file.
    ///
    /// This function uses the ftell() C++ standard library function but
    /// retains the current file position.
    ///
    /// @par Assumptions:
    ///   None.
    ///
    /// @par Side Effects:
    ///   None.
    ///
    /// @param stream
    ///   The FILE pointer representing the stream to access.
    /// @param pSize
    ///   Returns the size of the file, in bytes.
    ///
    /// @return (int): Returns 0 for success, non-zero for failure.  The error
    ///   codes are defined in errno.h.
    /// @retval EINVAL The @a stream or @a pSize parameter is NULL.
    //////////////////////////////////////////////////////////////////////////
    int getfilesize(FILE* stream, long* pSize);

    //////////////////////////////////////////////////////////////////////////
    //  Function: writefile
    //
    /// @brief Writes a buffer to the specified file.
    ///
    /// This function wraps the fwrite() C++ standard library function.
    ///
    /// @par Assumptions:
    ///   The stream was opened for writing.
    ///
    /// @par Side Effects:
    ///   The stream is updated with the contents of the buffer.
    ///
    /// @param buffer
    ///   The buffer to write to the stream.
    ///
    /// @param size
    ///   The size of each element in the buffer to write.
    ///
    /// @param count
    ///   The number of elements in the buffer to write.  The total bytes
    ///   written is @a size * @a count.
    ///
    /// @param stream
    ///   The FILE pointer representing the stream to write.
    ///
    /// @return (int): Returns 0 for success, non-zero for failure.  The error
    ///   codes are defined in errno.h.
    /// @retval EINVAL The @a buffer or @a stream parameter is NULL.
    //////////////////////////////////////////////////////////////////////////
    int writefile(const void *buffer, size_t size, size_t count, FILE *stream);

    //////////////////////////////////////////////////////////////////////////
    //  Function: readfile
    //
    /// @brief Reads a buffer from the specified file.
    ///
    /// This function wraps the fread() C++ standard library function.
    ///
    /// @par Assumptions:
    ///   The stream was opened for reading.
    ///
    /// @par Side Effects:
    ///   The file pointer in the stream is updated by the amount read.
    ///
    /// @param buffer
    ///   The buffer to fill from the stream.
    ///
    /// @param size
    ///   The size of each element in the buffer to read.
    ///
    /// @param count
    ///   The number of elements in the buffer to read.  The total bytes
    ///   read is @a size * @a count.
    ///
    /// @param stream
    ///   The FILE pointer representing the stream to read.
    ///
    /// @return (int): Returns 0 for success, non-zero for failure.  The error
    ///   codes are defined in errno.h.
    /// @retval EINVAL The @a buffer or @a stream parameter is NULL.
    //////////////////////////////////////////////////////////////////////////
    int readfile(void *buffer, size_t size, size_t count, FILE* stream);


    //////////////////////////////////////////////////////////////////////////
    //  Function: readline
    //
    /// @brief Reads a line of text from the specified file.
    ///
    /// This function reads up to the next newline, end of file, or the
    /// specified maximum number of characters.  The buffer is guaranteed
    /// to be zero-terminated on return.
    ///
    /// This function wraps the fgets() C++ standard library function.
    ///
    /// @par Assumptions:
    ///   The stream was opened for reading.
    ///
    /// @par Side Effects:
    ///   The file pointer in the stream is updated by the amount read.
    ///
    /// @param buffer
    ///   The buffer to fill from the stream.
    ///
    /// @param maxSize
    ///   The maximum number of characters to read looking for a newline.
    ///
    /// @param stream
    ///   The FILE pointer representing the stream to read.
    ///
    /// @return (int): Returns 0 for success, non-zero for failure.  The error
    ///   codes are defined in errno.h.
    /// @retval EINVAL The @a buffer or @a stream parameter is NULL or the
    ///                @a maxSize parameter is 0.
    /// @retval EOF    The end of file was encountered while trying to read
    ///                from the file.
    //////////////////////////////////////////////////////////////////////////
    int readline(char *buffer, size_t maxSize, FILE* stream);


    //////////////////////////////////////////////////////////////////////////
    //  Function: flushfile
    //
    /// @brief Flush data held in memory to the specified file.
    ///
    /// This function is a wrapper for the fflush() C++ standard library
    /// function.
    ///
    /// @par Assumptions:
    ///   The file is opened.
    ///
    /// @par Side Effects:
    ///   The file on disk may be updated.
    ///
    /// @param stream
    ///   The FILE pointer representin the stream to flush.
    ///
    /// @return (int): Returns 0 for success, non-zero for failure.  The error
    ///   codes are defined in errno.h.
    /// @retval EINVAL The @a stream parameter is NULL.
    //////////////////////////////////////////////////////////////////////////
    int flushfile(FILE* stream);

} // end namespace InternalUtils

#endif // __OPENFILE_H__
