/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
/// @internal
///
/// @brief Declaration of a function to locate a dynamic library.
///
//////////////////////////////////////////////////////////////////////////////

#ifndef __INTERNAL_FINDFILE_H__
#define __INTERNAL_FINDFILE_H__

#include <vector>
#include <string>

namespace InternalUtils
{
    //////////////////////////////////////////////////////////////////////////
    //  Function: findmodule
    //
    /// @brief Retrieve the location of the specified module.
    ///
    /// This helper function searches for the specified module and
    /// returns the path to it if found.  Otherwise, it returns
    /// the name of the module without extension to allow the
    /// operating system to locate it through normal means.  Either way,
    /// the call of this function is responsible for catching errors due to
    /// a missing module.
    ///
    /// @par Assumptions:
    ///   The @a moduleName parameter is the full name of the module
    ///   to look for but without a path.
    ///
    /// @par Side Effects:
    ///   None.
    ///
    /// @param[in] moduleName 
    ///    The name of the module to look for.  This is expected to
    ///    be the full name of the file, for example, "IpcApi_x86.dll" or
    ///    "libipcapi_x86.so".
    ///
    /// @return (std::string):
    ///    Returns a string containing a path that can be passed to the
    ///    LoadDynamicLibrary() function.  There is no guarantee the file
    ///    actually exists.
    //////////////////////////////////////////////////////////////////////////
    std::string findmodule(const char* moduleName);

	//////////////////////////////////////////////////////////////////////////
    //  Function: findopenipcfile
    //
    /// @brief Retrieve the location of the specified OpenIPC file.
    ///
    /// This helper function searches for the specified file and
    /// returns the path to it if found.  Otherwise, it returns
    /// the name of the file without extension. The call of this
	/// function is responsible for catching errors due to a
    /// missing file.
    ///
    /// @par Assumptions:
    ///   The @a fileName parameter is the full name of the file
    ///   to look for but without a path.
    ///
    /// @par Side Effects:
    ///   None.
    ///
    /// @param[in] fileName 
    ///    The name of the file to look for.  This is expected to
    ///    be the full name of the file, for example, "IpcApi_x86.dll" or
    ///    "libipcapi_x86.so".
    ///
    /// @return (std::string):
    ///    Returns a string containing a path. There is no guarantee the file
    ///    actually exists.
    //////////////////////////////////////////////////////////////////////////
	std::string findopenipcfile(const char* fileName);
	
	//////////////////////////////////////////////////////////////////////////
    //  Function: finddalfile
    //
    /// @brief Retrieve the location of the specified DAL file.
    ///
    /// This helper function searches for the specified file and
    /// returns the path to it if found.  Otherwise, it returns
    /// the name of the file without extension. The call of this
	/// function is responsible for catching errors due to a
    /// missing file.
    ///
    /// @par Assumptions:
    ///   The @a fileName parameter is the full name of the file
    ///   to look for but without a path.
    ///
    /// @par Side Effects:
    ///   None.
    ///
    /// @param[in] fileName 
    ///    The name of the file to look for.  This is expected to
    ///    be the full name of the file, for example, "IpcApi_x86.dll" or
    ///    "libipcapi_x86.so".
    ///
    /// @return (std::string):
    ///    Returns a string containing a path. There is no guarantee the file
    ///    actually exists.
    //////////////////////////////////////////////////////////////////////////
	std::string finddalfile(const char* fileName);


	//////////////////////////////////////////////////////////////////////////
    //  Function: finddalfile
    //
    /// @brief Find files matching a directory/file specification.
    ///
    /// This utility function searches a directory path for files matching
	/// a wildcard-able string.
    ///
    /// @par Assumptions:
    ///   The @a directoryfilespec parameter is a valid path and wildcard
	///   file specification.
    ///
    /// @par Side Effects:
    ///   None.
    ///
    /// @param[in] directory
    ///    The path of the directory to look in.
    ///
    /// @param[in] filespec 
    ///    File specifications accept wildcards to match files with.
    ///    For example "*.dll" would return all files with a "dll"
	///    extension.
	///
    /// @return (std::vector<std::string>):
    ///    Returns a vector of strings containing the fully-pathed file matches
    //////////////////////////////////////////////////////////////////////////
	std::vector<std::string> findfiles(const std::string& directory, const std::string& filespec);

} // end namespace InternalUtils

#endif // __INTERNAL_FINDFILE_H__
