/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
/// @internal
///
/// @brief Implementation of a function to locate the IPC's registered location.
///
//////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "fileops.h"
#include "direxists.h"
#include "fileexists.h"
#include "getenvvar.h"
#include "paths.h"
#include "locateimpl.h"
#include "geterrormessage.h"

#ifdef _WIN32
    #include <shlobj.h> // For SHGetKnownFolderPath()
    #include <codecvt>
#endif



namespace // anonymous
{

    //////////////////////////////////////////////////////////////////////////
    //  Function: _GetIPCLocation
    //
    // Read the IPC's registration file to find the registered location of
    // the IPC and return that location as a string.
    //////////////////////////////////////////////////////////////////////////
    std::string _GetIPCLocation(const char* registeredFilePath)
    {
        std::string path;
        std::string fileContents;
        FILE* pFile = NULL;

        int err = InternalUtils::openfile(&pFile, registeredFilePath, "r");
        if (err == 0)
        {
            long fileSize = 0;
            err = InternalUtils::getfilesize(pFile, &fileSize);
            if (err == 0)
            {
                fileContents.resize(fileSize);
                err = InternalUtils::readfile(&fileContents[0], 1, fileSize, pFile);
                fileContents[fileSize] = '\0';
            }
            (void)InternalUtils::closefile(pFile);
        }
        if (err == 0)
        {
            //  read first line of file to get path to MasterFrame.HostApplication.exe
            const char* pLineEnd = strchr(fileContents.c_str(), '\n');
            if (pLineEnd != NULL)
            {
                size_t lineLength = pLineEnd - fileContents.c_str();
                if (lineLength > 0)
                {
                    path = fileContents.substr(0, lineLength);
                    //  strip tail from path
                    path = InternalUtils::removetail(path.c_str());
                    //  return path
                }
            }
        }

        return path;
    }

} // end namespace anonymous



namespace InternalUtils
{

    // Return the registered location of the IPC.
    // Returns an empty path if the IPC is not registered.
    std::string locatedal()
    {
        std::string path;

        // Note: All of the code up to the call to _GetIPCLocation()
        // is just to find the IPC's registration file.

#ifdef _WIN32
        // (From IPCPathUtils.cs:)
        // If on Windows then
        //   Get special folder, CommonApplicationData
        wchar_t* buffer = nullptr;
        HRESULT hr = SHGetKnownFolderPath(FOLDERID_ProgramData, KF_FLAG_DEFAULT, NULL, &buffer);
        if (hr == S_OK)
        {
            std::wstring_convert<std::codecvt_utf8_utf16<wchar_t>, wchar_t> converter;
            path = converter.to_bytes(buffer);
            CoTaskMemFree(buffer); buffer = nullptr;
        }
#else
        // else
        //   get user's home folder
        //   append .dal to user's folder
        path = InternalUtils::getenvvar("HOME");
        path += "/.dal";
        //   if the .dal folder does not exist in the user's folder then
        if (!InternalUtils::direxists(path.c_str()))
        {
            //     get special folder, CommonApplicationData
            path = "/usr/share";
        }
#endif
        // append Intel/IPC/MasterFrame/sharedinfo.txt to folder
        const char* pathParts[] = { "Intel", "DAL", "MasterFrame", "sharedinfo.txt" };
        path = InternalUtils::combinepaths(path.c_str(), pathParts, _countof(pathParts));
        // if file exists then
        if (InternalUtils::fileexists(path.c_str()))
        {
            path = _GetIPCLocation(path.c_str());
        }
        else
        {
            // return NULL, IPC not registered
            path.clear();
        }
        
        return path;
    }

	// Return the registered location of the IPC.
    // Returns an empty path if the IPC is not registered.
    std::string locateopenipc()
    {
        std::string path;

        // Note: All of the code up to the call to _GetIPCLocation()
        // is just to find the IPC's registration file.

#ifdef _WIN32
        // (From IPCPathUtils.cs:)
        // If on Windows then
        //   Get special folder, CommonApplicationData
        wchar_t* buffer = nullptr;
        HRESULT hr = SHGetKnownFolderPath(FOLDERID_ProgramData, KF_FLAG_DEFAULT, NULL, &buffer);
        if (hr == S_OK)
        {
            std::wstring_convert<std::codecvt_utf8_utf16<wchar_t>, wchar_t> converter;
            path = converter.to_bytes(buffer);
            CoTaskMemFree(buffer); buffer = nullptr;
        }
#else
        // else
        //   get user's home folder
        //   append .dal to user's folder
        path = InternalUtils::getenvvar("HOME");
        path += "/.dal";
        //   if the .dal folder does not exist in the user's folder then
        if (!InternalUtils::direxists(path.c_str()))
        {
            //     get special folder, CommonApplicationData
            path = "/usr/share";
        }
#endif
        // append Intel/IPC/MasterFrame/sharedinfo.txt to folder
        const char* pathParts[] = { "Intel", "OpenIPC", "sharedinfo.txt" };
        path = InternalUtils::combinepaths(path.c_str(), pathParts, _countof(pathParts));
        // if file exists then
        if (InternalUtils::fileexists(path.c_str()))
        {
            path = _GetIPCLocation(path.c_str());
        }
        else
        {
            // return NULL, IPC not registered
            path.clear();
        }
        
        return path;
    }

} // end namespace InternalUtils
