/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
/// @internal
/// @brief Implementation of path utility functions.
///
/// These utility functions provide platform-independent way to work with
/// file paths.
///
/// @par Operations defined:
/// - removetail()
/// - appendtail()
///
//////////////////////////////////////////////////////////////////////////////
#include "stdafx.h"
#include <string>

#ifdef _WIN32
#  include <stdlib.h>
#else
#  include <libgen.h>
#endif

#include "paths.h"

namespace // anonymous
{

#ifdef _WIN32
    // Version of the linux function, dirname(), which returns the
    // directory of the given path (basically, removes the tail
    // path component).
    std::string dirname(const std::string& file)
    {
        std::string directory;

        size_t last_slash_idx = file.find_last_of("\\/");
        if (std::string::npos != last_slash_idx)
        {
            directory = file.substr(0, last_slash_idx);
        }

        return directory;
    }
#endif

    // Append a directory separator to the specified path if the last
    // character is not already a directory separator.
    //
    // Note: if the given path is empty, no separator is added.
    void _appendDirectorySeparator(std::string& path)
    {
        if (!path.empty())
        {
            char lastChar = path[path.size() - 1];
#ifdef _WIN32
            if (lastChar != '\\')
            {
                path.append("\\");
            }
#else
            if (lastChar != '/')
            {
                path.append("/");
            }
#endif
        }
    }

} // end namespace anonymous


namespace InternalUtils
{

    //////////////////////////////////////////////////////////////////////////
    // function removetail
    //
    //////////////////////////////////////////////////////////////////////////
    std::string removetail(
        const char* path)
    {
        std::string basePath;

        if (path != nullptr)
        {
            basePath = path;
            basePath = dirname(&basePath[0]);
        }

        return basePath;
    }


    //////////////////////////////////////////////////////////////////////////
    // function appendtail
    //
    //////////////////////////////////////////////////////////////////////////
    std::string appendtail(
        const char* path,
        const char* tail)
    {
        if (path == NULL)
        {
            path = "";
        }

        std::string newPath(path);

        if (tail != NULL)
        {
            _appendDirectorySeparator(newPath);
            newPath.append(tail);
        }

        return newPath;
    }


    //////////////////////////////////////////////////////////////////////////
    // function combinepaths
    //
    //////////////////////////////////////////////////////////////////////////
    std::string combinepaths(
         const char*  path,
        const char*  parts[],
              size_t numParts)
    {
        if (path == NULL)
        {
            path = "";
        }

        std::string finalPath(path);
        for (size_t i = 0; i < numParts; ++i)
        {
            if (parts[i] != NULL)
            {
                finalPath = appendtail(finalPath.c_str(), parts[i]);
            }
        }

        return finalPath;
    }


    //////////////////////////////////////////////////////////////////////////
    // function removeextension
    //
    //////////////////////////////////////////////////////////////////////////
    std::string removeextension(
        const char* path)
    {
        if (path == NULL)
        {
            path = "";
        }

        std::string basePath(path);

        size_t extensionStartIndex = basePath.find_last_of('.');
        if (extensionStartIndex != std::string::npos)
        {
            basePath = basePath.substr(0, extensionStartIndex);
        }
        return basePath;

    }


    //////////////////////////////////////////////////////////////////////////
    // function addextension
    //
    //////////////////////////////////////////////////////////////////////////
    std::string addextension(
        const char* path,
        const char* extension)
    {
        if (path == NULL)
        {
            path = "";
        }
        if (extension == NULL)
        {
            extension = "";
        }

        std::string newPath(path);

        if (!newPath.empty() && extension[0] != '\0' && extension[0] != '.')
        {
            newPath.append(".");
        }
        newPath.append(extension);

        return newPath;
    }

} // end namespace InternalUtils
