/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
/// @internal
/// @brief Declaration of path utility functions.
///
/// These utility functions provide platform-independent way to work with
/// file paths.
///
/// @par Operations defined:
/// - removetail()
/// - appendtail()
///
//////////////////////////////////////////////////////////////////////////////

#ifndef _PATHS_H__
#define _PATHS_H__

#include <string>

namespace InternalUtils
{

    //////////////////////////////////////////////////////////////////////////
    //  Function: removetail
    //
    /// @brief Return the given path with the last path component removed.
    ///
    /// This helper function removes the last path component from the given
    /// path.  For example, given "/some/path/here", the removetail()
    /// function returns "/some/path".
    ///
    /// @par Assumptions:
    ///   - On Windows, the path given is valid for Windows with \ as the
    ///     path separator.
    ///   - On Linux, the path given is valid for Linux with / as the path
    ///     separator.
    ///
    /// @par Side Effects:
    ///   None.
    ///
    /// @param path
    ///     The path to manipulate.
    ///
    /// @return (std::string):
    ///     Returns a new string containing the path without the last path
    ///     component.  If the @a path parameter is NULL, returns an empty
    ///     string.  If the path does not contain any additional path
    ///     path components, returns a empty string.
    //////////////////////////////////////////////////////////////////////////
    std::string removetail(
        const char* path);


    //////////////////////////////////////////////////////////////////////////
    //  Function: appendtail
    //
    /// @brief Return the given path with the specified component appended.
    ///
    /// This helper function adds the specified tail component to the given
    /// path.  For example, given "/some/path" and a tail of "there", the
    /// appendtail() function returns "/some/path/there".
    ///
    /// @par Assumptions:
    ///   - On Windows, the path given is valid for Windows with \ as the
    ///     path separator.
    ///   - On Linux, the path given is valid for Linux with / as the path
    ///     separator.
    ///
    /// @par Side Effects:
    ///   None.
    ///
    /// @param path
    ///     The path to manipulate.  If NULL, assume an empty string.
    /// @param tail
    ///     The path component to append to the path.  If NULL then the
    ///     original path is returned.
    ///
    /// @return (std::string):
    ///     Returns a new string containing the original path plus the tail
    ///     component appended.
    //////////////////////////////////////////////////////////////////////////
    std::string appendtail(
        const char* path,
        const char* tail);


    //////////////////////////////////////////////////////////////////////////
    //  Function: combinepaths
    //
    /// @brief Return the combination of the given path parts.
    ///
    /// This helper function combines two or more components into a single
    /// path.  This function works like appendtail() but can handle an
    /// arbitrary number of path components.
    ///
    /// @par Assumptions:
    ///   - On Windows, the path given is valid for Windows with \ as the
    ///     path separator.
    ///   - On Linux, the path given is valid for Linux with / as the path
    ///     separator.
    ///
    /// @par Side Effects:
    ///   None.
    ///
    /// @param path
    ///     The path to manipulate.  If NULL, assume an empty string.
    /// @param parts
    ///     An array of strings, each of which specifies one path component.
    ///     If an entry is NULL, it is skipped.
    /// @param numParts
    ///     The number of strings in the @a parts array.
    ///
    /// @return (std::string):
    ///     Returns a string containing the concatenation of all the path
    ///     components.
    //////////////////////////////////////////////////////////////////////////
    std::string combinepaths(
        const char*  path,
        const char*  parts[],
              size_t numParts);

    //////////////////////////////////////////////////////////////////////////
    //  Function: removeextension
    //
    /// @brief Remove any extension from the given path and return a new path.
    ///
    /// This helper function removes any extension (a part of the file name
    /// that starts with '.') and returns the modified path as a new string.
    ///
    /// @note
    /// This function looks for the last '.' in the path and treats that as
    /// the extension, regardless of where the '.' appears.
    ///
    /// @par Assumptions:
    ///   - The extension starts with '.'.
    ///
    /// @par Side Effects:
    ///   None.
    ///
    /// @param path
    ///     The path to manipulate.  If NULL, assume an empty string.  If the
    ///     path does not have a '.' in it, the path is returned unchanged.
    ///
    /// @return (std::string):
    ///     Returns a string containing the path with the extension removed.
    //////////////////////////////////////////////////////////////////////////
    std::string removeextension(
        const char* path);


    //////////////////////////////////////////////////////////////////////////
    //  Function: addextension
    //
    /// @brief Add an extension to the specified path and return a new path.
    ///
    /// This helper function adds the specified extension to the end of the
    /// given path and returns the new path.  The extension can include a
    /// leading '.', which is ignored.
    ///
    /// @note
    /// If the incoming extension is NULL or empty, no attempt is made to
    /// add an extension.  This means there won't be any extraneous '.'
    /// added.
    ///
    /// @par Assumptions:
    ///   None.
    ///
    /// @par Side Effects:
    ///   None.
    ///
    /// @param[in] path
    ///     The path to manipulate.  If NULL, assume an empty string.
    /// @param[in] extension
    ///     The extension to add.  If NULL, assume an empty string.
    ///
    /// @return (std::string):
    ///     Returns a string containing the path with the extension added.
    //////////////////////////////////////////////////////////////////////////
    std::string addextension(
        const char* path,
        const char* extension);

} // end namespace InternalUtils

#endif // _PATHS_H__
