/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
/// @internal
///
/// @brief Implementations of functions to replace strings or characters within
///        a source string, changing the source string in the process.
///
///////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "strstri.h"
#include <string.h>

namespace InternalUtils
{

    ///////////////////////////////////////////////////////////////////////////
    // function replace
    //
    // Replace narrow string with another narrow string
    ///////////////////////////////////////////////////////////////////////////
    bool replace(
        std::string& s,
        const char* str1,
        const char* str2,
        bool bCaseInsensitive/*=false*/)
    {
        bool breplaced = false;
        size_t str1Len = (str1 != NULL) ? strlen(str1) : 0;

        // If there is something to do then
        if (!s.empty() && str1Len > 0 && str2 != NULL)
        {
            std::string out; // build new string here first
            const char* pStart = s.c_str();
            const char* pEnd = pStart + s.length();

            // Start at the beginning of the source string
            // and continue to the end.
            while (pStart < pEnd)
            {
                const char* pFound = NULL;

                // Look for next occurrence of str1
                if (bCaseInsensitive)
                {
                    pFound = strstri(pStart, str1);
                }
                else
                {
                    pFound = strstr(pStart, str1);
                }

                // If str1 has been found, replace it
                if (pFound != NULL)
                {
                    // First, append the stuff between the search start
                    // and the just found str1
                    out.append(pStart, pFound - pStart);
                    // "Insert" str2 into the string (this is the replacement)
                    out.append(str2);
                    // Update the search start to point to after the found str1
                    pStart = pFound + str1Len;
                    breplaced = true;
                }
                else
                {
                    // str1 not found, copy rest of source to out
                    // and terminate the loop.
                    out += pStart;
                    pStart = pEnd;
                }
            }

            // If replacements made then set source string to new string.
            if (breplaced)
            {
                s = out;
            }
        }

        return (breplaced);
    }


    ///////////////////////////////////////////////////////////////////////////
    // function replace
    //
    // Replace wide string with another wide string
    ///////////////////////////////////////////////////////////////////////////
    bool replace(
        std::wstring& s,
        const wchar_t* str1,
        const wchar_t* str2,
        bool bCaseInsensitive/*=false*/)
    {
        bool breplaced = false;
        size_t str1Len = (str1 != NULL) ? wcslen(str1) : 0;

        // If there is something to do then
        if (!s.empty() && str1Len > 0 && str2 != NULL)
        {
            std::wstring out; // build new string here first
            const wchar_t* pStart = s.c_str();
            const wchar_t* pEnd = pStart + s.length();

            // Start at the beginning of the source string
            // and continue to the end.
            while (pStart < pEnd)
            {
                const wchar_t* pFound = NULL;

                // Look for next occurrence of str1
                if (bCaseInsensitive)
                {
                    pFound = wcsstri(pStart, str1);
                }
                else
                {
                    pFound = wcsstr(pStart, str1);
                }

                // If str1 has been found, replace it
                if (pFound != NULL)
                {
                    // First, append the stuff between the search start
                    // and the just found str1
                    out.append(pStart, pFound - pStart);
                    // "Insert" str2 into the string (this is the replacement)
                    out.append(str2);
                    // Update the search start to point to after the found str1
                    pStart = pFound + str1Len;
                    breplaced = true;
                }
                else
                {
                    // str1 not found, copy rest of source to out
                    // and terminate the loop.
                    out += pStart;
                    pStart = pEnd;
                }
            }

            // If replacements made then set source string to new string.
            if (breplaced)
            {
                s = out;
            }
        }

        return (breplaced);
    }


    ///////////////////////////////////////////////////////////////////////////
    // function replace
    //
    // Replace narrow character with another narrow character
    ///////////////////////////////////////////////////////////////////////////
    bool replace(
        std::string& s,
        char c1,
        char c2,
        bool bCaseInsensitive/*=false*/)
    {
        std::string str1(1, c1);
        std::string str2(1, c2);
        return (replace(s, str1.c_str(), str2.c_str(), bCaseInsensitive));
    }


    ///////////////////////////////////////////////////////////////////////////
    // function replace
    //
    // Replace wide character with another wide character
    ///////////////////////////////////////////////////////////////////////////
    bool replace(
        std::wstring& s,
        wchar_t c1,
        wchar_t c2,
        bool bCaseInsensitive/*=false*/)
    {
        std::wstring str1(1, c1);
        std::wstring str2(1, c2);
        return (replace(s, str1.c_str(), str2.c_str(), bCaseInsensitive));
    }

} // end namespace InternalUtils
