/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
/// @internal
/// @brief Implementation of the sprintf_safe() function.
///
/// This utility function is a "safe" version of the sprintf() standard C
/// library funtion that prevents buffer overruns.
///
//////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include <stdio.h>
#include <stdarg.h>
#include "sprintf_safe.h"

#ifdef _MSC_VER
//------------------------------------------------------------
// Only Windows needs the following invalid parameter override
//------------------------------------------------------------

namespace // Anonymous
{
    //////////////////////////////////////////////////////////////////////////
    // Function: _invalid_parameter
    //
    // Handler for invalid parameter errors.  In this case, we do nothing
    // and let the code continue to run, returning an error code.
    //////////////////////////////////////////////////////////////////////////
    void _invalid_parameter(
       const wchar_t * /*expression*/,
       const wchar_t * /*function*/, 
       const wchar_t * /*file*/, 
       unsigned int /*line*/,
       uintptr_t /*pReserved*/
    )
    {
        // Do nothing here.  We want the code to continue to run.
    }


    _invalid_parameter_handler _oldhandler = NULL;


    //////////////////////////////////////////////////////////////////////////
    // Function: _DisableInvalidParameterHandler
    //
    // Effectively turns off the default invalid parameter handler, which
    // normally aborts the program.
    //////////////////////////////////////////////////////////////////////////
    void _DisableInvalidParameterHandler()
    {
        // By disabling the invalid parameter handler, vsprintf_s() will return
        // -1 for any errors, including if the buffer is too small.  This is
        // somewhat different from vsnprintf() on Linux, which returns the
        // number of characters that would have been written to the buffer if
        // it were large enough.  Otherwise, vsnprintf() returns -1 for any
        // formatting errors.
        //
        // Note: On Windows, vsnprintf() always returns -1 if the buffer is
        // too small.
        _oldhandler = _set_invalid_parameter_handler(_invalid_parameter);
    }


    //////////////////////////////////////////////////////////////////////////
    // Function: _EnableInvalidParameterHandler
    //
    // Restores the default invalid parmater handler.
    //////////////////////////////////////////////////////////////////////////
    void _EnableInvalidParameterHandler()
    {
        _set_invalid_parameter_handler(_oldhandler);
    }

} // end namespace Anonymous

#endif // _MSC_VER




namespace InternalUtils
{
    //////////////////////////////////////////////////////////////////////////
    // Function: sprintf_safe
    //
    //////////////////////////////////////////////////////////////////////////
    int sprintf_safe(char* szBuffer, int nSize, const char* szFormatStr, ... )
    {
        int ret = 0;
        if (szBuffer != NULL && szFormatStr != NULL && nSize > 0)
        {
            va_list vaArgList;
            va_start(vaArgList,szFormatStr);
#ifdef _MSC_VER
            _DisableInvalidParameterHandler();
            ret = vsprintf_s(szBuffer, nSize, szFormatStr, vaArgList);
            _EnableInvalidParameterHandler();
#else
            ret = vsnprintf(szBuffer, nSize, szFormatStr, vaArgList);
            if (ret >= nSize)
            {
                // Convert virtual buffer overrun to an error to match Windows
                // behavior.
                ret = -1;
            }
#endif
            szBuffer[nSize - 1] = '\0';
            va_end(vaArgList);
        }
        return ret;
    }

} // end namespace InternalUtils
