/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
/// @internal
///
/// @brief Declaration of the event functions that handle signaling
///        across threads.
///
//////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include <condition_variable>
#include <atomic>

#include "threadevent.h"
#include "threadmutex.h"
#include "_threadmutexprivate.h" // For InternalUtilsPrivate::_FindMutexHandle()
#include "_HandleCollection.h"

#include <chrono>

namespace // anonymous
{
    class Event
    {
    private:
        std::condition_variable_any _cv;
        std::shared_ptr<bool> _signaled;
        std::mutex _mutex;

    public:
        Event()
            : _cv()
            , _signaled(new bool(false))
        {
        }

        ~Event()
        {
        }

        void notify_all()
        {
            std::unique_lock<decltype(_mutex)> lock(_mutex);
            *_signaled = true;
            //We only want to signal the threads currently waiting.
            //Any thread that wait after this call will need to be signaled again,
            //so we make sure that any subsequent calls to "wait" or "wait_for"
            //use a new signal.
            _signaled.reset(new bool(false));
            _cv.notify_all();
        }

        template<class Lock>
        void wait(Lock& lock)
        {
            //Grabbing a copy of the shared signal bool means that this thread is waiting.
            std::shared_ptr<bool> signaled;
            {
                std::unique_lock<decltype(_mutex)> _(_mutex);
                //This wait will use the current _signaled variable, which will be set
                //on the next notify.
                signaled = _signaled;
            }
            if (_signaled)
            {
                _cv.wait(lock, [signaled]() { return *signaled; });
            }
        }

        template<class Lock, class Rep, class Period>
        bool wait_for(Lock& lock, std::chrono::duration<Rep, Period> timeout)
        {
            //Grabbing a copy of the shared signal bool means that this thread is waiting.
            std::shared_ptr<bool> signaled;
            {
                std::unique_lock<decltype(_mutex)> _(_mutex);
                //This wait will use the current _signaled variable, which will be set
                //on the next notify.
                signaled = _signaled;
            }
            if (_signaled)
            {
                return _cv.wait_for(lock, timeout, [signaled]() { return *signaled; });
            }
            return false;
        }
    };

    typedef std::unique_ptr<Event> EventPointer;
    // The collection mapping handles to event pointers.
    InternalUtilsPrivate::HandleCollection<EventPointer::element_type> eventHandles(2000, 2999);
} // end namespace anonymous


namespace InternalUtils
{

    //////////////////////////////////////////////////////////////////////////
    // createthreadevent
    //
    //////////////////////////////////////////////////////////////////////////
    int createthreadevent(ThreadEventHandle* pThreadEventHandle)
    {
        int err = EINVAL;

        if (pThreadEventHandle != NULL)
        {
            try
            {
                *pThreadEventHandle = 0;

                EventPointer pEvent(new Event());
                InternalUtilsPrivate::HandleNumber handleNumber = eventHandles.AddHandle(std::move(pEvent));
                *pThreadEventHandle = handleNumber;
                err = 0;
            }
            catch (std::bad_alloc&)
            {
                err = ENOMEM;
            }
            catch (...)
            {
                err = EINVAL;
            }
        }

        return err;
    }


    //////////////////////////////////////////////////////////////////////////
    // destroythreadevent
    //
    //////////////////////////////////////////////////////////////////////////
    int destroythreadevent(ThreadEventHandle threadEventHandle)
    {
        int err = EINVAL;

        InternalUtilsPrivate::HandleNumber handleNumber = threadEventHandle;
        if (eventHandles.FindAndRemoveHandle(handleNumber))
        {
            err = 0;
        }

        return err;
    }


    //////////////////////////////////////////////////////////////////////////
    // waitthreadevent
    //
    //////////////////////////////////////////////////////////////////////////
    int waitthreadevent(ThreadEventHandle threadEventHandle, MutexHandle mutexHandle, TimeoutDelay milliseconds)
    {
        int err = EINVAL;

        InternalUtilsPrivate::HandleNumber handleNumber = threadEventHandle;
        EventPointer const& pEvent = eventHandles.FindHandle(handleNumber);
        if (pEvent)
        {
            InternalUtilsPrivate::MutexPointer const& pMutex = InternalUtilsPrivate::_FindMutexHandle(mutexHandle);
            err = EINVAL_MUTEX;
            if (pMutex)
            {
                try
                {
                    //std::unique_lock<InternalUtilsPrivate::MutexPointer::element_type> lock(*pMutex, std::adopt_lock);
                    if (milliseconds == static_cast<TimeoutDelay>(-1))
                    {
                        pEvent->wait(*pMutex);
                        err = 0;
                    }
                    else
                    {
                        if (pEvent->wait_for(*pMutex, std::chrono::milliseconds(milliseconds)))
                        {
                            err = 0;
                        }
                        else
                        {
                            err = ETIMEDOUT;
                        }
                    }
                }
                catch (std::system_error)
                {
                    err = eNOTOWNED;
                }
                catch (...)
                {
                    err = EINVAL;
                }
            }
        }

        return err;
    }


    //////////////////////////////////////////////////////////////////////////
    // signalthreadevent
    //
    //////////////////////////////////////////////////////////////////////////
    int signalthreadevent(ThreadEventHandle threadEventHandle)
    {
        int err = EINVAL;

        InternalUtilsPrivate::HandleNumber handleNumber = threadEventHandle;
        EventPointer const& pEvent = eventHandles.FindHandle(handleNumber);
        if (pEvent)
        {
            pEvent->notify_all();
            err = 0;
        }

        return err;
    }

} // end namespace InternalUtils
