/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
/// @internal
///
/// @brief Implementation of the mutex functions that handle synchronization
///        across threads.
///
//////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"

#include <mutex>

#include "threadmutex.h"
#include "_threadmutexprivate.h"
#include "_HandleCollection.h"


namespace // anonymous
{
    using InternalUtilsPrivate::MutexPointer;

    // The collection mapping handles to mutex pointers.
    InternalUtilsPrivate::HandleCollection<MutexPointer::element_type> mutexHandles(1000, 1999);

} // end namespace anonymous


namespace InternalUtilsPrivate
{
    MutexPointer const& _FindMutexHandle(HandleNumber mutexHandle)
    {
        return mutexHandles.FindHandle(mutexHandle);
    }
} // end namespace InternalUtilsPrivate



namespace InternalUtils
{

    //////////////////////////////////////////////////////////////////////////
    // createmutex
    //
    //////////////////////////////////////////////////////////////////////////
    int createmutex(MutexHandle* pMutexHandle)
    {
        int err = EINVAL;

        if (pMutexHandle != NULL)
        {
            *pMutexHandle = 0;

            try
            {
                MutexPointer pMutex(new std::recursive_timed_mutex());
                InternalUtilsPrivate::HandleNumber handleNumber = mutexHandles.AddHandle(std::move(pMutex));
                *pMutexHandle = handleNumber;
                err = 0;
            }
            catch (std::bad_alloc&)
            {
                err = ENOMEM;
            }
            catch (...)
            {
                err = EINVAL;
            }
        }

        return err;
    }


    //////////////////////////////////////////////////////////////////////////
    // destroymutex
    //
    //////////////////////////////////////////////////////////////////////////
    int destroymutex(MutexHandle mutexHandle)
    {
        int err = EINVAL;

        InternalUtilsPrivate::HandleNumber handleNumber = mutexHandle;
        if (mutexHandles.FindAndRemoveHandle(handleNumber))
        {
            err = 0;
        }

        return err;
    }


    //////////////////////////////////////////////////////////////////////////
    // lockmutex
    //
    //////////////////////////////////////////////////////////////////////////
    int lockmutex(MutexHandle mutexHandle, TimeoutDelay milliseconds)
    {
        int err = EINVAL;

        InternalUtilsPrivate::HandleNumber handleNumber = mutexHandle;
        MutexPointer const& pMutex = mutexHandles.FindHandle(handleNumber);
        if (pMutex)
        {
            if (milliseconds == static_cast<TimeoutDelay>(-1))
            {
                pMutex->lock();
                err = 0;
            }
            else
            {
                if (pMutex->try_lock_for(std::chrono::milliseconds(milliseconds)))
                {
                    err = 0;
                }
                else
                {
                    err = ETIMEDOUT;
                }
            }
        }

        return err;
    }


    //////////////////////////////////////////////////////////////////////////
    // unlockmutex
    //
    //////////////////////////////////////////////////////////////////////////
    int unlockmutex(MutexHandle mutexHandle)
    {
        int err = EINVAL;

        InternalUtilsPrivate::HandleNumber handleNumber = mutexHandle;
        MutexPointer const& pMutex = mutexHandles.FindHandle(handleNumber);
        if (pMutex)
        {
            pMutex->unlock();
            err = 0;
        }

        return err;
    }


    //////////////////////////////////////////////////////////////////////////
    // trylockmutex
    //
    //////////////////////////////////////////////////////////////////////////
    int trylockmutex(MutexHandle mutexHandle)
    {
        int err = EINVAL;

        InternalUtilsPrivate::HandleNumber handleNumber = mutexHandle;
        MutexPointer const& pMutex = mutexHandles.FindHandle(handleNumber);
        if (pMutex)
        {
            if (pMutex->try_lock())
            {
                err = 0;
            }
            else
            {
                err = EBUSY;
            }
        }

        return err;
    }

} // end namespace InternalUtils
