/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
/// @internal
///
/// @brief Implementation of the beginthread() function that runs a function
///        in another thread.
///
//////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"

/*
#ifdef _MSC_VER

// Windows only
#include <process.h>
typedef unsigned int (__stdcall *WindowsThreadFunc)(void*);

#else

// Non-Windows only
#include <pthread.h>
#include "darwinutils.h"
#include <chrono>
#endif
*/

#include <chrono>
#include <future>

#include "threads.h"
#include "_HandleCollection.h"

namespace // anonymous
{
    typedef std::unique_ptr<std::future<void*>> ThreadPointer;
    // The collection mapping handles to event pointers.
    InternalUtilsPrivate::HandleCollection<ThreadPointer::element_type> threadHandles(3000, 9999);

} // end namespace anonymous


namespace InternalUtils
{
    //////////////////////////////////////////////////////////////////////////
    // Function: beginthread
    //
    //////////////////////////////////////////////////////////////////////////
    int beginthread(ThreadProc function, unsigned int /*stackSize*/, void* userData, ThreadHandle* pThreadHandle)
    {
        int err = EINVAL;

        if (function != NULL)
        {
            try
            {
                ThreadPointer pThread(new std::future<void*>(std::async(std::launch::async, function, userData)));

                InternalUtilsPrivate::HandleNumber handleNumber = threadHandles.AddHandle(std::move(pThread));
                *pThreadHandle = handleNumber;
                err = 0;
            }
            catch (std::bad_alloc&)
            {
                err = ENOMEM;
            }
            catch (...)
            {
                err = EINVAL;
            }
        }

        return err;
    }


    //////////////////////////////////////////////////////////////////////////
    // Function: jointhread
    //
    //////////////////////////////////////////////////////////////////////////
    int jointhread(ThreadHandle threadHandle, unsigned int millisecTimeout, void** ppReturnValue)
    {
        int err = eINVAL;

        InternalUtilsPrivate::HandleNumber handleNumber = threadHandle;
        auto& pThread = threadHandles.FindHandle(handleNumber);
        if (pThread)
        {
            err = 0;

            if (millisecTimeout == static_cast<unsigned int>(-1))
            {
                pThread->wait();
            }
            else if (millisecTimeout > 0)
            {
                auto status = pThread->wait_for(std::chrono::milliseconds(millisecTimeout));
                if (status == std::future_status::timeout)
                {
                    err = eTIMEDOUT;
                }
            }
            if (pThread->valid() && ppReturnValue != NULL)
            {
                *ppReturnValue = pThread->get();
            }
        }

        return err;
    }

} // end namespace InternalUtils
