/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
/// @internal
///
/// @brief Declaration of functions for managing threads of execution.
///
//////////////////////////////////////////////////////////////////////////////

#ifndef __BEGINTHREAD_H__
#define __BEGINTHREAD_H__

#include <errno.h>
#include "threadtypes.h"

#ifdef _MSC_VER
/// Calling convention for thread function.
#define THREADPROC_CONVENTION __stdcall
#else
/// Calling convention for thread function.
#define THREADPROC_CONVENTION
#endif

namespace InternalUtils
{

    /// Definition of the function to call on the new thread.
    ///
    /// On Windows, the return value is ignored.
    typedef void* (THREADPROC_CONVENTION *ThreadProc)(void* data);


    //////////////////////////////////////////////////////////////////////////
    //  Function: beginthread
    //
    /// @brief Creates and starts a background thread running a specified
    ///        function.
    ///
    /// This helper function starts a background thread running the specified
    /// function.
    ///
    /// @par Assumptions:
    ///   None.
    ///
    /// @par Side Effects:
    ///   A thread is created.
    ///
    /// @param function
    ///     The function to execute on the new thread.
    /// @param stackSize
    ///     The size of the stack to use, in bytes.  Pass 0 to use the
    ///     default stack size for the operating system.
    /// @param userData
    ///     A pointer to data to be passed to the function in the other thread.
    ///     This could be a pointer to a structure, class, or a 64-bit integer.
    ///     Note: if passing a pointer to a structure or class, access to the
    ///     structure or class should be properly managed for multi-threaded
    ///     access.
    /// @param pThreadHandle
    ///     Returns a ThreadHandle representing the created thread.  This can
    ///     be passed to jointhread().
    ///
    /// @return (in):
    ///     Returns 0 if successful; otherwise, returns an error code (errno).
    //////////////////////////////////////////////////////////////////////////
    int beginthread(ThreadProc function, unsigned int stackSize, void* userData, ThreadHandle* pThreadHandle);


    //////////////////////////////////////////////////////////////////////////
    //  Function: jointhread
    //
    /// @brief Joins a specified thread and waits for it end.
    ///
    /// This helper function waits on a thread until it exits.
    ///
    /// @par Assumptions:
    ///   None.
    ///
    /// @par Side Effects:
    ///   None.
    ///
    /// @param threadHandle
    ///     The handle to the thread to wait on.
    /// @param millisecTimeout
    ///     The number of milliseconds to wait for the thread to terminate.
    ///     pass TIMEOUT_INFINITE to wait forever.
    /// @param ppReturnValue
    ///     Returns the return value from thread.  If this is NULL, the
    ///     return value from the thread is ignored.  On Windows, the return
    ///     value is limited to 32 bits in size.
    ///
    /// @return (int):
    ///     Returns 0 for success; otherwise, returns an error code.
    /// @retval eINVAL
    ///     The thread handle is invalid.
    /// @retval eSRCH
    ///     The given thread handle does not correspond to a valid thread.
    /// @retval eTIMEDOUT
    ///     The join timed out before the thread terminated.
    /// @retval eDEADLK
    ///     The thread being joined to is already trying to join the current thread.
    //////////////////////////////////////////////////////////////////////////
    int jointhread(ThreadHandle threadHandle, TimeoutDelay millisecTimeout, void** ppReturnValue);

} // end namespace InternalUtils

#endif // __BEGINTHREAD_H__
