/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#include "stdafx.h"

#include <algorithm>

#include <Foundation/Error/ErrorMacros.h>
#include <Components/Common/Common.h>
#include <Components/Configuration/Configuration.h>
#include <ExceptionBridge.h>
#include <SafeString.h>

#include "ConfigurationManager.h"

namespace
{

    ConfigurationManager _configurationManager;

}

OpenIPC_Error SelectByFile_Impl(
    IN const char* configurationFilePath)
{
    return CallAndTranslateError(
        [&]()
        {
            if (!configurationFilePath)
            {
                THROW_POSTED_EXCEPTION(OpenIPC_Error_Null_Pointer);
            }

            _configurationManager.SelectConfigFile(configurationFilePath);
        });
}

OpenIPC_Error SelectByName_Impl(
    IN const char* configurationName)
{
    return CallAndTranslateError(
        [&]()
        {
            if (!configurationName)
            {
                THROW_POSTED_EXCEPTION(OpenIPC_Error_Null_Pointer);
            }

            _configurationManager.SelectConfig(configurationName);
        });
}

OpenIPC_Error LoadProbeConfiguration_Impl(
    IN const char* configurationName)
{
    return CallAndTranslateError(
        [&]()
        {
            if (!configurationName)
            {
                THROW_POSTED_EXCEPTION(OpenIPC_Error_Null_Pointer);
            }

            _configurationManager.LoadReferencedProbeConfig(configurationName);
        });
}

OpenIPC_Error SetAutomaticConfigurationEnabled_Impl(
    IN OpenIPC_Bool enabled)
{
    return CallAndTranslateError(
        [&]()
        {
            _configurationManager.SetAutomaticConfigurationEnabled(enabled == OpenIPC_TRUE);
        });
}

OpenIPC_Error IsAutomaticConfigurationEnabled_Impl(
    OUT OpenIPC_Bool* enabled)
{
    return CallAndTranslateError(
        [&]()
        {
            if (!enabled)
            {
                THROW_POSTED_EXCEPTION(OpenIPC_Error_Null_Pointer);
            }

            *enabled = _configurationManager.IsAutomaticConfigurationEnabled() ? OpenIPC_TRUE : OpenIPC_FALSE;
        });
}

OpenIPC_Error SetParameterValues_Impl(
    IN uint32_t parameterCount,
    IN const char(*parameterNames)[OpenIPC_MAX_CONFIGURATION_PARAMETER_NAME_LENGTH],
    IN const char(*parameterValues)[OpenIPC_MAX_CONFIGURATION_PARAMETER_VALUE_LENGTH])
{
    return CallAndTranslateError(
        [&]()
        {
            if (!parameterNames || !parameterValues)
            {
                THROW_POSTED_EXCEPTION(OpenIPC_Error_Null_Pointer);
            }

            if (parameterCount > 0)
            {
                DataAttribute::ParameterValues parameterNamesToValues;
                for (uint32_t parameterIndex = 0; parameterIndex < parameterCount; ++parameterIndex)
                {
                    parameterNamesToValues[parameterNames[parameterIndex]] = parameterValues[parameterIndex];
                }

                _configurationManager.SetParameterValues(parameterNamesToValues);
            }
        });
}

OpenIPC_Error Finalize_Impl(
    void)
{
    return CallAndTranslateError(
        [&]()
        {
            _configurationManager.Finalize();
        });
}

OpenIPC_Error Reset_Impl(
    void)
{
    _configurationManager = ConfigurationManager();
    return OpenIPC_Error_No_Error;
}

OpenIPC_Error GetName_Impl(
    OUT char configurationName[OpenIPC_MAX_CONFIGURATION_NAME_LENGTH])
{
    return CallAndTranslateError(
        [&]()
        {
            if (!configurationName)
            {
                THROW_POSTED_EXCEPTION(OpenIPC_Error_Null_Pointer);
            }

            const std::string& name = _configurationManager.GetConfigName();
            CommonUtils::SafeStringCopy(configurationName, OpenIPC_MAX_CONFIGURATION_NAME_LENGTH, name);
        });
}

OpenIPC_Error GetPrimaryElement_Impl(
    OUT const OpenIPC_ConfigurationElement** element)
{
    return CallAndTranslateError(
        [&]()
        {
            if (!element)
            {
                THROW_POSTED_EXCEPTION(OpenIPC_Error_Null_Pointer);
            }

            const DataElement* dataElement = _configurationManager.GetPrimaryElement();
            *element = reinterpret_cast<const OpenIPC_ConfigurationElement*>(dataElement);
        });
}

OpenIPC_Error GetProbeElement_Impl(
    OUT const OpenIPC_ConfigurationElement** element)
{
    return CallAndTranslateError(
        [&]()
        {
            if (!element)
            {
                THROW_POSTED_EXCEPTION(OpenIPC_Error_Null_Pointer);
            }

            const DataElement* dataElement = _configurationManager.GetProbeElement();
            *element = reinterpret_cast<const OpenIPC_ConfigurationElement*>(dataElement);
        });
}

OpenIPC_Error GetProbePluginElement_Impl(
    IN const char*                           pluginName,
    OUT const OpenIPC_ConfigurationElement** element)
{
    return CallAndTranslateError(
        [&]()
        {
            if (!pluginName || !element)
            {
                THROW_POSTED_EXCEPTION(OpenIPC_Error_Null_Pointer);
            }

            const DataElement* dataElement = _configurationManager.GetProbePluginElement(pluginName);
            *element = reinterpret_cast<const OpenIPC_ConfigurationElement*>(dataElement);
        });
}

OpenIPC_Error GetRunControlElement_Impl(
    OUT const OpenIPC_ConfigurationElement** element)
{
    return CallAndTranslateError(
        [&]()
        {
            if (!element)
            {
                THROW_POSTED_EXCEPTION(OpenIPC_Error_Null_Pointer);
            }

            const DataElement* dataElement = _configurationManager.GetRunControlElement();
            *element = reinterpret_cast<const OpenIPC_ConfigurationElement*>(dataElement);
        });
}

OpenIPC_Error GetEnabledProbePluginCount_Impl(
    OUT uint32_t* pluginCount)
{
    return CallAndTranslateError(
        [&]()
        {
            if (!pluginCount)
            {
                THROW_POSTED_EXCEPTION(OpenIPC_Error_Null_Pointer);
            }

            *pluginCount = static_cast<uint32_t>(_configurationManager.GetEnabledProbePluginNames().size());
        });
}

OpenIPC_Error GetEnabledProbePluginNames_Impl(
    IN uint32_t pluginCount,
    OUT char (*pluginNames)[OpenIPC_MAX_IDENTIFIER_LENGTH])
{
    return CallAndTranslateError(
        [&]()
        {
            if (!pluginNames)
            {
                THROW_POSTED_EXCEPTION(OpenIPC_Error_Null_Pointer);
            }

            auto names = _configurationManager.GetEnabledProbePluginNames();
            uint32_t effectiveCount = std::min(pluginCount, static_cast<uint32_t>(names.size()));
            for (uint32_t pluginIndex = 0; pluginIndex < effectiveCount; ++pluginIndex)
            {
                CommonUtils::SafeStringCopy(pluginNames[pluginIndex], OpenIPC_MAX_IDENTIFIER_LENGTH, names[pluginIndex]);
            }
        });
}

OpenIPC_Error GetEnabledRunControlPluginCount_Impl(
    OUT uint32_t* pluginCount)
{
    return CallAndTranslateError(
        [&]()
        {
            if (!pluginCount)
            {
                THROW_POSTED_EXCEPTION(OpenIPC_Error_Null_Pointer);
            }

            *pluginCount = static_cast<uint32_t>(_configurationManager.GetEnabledRunControlPluginNames().size());
        });
}

OpenIPC_Error GetEnabledRunControlPluginNames_Impl(
    IN uint32_t pluginCount,
    OUT char (*pluginNames)[OpenIPC_MAX_IDENTIFIER_LENGTH])
{
    return CallAndTranslateError(
        [&]()
        {
            if (!pluginNames)
            {
                THROW_POSTED_EXCEPTION(OpenIPC_Error_Null_Pointer);
            }

            auto names = _configurationManager.GetEnabledRunControlPluginNames();
            uint32_t effectiveCount = std::min(pluginCount, static_cast<uint32_t>(names.size()));
            for (uint32_t pluginIndex = 0; pluginIndex < effectiveCount; ++pluginIndex)
            {
                CommonUtils::SafeStringCopy(pluginNames[pluginIndex], OpenIPC_MAX_IDENTIFIER_LENGTH, names[pluginIndex]);
            }
        });
}

static OpenIPC_ConfigurationComponent _configurationComponent =
{
    SelectByFile_Impl,
    SelectByName_Impl,
    LoadProbeConfiguration_Impl,
    SetAutomaticConfigurationEnabled_Impl,
    IsAutomaticConfigurationEnabled_Impl,
    SetParameterValues_Impl,
    Finalize_Impl,
    Reset_Impl,
    GetName_Impl,
    GetPrimaryElement_Impl,
    GetProbeElement_Impl,
    GetProbePluginElement_Impl,
    GetRunControlElement_Impl,
    GetEnabledProbePluginCount_Impl,
    GetEnabledProbePluginNames_Impl,
    GetEnabledRunControlPluginCount_Impl,
    GetEnabledRunControlPluginNames_Impl
};

OpenIPC_Error OpenIPC_GetConfiguration(OpenIPC_ConfigurationComponent** component)
{
    OpenIPC_Error error = OpenIPC_Error_No_Error;

    if (component == nullptr)
    {
        error = OpenIPC_Error_Null_Pointer;
        POST_ERROR_MESSAGE(error, "OpenIPC_GetConfiguration() argument error (component) : Null pointer");
    }
    else
    {
        *component = &_configurationComponent;
    }

    return error;
}
