/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#include "stdafx.h"
#include "ConfigurationManager.h"

#include <boost/filesystem.hpp>
#include <boost/algorithm/string.hpp>
#include <regex>

#include <Foundation/Error/ErrorMacros.h>
#include <Foundation/Logging/OpenIPC_Logger.h>
#include <Foundation/Telemetry/Telemetry.h>
#include <Components/Configuration/ConfigurationErrors.h>
#include <DataStore.h>
#include <ManipulatePath.h>
#include <ExceptionBridge.h>

namespace
{

    static const std::string _defaultIpcConfigSchemaName    = "OpenIpcConfig.Default.xsd";
    static const std::string _defaultConfigOuterElement     = "DefaultConfigs";
    static const std::string _defaultIpcConfigElement       = "DefaultIpcConfig";
    static const std::string _defaultIpcConfigNameAttribute = "Name";
    static const std::string _defaultIpcConfigPathAttribute = "ConfigFile";
    static const std::string _defaultIpcConfigFile = "../Config/OpenIpcConfig.xml";

    static std::unique_ptr<OpenIPC_Logger> _logger(OpenIPC_Logger::GetLogger(OpenIPC_LoggingLoggerId::OpenIPC_LoggingLoggerId_Configuration));

}

void ConfigurationManager::SetConfigName(const std::string& configName)
{
    _configName = configName;
}

const std::string& ConfigurationManager::GetConfigName() const
{
    return _configName;
}

void ConfigurationManager::Finalize()
{
    if (!_configurationLoaded)
    {
        // If no config name was set and no config was selected by file
        if (_configName.empty() && _configFilePath.empty())
        {
            // Select the default config file
            SelectConfigFile(_defaultIpcConfigFile);
        }

        // If a config name was set and no config was selected
        else if (!_configName.empty() && _configFilePath.empty())
        {
            // Select the config with the set name
            SelectConfig(_configName);
        }
    }
}

void ConfigurationManager::SelectConfig(const std::string& configName, const std::string& configFilePath)
{
    if (_configurationLoaded)
    {
        THROW_POSTED_EXCEPTION(OpenIPC_Error_Already_Initialized);
    }

    _IncludeConfigDirectory();

    // Load the config
    _LoadOpenIpcConfig(configFilePath, configName);
}

void ConfigurationManager::SelectConfigFile(const std::string& configFilePath)
{
    if (_configurationLoaded)
    {
        THROW_POSTED_EXCEPTION(OpenIPC_Error_Already_Initialized);
    }

    // Resolve the path to the config file and the associated schema file
    std::string resolvedConfigFilePath;
    std::string schemaFilePath;
    _ResolveConfigFilePath(configFilePath, resolvedConfigFilePath, schemaFilePath);

    std::vector<DataStoreException> dataStoreExceptions;

    // Add the config file to Data Store
    DataStore* dataStore = DataStore::Get();
    dataStore->AddDocument(resolvedConfigFilePath, false, true, schemaFilePath, dataStoreExceptions);
    _ReportDataStoreExceptions(dataStoreExceptions);

    // Extract the config name and optional referenced config file path
    std::string configName;
    std::string referencedConfigFilePath;
    if (!_automaticConfigurationEnabled)
    {
        _ProcessDefaultConfigFile(resolvedConfigFilePath, schemaFilePath, configName, referencedConfigFilePath, _automaticConfigurationEnabled);
    }

    if (_automaticConfigurationEnabled)
    {
        _IncludeConfigDirectory();
        _configurationLoaded = true;
    }
    else
    {
        // Get the config directory path for OpenIPC
        const std::string configDirectoryPath = _GetConfigDirectoryPath();

        // If the config file did not reference another config file
        if (referencedConfigFilePath.empty())
        {
            SelectConfig(configName, resolvedConfigFilePath);
        }
        else
        {
            // Resolve the path to the referenced config file
            _ResolveConfigFilePath(referencedConfigFilePath, resolvedConfigFilePath, schemaFilePath);

            boost::filesystem::path path = resolvedConfigFilePath;
            if (path.is_relative())
            {
                path = boost::filesystem::path(configFilePath).parent_path() / referencedConfigFilePath;
            }

            if (!boost::filesystem::exists(path))
            {
                THROW_POSTED_EXCEPTION_MESSAGE(OpenIPC_Error_Invalid_Configuration, "Cannot find the configuration file '" << path.string() << "' specified by '" << _defaultIpcConfigPathAttribute << "' in " << _defaultIpcConfigElement);
            }

            // Add the referenced config file to data store
            dataStore->AddDocument(path.string(), false, true, schemaFilePath, dataStoreExceptions);
            dataStore->IncludeDirectory(configDirectoryPath, false, false, dataStoreExceptions);
            _ReportDataStoreExceptions(dataStoreExceptions);

            // Load the config
            _LoadOpenIpcConfig(resolvedConfigFilePath, configName);
        }
    }
}

void ConfigurationManager::LoadReferencedProbeConfig(const std::string& probeConfigName)
{
    // Query for the probe config by name
    DataQuery query;
    query.SelectElements("ProbeConfigs", "ProbeConfig")
        .WhereAttributeHas("Name", probeConfigName);

    DataStore* dataStore = DataStore::Get();
    DataQueryResult    result = dataStore->Find(query);
    const DataElement* firstElement = result.FirstDataElement();
    if (result.ErrorOccurred())
    {
        THROW_POSTED_EXCEPTION_MESSAGE(OpenIPC_Error_Invalid_Configuration, result.ErrorMessage());
    }
    else if (result.DataElementCount() == 0 || firstElement == nullptr)
    {
        THROW_POSTED_EXCEPTION_MESSAGE(OpenIPC_Error_Invalid_Configuration, "Unable to find OpenIPC probe configuration '" << probeConfigName << "'");
    }
    else if (result.DataElementCount() > 1)
    {
        THROW_POSTED_EXCEPTION_MESSAGE(OpenIPC_Error_Invalid_Configuration, "Multiple OpenIPC probe configurations named '" << probeConfigName << "' were found");
    }
    else
    {
        _probeConfigDocuments.push_back(std::make_shared<DataDocument>(*firstElement));

        DataElement* probeConfigElement = _probeConfigDocuments.back()->RootElement();
        if (!_probeElement)
        {
            _probeElement = probeConfigElement;
        }
        _probeElements.push_back(probeConfigElement);

        _ParseParameters(*probeConfigElement);

        DataElement* pluginsElement = probeConfigElement->FindChildByName("Plugins");
        if (pluginsElement)
        {
            // Remember each plug-in element
            DataElement* pluginElement = pluginsElement->FirstChild();
            while (pluginElement)
            {
                // Check if there are any plug-ins in the referenced
                // config that were already referenced in another
                // probe config
                const bool pluginAlreadyReferenced = std::any_of(_probePluginElements.begin(), _probePluginElements.end(),
                                                                 [&](const DataElement* e)
                                                                 {
                                                                     return e->Name() == pluginElement->Name();
                                                                 });

                if (pluginAlreadyReferenced)
                {
                    THROW_POSTED_EXCEPTION_MESSAGE(OpenIPC_Error_Invalid_Configuration, "Probe plug-in '" << pluginElement->Name() << "' used in multiple probe plug-ins referenced in the OpenIPC configuration XML");
                }

                _probePluginElements.push_back(pluginElement);
                _ResolveProbeConfigIncludes(*pluginElement);

                pluginElement = pluginElement->NextSibling();
            }
        }
    }
}

void ConfigurationManager::SetAutomaticConfigurationEnabled(bool enabled)
{
    _automaticConfigurationEnabled = enabled;
}

bool ConfigurationManager::IsAutomaticConfigurationEnabled() const
{
    return _automaticConfigurationEnabled;
}

void ConfigurationManager::SetParameterValues(const DataAttribute::ParameterValues& parameterValues)
{
    Finalize();
    for (auto& item : parameterValues)
    {
        _parameterValues[item.first] = item.second;
    }
    _EnsureParametersResolved();
}

const DataElement* ConfigurationManager::GetPrimaryElement()
{
    _EnsureConfigurationLoadedAndParametersResolved();
    return _primaryElement;
}

const DataElement* ConfigurationManager::GetProbeElement()
{
    _EnsureConfigurationLoadedAndParametersResolved();
    return _probeElement;
}

const DataElement* ConfigurationManager::GetProbePluginElement(const std::string& pluginName)
{
    _EnsureConfigurationLoadedAndParametersResolved();

    for (const DataElement* element : _probePluginElements)
    {
        if (element->Name() == pluginName)
        {
            return element;
        }
    }

    return nullptr;
}

const DataElement* ConfigurationManager::GetRunControlElement()
{
    _EnsureConfigurationLoadedAndParametersResolved();
    return _runControlElement;
}

std::vector<std::string> ConfigurationManager::GetEnabledProbePluginNames() const
{
    std::vector<std::string> pluginNames;

    for (const DataElement* pluginElement : _probePluginElements)
    {
        const DataAttribute* enableAttribute = pluginElement->FindAttribute("Enable");
        if (enableAttribute && enableAttribute->ToBool())
        {
            pluginNames.push_back(pluginElement->Name());
        }
    }

    return pluginNames;
}

std::vector<std::string> ConfigurationManager::GetEnabledRunControlPluginNames() const
{
    std::vector<std::string> pluginNames;

    if (_runControlElement)
    {
        const DataElement* vendorsElement = _runControlElement->FindDescendantByName("Vendors");
        if (vendorsElement)
        {
            const DataElement::Array pluginElements = vendorsElement->FindChildrenByName("Plugin");
            for (DataElement* pluginElement : pluginElements)
            {
                const DataAttribute* nameAttribute = pluginElement->FindAttribute("Name");
                if (nameAttribute)
                {
                    pluginNames.push_back(nameAttribute->Value());
                }
            }
        }
    }

    return pluginNames;
}

void ConfigurationManager::_ParseParameters(const DataElement& element)
{
    // Find the parameters element
    DataElement* parametersElement = element.FindChildByName("Parameters");
    if (parametersElement)
    {
        // For each parameter
        DataElement::Array parameterElements = parametersElement->FindChildrenByName("Parameter");
        for (DataElement* parameterElement : parameterElements)
        {
            // Get the name
            DataAttribute* nameAttribute = parameterElement->FindAttribute("Name");
            if (nameAttribute)
            {
                const std::string& name = nameAttribute->Value();

                // Get the default value
                bool hasDefault = false;
                std::string    defaultValue;
                DataAttribute* defaultValueAttribute = parameterElement->FindAttribute("DefaultValue");
                if (defaultValueAttribute)
                {
                    hasDefault   = true;
                    defaultValue = defaultValueAttribute->Value();
                }

                // Get the valid values
                std::string    validValues;
                DataAttribute* validValuesAttribute = parameterElement->FindAttribute("ValidValues");
                if (validValuesAttribute)
                {
                    validValues = validValuesAttribute->Value();
                }

                auto it = _parameters.find(name);
                if (it != _parameters.end())
                {
                    THROW_POSTED_EXCEPTION_MESSAGE(OpenIPC_Error_Invalid_Configuration, "Duplicate configuration parameter '" << name << "' declared in '" << LOG_ELEMENT_CONTEXT(parameterElement) << "; first declared in " << LOG_ELEMENT_CONTEXT(it->second.Element));
                }
                else
                {
                    _parameters[name]   = Parameter { name, hasDefault, defaultValue, validValues, parameterElement };
                    _parametersResolved = false;
                }
            }
        }
    }
}

void ConfigurationManager::_ParseParameterValues(const DataElement& element)
{
    // Find the parameters element
    DataElement* parametersElement = element.FindChildByName("Parameters");
    if (parametersElement)
    {
        // For each parameter
        DataElement::Array parameterElements = parametersElement->FindChildrenByName("Parameter");
        for (DataElement* parameterElement : parameterElements)
        {
            // Get the name
            DataAttribute* nameAttribute = parameterElement->FindAttribute("Name");
            if (nameAttribute)
            {
                const std::string& name = nameAttribute->Value();
                std::string value;

                // Get the default value
                DataAttribute* valueAttribute = parameterElement->FindAttribute("Value");
                if (valueAttribute)
                {
                    value = valueAttribute->Value();
                }

                // Remember the value for this parameter to be resolved later if
                // it didn't already have a value
                if (_parameterValues.count(name) == 0)
                {
                    _parameterValues[name] = value;
                }
            }
        }
    }
}

void ConfigurationManager::_ValidateParameterValue(const Parameter& parameter, const std::string& value)
{
    // If the valid values pattern was specified
    if (!parameter.ValidValues.empty())
    {
        std::regex regex(parameter.ValidValues);
        if (!std::regex_match(value, regex))
        {
            THROW_POSTED_EXCEPTION_MESSAGE(OpenIPC_Error_Invalid_Configuration, "Configuration parameter value '" << value << "' does not match validation pattern '" << parameter.ValidValues << "' for parameter '" << parameter.Name << "' declared in '" << LOG_ELEMENT_CONTEXT(parameter.Element));
        }
    }
}

void ConfigurationManager::_LoadOpenIpcConfig(const std::string& configFilePath, const std::string& configName)
{
    _configName = configName;

    // Verify that the config name is not empty
    if (configName.empty())
    {
        TELEMETRY_BUILD_DATA(Telemetry_Event_Server_Init, "configstatus", "fail_noconfig");
        if (configFilePath.empty())
        {
            //The way the rest of this class works makes it impossible to get here today. But there's nothing preventing
            //anyone from changing that.
            THROW_POSTED_EXCEPTION_MESSAGE(OpenIPC_Error_Not_Configured, "The OpenIPC is not configured. A configuration name must be specified.");
        }
        else
        {
            THROW_POSTED_EXCEPTION_MESSAGE(OpenIPC_Error_Not_Configured, "The OpenIPC is not configured. A configuration name must be specified in: '" << boost::filesystem::canonical(configFilePath).string() << "'");
        }
    }

    OpenIPC_LOG(_logger, OpenIPC_LoggingSeverityLevel_Debug, "Initializing TargetManager with configuration '" << configName << "'...");
    if (configName == "LastSaved")
    {
        TELEMETRY_BUILD_DATA(Telemetry_Event_Server_Init, "configstatus", "last_saved");
        _configurationLoaded = true;
        return;
    }
    DataStore* dataStore = DataStore::Get();

    DataQuery query;
    query.SelectElements("OpenIpcConfigs", "OpenIpcConfig")
        .WhereAttributeHas("Name", configName);

    DataQueryResult    result = dataStore->Find(query);
    const DataElement* firstElement = result.FirstDataElement();
    if (result.ErrorOccurred())
    {
        TELEMETRY_BUILD_DATA(Telemetry_Event_Server_Init, "configstatus", "fail_invalid");
        THROW_POSTED_EXCEPTION_MESSAGE(OpenIPC_Error_Invalid_Configuration, result.ErrorMessage());
    }
    else if (result.DataElementCount() == 0 || firstElement == nullptr)
    {
        // No config found
        TELEMETRY_BUILD_DATA(Telemetry_Event_Server_Init, "configstatus", "fail_notfound");
        THROW_POSTED_EXCEPTION_MESSAGE(OpenIPC_Error_Unknown_Configuration, "Unable to find OpenIPC configuration named '" << configName << "'");
    }
    else if (result.DataElementCount() > 1)
    {
        // Multiple configurations found
        TELEMETRY_BUILD_DATA(Telemetry_Event_Server_Init, "configstatus", "fail_multiple");
        THROW_POSTED_EXCEPTION_MESSAGE(OpenIPC_Error_Invalid_Configuration, "Found multiple OpenIPC configurations named '" << configName << "'");
    }
    else
    {
        TELEMETRY_BUILD_DATA(Telemetry_Event_Server_Init, "configstatus", "ok");

        // Create a local copy of the OpenIPC config XML document and keep
        // a pointer to the config element
        _primaryConfigDocument = DataDocument(*firstElement->Document());
        _primaryElement = _primaryConfigDocument.RootElement()->FindDescendant(
            [&configName](const DataElement* element)
            {
                if (element->Name() == "OpenIpcConfig")
                {
                    DataAttribute* nameAttribute = element->FindAttribute("Name");
                    if (nameAttribute)
                    {
                        return boost::iequals(nameAttribute->Value(), configName);
                    }
                }

                return false;
            });

        // Parse the parameters of the OpenIPC config
        _ParseParameters(*_primaryElement);
        _LoadProbeConfig();
        _LoadRunControlConfig();
        _configurationLoaded = true;
    }
}

void ConfigurationManager::_LoadProbeConfig()
{
    // Use the inline probe config if it is present
    DataElement* probeElement = _primaryElement->FindChildByName("Probe");

    // If there is no inline config
    if (!probeElement)
    {
        // If an external probe config is referenced
        DataElement::Array probeConfigElements = _primaryElement->FindDescendantsByName("ProbeConfig");
        for (DataElement* probeConfigElement : probeConfigElements)
        {
            // Parse the specified parameter values
            _ParseParameterValues(*probeConfigElement);

            // Get the config name
            DataAttribute* nameAttribute = probeConfigElement->FindAttribute("Name");
            if (nameAttribute)
            {
                const std::string& configName = nameAttribute->Value();

                // Load the config by name
                LoadReferencedProbeConfig(configName);
            }
        }
    }
    else
    {
        _probeConfigDocuments.push_back(std::make_shared<DataDocument>(*probeElement));

        DataElement* probeConfigElement = _probeConfigDocuments.back()->RootElement();
        if (!_probeElement)
        {
            _probeElement = probeConfigElement;
        }

        DataElement* pluginsElement = probeConfigElement->FindChildByName("Plugins");
        DataElement* pluginElement  = pluginsElement->FirstChild();
        while (pluginElement)
        {
            _probePluginElements.push_back(pluginElement);
            _ResolveProbeConfigIncludes(*pluginElement);

            pluginElement = pluginElement->NextSibling();
        }
    }
}

void ConfigurationManager::_ResolveProbeConfigIncludes(DataElement& probePluginElement)
{
    DataAttribute* includeAttribute = probePluginElement.FindAttribute("Include");
    if (includeAttribute)
    {
        DataDocument doc("Included");
        auto filePath = boost::filesystem::path(probePluginElement.Document()->FilePath()).remove_filename() / includeAttribute->Value();

        doc.LoadXmlFile(filePath.string(), true);
        auto child = doc.RootElement()->FirstChild();
        while (child)
        {
            auto newChild = probePluginElement.CreateChild("Temporary");
            newChild->Copy(child);
            child = child->NextSibling();
        }
    }
}

void ConfigurationManager::_LoadRunControlConfig()
{
    // Use the inline run control config if it is present
    _runControlElement = _primaryElement->FindDescendantByName("RunControl");

    // If there is no inline config
    if (!_runControlElement)
    {
        // If an external run control config is referenced
        DataElement* runControlConfigElement = _primaryElement->FindDescendantByName("RunControlConfig");
        if (runControlConfigElement)
        {
            // Parse the specified parameter values
            _ParseParameterValues(*runControlConfigElement);

            // Get the config name
            DataAttribute* nameAttribute = runControlConfigElement->FindAttribute("Name");
            if (nameAttribute)
            {
                const std::string& configName = nameAttribute->Value();

                // Load the config by name
                _LoadReferencedRunControlConfig(configName);
            }
        }
    }
}

void ConfigurationManager::_LoadReferencedRunControlConfig(const std::string& runControlConfigName)
{
    // Query for the run control config by name
    DataQuery query;
    query.SelectElements("RunControlConfigs", "RunControlConfig")
        .WhereAttributeHas("Name", runControlConfigName);

    DataStore* dataStore = DataStore::Get();
    DataQueryResult    result = dataStore->Find(query);
    const DataElement* firstElement = result.FirstDataElement();
    if (result.ErrorOccurred())
    {
        THROW_POSTED_EXCEPTION_MESSAGE(OpenIPC_Error_Invalid_Configuration, result.ErrorMessage());
    }
    else if (result.DataElementCount() == 0 || firstElement == nullptr)
    {
        THROW_POSTED_EXCEPTION_MESSAGE(OpenIPC_Error_Invalid_Configuration, "Unable to find OpenIPC run control configuration '" << runControlConfigName << "'");
    }
    else if (result.DataElementCount() > 1)
    {
        THROW_POSTED_EXCEPTION_MESSAGE(OpenIPC_Error_Invalid_Configuration, "Multiple OpenIPC run control configurations named '" << runControlConfigName << "' were found");
    }
    else
    {
        _runControlConfigDocument = DataDocument(*firstElement);
        _runControlElement = _runControlConfigDocument.RootElement();

        _ParseParameters(*_runControlElement);
    }
}

void ConfigurationManager::_EnsureParametersResolved()
{
    if (_parametersResolved)
    {
        return;
    }

    DataAttribute::ParameterValues parameterValues;

    // For each specified parameter
    for (auto& pair : _parameterValues)
    {
        const std::string& name  = pair.first;
        const std::string& value = pair.second;

        auto it = _parameters.find(name);
        if (it != _parameters.end())
        {
            const Parameter& parameter = it->second;

            _ValidateParameterValue(parameter, value);
            parameterValues[name] = value;
        }
        else
        {
            THROW_POSTED_EXCEPTION_MESSAGE(OpenIPC_Error_Invalid_Configuration, "Value specified for unknown configuration parameter '" << name << "'");
        }
    }

    // Use the default value for unspecified paramters
    for (auto& pair : _parameters)
    {
        const Parameter& parameter = pair.second;

        auto it = parameterValues.find(parameter.Name);
        if (it == parameterValues.end())
        {
            if (parameter.HasDefaultValue)
            {
                // The parameter was not specified but has a default value
                _ValidateParameterValue(parameter, parameter.DefaultValue);
                parameterValues[parameter.Name] = parameter.DefaultValue;
            }
            else
            {
                THROW_POSTED_EXCEPTION_MESSAGE(OpenIPC_Error_Invalid_Configuration, "Missing value for configuration parameter '" << parameter.Name << "' declared in '" << LOG_ELEMENT_CONTEXT(parameter.Element) << "'");
            }
        }
    }

    if (_primaryElement)
    {
        _primaryElement->ResolveAttributeParameters(parameterValues);
    }

    for (DataElement* probeElement : _probeElements)
    {
        probeElement->ResolveAttributeParameters(parameterValues);
    }

    if (_runControlElement)
    {
        _runControlElement->ResolveAttributeParameters(parameterValues);
    }

    _parametersResolved = true;
}

void ConfigurationManager::_EnsureConfigurationLoadedAndParametersResolved()
{
    if (!_configurationLoaded)
    {
        THROW_POSTED_EXCEPTION(OpenIPC_Error_Not_Configured);
    }

    _EnsureParametersResolved();
}

void ConfigurationManager::_ProcessDefaultConfigFile(const std::string& configFilePath, const std::string& schemaFilePath, std::string& configName, std::string& referencedConfigFilePath, bool& automaticConfigurationEnabled)
{
    automaticConfigurationEnabled = false;

    // Load the config XML file
    DataDocument document;
    try
    {
        document.LoadXmlFile(configFilePath, schemaFilePath);
    }
    catch (const DataStoreException& ex)
    {
        THROW_POSTED_EXCEPTION_MESSAGE(OpenIPC_Error_Invalid_Configuration, "The given configuration file is malformed: '" << ex.Message() << "'");
    }
    catch (const std::exception& ex)
    {
        THROW_POSTED_EXCEPTION_MESSAGE(OpenIPC_Error_Invalid_Configuration, "Received an unexpected exception while parsing the configuration file: '" << ex.what() << "'");
    }

    const DataElement* rootElement = document.RootElement();
    if (!rootElement)
    {
        THROW_POSTED_EXCEPTION_MESSAGE(OpenIPC_Error_Invalid_Configuration, "The given configuration file is empty");
    }

    if (rootElement->Name() != _defaultConfigOuterElement)
    {
        THROW_POSTED_EXCEPTION_MESSAGE(OpenIPC_Error_Invalid_Configuration, "The given configuration file is not an OpenIPC configuration file."
                                       << "The root element was expected to be 'DefaultConfigs', but is '" << rootElement->Name() << "'");
    }

    DataElement* configElement = rootElement->FindChildByName(_defaultIpcConfigElement);
    if (!configElement)
    {
        THROW_POSTED_EXCEPTION_MESSAGE(OpenIPC_Error_Invalid_Configuration, "The given configuration file is not an OpenIPC configuration file."
                                       << "Missing required element '" << _defaultIpcConfigElement << "'");
    }

    DataAttribute* configNameAttibute = configElement->FindAttribute(_defaultIpcConfigNameAttribute);
    if (configNameAttibute)
    {
        configName = configNameAttibute->Value();

        // Check if automatic configuration is enabled
        if (boost::iequals(configName, "Auto"))
        {
            configName.clear();
            automaticConfigurationEnabled = true;
        }
    }

    DataAttribute* configPathAttibute = configElement->FindAttribute(_defaultIpcConfigPathAttribute);
    if (configPathAttibute)
    {
        referencedConfigFilePath = configPathAttibute->Value();
    }

    _ParseParameterValues(*configElement);
}

void ConfigurationManager::_IncludeConfigDirectory()
{
    // Get the config directory path for OpenIPC
    const std::string configDirectoryPath = _GetConfigDirectoryPath();

    // Add the config directory to Data Store
    std::vector<DataStoreException> dataStoreExceptions;
    DataStore* dataStore = DataStore::Get();
    dataStore->IncludeDirectory(configDirectoryPath, true, true, dataStoreExceptions);
    _ReportDataStoreExceptions(dataStoreExceptions);
}

void ConfigurationManager::_ResolveConfigFilePath(const std::string& configFilePath, std::string& resolvedConfigFilePath, std::string& schemaFilePath)
{
    const std::string openIpcPathString = _GetOpenIpcPath();

    boost::filesystem::path openIpcPath(openIpcPathString.data());
    boost::filesystem::path fullFilePath = configFilePath;

    OpenIPC_Error error = CommonUtils::SanitizePath(fullFilePath);
    if (!OpenIPC_PASS(error))
    {
        THROW_POSTED_EXCEPTION_MESSAGE(OpenIPC_Error_Invalid_Configuration, "Failed to sanitize the given configuration path '" << fullFilePath.string() << "'");
    }

    if (fullFilePath.is_relative())
    {
        fullFilePath = openIpcPath / fullFilePath;
    }

    if (!boost::filesystem::exists(fullFilePath))
    {
        THROW_POSTED_EXCEPTION_MESSAGE(OpenIPC_Error_Cannot_Open_File, "The given configuration file cannot be found: '" << configFilePath << "'");
    }

    boost::system::error_code errorCode;
    boost::filesystem::file_status fileStatus = boost::filesystem::status(fullFilePath, errorCode);

    if (errorCode)
    {
        THROW_POSTED_EXCEPTION_MESSAGE(OpenIPC_Error_Cannot_Open_File, "Failed to resolve the type of the the file: '" << configFilePath << "'. Error: " << errorCode.message());
    }
    else if (OpenIPC_PASS(error) && fileStatus.type() != boost::filesystem::regular_file)
    {
        THROW_POSTED_EXCEPTION_MESSAGE(OpenIPC_Error_Cannot_Open_File, "The given path is not a file: '" << configFilePath << "'");
    }

    resolvedConfigFilePath = fullFilePath.string();
    schemaFilePath = (openIpcPath.parent_path() / "Config" / "Schemas" / _defaultIpcConfigSchemaName).string();
}

std::string ConfigurationManager::_GetOpenIpcPath()
{
    OpenIPC_CommonComponent* common;
    ThrowIfError(OpenIPC_GetCommon(&common));

    size_t pathSize = 1024;
    std::vector<char> path(pathSize);

    OpenIPC_Error error = common->GetPath(&pathSize, path.data());
    if (error == OpenIPC_Error_Bad_Argument)
    {
        path.resize(pathSize);
        error = common->GetPath(&pathSize, path.data());
    }

    ThrowIfError(error);

    return std::string(path.data());
}

std::string ConfigurationManager::_GetConfigDirectoryPath()
{
    const std::string openIpcPath = _GetOpenIpcPath();
    return (boost::filesystem::path(openIpcPath).parent_path() / "Config").string();
}

void ConfigurationManager::_ReportDataStoreExceptions(const std::vector<DataStoreException>& dataStoreExceptions)
{
    // If any malformed XML was encountered
    if (dataStoreExceptions.size() > 0)
    {
        // Build a string containing all of the XML errors
        std::stringstream ss;
        ss << "Malformed XML: ";
        for (const DataStoreException& e : dataStoreExceptions)
        {
            ss << e.Message() << std::endl;
        }

        // Post the error
        THROW_POSTED_EXCEPTION_MESSAGE(OpenIPC_Error_Invalid_Configuration, ss.str());
    }
}
