//////////////////////////////////////////////////////////////////////////////
//
//                      INTEL CONFIDENTIAL
//       Copyright 2016-2017 Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its
// suppliers. Title to the Material remains with Intel Corporation, its
// suppliers, or licensors. The Material contains trade secrets and
// proprietary and confidential information of Intel Corporation, its
// suppliers, and licensors, and is protected by worldwide copyright and trade
// secret laws and treaty provisions. No part of the Material may be used,
// copied, reproduced, modified, published, uploaded, posted, transmitted,
// distributed, or disclosed in any way without Intel's prior express written
// permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or
// delivery of the Materials, either expressly, by implication, inducement,
// estoppel or otherwise. Any license under such intellectual property rights
// must be express and approved by Intel in writing.
//
// Unless otherwise agreed by Intel in writing, you may not remove or alter
// this notice or any other notice embedded in Materials by Intel or Intel's
// suppliers or licensors in any way.
//
//////////////////////////////////////////////////////////////////////////////
///  @file
///
///  @brief Contains methods for obtaining instances of the Probe interfaces
///
///  For additional information on obtaining and using instances, see @ref probeusage.
///
//////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include <unordered_map>

#undef min
#undef max


#include "ASD_Connection.h"
#include "ProbePluginErrors.h"

/// NOTE: Copied from Probe Plugin infrastructure
// Error contexts arranged by thread id.
thread_local ErrorContext* _threadErrorContext = nullptr;

// Last error message by thread id. Note this list will leak slightly as the number of threads grows.
thread_local std::pair<OpenIPC_Error, std::string> _threadLastErrorMessages;

ErrorContext::ErrorContext()
{
	_threadErrorContext = this;
}

ErrorContext::~ErrorContext()
{
	OpenIPC_Error error = OpenIPC_Error_No_Error;
	if (!_errorStack.empty())
	{
		error = _errorStack.back().first;
	}

	if (error != OpenIPC_Error_No_Error)
	{
		_threadLastErrorMessages = std::make_pair(error, GetErrorMessage());
	}
	else
	{
		_threadLastErrorMessages.first = OpenIPC_Error_No_Error;
		_threadLastErrorMessages.second.clear();
	}

	// Remove this error context from the stack
	if (_threadErrorContext == this)
	{
		_threadErrorContext = nullptr; // set to null instead of erase to save time on repeated alloc/dealloc
	}
}

// Note: A raw pointer is guaranteed to be safe here since the ErrorContext exists in a higher stack
// frame on the same thread. i.e. The pointer is registered in _contexts in its constructor and removed
// in its destructor, so if its in this list it definately exists.
ErrorContext* ErrorContext::GetErrorContext()
{
	return _threadErrorContext;
}

void ErrorContext::_PostInternal(OpenIPC_Error errorCode, std::string message)
{
	if (errorCode != OpenIPC_Error_No_Error)
	{
		_errorStack.push_back(std::pair<OpenIPC_Error, std::string>(errorCode, message));
	}
}

std::string ErrorContext::GetErrorMessage() const
{
	std::stringstream ss;

	if (!_errorStack.empty())
	{
		for (int i = static_cast<int>(_errorStack.size() - 1); i >= 0; --i)
		{
			if (!_errorStack[i].second.empty())
			{
				ss << _errorStack[i].second;
				if (i != 0)
				{
					ss << ".  ";
				}
			}
		}
	}

	return ss.str();
}

OpenIPC_Error PluginNotifications::Handler(PluginNotificationCallbackHandler notificationHandlerFunction)
{
	OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

	if (_notificationHandlerFunction == nullptr)
	{
		_notificationHandlerFunction = notificationHandlerFunction;
	}
	else
	{
		openIPCError = OpenIPC_Error_Callback_Already_Initialized;
		PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "A handler is already installed for this logger.");
	}

	return openIPCError;
}

OpenIPC_Error PluginLogger::Handler(PluginStreamLogCallbackHandler logHandlerFunction)
{
	OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

	if (_logStreamHandlerFunction == nullptr)
	{
		_logStreamHandlerFunction = logHandlerFunction;
	}
	else
	{
		openIPCError = OpenIPC_Error_Callback_Already_Initialized;
		PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "A handler is already installed for this logger.");
	}

	return openIPCError;
}

OpenIPC_Error PluginLogger::Handler(PluginLogCallbackHandler logHandlerFunction)
{
	OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

	if (_logHandlerFunction == nullptr)
	{
		_logHandlerFunction = logHandlerFunction;
	}
	else
	{
		openIPCError = OpenIPC_Error_Callback_Already_Initialized;
		PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
	}

	return openIPCError;
}

extern std::mutex  remoteConnectionAcccessControl;  // RemoteConnection.cpp
EXPORT_PREFIX Connection_Error RegisterLogHandler(PluginLogCallbackHandler handler)
{
	std::lock_guard<std::mutex> lock(remoteConnectionAcccessControl);
	if (OpenIPC_PASS(PluginLogger::GetInstance().Handler(handler))) {
		return No_Error;
	}
	// Handler returns PROBE_PLUGIN_ERRORS
	return Failed_To_Register;
}

EXPORT_PREFIX Connection_Error RegisterStreamLogHandler(PluginStreamLogCallbackHandler handler)
{
	std::lock_guard<std::mutex> lock(remoteConnectionAcccessControl);
	if (OpenIPC_PASS(PluginLogger::GetInstance().Handler(handler))) {
		return No_Error;
	}
	// Handler returns PROBE_PLUGIN_ERRORS
	return Failed_To_Register;
}

EXPORT_PREFIX Connection_Error RegisterNotificationHandler(PluginNotificationCallbackHandler handler)
{
	std::lock_guard<std::mutex> lock(remoteConnectionAcccessControl);
	if (OpenIPC_PASS(PluginNotifications::GetInstance().Handler(handler))) {
		return No_Error;
	}
	// Handler returns PROBE_PLUGIN_ERRORS
	return Failed_To_Register;
}

EXPORT_PREFIX Connection_Error ChangeNotificationLevel(PPI_NotificationLevel level)
{
	std::lock_guard<std::mutex> lock(remoteConnectionAcccessControl);
	PluginNotifications::GetInstance().Level(level);
	return No_Error;
}
