#include "ASDBaseProtocol.h"
#include "PluginLogger.h"


ASDBaseProtocol::ASDBaseProtocol()
{
	InitLocals();
}

ASDBaseProtocol::~ASDBaseProtocol()
{
	if (readMonitorThread) {
#if defined(HOST_LINUX) || defined(HOST_DARWIN)
		pthread_cancel(readMonitorThread);
		pthread_cond_destroy(&condition);
#else
		TerminateThread(readMonitorThread, 0);
#endif // DARWIN
	}

	if (write_message.buffer) {
		free(write_message.buffer);
		write_message.buffer = nullptr;
	}
	if (read_message.buffer) {
		free(read_message.buffer);
		read_message.buffer = nullptr;
	}
	if (send_buffer) {
		free(send_buffer);
		send_buffer = nullptr;
	}
}

void ASDBaseProtocol::ReceiveData(void)
{
	uint16_t dataSize = 0;
	unsigned int tag_queue_front = INVALID_TAG_NUMBER;
	bool assignedBufferList = false;
	std::shared_ptr<std::list<std::shared_ptr<OutputBuffers>>> tmp_input_list = nullptr;
	Connection_Error error = No_Error;

	auto logFuncRead = [&](std::ostream& log)
	{
		int i = 0;
		int bytes_per_line = 16;

		log << "Receiving packet tag:" << std::dec << read_message.header.tag << ". ";
		if (dataSize != 0) {
			log << "Showing data ONLY, no header." << std::endl;
			while (i < dataSize) {
				log << "0x" << std::setfill('0') << std::setw(2) << std::hex << (read_message.buffer[i++] & 0xff);
				if (i < dataSize) {
					if (i % bytes_per_line == 0) {
						log << std::endl;
					}
					else {
						log << " ";
					}
				}
			}
		}
		else {
			log << "No data.";
		}
	};

	switch (readPacket()) {
	case PacketRead_Complete:
		dataSize = (uint16_t)(read_message.header.size_lsb | (read_message.header.size_msb << 8));
		break;
	case PacketRead_NotComplete:
		dataSize = 0;
		goto release;
	case PacketRead_Error:
		dataSize = 0;
		error = Socket_Failed_Recv;
		goto release;
	}

	if (!mutex_lock(list_mutex)) {
		error = Could_Not_Handle_Mutex;
		goto release;
	}

	if (!payload_desc_queue.empty())
		tag_queue_front = payload_desc_queue.front().tag;

	if (!mutex_unlock(list_mutex)) {
		error = Could_Not_Handle_Mutex;
		goto release;
	}

	if (ProcessError(read_message) != No_Error) {
		goto release;
	}

	if (PluginLogger::GetInstance().IsLogEnabled(PPI_infoNotification)) {
		PPI_LOG_FUNC(probeInstance->deviceID, PPI_traceNotification, logFuncRead);
	}

	read_message.buffer_size = dataSize;
	if (read_message.header.enc_bit) {
		error = DecryptData(&read_message);
		if (!CONNECTION_PASS(error)) {
			goto release;
		}
	}

	if (read_message.header.type == AGENT_CONTROL_MESSAGE_TYPE) {
		char cmdCode = read_message.buffer[0];
		char data = read_message.buffer[1];
		uint16_t data_size =
			static_cast<uint16_t>(read_message.buffer_size - 1);
		switch (cmdCode)
		{
		case QUERY_SUPPORTED_MESSAGE_TYPES:
			break;
		case INIT_SECURE_CHANNEL:
			break;
		case TERMINATE_SECURE_CHANNEL:
			break;
		case QUERY_NUM_IN_FLIGHT_MSGS:
			if (data_size == 1) {
				if (probeInstance)
					PPI_LOG(probeInstance->deviceID, PPI_traceNotification,
						"Reading BMC Buffer Number=" << std::dec << int(data));
				if (maxNumBuffers >= uint8_t(data))
					maxNumBuffers = uint8_t(data);
				if (probeInstance)
					PPI_LOG(probeInstance->deviceID, PPI_traceNotification,
						"Setting max_buffers number =" << std::dec << maxNumBuffers);
			}
			break;
		case QUERY_MAX_MESSAGE_SIZE:
			{
				if (data_size == 2 ) {
					char data_msb = read_message.buffer[2];
					if (probeInstance)
						PPI_LOG(probeInstance->deviceID, PPI_traceNotification,
							"Reading BMC message size=" << std::dec << uint16_t((data & 0xFF) | (data_msb << 8)));
					if (payloadSize>=uint16_t((data & 0xFF) | (data_msb << 8)))
						payloadSize = uint16_t((data & 0xFF) | (data_msb << 8));
					if (probeInstance)
						PPI_LOG(probeInstance->deviceID, PPI_traceNotification,
							"Setting max message size=" << std::dec << payloadSize);
				}
			}
			break;
		case QUERY_SUPPORTED_JTAG_CHAINS:
			{
				if (data_size == 1 ) {
					supportedJtagChains = uint8_t(data);
					PPI_LOG(probeInstance->deviceID, PPI_traceNotification,
							"Reading Supported JTAG chains BMC message chains=" << supportedJtagChains);
				}
			}
			break;
		case QUERY_SUPPORTED_I2C_BUSES:
			{
				if (data_size >= 1 ) {
					supportedI2cBusesCount = uint8_t(data);
					PPI_LOG(probeInstance->deviceID, PPI_traceNotification,
							"Reading Supported I2C buses BMC message buses=" << supportedI2cBusesCount);

					if(supportedI2cBusesCount != data_size -1) {
						error = Error_Processing_Received_Data;
					} else {
						for(unsigned int i=0; i<supportedI2cBusesCount; i++) {
							supportedI2cBusAddresses[i].busSelect = read_message.buffer[i+2];
							supportedI2cBusAddresses[i].busSpeed = 100000;
							PPI_LOG(probeInstance->deviceID, PPI_traceNotification,
							"Reading Supported I2C buses BMC message address=" << read_message.buffer[i+1]);
						}
					}
				}
			}
			break;
		case OBTAIN_DOWNSTREAM_VERSION:
			if (data_size > 256)
				data_size = 256;
			if (probeInstance != nullptr && data_size > 0) {
				probeInstance->AppendDownstreamVersion(
					&read_message.buffer[1], data_size);
			}
			break;
		case HARDWARE_CONFIGURATION:
			// Do not log here! There appears to be a deadlock you will hit in the OpenIPC code.
			// We are currently blocking some OpenIPC logger code PPI_PluginSetNotificationRequiredForStreamLogCallBack()
			// and logging a message causes OpenIPC to hang.
			// PPI_LOG(probeInstance->deviceID, PPI_infoNotification, "Log level successfully sent to BMC.");
			break;
		case SOCKET_TIMEOUT:
			break;
		}
	}
	else if (read_message.header.type == HARDWARE_LOG_EVENT) {
		int data_size = read_message.buffer_size - 2; // First two bytes are command and level
													  // Extract log level
		LoggingConfiguration log_config = { {0} };
		log_config.value = read_message.buffer[1];
		// Extract message
		char *data = &read_message.buffer[2];
		data[data_size] = 0; // Terminate the message char string
		PPI_LOG_STREAM(probeInstance->deviceID, (PPI_Stream)log_config.Parsed.logging_stream, (PPI_NotificationLevel)log_config.Parsed.logging_level, data);
		goto release;
	}

	if (read_message.header.origin_id == BROADCAST_MESSAGE_ORIGIN_ID) {
		for (uint16_t i = 0; i < read_message.buffer_size; i++)
		{
			if (probeInstance != nullptr) {
				switch (read_message.buffer[i])
				{
				case TARGET_RESET_ASSERT:
					PPI_LOG(probeInstance->deviceID, PPI_traceNotification, "Received TARGET_RESET_ASSERT event");
					probeInstance->callEventHandler(PPI_targetReset, 1);
					break;
				case TARGET_RESET_DEASSERT:
					PPI_LOG(probeInstance->deviceID, PPI_traceNotification, "Received TARGET_RESET_DEASSERT event");
					probeInstance->callEventHandler(PPI_targetReset, 0);
					break;
				case TARGET_PRDY_EVENT:
					PPI_LOG(probeInstance->deviceID, PPI_traceNotification, "Received TARGET_PRDY_EVENT event");
					probeInstance->callEventHandler(PPI_targetPrdy, 1);
					probeInstance->callEventHandler(PPI_targetPrdy, 0);
					break;
				case TARGET_POWER_ASSERT:
					PPI_LOG(probeInstance->deviceID, PPI_traceNotification, "Received TARGET_POWER_ASSERT event");
					probeInstance->callEventHandler(PPI_targetCpuPower, 1);
					break;
				case TARGET_POWER_DEASSERT:
					PPI_LOG(probeInstance->deviceID, PPI_traceNotification, "Received TARGET_POWER_DEASSERT event");
					probeInstance->callEventHandler(PPI_targetCpuPower, 0);
					break;
				case WAITPRDY_TIMEOUT_EVENT:
					PPI_LOG(probeInstance->deviceID, PPI_traceNotification, "Received WAITPRDY_TIMEOUT event");
					clear_prdy_timeout = true;
					break;
				case MBP_EVENT:
					break;
				default:
					PPI_LOG(probeInstance->deviceID, PPI_traceNotification, "Received unknown event " << read_message.buffer[i]);
					break;
				}
			}
		}
	}
	else if (read_message.header.tag == tag_queue_front) {
		if (!mutex_lock(list_mutex)) {
			error = Could_Not_Handle_Mutex;
			goto release;
		}
		if (!assignedBufferList) {
			assignedBufferList = true;
			tmp_input_list = payload_desc_queue.front().out_buffers;
		}
		if (cmd_stat == 0 || read_message.header.cmd_stat == 0x80 || read_message.header.type == AGENT_CONTROL_MESSAGE_TYPE) {
			if (read_message.header.type == I2C_MESSAGE_TYPE)
				error = ProcessReadBuffersI2c(&read_message, tmp_input_list);
			else
				error = ProcessReadBuffers(&read_message, tmp_input_list);
			if (error != No_Error) {
				mutex_unlock(list_mutex);
				goto release;
			}

			if (read_message.header.cmd_stat != 0x80) {
				payload_desc_queue.pop();

				this->WakeReadEvent();

				if (tmp_input_list && !tmp_input_list->empty())
					error = Error_Processing_Received_Data;
			}
		}
		if (!mutex_unlock(list_mutex)) {
			error = Could_Not_Handle_Mutex;
			goto release;
		}
		if (!CONNECTION_PASS(readResult)) {
			goto release;
		}
	}

release:
	if (CONNECTION_PASS(readResult) && !CONNECTION_PASS(error)) {
		readResult = error;
	}
	return;
}

Connection_Error ASDBaseProtocol::ProcessError(message read_message)
{
	Connection_Error error = No_Error;
	std::string errorMessage;
	if (read_message.header.type != AGENT_CONTROL_MESSAGE_TYPE) {
		switch (read_message.header.cmd_stat) {
			case ASD_FAILURE_XDP_PRESENT:
				error = ASD_FAILURE_XDP_PRESENT;
				errorMessage = "XDP Presence Detected - Terminating ASD BMC process";
				break;
			case ASD_MSG_CRYPY_NOT_SUPPORTED:
				error = ASD_MSG_CRYPY_NOT_SUPPORTED;
				errorMessage = "ASD message encryption not supported";
				break;
			case ASD_FAILURE_INIT_JTAG_HANDLER:
				error = ASD_FAILURE_INIT_JTAG_HANDLER;
				errorMessage = "Failed to initialize the jtag handler";
				break;
			case ASD_FAILURE_INIT_I2C_HANDLER:
				error = ASD_FAILURE_INIT_I2C_HANDLER;
				errorMessage = "Failed to initialize the i2c handler";
				break;
			case ASD_FAILURE_DEINIT_JTAG_HANDLER:
				error = ASD_FAILURE_DEINIT_JTAG_HANDLER;
				errorMessage = "Failed to de-initialize the JTAG handler";
				break;
			case ASD_FAILURE_PROCESS_JTAG_MSG:
				error = ASD_FAILURE_PROCESS_JTAG_MSG;
				errorMessage = "Failed to process JTAG message";
				break;
			case ASD_MSG_NOT_SUPPORTED:
				error = ASD_MSG_NOT_SUPPORTED;
				errorMessage = "Received the message that is not supported by this host";
				break;
			case ASD_FAILURE_PROCESS_I2C_MSG:
				error = ASD_FAILURE_PROCESS_I2C_MSG;
				errorMessage = "Failed to process I2C message";
				break;
			case ASD_FAILURE_PROCESS_I2C_LOCK:
				error = ASD_FAILURE_PROCESS_I2C_LOCK;
				errorMessage = "Failed to apply i2c lock";
				break;
			case ASD_I2C_MSG_NOT_SUPPORTED:
				error = ASD_I2C_MSG_NOT_SUPPORTED;
				errorMessage = "Received I2C message that is not supported";
				break;
			case ASD_FAILURE_REMOVE_I2C_LOCK:
				error = ASD_FAILURE_REMOVE_I2C_LOCK;
				errorMessage = "Failed to remove i2c lock";
				break;
			case ASD_FAILURE_HEADER_SIZE:
				error = ASD_FAILURE_HEADER_SIZE;
				errorMessage = "Failed to read header size";
				break;
			case ASD_UNKNOWN_ERROR:
				error = ASD_UNKNOWN_ERROR;
				errorMessage = "Unknown Error";
				break;
			break;
		default:
			return No_Error;
		}
	}
	else {
		return No_Error;
	}
  
	PPI_LOG(probeInstance->deviceID, PPI_errorNotification, errorMessage);
	PluginNotifications::GetInstance().Notify(
		probeInstance->deviceID,
		error,
		PPI_warningNotification,
		[&](std::basic_ostream<char>& stream)
		{
			stream << errorMessage;
		});
	return error;
}

Connection_Error ASDBaseProtocol::SendData(bool flush, Connection_Error readResultInput)
{
	int iResult = 0;
	Connection_Error error = No_Error;
	struct payload_descritpor payload_desc;
	bool payload_desc_queue_monitor;
	int send_buffer_size = 0;
	auto logFuncSend = [&](std::ostream& log)
	{
		int i = 0;
		int bytes_per_line = 16;

		log << "Sending packet tag:" << std::dec << tag << ".  Sent Header:";
		while (i < send_buffer_size && i < sizeof(message_header)) {
			log << " 0x" << std::setfill('0') << std::setw(2) << std::hex << (send_buffer[i++] & 0xff);
		}
		if (i < send_buffer_size) {
			log << std::endl;
		}
		while (i < send_buffer_size) {
			log << "0x" << std::setfill('0') << std::setw(2) << std::hex << (send_buffer[i++] & 0xff);
			if (i < send_buffer_size) {
				if ((i - sizeof(message_header)) % bytes_per_line == 0) {
					log << std::endl;
				}
				else {
					log << " ";
				}
			}
		}
	};

	uint16_t sizeSending = write_message_offset;

	if (!CONNECTION_PASS(readResultInput)) {
		error = readResultInput;
		flush = true;
		goto flush;
	}

	write_message.header.type = msgType;
	write_message.header.reserved = 0x0;
	write_message.header.origin_id = 0x0;
	tag = (tag + 1) & 0x7;
	if (tag == 0)
		tag = 1;
	write_message.header.tag = tag;
	write_message.header.size_lsb = write_message_offset & 0xFF;
	write_message.header.size_msb = (write_message_offset >> 8) & 0x1F;
	write_message.header.cmd_stat = cmd_stat;
	write_message_offset = 0;

	if (msgType == DMA_MESSAGE_TYPE_WRITE || msgType == DMA_MESSAGE_TYPE_READ) {
		write_message.header.type = DMA_MESSAGE_TYPE;
		//for dma write message type needs to be 1 and for read 0 so just and with 1 and done
		write_message.header.cmd_stat = msgType & 1;
	}
	error = this->CheckReadAlive();
	if (!CONNECTION_PASS(error)) {
		goto end;
	}

	if (encryptionEnabled) {
		write_message.header.enc_bit = 0x1;
		error = EncryptData(&write_message);
		if (!CONNECTION_PASS(error))
			goto end;
	}
	else
		write_message.header.enc_bit = 0x0;

	if (payload_desc_queue.size() >= maxNumBuffers) {
		error = this->WaitForReadEvent();
		if (!mutex_lock(list_mutex)) {
			error = Could_Not_Handle_Mutex;
			goto end;
		}
		payload_desc_queue_monitor = payload_desc_queue.empty();
		if (!mutex_unlock(list_mutex)) {
			error = Could_Not_Handle_Mutex;
			goto end;
		}

		if (!CONNECTION_PASS(error) && !payload_desc_queue_monitor)
			goto end;
	}

	send_buffer_size = sizeof(struct message_header) + sizeSending;

	if (write_message.header.enc_bit != 0)
		send_buffer_size += sizeof(write_message.auth_code);

	memcpy_s(send_buffer, MAX_PACKET_PAYLOAD,
		(char*)(&write_message.header), sizeof(struct message_header));

	memcpy_s(send_buffer + sizeof(struct message_header),
		MAX_PACKET_PAYLOAD - sizeof(struct message_header),
		write_message.buffer, sizeSending);

	if (write_message.header.enc_bit != 0)
		memcpy_s(send_buffer + sizeof(struct message_header) + sizeSending,
			MAX_PACKET_PAYLOAD - sizeof(struct message_header) - sizeSending,
			write_message.auth_code, sizeof(write_message.auth_code));

	if (!mutex_lock(list_mutex)) {
		return Could_Not_Handle_Mutex;
	}
	iResult = UseTransport->Send(send_buffer, send_buffer_size);
	if (iResult < 0 || iResult != send_buffer_size) {
		error = Socket_Failed_Send;
		mutex_unlock(list_mutex);
		goto end;
	}

	if (PluginLogger::GetInstance().IsLogEnabled(PPI_infoNotification)) {
		PPI_LOG_FUNC(probeInstance->deviceID, PPI_traceNotification, logFuncSend);
	}

	payload_desc.tag = write_message.header.tag;
	payload_desc.out_buffers = out_buffers_list;
	payload_desc_queue.push(payload_desc);
	out_buffers_list = std::make_shared<std::list<std::shared_ptr<OutputBuffers>>>();
	if (!mutex_unlock(list_mutex)) {
		return Could_Not_Handle_Mutex;
	}

flush:
	if (flush) {
		if (!mutex_lock(list_mutex)) {
			error = Could_Not_Handle_Mutex;
			goto end;
		}
		payload_desc_queue_monitor = payload_desc_queue.empty();
		if (!mutex_unlock(list_mutex)) {
			error = Could_Not_Handle_Mutex;
			goto end;
		}

		while (!payload_desc_queue_monitor && CONNECTION_PASS(error)) {
			error = this->WaitForReadEvent();
			if (!mutex_lock(list_mutex)) {
				error = Could_Not_Handle_Mutex;
				goto end;
			}
			payload_desc_queue_monitor = payload_desc_queue.empty();
			if (!mutex_unlock(list_mutex)) {
				error = Could_Not_Handle_Mutex;
				goto end;
			}
			if (!CONNECTION_PASS(error) && !payload_desc_queue_monitor)
				goto end;
			error = readResult;
		}
	}

end:
	if (!CONNECTION_PASS(error)) {
		out_buffers_list = std::make_shared<std::list<std::shared_ptr<OutputBuffers>>>();
		switch (error) {
		case Connection_Error::Socket_Failed_Recv:
			PPI_LOG(probeInstance->deviceID, PPI_errorNotification, "Socket Failed Receive");  // Currently happens when trying to connect to a busy BMC
			break;
		case Connection_Error::Socket_Read_Wait_Failure:
			PPI_LOG(probeInstance->deviceID, PPI_errorNotification, "Socket Read Wait Failed");
			break;
		default:
			if (probeInstance != nullptr)
				PPI_LOG(probeInstance->deviceID, PPI_errorNotification, "Send Data Error " << error);
		}
	}
	if (clear_prdy_timeout)
		return WAIT_PRDY_Timeout;
	return error;
}

#if defined(HOST_LINUX) || defined(HOST_DARWIN)
int ASDBaseProtocol::mutex_lock(pthread_mutex_t *mutex)
{
	//windows api returns 0 for error
	if (!pthread_mutex_lock(mutex))
		return 1;
	return 0;
}

int ASDBaseProtocol::mutex_unlock(pthread_mutex_t *mutex)
{
	//windows api returns 0 for error
	if (!pthread_mutex_unlock(mutex))
		return 1;
	return 0;
}
#else
BOOL ASDBaseProtocol::mutex_lock(HANDLE mutex)
{
	DWORD mutexWaitResult;
	mutexWaitResult = WaitForSingleObject(mutex, INFINITE);
	if (mutexWaitResult != WAIT_OBJECT_0)
		return false;
	return true;
}

BOOL ASDBaseProtocol::mutex_unlock(HANDLE mutex)
{
	return ReleaseMutex(mutex);
}
#endif //HOST_DARWIN

Connection_Error ASDBaseProtocol::WaitForReadEvent() {
	Connection_Error status = No_Error;
	// TODO: Use OS Dependency Injected methods
#if defined(HOST_LINUX) || defined(HOST_DARWIN)
	struct timespec time_to_wait = { 0, 0 };
	int timeout_seconds = this->Timeout / 1000 + 1;//pthread rounds to the lower number it seems so when is 1 it exits immediately
	time_to_wait.tv_sec = time(NULL) + timeout_seconds;
	int waitResult;
	if (!mutex_lock(&mutex)) {
		status = Could_Not_Handle_Mutex;
	}
	else {
		waitResult = pthread_cond_timedwait(&condition, &mutex, &time_to_wait);
		if (waitResult != 0) {
			status = Socket_Read_Wait_Failure;
		}
	}
	if (!mutex_unlock(&mutex)) {
		status = Could_Not_Handle_Mutex;
	}
#else
	BOOL waitResult;
	EnterCriticalSection(&readCriticalSection);
	waitResult = SleepConditionVariableCS(&readEvent, &readCriticalSection, Timeout);
	LeaveCriticalSection(&readCriticalSection);
	if (waitResult == 0) {
		status = Socket_Read_Wait_Failure;
	}
#endif
	return status;
}

Connection_Error ASDBaseProtocol::CheckReadAlive() {
	Connection_Error status = No_Error;
	if (readMonitorThread) {
#if defined(HOST_LINUX) || defined(HOST_DARWIN)
		if (pthread_kill(readMonitorThread, 0) != 0) {
			status = Error_Receiveing_Thread_Died;
		}
#else
		DWORD exitCode;
		auto ret = GetExitCodeThread(readMonitorThread, &exitCode);
		if (!ret || exitCode != STILL_ACTIVE) {
			status = Error_Receiveing_Thread_Died;
		}
#endif //HOST_DARWIN
	}
	else {
		status = Error_Receiveing_Thread_Died;
	}
	return status;
}

void ASDBaseProtocol::WakeReadEvent() {
#if defined(HOST_LINUX) || defined(HOST_DARWIN)
	mutex_lock(&mutex);
	pthread_cond_signal(&condition);
	mutex_unlock(&mutex);
#else
	WakeConditionVariable(&readEvent);
#endif //HOST_DARWIN
}

Connection_Error ASDBaseProtocol::CreateReadMonitorThread() {
	Connection_Error status = Connection_Error::No_Error;

#if defined(HOST_LINUX) || defined(HOST_DARWIN)
	if (pthread_mutex_init(&mutex, NULL) != 0) {
		status = Connection_Error::Read_Thread_Init_Failure;
	}

	if (CONNECTION_PASS(status) && pthread_cond_init(&condition, NULL) != 0) {
		status = Connection_Error::Read_Thread_Init_Failure;
	}

	if (CONNECTION_PASS(status) && pthread_create(&readMonitorThread, NULL, ASDBaseProtocol::ReadMonitorThread, this)) {
		if (pthread_cond_destroy(&condition) != 0) {
			;
			status = Connection_Error::Read_Thread_Init_Failure;
		}
		else {
			status = Connection_Error::Read_Thread_Init_Failure;
		}
	}
#else
	readMonitorThread = CreateThread(NULL, 0, ASDBaseProtocol::ReadMonitorThread, this, 0, NULL);
	if (readMonitorThread == NULL) {
		status = Connection_Error::Read_Thread_Init_Failure;
	}
	else {
		InitializeConditionVariable(&readEvent);
		InitializeCriticalSection(&readCriticalSection);
	}

#endif //HOST_DARWIN
	return status;
}

// Read a packet into the read_message header and buffer.  Returns:
//   PacketRead_Complete    if the packet is complete
//   PacketRead_NotComplete if more reads are needed
//   PacketRead_Error       on a read error.
// We're in one of three states, reading into the header, buffer or
// auth_code depending on how much we've already read.

unsigned int ASDBaseProtocol::readPacket(void)
{
	int left_to_read, actual_read = 0, payload_size;
	char *buffer_progress;

	if (UseTransport == nullptr)
		return PacketRead_Error;

	// Read header
	if (read_used < sizeof(read_message.header)) {
		buffer_progress = ((char *)&read_message.header) + read_used;
		left_to_read = sizeof(read_message.header) - read_used;

		actual_read = UseTransport->Receive(buffer_progress, left_to_read);
		if (actual_read < 0)
			return PacketRead_Error;

		if (actual_read < left_to_read || read_message.header.enc_bit ||
			(read_message.header.size_lsb |
			(read_message.header.size_msb << 8)) != 0) {
			read_used += actual_read;
			return PacketRead_NotComplete;
		}

		read_used = 0;
		return PacketRead_Complete;
	}

	// Read payload
	payload_size = read_message.header.size_lsb |
		(read_message.header.size_msb << 8);

	left_to_read = sizeof(read_message.header) + payload_size - read_used;
	if (left_to_read > 0) {
		buffer_progress = read_message.buffer + read_used - sizeof(read_message.header);
		// Make Unit tests, then refactor out this repeated pattern
		actual_read = UseTransport->Receive(buffer_progress, left_to_read);
		if (actual_read < 0)
			return PacketRead_Error;

		if (actual_read < left_to_read || read_message.header.enc_bit) {
			read_used += actual_read;
			return PacketRead_NotComplete;
		}

		read_used = 0;
		return PacketRead_Complete;
	}

	// Read auth_code, enc_bit must be set to get here
	left_to_read = sizeof(read_message.header) + payload_size + sizeof(read_message.auth_code) - read_used;
	buffer_progress = read_message.auth_code + sizeof(read_message.auth_code) - left_to_read;

	if (left_to_read > 0) {
		actual_read = UseTransport->Receive(buffer_progress, left_to_read);
	}

	if (actual_read < 0)
		return PacketRead_Error;

	if (actual_read < left_to_read) {
		read_used += actual_read;
		return PacketRead_NotComplete;
	}

	read_used = 0;
	return PacketRead_Complete;
}

Connection_Error ASDBaseProtocol::Flush(Connection_Error readResult)
{
	return SendData(true, readResult);
}

Connection_Error ASDBaseProtocol::ShiftI2C(OpenIPC_DeviceId deviceId, JtagStateEncode state, char *writeBuffer, size_t writeSize, char* readBuffer, size_t readSize, GotoStateOptions *goto_state_opt)
{
	return ShiftI2cProtocol(deviceId, writeBuffer, writeSize, readBuffer, readSize, goto_state_opt);
}

Connection_Error ASDBaseProtocol::ShiftJTAG(OpenIPC_DeviceId deviceId, JtagStateEncode state, char *writeBuffer, size_t writeSize, char* readBuffer, size_t readSize, GotoStateOptions *goto_state_opt)
{
	Connection_Error error = No_Error;
	error = doubleShiftCheck(state);
	if (!CONNECTION_PASS(error)) {
		return error;
	}
	return ShiftJTAGProtocol(deviceId, state, writeBuffer, writeSize, readBuffer, readSize, goto_state_opt);
}

void Exit_Thread() {
#if defined(HOST_LINUX) || defined(HOST_DARWIN)
	exit(1);
#else
	ExitThread(0);
#endif //HOST_DARWIN
}

#if defined(HOST_LINUX) || defined(HOST_DARWIN)
void *ASDBaseProtocol::ReadMonitorThread(void *lpParam)
#else
DWORD WINAPI ASDBaseProtocol::ReadMonitorThread(LPVOID lpParam)
#endif //HOST_DARWIN
{
	ASDBaseProtocol *connection = (ASDBaseProtocol *)lpParam;
	if (connection == nullptr)
		Exit_Thread();

	while (1)
	{
		if (connection->UseTransport->AwaitData()) {
			connection->ReceiveData();
			continue;
		}
		//signal the object and exit
		Exit_Thread();
	}
}

Connection_Error ASDBaseProtocol::validMsgType(unsigned int _msgType)
{
	if (_msgType == JTAG_MESSAGE_TYPE || _msgType == PM_MESSAGE_TYPE || _msgType == DMA_MESSAGE_TYPE_WRITE || _msgType == DMA_MESSAGE_TYPE_READ ||
		_msgType == AGENT_CONTROL_MESSAGE_TYPE || _msgType == I2C_MESSAGE_TYPE)
		return No_Error;
	return Invalid_Message_Type;
}

Connection_Error ASDBaseProtocol::setMsgType(unsigned int _msgType)
{
	if (validMsgType(_msgType) == No_Error) {
		this->msgType = _msgType;
		setMsgSize();
		return No_Error;
	}
	return Bad_Argument;
}

void ASDBaseProtocol::setMsgSize()
{
	if (msgType == JTAG_MESSAGE_TYPE || msgType == PM_MESSAGE_TYPE)
		write_message.max_payload = payloadSize;
	else if (msgType == DMA_MESSAGE_TYPE_WRITE || msgType == DMA_MESSAGE_TYPE_READ)
		write_message.max_payload = MAX_DATA_PAYLOAD;
	else if (msgType == I2C_MESSAGE_TYPE)
		write_message.max_payload = payloadSize;
	else
		write_message.max_payload = 0;
}

Connection_Error ASDBaseProtocol::SetClientTimeout()
{
	setMsgType(AGENT_CONTROL_MESSAGE_TYPE);
	cmd_stat = SOCKET_TIMEOUT;
	write_message.buffer[write_message_offset++] = (uint8_t)(this->Timeout / 1000);
	return SendData(true, No_Error);
}

Connection_Error ASDBaseProtocol::ObtainDownstreamVersion()
{
	setMsgType(AGENT_CONTROL_MESSAGE_TYPE);
	cmd_stat = OBTAIN_DOWNSTREAM_VERSION;
	return SendData(true, No_Error);
}

Connection_Error ASDBaseProtocol::ObtainBMCNumInFlightMsgs()
{
	setMsgType(AGENT_CONTROL_MESSAGE_TYPE);
	cmd_stat = QUERY_NUM_IN_FLIGHT_MSGS;
	return SendData(true, No_Error);
}

Connection_Error ASDBaseProtocol::ObtainBmcMessageSize()
 {
	setMsgType(AGENT_CONTROL_MESSAGE_TYPE);
	cmd_stat = QUERY_MAX_MESSAGE_SIZE;
	return SendData(true, No_Error);
}

Connection_Error ASDBaseProtocol::ObtainSupportedJtagChains()
 {
	setMsgType(AGENT_CONTROL_MESSAGE_TYPE);
	cmd_stat = QUERY_SUPPORTED_JTAG_CHAINS;
	return SendData(true, No_Error);
}

Connection_Error ASDBaseProtocol::ObtainSupportedI2cBuses()
 {
	setMsgType(AGENT_CONTROL_MESSAGE_TYPE);
	cmd_stat = QUERY_SUPPORTED_I2C_BUSES;
	return SendData(true, No_Error);
}

void ASDBaseProtocol::InitializeData(void)
{
	std::queue<struct payload_descritpor> empty;
	std::swap(payload_desc_queue, empty);
	readResult = No_Error;
	out_buffers_list = nullptr;
	write_message_offset = 0;
}

void ASDBaseProtocol::InitLocals(void)
{
	maxNumBuffers = 1;
	supportedJtagChains = DEFAULT_SUPPORTED_JTAG_CHAINS;
	supportedI2cBusesCount = 0;
	for(unsigned int i=0; i<I2C_MAX_BUSES; i++) {
		supportedI2cBusAddresses[i].busSelect = 0;
		supportedI2cBusAddresses[i].busSpeed = 0;
	}
	payloadSize = DEFAULT_PAYLOAD_SIZE;
	readMonitorThread = 0;
	// Just in case clear prdy timeout after the connection negotiating is done
	clear_prdy_timeout = false;
	write_message_offset = 0;
	send_buffer = NULL;
	out_buffers_list = NULL;
	readResult = No_Error;
	write_message.max_payload = MAX_DATA_PAYLOAD;
	write_message.buffer_size = 0;
	read_message.max_payload = MAX_DATA_PAYLOAD;
	read_message.buffer_size = 0;
	tag = 0;
	connect_setup = false;
	encryptionEnabled = false;
	// Things that are cleaned up by destructor
	//TODO: indicated error if mallocs fail
	write_message.buffer = (char*)malloc(MAX_DATA_PAYLOAD);
	if (!write_message.buffer) {
		return;
	}
	read_used = 0;
	read_message.buffer = (char*)malloc(MAX_DATA_PAYLOAD);
	if (!read_message.buffer) {
		return;
	}
	send_buffer = (char *)malloc(MAX_PACKET_PAYLOAD);
	if (!send_buffer) {
		return;
	}
	connectionEstablished = ConnectSetup();
}

void ASDBaseProtocol::connect()
{
	if (CONNECTION_PASS(connectionEstablished)) {
		connectionEstablished = CreateReadMonitorThread();
	}
	if (CONNECTION_PASS(connectionEstablished)) {
		connectionEstablished = InitClient();
	}
}

Connection_Error ASDBaseProtocol::InitClient()
{
	Connection_Error error = ObtainDownstreamVersion();
	if (CONNECTION_PASS(error)) {
		error = SetClientTimeout();
	}
	if (CONNECTION_PASS(error)) {
		error = ObtainBMCNumInFlightMsgs();
	}
	if (CONNECTION_PASS(error)) {
		error = ObtainBmcMessageSize();
	}
	if (CONNECTION_PASS(error)) {
		error = ObtainSupportedJtagChains();
	}
	if (CONNECTION_PASS(error)) {
		error = ObtainSupportedI2cBuses();
	}
	return error;
}

bool ASDBaseProtocol::DataFlushed(void)
{
	return payload_desc_queue.empty();
}

Connection_Error ASDBaseProtocol::StartDataTransfer(unsigned int messageType, JtagChainParameters *_params)
{
	Connection_Error error = No_Error;

	InitializeData();
	error = setMsgType(messageType);
	cmd_stat = 0;
	if (error != No_Error)
		goto end;

	if (clear_prdy_timeout) {
		clear_prdy_timeout = false;
		ClearPRDYTimeout();
	}
	if (_params) {
		error = SetHWStateToTheOutMsg(_params);
	}

end:
	return error;
}

Connection_Error ASDBaseProtocol::EndDataTransfer(JtagChainParameters *params)
{
	Connection_Error error = No_Error;

	if (params) {
		error = SetHWStateToTheOutMsg(params);
	}

	if (write_message_offset > 0) {
		// Final check to ensure data is sent.
		error = Flush();
	}
	write_message_offset = 0;

	if (CONNECTION_PASS(error) && !DataFlushed()) {
		error = Data_Not_Flushed;
	}
	return error;
}

Connection_Error ASDBaseProtocol::ConnectSetup()
{
	auto error = No_Error;
	if (!connect_setup) {
#if defined(HOST_LINUX) || defined(HOST_DARWIN)
		list_mutex = &list_mutex_m;
		if (pthread_mutex_init(list_mutex, NULL) != 0) {
			error = Could_Not_Handle_Mutex;
		}
#else
		list_mutex = CreateMutex(NULL, false, NULL);
		if (!list_mutex) {
			error = Could_Not_Handle_Mutex;
		}
#endif
		connect_setup = true;
	}
	return error;
}

Connection_Error ASDBaseProtocol::ConnectionEstablished()
{
	return connectionEstablished;
}
