//////////////////////////////////////////////////////////////////////////////
//
//                      INTEL CONFIDENTIAL
//       Copyright 2016-2017 Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its
// suppliers. Title to the Material remains with Intel Corporation, its
// suppliers, or licensors. The Material contains trade secrets and
// proprietary and confidential information of Intel Corporation, its
// suppliers, and licensors, and is protected by worldwide copyright and trade
// secret laws and treaty provisions. No part of the Material may be used,
// copied, reproduced, modified, published, uploaded, posted, transmitted,
// distributed, or disclosed in any way without Intel's prior express written
// permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or
// delivery of the Materials, either expressly, by implication, inducement,
// estoppel or otherwise. Any license under such intellectual property rights
// must be express and approved by Intel in writing.
//
// Unless otherwise agreed by Intel in writing, you may not remove or alter
// this notice or any other notice embedded in Materials by Intel or Intel's
// suppliers or licensors in any way.
//
//////////////////////////////////////////////////////////////////////////////
///  @file
///
///  @brief Contains methods for obtaining instances of the Probe interfaces
///
///  For additional information on obtaining and using instances, see @ref probeusage.
///
//////////////////////////////////////////////////////////////////////////////

#include "Connection.h"
#include "RemoteConnection_Static.h"
#include "GotoStateOptions.h"
#include "TransportSocket.hpp"
#include "SSLMethodOpenSSL.hpp"
#include "TransportTLSPassPhrase.hpp"
#include "ASDProtocol.h"
#include "OSDependenciesImpl.h"

#ifdef NEED_SAFE_CLIB
#include "safe_lib.h"
#endif


bool Connection::SetRCDeviceId(OpenIPC_DeviceId deviceId)
{
	if (associatedRCPlugin == 0) {
		associatedRCPlugin = deviceId;
		return true;
	}
	return false;
}

// Standard connection key generation
std::string Connection::GenerateKey(std::string &ipaddress, std::string &port)
{
	std::stringstream key;
	key << ipaddress << port;
	return key.str();
}

void Connection::init() {
	connectionEstablished = Connection_Busy;
	probeDeviceId = OpenIPC_INVALID_DEVICE_ID;
	currentUser = OpenIPC_INVALID_DEVICE_ID;
	msgType = 1;

	// TODO: These may already default to values being set
	connect_setup = false;

	associatedRCPlugin = 0;
	encryptionEnabled = false;
}

Connection::Connection() {
	init();
}

Connection::Connection(std::shared_ptr<ProbeInstanceASD> probe)
{
	init();
	probeInstance = probe;
	this->connectionParameters = probe->connectionParameters;
	InitializeTransport();
	connectionEstablished = (Connection_Error)UseTransport->Connect();
	if (CONNECTION_PASS(connectionEstablished)) {
		if (probeInstance)
			PPI_LOG(probeInstance->deviceID, PPI_traceNotification, "Socket Open");
		connectionEstablished = InitializeProtocol();
	}

	// Pass along all Transport generated warnings and errors
	UseTransport->ForEachWarningAndError(PPI_warningNotification, PPI_errorNotification,
		[&](unsigned int notification_type, std::string message) {
		PluginNotifications::GetInstance().Notify(
			probeInstance->deviceID,
			OpenIPC_Error_No_Error,
			notification_type,
			[&](std::basic_ostream<char>& stream)
		{
			stream << message;
		});
	});
	Key = GenerateKey(connectionParameters->transportParameters->ipAddress, connectionParameters->transportParameters->ipPort);
}

Connection_Error Connection::InitializeTransport()
{
	if (connectionParameters->transportParameters->socketType == BASIC_SOCKET) {
		UseTransport = std::make_shared<TransportSocket>(
			connectionParameters->transportParameters->ipAddress.c_str(),
			connectionParameters->transportParameters->ipPort.c_str(),
			std::make_shared<OSDependenciesImpl>());
	}
	else {
		// Use SetConfig to configure SSL settings
		std::shared_ptr<SSLMethodOpenSSL> ssl = std::make_shared<SSLMethodOpenSSL>();
		if (!ssl)
			return Connection_Error::Initialize_Error;
		SSL_Config config;
		config.chainFileOption = connectionParameters->transportParameters->chainFileVerification;
		ssl->SetConfig(config);
		UseTransport = std::make_shared<TransportTLSPassPhrase>(
			connectionParameters->transportParameters->ipAddress.c_str(),
			connectionParameters->transportParameters->ipPort.c_str(),
			connectionParameters->transportParameters->targetChainFile.c_str(),
			ssl, connectionParameters->transportParameters->passphrase,
			std::make_shared<OSDependenciesImpl>());
	}
	return No_Error;
}

Connection_Error Connection::InitializeProtocol()
{
	std::shared_ptr<ASDProtocol> ASDProt = std::make_shared<ASDProtocol>(UseTransport, probeInstance, connectionParameters);
	protocol = std::static_pointer_cast<Protocol>(ASDProt);
	return protocol->ConnectionEstablished();
}

Connection::~Connection()
{
}

Connection_Error Connection::Shift(OpenIPC_DeviceId deviceId, JtagStateEncode state, char *writeBuffer, size_t writeSize, char* readBuffer, size_t readSize, GotoStateOptions *goto_state_opt)
{
	Connection_Error result = Invalid_Message_Type;

	if (deviceId != currentUser)
		result = Connection_Busy;
	else if (msgType == JTAG_MESSAGE_TYPE)
		result = ShiftJTAG(deviceId, state, writeBuffer, writeSize, readBuffer, readSize, goto_state_opt);
	else if (msgType == PM_MESSAGE_TYPE || msgType == DMA_MESSAGE_TYPE_WRITE || msgType == DMA_MESSAGE_TYPE_READ)
		result = ShiftPM(deviceId, writeBuffer, writeSize, readBuffer, readSize);
	else if (msgType == I2C_MESSAGE_TYPE)
		result = ShiftI2C(deviceId, state, writeBuffer, writeSize, readBuffer, readSize, goto_state_opt);

	if (result != No_Error || (goto_state_opt != nullptr && goto_state_opt->flush)) {
		if (result != No_Error)
			PPI_LOG(deviceId, PPI_errorNotification, "Shift Error " << result << " flushing buffers");
		protocol->Flush(result);
		return result;
	}

	return result;
}

Connection_Error Connection::ShiftPM(OpenIPC_DeviceId deviceId, char *writeBuffer, size_t writeSize, char *readBuffer, size_t readSize)
{
	return No_Error;
}

Connection_Error Connection::ShiftJTAG(OpenIPC_DeviceId deviceId, JtagStateEncode state, char *writeBuffer, size_t writeSize, char* readBuffer, size_t readSize, GotoStateOptions *goto_state_opt)
{
	return protocol->ShiftJTAG(deviceId, state, writeBuffer, writeSize, readBuffer, readSize, goto_state_opt);
}

Connection_Error Connection::ShiftI2C(OpenIPC_DeviceId deviceId, JtagStateEncode state, char *writeBuffer, size_t writeSize, char* readBuffer, size_t readSize, GotoStateOptions *goto_state_opt)
{
	return protocol->ShiftI2C(deviceId, state, writeBuffer, writeSize, readBuffer, readSize, goto_state_opt);
}

Connection_Error Connection::setMsgType(unsigned int _msgType)
{
	if (validMsgType(_msgType) == No_Error) {
		this->msgType = _msgType;
		return No_Error;
	}
	return Bad_Argument;
}

Connection_Error Connection::StartDataTransfer(OpenIPC_DeviceId deviceId, unsigned int messageType, JtagChainParameters *_params)
{
	Connection_Error error = No_Error;
	if (currentUser != OpenIPC_INVALID_DEVICE_ID) {
		error =  Connection_Busy;
		goto end;
	}

	error = setMsgType(messageType);
	if (error != No_Error)
		goto end;
	
	error = protocol->StartDataTransfer(messageType, _params);
	if (error != No_Error)
		goto end;

	currentUser = deviceId;

end:
	return error;
}

Connection_Error Connection::EndDataTransfer(OpenIPC_DeviceId deviceId, JtagChainParameters *params)
{
	Connection_Error error = No_Error;

	if (currentUser != deviceId) {
		error = Invalid_Device_Id;
	} 
	else {
		error = protocol->EndDataTransfer(params);
		currentUser = OpenIPC_INVALID_DEVICE_ID;
	}

	return error;
}

Connection_Error Connection::UpdateInterfacePaddingSettings(InterfacePadding *interfacePaddingSettings)
{
	return protocol->UpdateInterfacePaddingSettings(interfacePaddingSettings);
}

Connection_Error Connection::validMsgType(unsigned int _msgType)
{
	if (_msgType == JTAG_MESSAGE_TYPE || _msgType == PM_MESSAGE_TYPE || _msgType == DMA_MESSAGE_TYPE_WRITE || _msgType == DMA_MESSAGE_TYPE_READ ||
		_msgType == AGENT_CONTROL_MESSAGE_TYPE || _msgType == I2C_MESSAGE_TYPE)
		return No_Error;
	return Invalid_Message_Type;
}

Connection_Error Connection::RegisterProbeInstance(std::shared_ptr<ProbeInstanceASD> probe)
{
	if (probeInstance == nullptr) {
		probeInstance = probe;
		return No_Error;
	}
	return User_Already_Registered;
}

bool Connection::ProbeRegistered()
{
	return probeInstance != nullptr;
}

bool Connection::InitSuccess()
{
	bool result = false;
	if (protocol)
		result = protocol->ConnectionEstablished() == Connection_Error::No_Error;
	return result;
}

Connection_Error Connection::GetConnectionError()
{
	return protocol
		? protocol->ConnectionEstablished()
		: connectionEstablished;
}

void Connection::EnableEncryption(bool enable)
{
	encryptionEnabled = enable;
}

void Connection::SetProbeDeviceId(OpenIPC_DeviceId deviceId)
{
	probeDeviceId = deviceId;
}

Connection_Error Connection::SetHardwareLogLevel(LoggingConfiguration log_config)
{
	if (protocol)
		return protocol->SetHardwareLogLevel(log_config);
	return Null_Pointer;
}

Connection_Error Connection::SetHardwareGpioConfig(const uint8_t gpio_config)
{
	if (protocol)
		return protocol->SetHardwareGpioConfig(gpio_config);
	return Null_Pointer;
}

Connection_Error Connection::SetHardwareJTAGDriverMode(JtagChainParameters params)
{
	if (protocol)
		return protocol->SetHardwareJTAGDriverMode(params);
	return Null_Pointer;
}

void Connection::ReceiveData()
{
	if (protocol)
		protocol->ReceiveData();
}
