//////////////////////////////////////////////////////////////////////////////
//
//                      INTEL CONFIDENTIAL
//       Copyright 2016-2017 Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its
// suppliers. Title to the Material remains with Intel Corporation, its
// suppliers, or licensors. The Material contains trade secrets and
// proprietary and confidential information of Intel Corporation, its
// suppliers, and licensors, and is protected by worldwide copyright and trade
// secret laws and treaty provisions. No part of the Material may be used,
// copied, reproduced, modified, published, uploaded, posted, transmitted,
// distributed, or disclosed in any way without Intel's prior express written
// permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or
// delivery of the Materials, either expressly, by implication, inducement,
// estoppel or otherwise. Any license under such intellectual property rights
// must be express and approved by Intel in writing.
//
// Unless otherwise agreed by Intel in writing, you may not remove or alter
// this notice or any other notice embedded in Materials by Intel or Intel's
// suppliers or licensors in any way.
//
//////////////////////////////////////////////////////////////////////////////
#include "SSLMethodOpenSSL.hpp"
#include <boost/thread.hpp>

SSLMethodOpenSSL::SSLMethodOpenSSL() {
	SSL_library_init();
	SSL_load_error_strings();
	OPENSSL_init_crypto(OPENSSL_INIT_NO_ADD_ALL_CIPHERS | OPENSSL_INIT_NO_ADD_ALL_DIGESTS, NULL);

	method = SSLv23_method();
	ssl_error = 0;
	ctx = nullptr;
	response = 1;
	bmcConnection = nullptr;
	ssl = nullptr;
	config.chainFileOption = "abort";
}

// todo: are socket options set? nagle algorithm?

bool SSLMethodOpenSSL::Open(std::string &address, std::string &port, std::string &targetChainFile) {
	bool error = false;

	if (method != nullptr) {
		ctx = SSL_CTX_new(method);
	}
	if (ctx != nullptr) {
		SSL_CTX_set_verify(ctx, SSL_VERIFY_PEER, SSLMethodOpenSSL::VerifyCallback);
		SSL_CTX_set_verify_depth(ctx, 5);
		const long flags = SSL_OP_ALL | SSL_OP_NO_SSLv2 | SSL_OP_NO_SSLv3 | SSL_OP_NO_COMPRESSION;
		(void)SSL_CTX_set_options(ctx, flags);

		// Try and load the target BMC chain PEM
		response = SSL_CTX_load_verify_locations(ctx, targetChainFile.c_str(), NULL);
		bmcConnection = BIO_new_ssl_connect(ctx);
	}

	// Error out with message
	if (config.chainFileOption == "abort" && response != 1) {
		if (errorHandler) {
			errorHandler(SSL_Error::Chain_File_Not_Found, "");
			response = 0;
		}
	}
	else {
		// Do not error out, but display warning message
		if (config.chainFileOption == "warning" && response == 1) {
			if (errorHandler) {
				errorHandler(SSL_Error::Verify_Target_Off, "");
			}
		}
		if (bmcConnection != nullptr) {
			std::string host;
			host.append(address).append(":").append(port);
			response = BIO_set_conn_hostname(bmcConnection, host.c_str());
		}
	}

	if (response == 1) {
		BIO_get_ssl(bmcConnection, &ssl);
	}
	if (ssl != nullptr) {
		response = SSL_set_cipher_list(ssl, PREFERRED_CIPHERS);
	}
	if (response == 1) {
		response = SSL_set_tlsext_host_name(ssl, address.c_str());
		if (response != 1) {
			// Warn
			if (errorHandler) {
				errorHandler(SSL_Error::General_Failure, "");
			}
		}

		// Notes: Use thread to check/exit blocking call BIO_do_connect
		// Attempted to fix with BIO_set_nbio(bmcConnection, 1) without success.
		// Currently the simplest solution is to use boost thread that will cover
		// both Windows/Linux.
		boost::thread connectionThread(&SSLMethodOpenSSL::CheckConnectionThread, this);
		if (!connectionThread.timed_join(boost::posix_time::seconds(5)))
			response = -1;
	}

	// Error out when verification option is "abort"
	if (!SSL_PASS(GetVerifyResult()) && config.chainFileOption == "abort") {
		if (errorHandler) {
			// Warn
			errorHandler(GetVerifyResult(), "");
			response = -1;
		}
	}
	if (response == 1) {
		response = BIO_do_handshake(bmcConnection);
	}

	if (response == 1) {
		X509* cert = SSL_get_peer_certificate(ssl);
		if (cert) { X509_free(cert); } /* Free immediately */
		if (cert == nullptr) {
			if (errorHandler) {
				// TODO: decide what to report here, if anything.
			}
		}
	} else {
		error = true;
	}

	// Notes: ERR_get_error() is use for debug only
	// Internal ssl error string is not what we want to report back to end user.
	// All errors should have captured from above.
	ssl_error = ERR_get_error();
	if (ssl_error) {
		if (errorHandler) {
			const char* const str = ERR_reason_error_string(ssl_error);
			std::string message = "";
			if (str) {
				message.assign(str);
			}
		}
	}
	if (response != 1) {
		error = true;
		if (bmcConnection) {
			BIO_free_all(bmcConnection);
			bmcConnection = nullptr;
		}
	}

	return error;
}

int SSLMethodOpenSSL::Close()
{
	if (ssl) {
		response = SSL_shutdown(ssl);
		if (response == 0) {
			SSL_shutdown(ssl);
			return 0;
			}
		if (response < 0) {
			ssl_error = ERR_get_error();
			if (ssl_error) {
				if (errorHandler) {
					const char* const str = ERR_reason_error_string(ssl_error);
					std::string message = "";
					if (str) {
						message.assign(str);
					}
				}
			}
		}
	}
	return 1;
}

void SSLMethodOpenSSL::RegisterErrorHandler(std::function<void(SSL_Error error, std::string message)> errorHandlerFunction) {
	errorHandler = errorHandlerFunction;
}

int SSLMethodOpenSSL::Write(char *buffer, int length) {
	auto written = BIO_write(bmcConnection, buffer, length);
	return written;
}

int SSLMethodOpenSSL::Read(char *buffer, int size) {
	auto read = BIO_read(bmcConnection, buffer, size);
	return (read == -1) ? 0 : read;
}

int SSLMethodOpenSSL::Getfd()
{
	int socket_fd;
	int ret;
	ret = BIO_get_fd(this->bmcConnection, &socket_fd);
	if (ret < 0) {
		socket_fd = ret;
	}
	return socket_fd;
}

int SSLMethodOpenSSL::Pending()
{
	return BIO_pending(this->bmcConnection);
}

void SSLMethodOpenSSL::SetConfig(SSL_Config config)
{
	if (config.chainFileOption == "") {
		this->config.chainFileOption = "abort";
	}
	else {
		this->config.chainFileOption = config.chainFileOption;
	}
}

SSL_Config SSLMethodOpenSSL::GetConfig()
{
	return this->config;
}

SSLMethodOpenSSL::~SSLMethodOpenSSL() {
}

SSL_Error SSLMethodOpenSSL::GetVerifyResult() {
	return VerifyResult; // global
}

int SSLMethodOpenSSL::VerifyCallback(int preverify, X509_STORE_CTX * x509_ctx) {
	// Emulate result reporting using a global variable

	if (preverify == 0)	{
		int err = X509_STORE_CTX_get_error(x509_ctx);

		if (err != X509_V_OK) {
			VerifyResult = SSL_Error::Unable_To_Verify_Target;
		}
	}

	// Do not fail if server certificate does not verify
	return 1;
}

void SSLMethodOpenSSL::CheckConnectionThread()
{
	this->response = BIO_do_connect(bmcConnection);
}
