//////////////////////////////////////////////////////////////////////////////
//
//                      INTEL CONFIDENTIAL
//       Copyright 2017 Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its
// suppliers. Title to the Material remains with Intel Corporation, its
// suppliers, or licensors. The Material contains trade secrets and
// proprietary and confidential information of Intel Corporation, its
// suppliers, and licensors, and is protected by worldwide copyright and trade
// secret laws and treaty provisions. No part of the Material may be used,
// copied, reproduced, modified, published, uploaded, posted, transmitted,
// distributed, or disclosed in any way without Intel's prior express written
// permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or
// delivery of the Materials, either expressly, by implication, inducement,
// estoppel or otherwise. Any license under such intellectual property rights
// must be express and approved by Intel in writing.
//
// Unless otherwise agreed by Intel in writing, you may not remove or alter
// this notice or any other notice embedded in Materials by Intel or Intel's
// suppliers or licensors in any way.
//
//////////////////////////////////////////////////////////////////////////////

#include <utility>

#include "TransportSocket.hpp"

#if defined(HOST_LINUX) || defined(HOST_DARWIN)
#include <netdb.h>
#include <netinet/tcp.h>
#else
#include <winsock2.h>
#endif

TransportSocket::TransportSocket(const char* address, const char* port,
				 std::shared_ptr<OSDependencies> deps) {
	Address = address;
	Port = port;
	Dependencies = std::move(deps);
	ConnectSocket = Dependencies->InitSocket();
}

TransportSocket::~TransportSocket()
{
	Close();
}

Connection_Error TransportSocket::Connect()
{
	struct addrinfo *address_info_result;
	struct addrinfo hints = { 0 };
	hints.ai_family = AF_UNSPEC;
	hints.ai_socktype = SOCK_STREAM;
	hints.ai_protocol = IPPROTO_TCP;
	Connection_Error result = User_Not_Registered;

	if (Dependencies->GetAddressInfo(Address, Port,
		&hints, &address_info_result) != 0) {
		result = Could_Not_Contact_BMC;
	} else {
		struct addrinfo* r;
		for (r = address_info_result; r; r = r->ai_next) {
			OSDepSocket socket = Dependencies->Socket(
				r->ai_family, r->ai_socktype, r->ai_protocol);
			if (!Dependencies->IsSocketValid(socket)) {
				result = Could_Not_Connect_To_BMC;
				break;
			}

			if (!Dependencies->SetSockOptKeepAlive(socket)) {
				result = Could_Not_Connect_To_BMC;
				break;
			}

			if (!Dependencies->SetSockOptNoDelay(socket)) {
				result = Could_Not_Connect_To_BMC;
				break;
			}

			if (Dependencies->Connect(socket, r->ai_addr, (unsigned int)r->ai_addrlen) < 0) {
				if (!r->ai_next) {
					result = Could_Not_Connect_To_BMC;
				}
				continue;
			} else {
				ConnectSocket = socket;
				result = No_Error;
				break;
			}
		}
		Dependencies->FreeAddressInfo(address_info_result);
	}
	return result;
}

int TransportSocket::Send(char *send_buffer, int length) {
	int result = -1;
	if(Dependencies->IsSocketValid(ConnectSocket))
		result = Dependencies->Send(ConnectSocket, send_buffer, length, 0);
	return result;
}

int TransportSocket::Receive(char *buffer, int length) {
	int result = -1;
	if(Dependencies->IsSocketValid(ConnectSocket))
		result = Dependencies->Receive(ConnectSocket, buffer, length, 0);
	return result;
}

bool TransportSocket::AwaitData()
{
	bool result = false;
	if(Dependencies->IsSocketValid(ConnectSocket))
		result = Dependencies->AwaitData(ConnectSocket);
	return result;
}

void TransportSocket::Close() {
	if(Dependencies->IsSocketValid(ConnectSocket))
		Dependencies->Close(ConnectSocket);
}

void TransportSocket::ForEachWarningAndError(
	unsigned int warning_notificaiton_type,
	unsigned int error_notification_type,
	std::function<void(unsigned int, std::string)> log_function)
{
	(void)warning_notificaiton_type;
	(void)error_notification_type;
	(void)log_function;
}
