//////////////////////////////////////////////////////////////////////////////
//
//                      INTEL CONFIDENTIAL
//       Copyright 2016-2017 Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its
// suppliers. Title to the Material remains with Intel Corporation, its
// suppliers, or licensors. The Material contains trade secrets and
// proprietary and confidential information of Intel Corporation, its
// suppliers, and licensors, and is protected by worldwide copyright and trade
// secret laws and treaty provisions. No part of the Material may be used,
// copied, reproduced, modified, published, uploaded, posted, transmitted,
// distributed, or disclosed in any way without Intel's prior express written
// permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or
// delivery of the Materials, either expressly, by implication, inducement,
// estoppel or otherwise. Any license under such intellectual property rights
// must be express and approved by Intel in writing.
//
// Unless otherwise agreed by Intel in writing, you may not remove or alter
// this notice or any other notice embedded in Materials by Intel or Intel's
// suppliers or licensors in any way.
//
//////////////////////////////////////////////////////////////////////////////
#if defined(HOST_LINUX) || defined(HOST_DARWIN)
#include <string.h>
#endif
#include "BundleHelpersASD.h"
#include "ProbePluginErrors.h"
#include "Foundation/Types.h"
#include <assert.h>
#include "CProbePlugin.hpp"

#include "Connection.h"
#include "RemoteConnection.h"
#include "Components/Configuration/ConfigurationErrors.h"

#ifdef NEED_SAFE_CLIB
#include "safe_lib.h"
#endif

extern CProbePlugin* GetProbePluginInstance();

static Local_Handle locked_handle;
static uint64_t number_of_clocks_in_tlr = 0;
static PPI_JTAG_RESET_OPTIONS_ET reset_method;

static bool in_lock = false;

uint64_t* getNumberOfClocksInTlR(){
	return &number_of_clocks_in_tlr;
}

PPI_JTAG_RESET_OPTIONS_ET* getResetMethod(){
	return & reset_method;
}


static inline void move_slots(Local_Handle* src, Local_Handle* dest){
	for (uint32_t i = 0;i< dest->slots.size();i++){
		BitData_FreeManaged(dest->slots[i]);
	}
	dest->slots.clear();
	for (uint32_t i = 0;i<src->slots.size();i++){
		BitData* newData = BitData_CreateManaged(src->slots[i]->bitsize);
		BitData_Copy(src->slots[i],0,newData,0,src->slots[i]->bitsize);
		dest->slots.push_back(newData);
	}
}

OpenIPC_Error ExecuteAllCommands(Local_Handle* current, OpenIPC_DeviceId deviceInterface, Command *start)
{
	Command *currentCommand = NULL;
	OpenIPC_Error result = OpenIPC_Error_No_Error;

	std::shared_ptr<InterfaceInstance> interface_ = GetProbePluginInstanceASD()->InterfaceFromId(deviceInterface);
	if (interface_ == nullptr) {
		result = OpenIPC_Error_Internal_Error;  // Invalid deviceInterface argument
	}

	InterfacePadding padding = {0};
	JtagChainInfo jtagSettings = {0};
	JtagChainParameters params;
	params.InterfacePaddingSettings = &padding;
	params.JtagChainSettings = &jtagSettings;

	if (start != NULL)
		currentCommand = start;
	else
		currentCommand = current->start;

	std::shared_ptr<InterfaceInstanceJtagASD> jtagInterface = std::dynamic_pointer_cast<InterfaceInstanceJtagASD>(interface_);
	std::shared_ptr<InterfaceInstanceI2cASD> i2cInterface = std::dynamic_pointer_cast<InterfaceInstanceI2cASD>(interface_);
	std::shared_ptr<InterfaceInstancePins> pinInterface = std::dynamic_pointer_cast<InterfaceInstancePins>(interface_);

	if (!OpenIPC_PASS(result))
		goto end;

	if (i2cInterface != nullptr)
		result = To_OpenIPC_Error(StartDataTransfer(interface_->probe->deviceID, I2C_MESSAGE_TYPE, nullptr));
	else if (jtagInterface != nullptr) { // successfull downcast
		jtagInterface->ObtainJtagChainParameters(&params);
		result = To_OpenIPC_Error(StartDataTransfer(interface_->probe->deviceID, JTAG_MESSAGE_TYPE, &params));
	} // else using 0 padding and interface 0 in params
	else if (pinInterface != nullptr)
		result = To_OpenIPC_Error(StartDataTransfer(interface_->probe->deviceID, JTAG_MESSAGE_TYPE, nullptr));
	else {
		result = OpenIPC_Error_Invalid_Device;
		goto end;
	}

	while (currentCommand != NULL && OpenIPC_PASS(result)) {
		result = currentCommand->Execute(current, deviceInterface, &currentCommand, false);
		if (!OpenIPC_PASS(result)) {
			EndDataTransfer(interface_->probe->deviceID, &params);
			goto end;
		}
	}
	if(!OpenIPC_PASS(result)) {
		switch (result) {
		case User_Not_Registered:
			PPI_LOG(deviceInterface, PPI_traceNotification, "User not registered");
			result = OpenIPC_Error_Not_Configured;
			break;
		case Connection_Busy:
			PPI_LOG(deviceInterface, PPI_traceNotification, "Connection Busy");
			result = OpenIPC_Error_Not_Configured;
			break;
		}
		goto end;
	}
	if (i2cInterface != nullptr)
		result = To_OpenIPC_Error(EndDataTransfer(interface_->probe->deviceID, nullptr));
	else if (jtagInterface != nullptr)
		result = To_OpenIPC_Error(EndDataTransfer(interface_->probe->deviceID, &params));
	else if (pinInterface != nullptr)
		result = To_OpenIPC_Error(EndDataTransfer(interface_->probe->deviceID, nullptr));
	else {
		result = OpenIPC_Error_Invalid_Device;
		goto end;
	}

	currentCommand = current->start;
	while (currentCommand != NULL && OpenIPC_PASS(result)) {
		result = currentCommand->Execute(current, deviceInterface, &currentCommand, true);
		if (!OpenIPC_PASS(result))
			goto end;
	}
end:
	return result;
}

OpenIPC_Error ExecuteHandle(Local_Handle* current, OpenIPC_DeviceId deviceInterface,  bool keepLock){

	OpenIPC_Error result = OpenIPC_Error_No_Error;

	if (in_lock){
		move_slots(&locked_handle, current);
	}

	result = ExecuteAllCommands(current, deviceInterface, NULL);

	if (keepLock){
		move_slots(current,&locked_handle);
		in_lock = true;
	}else {
		in_lock = false;
	}
	return result;
}

Local_Handle* getLockedHandle(){
	return &locked_handle;
}
bool isInLock(){
	return in_lock;
}

OpenIPC_Error StartCommand::LocalExecute(Local_Handle* handle, OpenIPC_DeviceId deviceInterface, bool doneExecuting){
	return OpenIPC_Error_No_Error;
}

uint32_t StartCommand::length_of_output() {
	return 0;
}

uint32_t SlotComparisonCommand::length_of_output() {
	return 0;
}

uint32_t SlotInfoCommand::length_of_output() {
	return 0;
}

uint32_t SlotCountCommand::length_of_output() {
	return 0;
}

uint32_t SlotModificationCommand::length_of_output() {
	return 0;
}

uint32_t LoopCaptureAllCommand::length_of_output() {
	return 0;
}

uint32_t LoopBreakCommand::length_of_output() {
	return 0;
}

OpenIPC_Error Command::Execute(Local_Handle* handle, OpenIPC_DeviceId deviceInterface, Command **nextCommand, bool doneExecuting){
	OpenIPC_Error result = this->LocalExecute(handle, deviceInterface, doneExecuting);

	*nextCommand = this->next;

	return result;
}

OpenIPC_Error Command::Append(Command* cmd){
	if (next == NULL){
		next = cmd;
		next->previous = this;
		return OpenIPC_Error_No_Error;
	} else {
		return next->Append(cmd);
	}
}

OpenIPC_Error Command::Clear(){
	Command *current = this;
	while (current->next != NULL) current = current->next;
	while (current != this) {
		current = current->previous;
		delete current->next;
		current->next = NULL;
	}
	return OpenIPC_Error_No_Error;
}

void Command::set_handle(Local_Handle* handle){
	context = handle;
}

OpenIPC_Error LoopBreakCommand::LocalExecute(Local_Handle* handle, OpenIPC_DeviceId deviceInterface, bool doneExecuting){
	OpenIPC_Error result = OpenIPC_Error_No_Error;
	for (uint32_t i = 0;i<this->maxNumberOfIterations && body->status == RUNNING;i++){
		result = ExecuteAllCommands(body, deviceInterface, NULL);
	}
	switch (body->status){
	case BREAK_NO_ERROR:
		// No error
		break;
	case BREAK_ERROR:
		// result <- some new error condition
		break;
	case RUNNING:
		// not possible!!
		break;
	default:
		// Throw an error
		break;
	};
	return result;
}

OpenIPC_Error LoopCaptureAllCommand::Copy(uint32_t& current_position){
	OpenIPC_Error result = OpenIPC_Error_No_Error;
	uint32_t numberOfBitsInBuffer = this->bufferLengthInBytes*8;
	BitData buffer = BitData_CreateLocalFromBuffer(numberOfBitsInBuffer, this->bufferLengthInBytes*8,this->bufferForIterations);
	BitData_ShiftRight(this->body->last_data, 0, this->body->last_data->bitsize, 1);
	BitData_Resize(this->body->last_data,this->body->last_data->bitsize-1);
	if (current_position + this->body->last_data->bitsize < numberOfBitsInBuffer){
		BitData_Copy(this->body->last_data,0,&buffer,current_position,this->body->last_data->bitsize);
		// Don't *really* need to mod here
		current_position = (current_position + this->body->last_data->bitsize) % numberOfBitsInBuffer;
	} else if (this->body->last_data->bitsize <= numberOfBitsInBuffer){
		// Ok, so now we know that we need to copy directly over the buffers (we can't over-run anything)
		uint32_t numberToCopy = numberOfBitsInBuffer - current_position;
		BitData_Copy(this->body->last_data, 0, &buffer, current_position, numberToCopy);
		BitData_Copy(this->body->last_data, numberToCopy, &buffer,0,this->body->last_data->bitsize - numberToCopy);
		// This update is because we very well might not write all of the buffer, so we might need to move current_pointer!
		// this->body->last_data->bitsize <= numberOfBitsInBuffer, so no chance of over running the buffer
		current_position = static_cast<uint32_t>(this->body->last_data->bitsize - numberToCopy);
	} else{
		uint32_t lastBufferSizePosition = static_cast<uint32_t>(this->body->last_data->bitsize - numberOfBitsInBuffer); // Position in last_data
		uint32_t lastCircularBufferPosition = static_cast<uint32_t>((this->body->last_data->bitsize)/ numberOfBitsInBuffer); // Position in last_data
		lastCircularBufferPosition *= numberOfBitsInBuffer;
		if (lastCircularBufferPosition + numberOfBitsInBuffer > this->body->last_data->bitsize){
			lastCircularBufferPosition -= numberOfBitsInBuffer;
		}
		// Picture-time (because I can't understand this without one!!):
		// buffer:   LSB |      |              | MSB (length L)
		//                   current_pointer
		// last_data LSB |                        |      |       |          | MSB Length (M)
		//               0                       X*L     M-L    (X+1)*L
		// lastBufferSizePosition is set to (M - L)
		// lastCircularBufferPosition is set to X*L
		// Notice that if lastBufferSizePosition == lastCircularBufferPosition then we can just do 2 copies and not change current_pointer;
		// Otherwise, if we move current_pointer to the right lastBufferSizePosition - lastCircularBufferPosition bits, then we reduce to the above case :)
		// We always write exactly 1 bufer size of data, so we don't need to worry about the current_pointer moving copying anything
		if (lastBufferSizePosition - lastCircularBufferPosition > 0){
			// I don't need to do this copy, but it helps me understand this code, so that's why it's commented out :)
			// BitData_Copy(this->body->last_data,lastBufferSizePosition,&buffer,current_position, lastBufferSizePosition - lastCircularBufferPosition);
			// This data is overwritten anyways, since we always write numberOfBitsInBuffer bits
			current_position += lastBufferSizePosition - lastCircularBufferPosition;
			current_position %= numberOfBitsInBuffer; // we might have wrapped around, so we *need* to take the mod
		}
		uint32_t numberToCopy = numberOfBitsInBuffer - current_position;
		BitData_Copy(this->body->last_data, lastBufferSizePosition, &buffer, current_position, numberToCopy);
		BitData_Copy(this->body->last_data, lastBufferSizePosition + numberToCopy, &buffer, 0 ,this->body->last_data->bitsize - lastBufferSizePosition - numberToCopy);
	}
	BitData_Resize(this->body->last_data,1);
	this->body->number_of_bits_in_last_data = 0;
	return result;
}

OpenIPC_Error LoopCaptureAllCommand::LocalExecute(Local_Handle* handle, OpenIPC_DeviceId deviceInterface, bool doneExecuting){
	OpenIPC_Error result = OpenIPC_Error_No_Error;
	uint32_t bufferPosition = 0;
	if (this->body->last_data != NULL){
		BitData_Delete(this->body->last_data,0, this->body->last_data->bitsize);
		BitData_FreeManaged(this->body->last_data);
	}
	// We need to create this with 1 bit of data (we correct this when we actually look at the data contained there)
	this->body->last_data = BitData_CreateManaged(1);
	this->body->enclosingContext = this;
	for (uint32_t i = 0;i<this->numberOfIterations;i++){
		result = ExecuteAllCommands(body, deviceInterface, NULL);
		// copy into the buffer
		this->Copy(bufferPosition);
	}
	// Update the nextPosition
	*this->currentBit =bufferPosition;
	if (handle->enclosingContext != nullptr){
		BitData_GetBuffer(this->body->last_data,0,this->bufferForIterations,this->bufferLengthInBytes);
		handle->number_of_bits_in_last_data = this->bufferLengthInBytes*8;
	}
	BitData_FreeManaged(this->body->last_data);
	this->body->last_data = NULL;
	return result;
}

OpenIPC_Error SlotModificationCommand::LocalExecute(Local_Handle* handle, OpenIPC_DeviceId deviceInterface, bool doneExecuting){	
	uint32_t number_of_bytes = (numberOfBits+7)/8;
	for (uint32_t i = 0;i<number_of_bytes;i++){
		BitData* bd = handle->slots[savedSlot];
		BitData maskBd = BitData_CreateLocalFromBuffer(8,8,&mask[i]);
		BitData orBd = BitData_CreateLocalFromBuffer(8,8,&valueToOrIn[i]);
		BitData_BitwiseAnd(bd,i*8,bd,i*8,&maskBd,0,8);
		BitData_BitwiseOr(bd,i*i,bd,i*8,&orBd,0,8);
	}
	return OpenIPC_Error_No_Error;
}


OpenIPC_Error SlotInfoCommand::LocalExecute(Local_Handle* handle, OpenIPC_DeviceId deviceInterface, bool doneExecuting){
	*lengthOfDataInSlot = (uint32_t) handle->slots[savedSlot]->bitsize;
	return OpenIPC_Error_No_Error;
}

OpenIPC_Error SlotComparisonCommand::LocalExecute(Local_Handle* handle, OpenIPC_DeviceId deviceInterface, bool doneExecuting){
	uint32_t number_of_bytes = (numberOfBits+7)/8;
	assert(handle->slots[savedSlot]->bitsize == number_of_bytes);
	bool result = true;
	for (uint32_t i = 0;i<number_of_bytes && result;i++){
		result &= (BitData_GetUpToUInt64(handle->slots[savedSlot],i*8,8) & mask[i]) == (match[i] & mask[i]);
	} // Note: need to fix the end case as well
	*comparisonResult = result;
	return OpenIPC_Error_No_Error;
}

OpenIPC_Error PinsDelayCommand::LocalExecute(Local_Handle* handle, OpenIPC_DeviceId deviceInterface, bool doneExecuting)
{
	OpenIPC_Error returnError = OpenIPC_Error_No_Error;
	if (!doneExecuting) {
		std::shared_ptr<InterfaceInstancePins> pinInterface = std::dynamic_pointer_cast<InterfaceInstancePins>(this->plugin->InterfaceFromId(deviceInterface));
		if (pinInterface != nullptr && pinInterface.get() != nullptr) {
			auto _probe = (ProbeInstanceASD *)pinInterface->probe;
			returnError = _probe->DelayUseconds(timeInMicroSeconds);
		}
	}
	return returnError;
}

OpenIPC_Error DeassertPinCommand::LocalExecute(Local_Handle* handle, OpenIPC_DeviceId deviceInterface, bool doneExecuting)
{
	OpenIPC_Error returnError = OpenIPC_Error_No_Error;
	if (!doneExecuting) {
		std::shared_ptr<InterfaceInstanceRemotePins> pinInterface = std::dynamic_pointer_cast<InterfaceInstanceRemotePins>(this->plugin->InterfaceFromId(deviceInterface));
		if (pinInterface != nullptr && pinInterface.get() != nullptr) {
			returnError = pinInterface->DrivePin(0, pin);
		}
		else
			returnError = OpenIPC_Error_Probe_Invalid_Parameter;
	}
	return returnError;
}

OpenIPC_Error AssertPinCommand::LocalExecute(Local_Handle* handle, OpenIPC_DeviceId deviceInterface, bool doneExecuting)
{
	OpenIPC_Error returnError = OpenIPC_Error_No_Error;
	if (!doneExecuting) {
		std::shared_ptr<InterfaceInstanceRemotePins> pinInterface = std::dynamic_pointer_cast<InterfaceInstanceRemotePins>(this->plugin->InterfaceFromId(deviceInterface));
		if (pinInterface != nullptr && pinInterface.get() != nullptr) {
			returnError = pinInterface->DrivePin(1, pin);
		}
		else
			returnError = OpenIPC_Error_Probe_Invalid_Parameter;
	}
	return returnError;
}

OpenIPC_Error SetPinCommand::LocalExecute(Local_Handle* handle, OpenIPC_DeviceId deviceInterface, bool doneExecuting)
{
	OpenIPC_Error returnError = OpenIPC_Error_No_Error;
	if (!doneExecuting) {
		std::shared_ptr<InterfaceInstanceRemotePins> pinInterface = std::dynamic_pointer_cast<InterfaceInstanceRemotePins>(this->plugin->InterfaceFromId(deviceInterface));
		if (pinInterface != nullptr && pinInterface.get() != nullptr) {
			returnError = pinInterface->DrivePin(pinValue, pin);
		}
		else
			returnError = OpenIPC_Error_Probe_Invalid_Parameter;
	}
	return returnError;
}

OpenIPC_Error ReadPinCommand::LocalExecute(Local_Handle* handle, OpenIPC_DeviceId deviceInterface, bool doneExecuting)
{
	OpenIPC_Error returnError = OpenIPC_Error_No_Error;
	if (!doneExecuting) {
		std::shared_ptr<InterfaceInstanceRemotePins> pinInterface = std::dynamic_pointer_cast<InterfaceInstanceRemotePins>(this->plugin->InterfaceFromId(deviceInterface));
		if (pinInterface != nullptr && pinInterface.get() != nullptr) {
			returnError = pinInterface->ReadPin(pin, pinValue);
		}
		else
			returnError = OpenIPC_Error_Probe_Invalid_Parameter;
	}
	return returnError;
}

OpenIPC_Error GotoStateJTAGCommandASD::LocalExecute(Local_Handle* handle, OpenIPC_DeviceId deviceInterface, bool doneExecuting) {
	OpenIPC_Error returnError = OpenIPC_Error_No_Error;
	if (doneExecuting)
		return returnError;
	// TODO: Also, handle the options flags here (as well as check the deviceid!)
	std::shared_ptr<InterfaceInstanceJtagASD> jtagInterface = std::dynamic_pointer_cast<InterfaceInstanceJtagASD>(this->plugin->InterfaceFromId(deviceInterface));
	if (jtagInterface == nullptr) {
		return OpenIPC_Error_Null_Pointer;
	} else {
		returnError = jtagInterface->GotoState(this->goto_state, this->number_of_clocks_in_state, this->options);
		return returnError;
	}
}

GotoStateJTAGCommandASD::~GotoStateJTAGCommandASD() {
	if(options != NULL) {
		delete options;
		options = NULL;
	}
}

OpenIPC_Error IRShiftJTAGCommandASD::LocalExecute(Local_Handle* handle, OpenIPC_DeviceId deviceInterface, bool doneExecuting) {
	OpenIPC_Error returnError = OpenIPC_Error_No_Error;
	uint32_t number_of_bytes = (this->shiftLengthBits+7)/8;
	std::vector<uint8_t> *shift_reg_in = &this->inBits, tdo_out(number_of_bytes);
	std::shared_ptr<InterfaceInstanceJtagASD> jtagInterface = std::dynamic_pointer_cast<InterfaceInstanceJtagASD>(this->plugin->InterfaceFromId(deviceInterface));
	if (jtagInterface == nullptr)
		return OpenIPC_Error_Null_Pointer;
	if (!doneExecuting) {
		if ((TdiTdoOptions & JtagOption_TDI_Restore_From_Slot) != 0){
			BitData_GetBuffer(handle->slots[this->savedSlot],0,&(*shift_reg_in)[0],static_cast<uint32_t>(shift_reg_in->size()));
		}
		if (this->outBits == nullptr)
			returnError = jtagInterface->Shift(JtagShfIR, (char*)shift_reg_in->data(),this->shiftLengthBits, nullptr, 0);
		else if (this->inBits.size() == 0)
			returnError = jtagInterface->Shift(JtagShfIR, NULL , 0, (char*)this->outBits, this->shiftLengthBits);
		else
			returnError = jtagInterface->Shift(JtagShfIR, (char*)shift_reg_in->data(),this->shiftLengthBits, (char*)this->outBits, this->shiftLengthBits);
	}
	else {
		if (this->outBits != nullptr){
			memcpy_s(tdo_out.data(), number_of_bytes, this->outBits, tdo_out.size());
		}

		if (handle->enclosingContext != nullptr && this->outBits != nullptr){
			// Make sure we have enough space to hold everything
			BitData localTdo = BitData_CreateLocalFromBuffer(this->shiftLengthBits,tdo_out.size()*8,tdo_out.data());
			BitData_Append(handle->last_data,&localTdo);
			handle->number_of_bits_in_last_data +=this->shiftLengthBits;
		}

		if ((TdiTdoOptions & JtagOption_TDO_Save_To_Slot) != 0){
			BitData localBd = BitData_CreateLocalFromBuffer(this->shiftLengthBits,tdo_out.size()*8,tdo_out.data());
			BitData_Resize(handle->slots[this->savedSlot],this->shiftLengthBits);
			BitData_Copy(&localBd,0,handle->slots[this->savedSlot],0,this->shiftLengthBits);
		}
	}
	return returnError;
}

OpenIPC_Error DRShiftJTAGCommandASD::LocalExecute(Local_Handle* handle, OpenIPC_DeviceId deviceInterface, bool doneExecuting) {
	OpenIPC_Error returnError = OpenIPC_Error_No_Error;
	uint32_t number_of_bytes = (this->shiftLengthBits+7)/8;
	std::vector<uint8_t>* shift_reg_in = &this->inBits, tdo_out(number_of_bytes);
	std::shared_ptr<InterfaceInstanceJtagASD> jtagInterface = std::dynamic_pointer_cast<InterfaceInstanceJtagASD>(this->plugin->InterfaceFromId(deviceInterface));
	if (jtagInterface == nullptr)
		return OpenIPC_Error_Null_Pointer;
	if (!doneExecuting) {
		if ((TdiTdoOptions & JtagOption_TDI_Restore_From_Slot) != 0){
			BitData_GetBuffer(handle->slots[this->savedSlot],0,&(*shift_reg_in)[0],static_cast<uint32_t>(shift_reg_in->size()));
		}
		if (this->outBits == nullptr)
			returnError = jtagInterface->Shift(JtagShfDR, (char*)shift_reg_in->data(),this->shiftLengthBits, nullptr, 0);
		else if (this->inBits.size() == 0)
			returnError = jtagInterface->Shift(JtagShfDR, NULL , 0 , (char*)this->outBits, this->shiftLengthBits);
		else
			returnError = jtagInterface->Shift(JtagShfDR, (char*)shift_reg_in->data(),this->shiftLengthBits, (char*)this->outBits, this->shiftLengthBits);
	}
	else {
		if (this->outBits != nullptr){
			memcpy_s(tdo_out.data(), number_of_bytes, this->outBits, tdo_out.size());
		}
		if (handle->enclosingContext != nullptr && this->outBits != nullptr){
			// Make sure we have enough space to hold everything
			BitData localTdo = BitData_CreateLocalFromBuffer(this->shiftLengthBits,tdo_out.size()*8,tdo_out.data());
			BitData_Append(handle->last_data,&localTdo);
			handle->number_of_bits_in_last_data +=this->shiftLengthBits;
		}

		if ((TdiTdoOptions & JtagOption_TDO_Save_To_Slot) != 0){
			BitData localBd = BitData_CreateLocalFromBuffer(this->shiftLengthBits,tdo_out.size()*8,tdo_out.data());
			BitData_Resize(handle->slots[this->savedSlot],this->shiftLengthBits);
			BitData_Copy(&localBd,0,handle->slots[this->savedSlot],0,this->shiftLengthBits);
		}
	}
	return returnError;
}

OpenIPC_Error To_OpenIPC_Error(Connection_Error error) {
	// Before this function, Connection Errors were just passed back as OpenIPC errors
	// resulting in confustion since OpenIPC_Error is a sub-set of Connection_Error
	OpenIPC_Error result = (OpenIPC_Error)error;

	// Map groups of Connection Errors to OpenIPC Errors
	switch (error) {
	case Invalid_Device_Id:
		result = OpenIPC_Error_Invalid_Device_ID;
		break;
	case Bad_Argument:
	case RC_Plugin_Not_Registered:
	case Wrong_Socket_Data_Size:
	case Read_Thread_Init_Failure:
	case Mutex_Error:
	case Null_Pointer:
	case Initialize_Error:
	case I2C_Response_Error:
	case I2C_Request_Error:
	case I2C_Nack_Received:
	case I2C_Write_Acks_Not_Received:
	case I2C_Unknown_Request:
	case I2C_Unknown_Response:
	case Not_Implemented:
	case WAIT_PRDY_Timeout:
	case Invalid_Data_Size:
	case Socket_Failed_Send:
	case Socket_Failed_Recv:
	case Socket_Received_Error_From_Tap:
	case Error_Processing_Received_Data:
	case Error_Receiveing_Thread_Died:
	case Connection_Busy:
	case Failed_To_UnRegister:
	case Invalid_Message_Type:
	case Data_Not_Flushed:
	case Could_Not_Handle_Mutex:
	case Could_Not_Contact_BMC:
	case Could_Not_Connect_To_BMC:
	case Connection_To_BMC_Rejected:
	case Socket_Read_Wait_Failure:
	case Multi_Bundle_Error:
	case Socket_Send_Recv_Wrong_Tag:
		result = OpenIPC_Error_Internal_Error;
		break;
	case User_Not_Registered:
	case User_Already_Registered:
	case Failed_To_Register:
	case Probe_Plugin_Not_Registered:
	case ASD_MSG_CRYPY_NOT_SUPPORTED:
	case ASD_FAILURE_INIT_JTAG_HANDLER:
	case ASD_FAILURE_INIT_I2C_HANDLER:
	case ASD_FAILURE_DEINIT_JTAG_HANDLER:
	case ASD_MSG_NOT_SUPPORTED:
	case ASD_FAILURE_PROCESS_JTAG_MSG:
	case ASD_FAILURE_PROCESS_I2C_MSG:
	case ASD_FAILURE_PROCESS_I2C_LOCK:
	case ASD_I2C_MSG_NOT_SUPPORTED:
	case ASD_FAILURE_REMOVE_I2C_LOCK:
	case ASD_FAILURE_HEADER_SIZE:
	case ASD_FAILURE_XDP_PRESENT:
	case ASD_UNKNOWN_ERROR:
	case Error_Connection_Taken:
		result = OpenIPC_Error_Initialization_Failed;
		break;
	case No_Error:
		result = OpenIPC_Error_No_Error;
	}

	return result;
}


// Translate a Connection_Error to text.

std::string Connection_Error_text(Connection_Error error) {
	std::string text;

	switch (error) {
	case Invalid_Device_Id:
		text = "Invalid Device identifier";
		break;

	case Invalid_Data_Size:
		text = "Invalid Data Size";
		break;

	case Socket_Failed_Send:
		text = "Socket send failure";
		break;

	case Socket_Failed_Recv:
		text = "Socket receive failure";
		break;

	case Socket_Received_Error_From_Tap:
		text = "Socket receive error from tap";
		break;

	case Error_Processing_Received_Data:
		text = "Error processing received data";
		break;

	case Error_Receiveing_Thread_Died:
		text = "Receiving thread died";
		break;

	case Connection_Busy:
		text = "Connection Busy";
		break;

	case Failed_To_UnRegister:
		text = "Failed to unregister";
		break;

	case Invalid_Message_Type:
		text = "Invalid message type";
		break;

	case Data_Not_Flushed:
		text = "Data not flushed";
		break;

	case Could_Not_Handle_Mutex:
		text = "Could not handle mutex";
		break;

	case Could_Not_Contact_BMC:
		text = "Could not contact BMC";
		break;

	case Could_Not_Connect_To_BMC:
		text = "Could not connect to BMC";
		break;

	case Connection_To_BMC_Rejected:
		text = "Connection to BMC rejected";
		break;

	case Socket_Read_Wait_Failure:
		text = "Socket read wait failure";
		break;

	case User_Not_Registered:
		text = "User not registered";
		break;

	case User_Already_Registered:
		text = "User already registered";
		break;

	case Failed_To_Register:
		text = "Failed to register";
		break;

	case Probe_Plugin_Not_Registered:
		text = "Probe plugin not registered";
		break;

	case Error_Connection_Taken:
		text = "Error connection taken";
		break;

	default:
		text = "Unknown Connection_Error";
		break;
	}

	return text;
}


OpenIPC_Error handle_command(Command* cmd, PPI_ProbeBundleHandle handle) {
	OpenIPC_Error result = OpenIPC_Error_No_Error;

	if (handle == PPI_PROBE_LOCK_RELEASE) {
		if (locked_handle.target_valid) {
			result = ExecuteAllCommands(&locked_handle, locked_handle.target, cmd);
		}
		else {
			result = OpenIPC_Error_Probe_Bundle_Invalid;
		}
		delete cmd;

		for (uint32_t i = 0; i<locked_handle.slots.size(); i++) {
			uint32_t bitSize = static_cast<uint32_t>(locked_handle.slots[i]->bitsize);
			BitData_Resize(locked_handle.slots[i], 0);
			BitData_Resize(locked_handle.slots[i], bitSize);
		}
		in_lock = false;
	} else if (handle == PPI_PROBE_LOCK_HOLD) {
		if (locked_handle.target_valid) {
			result = ExecuteAllCommands(&locked_handle, locked_handle.target, cmd);
		}
		else {
			result = OpenIPC_Error_Probe_Bundle_Invalid;
		}
		delete cmd;
		in_lock = true;
	} else {
		Local_Handle* current = (Local_Handle*)handle;
		// Check for the type of operation here...
		current->last->Append(cmd);
		current->last = cmd;
	}

	return result;
}

uint32_t GotoStateJTAGCommandASD::length_of_output() {
	return (number_of_clocks_in_state + 7)/8;
}

uint32_t IRShiftJTAGCommandASD::length_of_output(){
	return (shiftLengthBits + 7)/8;
}

uint32_t DRShiftJTAGCommandASD::length_of_output(){
	return (shiftLengthBits + 7)/8;
}

OpenIPC_Error JTAGDelay::LocalExecute(Local_Handle* handle, OpenIPC_DeviceId deviceInterface, bool doneExecuting){
	return OpenIPC_Error_No_Error;
}

uint32_t JTAGDelay::length_of_output(){
	return 0;
}

uint32_t DeassertPinCommand::length_of_output(){
	return 1;
}

uint32_t AssertPinCommand::length_of_output(){
	return 1;
}

uint32_t PinsDelayCommand::length_of_output(){
	return 1;
}

uint32_t ReadPinCommand::length_of_output(){
	return 1;
}

uint32_t SetPinCommand::length_of_output(){
	return 1;
}

OpenIPC_Error JTAGUpdateBundlePadding::LocalExecute(Local_Handle* handle, OpenIPC_DeviceId deviceInterface, bool doneExecuting)
{
	OpenIPC_Error result = OpenIPC_Error_No_Error;
	if (!doneExecuting) {
		handle->dr_pre = this->drPaddingNearTDO;
		handle->dr_post = this->drPaddingNearTDI;
		handle->ir_pre = this->irPaddingNearTDO;
		handle->ir_post = this->irPaddingNearTDI;

		std::shared_ptr<InterfaceInstance> interface_ = GetProbePluginInstanceASD()->InterfaceFromId(deviceInterface);
		if (interface_ == nullptr) {
			result = OpenIPC_Error_Internal_Error;  // Invalid deviceInterface argument
		}
		InterfacePadding padding = { 0 };
		JtagChainInfo jtagSettings = { 0 };
		JtagChainParameters params;
		params.InterfacePaddingSettings = &padding;
		params.JtagChainSettings = &jtagSettings;

		std::shared_ptr<InterfaceInstanceJtagASD> jtagInterface = std::dynamic_pointer_cast<InterfaceInstanceJtagASD>(interface_);
		if (jtagInterface != nullptr) {
			jtagInterface->ObtainJtagChainParameters(&params);
		}

		params.InterfacePaddingSettings->drPaddingNearTDI = handle->dr_pre;
		params.InterfacePaddingSettings->drPaddingNearTDO = handle->dr_post;
		params.InterfacePaddingSettings->irPaddingNearTDI = handle->ir_pre;
		params.InterfacePaddingSettings->irPaddingNearTDO = handle->ir_post;

		result = To_OpenIPC_Error(UpdateInterfacePaddingSettings(interface_->probe->deviceID, params.InterfacePaddingSettings));
	}
	return result;
}

uint32_t JTAGUpdateBundlePadding::length_of_output()
{
	return 0;
}

OpenIPC_Error I2cWriteCfgCommandASD::LocalExecute(Local_Handle* handle, OpenIPC_DeviceId deviceInterface, bool doneExecuting){
	// TODO
	return OpenIPC_Error_No_Error;
}

uint32_t I2cWriteCfgCommandASD::length_of_output(){
	// TODO
	return 1;
}

OpenIPC_Error I2cReadCommandASD::LocalExecute(Local_Handle* handle, OpenIPC_DeviceId deviceInterface, bool doneExecuting){
	OpenIPC_Error returnError = OpenIPC_Error_No_Error;
	std::shared_ptr<InterfaceInstanceI2cASD> i2cInterface = std::dynamic_pointer_cast<InterfaceInstanceI2cASD>(this->plugin->InterfaceFromId(deviceInterface));
	if (i2cInterface == nullptr)
		return OpenIPC_Error_Null_Pointer;
	if (!doneExecuting) {
		returnError = i2cInterface->Read(deviceId, bufferLength, readBuffer, forceStop, addressAck);
	}

	return returnError;
}

uint32_t I2cReadCommandASD::length_of_output(){
	// TODO
	return 1;
}

OpenIPC_Error I2cWriteCommandASD::LocalExecute(Local_Handle* handle, OpenIPC_DeviceId deviceInterface, bool doneExecuting)
{
	OpenIPC_Error returnError = OpenIPC_Error_No_Error;
	std::shared_ptr<InterfaceInstanceI2cASD> i2cInterface = std::dynamic_pointer_cast<InterfaceInstanceI2cASD>(this->plugin->InterfaceFromId(deviceInterface));
	if (i2cInterface == nullptr)
		return OpenIPC_Error_Null_Pointer;
	if (!doneExecuting) {
		returnError = i2cInterface->Write(deviceId, bufferLength, writeBuffer, forceStop, addressAck, lastDataAck);
	}
	return returnError;
}

uint32_t I2cWriteCommandASD::length_of_output()
{
	// TODO
	return 1;
}
