//////////////////////////////////////////////////////////////////////////////
//
//                      INTEL CONFIDENTIAL
//       Copyright 2017 Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its
// suppliers. Title to the Material remains with Intel Corporation, its
// suppliers, or licensors. The Material contains trade secrets and
// proprietary and confidential information of Intel Corporation, its
// suppliers, and licensors, and is protected by worldwide copyright and trade
// secret laws and treaty provisions. No part of the Material may be used,
// copied, reproduced, modified, published, uploaded, posted, transmitted,
// distributed, or disclosed in any way without Intel's prior express written
// permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or
// delivery of the Materials, either expressly, by implication, inducement,
// estoppel or otherwise. Any license under such intellectual property rights
// must be express and approved by Intel in writing.
//
// Unless otherwise agreed by Intel in writing, you may not remove or alter
// this notice or any other notice embedded in Materials by Intel or Intel's
// suppliers or licensors in any way.
//
//////////////////////////////////////////////////////////////////////////////
#include "InterfaceInstanceRemotePins.h"
#include "RemoteConnection.h"
#include <Foundation/Error/ErrorTypes.h>
#include <Foundation/Types.h>
#include "ProbePluginErrors.h"
#include "BundleHelpersASD.h"

#include <algorithm>

void InterfaceInstanceRemotePins::AddTriggerResponse(std::shared_ptr<PPI_InterfaceTriggerResponse> triggerResponse)
{
	_triggerResponsesSupported.push_back(triggerResponse);
}

void InterfaceInstanceRemotePins::SettingsUpdated() {
	bool prdyBankAAssert = false;
	bool prdyBankBAssert = false;
	bool prdyBankCAssert = false;
	bool prdyBankDAssert = false;
	bool _checkPrdy = false;
	unsigned char buf[4];
	buf[0] = 0x0; //write config cmd Probe Config reg
	buf[1] = 0x2; // prdy report setting default off
	buf[2] = 0x0; //write config cmd Probe Config reg
	buf[3] = 0x83; // reset reporting setting default on

	if (settings.GetBool("Pins.PrdyBankAAssert"))
	{
		PPI_LOG(deviceID, PPI_debugNotification, "Pins.PrdyBankAAssert Set to true");
		prdyBankAAssert = true;
	}
	else {
		PPI_LOG(deviceID, PPI_debugNotification, "Pins.PrdyBankAAssert Set to false");
	}

	if (settings.GetBool("Pins.PrdyBankBAssert"))
	{
		PPI_LOG(deviceID, PPI_debugNotification, "Pins.PrdyBankBAssert Set to true");
		prdyBankBAssert = true;
	}
	else {
		PPI_LOG(deviceID, PPI_debugNotification, "Pins.PrdyBankBAssert Set to false");
	}

	if (settings.GetBool("Pins.PrdyBankCAssert"))
	{
		PPI_LOG(deviceID, PPI_debugNotification, "Pins.PrdyBankCAssert Set to true");
		prdyBankCAssert = true;
	}
	else {
		PPI_LOG(deviceID, PPI_debugNotification, "Pins.PrdyBankCAssert Set to false");
	}

	if (settings.GetBool("Pins.PrdyBankDAssert"))
	{
		PPI_LOG(deviceID, PPI_debugNotification, "Pins.PrdyBankDAssert Set to true");
		prdyBankDAssert = true;
	}
	else {
		PPI_LOG(deviceID, PPI_debugNotification, "Pins.PrdyBankDAssert Set to false");
	}
	_checkPrdy = prdyBankAAssert || prdyBankBAssert || prdyBankCAssert || prdyBankDAssert;

	if (_checkPrdy) {
		PPI_LOG(deviceID, PPI_debugNotification, "Enabling PRDY reporting");
		buf[1] = 0x82;
	}
	else {
		PPI_LOG(deviceID, PPI_debugNotification, "Disabling PRDY reporting");
	}
	StartDataTransfer(probe->deviceID, JTAG_MESSAGE_TYPE, NULL);
	RemoteShift(probe->deviceID, JtagNoStateChange, (char*)buf, 32, NULL, 0);
	EndDataTransfer(probe->deviceID);
}

OpenIPC_Error InterfaceInstanceRemotePins::DrivePin(uint8_t state, PPI_Pins_TypeEncode pin)
{
	char buf[2], pinState;
	if (std::find(drivePins.begin(), drivePins.end(), pin) == drivePins.end()) {
		return OpenIPC_Error_Probe_Invalid_Parameter;
	}

	//bits 6:0 are determining which pin will be set
	//bit 7 determines if it is assert or deassert LOGICALLY
	pinState = (openIPC2Protocol[pin] & 0x7F) | ((state & 0x1) << 7);

	PPI_LOG(deviceID, PPI_debugNotification, "Drive Pin 0x" << std::hex << pin << " to " << (state & 0x1));

	buf[0] = 0x7;
	buf[1] = pinState;
	return To_OpenIPC_Error(RemoteShift(probe->deviceID, JtagNoStateChange, buf, 16, NULL, 0));
}

OpenIPC_Error InterfaceInstanceRemotePins::ReadPin(PPI_Pins_TypeEncode pin, uint32_t *pinValue)
{
	OpenIPC_Error returnError = OpenIPC_Error_No_Error;
	char buf[2];
	GotoStateOptions options = GotoStateOptions();
	options.flush = true;

	if (std::find(readPins.begin(), readPins.end(), pin) == readPins.end()) {
		return OpenIPC_Error_Probe_Invalid_Parameter;
	}

	buf[0] = READ_CMD_BIT | CFGREG_PIN_STATUS;
	buf[1] = openIPC2Protocol[pin] & 0x7F;

	returnError = To_OpenIPC_Error(RemoteShift(probe->deviceID, JtagNoStateChange, buf, 16, (char*)pinValue, 1, &options));
	if (!OpenIPC_PASS(returnError)) {
		POST_ERROR(returnError);
		return returnError;
	}
	if ((openIPC2Protocol[pin] & 0x7f) != (*pinValue & 0x7F)) {
		return OpenIPC_Error_Bad_Argument;
	}
	*pinValue = *pinValue >> 7;
	PPI_LOG(deviceID, PPI_debugNotification, "Read Pin: " << pin << " is " << *pinValue);
	return OpenIPC_Error_No_Error;

}

OpenIPC_Error InterfaceInstanceRemotePins::AddPin(PPI_Pins_TypeEncode openIPCValue, uint32_t protocolValue, uint32_t readWrite)
{
	openIPC2Protocol[openIPCValue] = protocolValue;
	if (readWrite == InterfaceInstanceRemotePins::READ || readWrite == InterfaceInstanceRemotePins::READWRITE)
		readPins.push_back(openIPCValue);
	if (readWrite == InterfaceInstanceRemotePins::WRITE || readWrite == InterfaceInstanceRemotePins::READWRITE)
		drivePins.push_back(openIPCValue);
	return OpenIPC_Error_No_Error;
}

OpenIPC_Error InterfaceInstanceRemotePins::GetReadablePins(std::vector<PPI_Pins_TypeEncode>& pins) const
{
	pins = readPins;
	return OpenIPC_Error_No_Error;
}

OpenIPC_Error InterfaceInstanceRemotePins::GetDrivablePins(std::vector<PPI_Pins_TypeEncode>& pins) const
{
	pins = drivePins;
	return OpenIPC_Error_No_Error;
}

InterfaceInstanceRemotePins::InterfaceInstanceRemotePins(uint32_t interface_refid, uint32_t instance_id, PPI_InterfacePinsCapabilities caps)
    :InterfaceInstancePins(interface_refid, instance_id, caps),
     readPins(), drivePins(), openIPC2Protocol({})
{
}
