//////////////////////////////////////////////////////////////////////////////
//
//                      INTEL CONFIDENTIAL
//       Copyright 2017 Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its
// suppliers. Title to the Material remains with Intel Corporation, its
// suppliers, or licensors. The Material contains trade secrets and
// proprietary and confidential information of Intel Corporation, its
// suppliers, and licensors, and is protected by worldwide copyright and trade
// secret laws and treaty provisions. No part of the Material may be used,
// copied, reproduced, modified, published, uploaded, posted, transmitted,
// distributed, or disclosed in any way without Intel's prior express written
// permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or
// delivery of the Materials, either expressly, by implication, inducement,
// estoppel or otherwise. Any license under such intellectual property rights
// must be express and approved by Intel in writing.
//
// Unless otherwise agreed by Intel in writing, you may not remove or alter
// this notice or any other notice embedded in Materials by Intel or Intel's
// suppliers or licensors in any way.
//
//////////////////////////////////////////////////////////////////////////////
#include <cstring>

#include "Public/StructuredData.h"					// For XML support

#include "ProbeInstanceASD.h"
#include "OutputBuffers.h"
#include "RemoteConnection.h"
#include "ProbePluginErrors.h"
#include "ProbePluginASD_JTAG.hpp"
#include "BundleHelpersASD.h"
#include "ASDBaseProtocolDefs.h"
#include "PPI_InterfaceTriggerResponse.h"

#ifdef NEED_SAFE_CLIB
#include "safe_lib.h"
#endif

ProbeInstanceASD::ProbeInstanceASD(uint32_t probe_refid, std::string typeString, uint32_t typeID)
	: ProbeInstance(probe_refid, typeString, typeID),
	clockCycleTimeinUs(0), SocketServerMajorMinor(""), ProbeMajorMinor(""), fwVersion(""),
	socketServerVersion(""), pinInterface(nullptr),
	jtagInterfaces({}), i2cInterfaces({}),
	_eventHandlerFunction(nullptr) {

	jtagInterfaces.reserve(ASD_JTAG_PROTOCOL::MAX);
	i2cInterfaces.reserve(ASD_JTAG_PROTOCOL::MAX);
	connectionParameters = std::make_shared<ConnectionParameters>();
}

void ProbeInstanceASD::SettingsUpdated()
{
	// Pass any Device/Probe settings to each registered protocol's bundle filter
	auto plugin = ProbePluginASD_JTAG::GetInstance();
	for (auto itr = plugin->BundleFilters().begin(); itr != plugin->BundleFilters().end(); itr++) {
		itr->second->UpdateSettings(&settings);
	}
}

//////////////////////////////////////////////////////////////////////////////
// ProbeRegisterEventHandler
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error ProbeInstanceASD::ProbeRegisterEventHandler(
	PluginEventCallbackHandler eventHandlerFunction
)
{
	OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

	if (eventHandlerFunction == nullptr)
		return openIPCError;

	if (_eventHandlerFunction == nullptr)
	{
		_eventHandlerFunction = eventHandlerFunction;
	}
	else
	{
		openIPCError = OpenIPC_Error_Callback_Already_Initialized;
		PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
	}

	return openIPCError;
}

bool ProbeInstanceASD::ProcessSpecialCommands(char command, std::shared_ptr<OutputBuffers> buffer, char data, uint16_t *size)
{
	if ((buffer->cmd & READ_CMD_BIT) != 0 &&
		(buffer->cmd & ~READ_CMD_MASK) == 0) {
		*(buffer->buffer) = data;
		*size = 1;
		return true;
	}
	return false;
}

void ProbeInstanceASD::callEventHandler(PPI_PluginEvent pluginEvent, uint64_t value)
{
	// We dont directly use PPI_TYPE_EVENTS in our plugin, but other plugins do.
	// This line is to silence compiler warnings, particularly on the Linux side.
	(void)PPI_TYPE_EVENTS;

	if (_eventHandlerFunction != nullptr && deviceID != -1) {
		PPI_LOG(deviceID, PPI_traceNotification, "event handler function being called. (1 of 2)");
		_eventHandlerFunction(deviceID, pluginEvent, value);
		PPI_LOG(deviceID, PPI_traceNotification, "event handler function returned. (2 of 2)");
	}
}

OpenIPC_Error ProbeInstanceASD::DelayUseconds(uint32_t numberOfUseconds)
{
	uint32_t counter = uint32_t(numberOfUseconds / clockCycleTimeinUs);
	OpenIPC_Error ret = OpenIPC_Error_No_Error;
	char *buf;
	int i = 0;

	if (counter == 0) return ret;

	buf = (char*)malloc((counter / 256 + 1) * 2);
	if (buf == nullptr)
		return OpenIPC_Error_Null_Pointer;
	PPI_LOG(deviceID, PPI_traceNotification, "Sending Wait Cycles Cmd, TCKenable = 0, Cycles = " << counter);
	while (counter > 0)
	{
		uint32_t tmp_count = counter;
		if (tmp_count > 256) tmp_count = 256;
		counter -= tmp_count;
		if (tmp_count == 256) tmp_count = 0;
		buf[i++] = CMD_WAITCYCLES;
		buf[i++] = char(tmp_count);
	}
	ret = RemoteShift(deviceID, JtagNoStateChange, buf, 8 * i, NULL, 0);
	free(buf);
	return ret;
}

OpenIPC_Error ProbeInstanceASD::ConfigureTap(JtagChainParameters* params)
{
	Connection_Error ret = No_Error;
	//this will force configuration to be applied
	ret = StartDataTransfer(deviceID, JTAG_MESSAGE_TYPE, params);
	if (CONNECTION_PASS(ret)) {
		EndDataTransfer(deviceID);
	}
	return To_OpenIPC_Error(ret);
}

OpenIPC_Error ProbeInstanceASD::SetPinsInterface(std::shared_ptr<InterfaceInstanceRemotePins> probeinterface)
{
	pinInterface = probeinterface;
	AddInterface(probeinterface);
	return OpenIPC_Error_No_Error;
}

OpenIPC_Error ProbeInstanceASD::GetReadablePins(std::vector<PPI_Pins_TypeEncode>& pins) const
{
	return pinInterface->GetReadablePins(pins);
}

OpenIPC_Error ProbeInstanceASD::GetDrivablePins(std::vector<PPI_Pins_TypeEncode>& pins) const
{
	return pinInterface->GetDrivablePins(pins);
}

void  ProbeInstanceASD::SetVersion() {
	fwVersion.assign(PROBE_PLUGIN_VERSION);
	unsigned int length = (unsigned int)strlen(PROBE_PLUGIN_VERSION) + 1;
	std::string temp;
	MajorMinor(PROBE_PLUGIN_VERSION, length, (char*)temp.c_str(), LEFT);
	ProbeMajorMinor.assign(temp);
}

std::string  ProbeInstanceASD::GetVersion() {
	return fwVersion;
}

void ProbeInstanceASD::AppendDownstreamVersion(char *buffer, uint16_t length) {
	// The version of the JTAG Socket Server we are connected to
	fwVersion.append(buffer, length);
	socketServerVersion.assign(buffer, length);
	std::string temp;
	MajorMinor(buffer, length, (char*)temp.c_str(), RIGHT);
	SocketServerMajorMinor.assign(temp);
}

bool is_char_a_version_number_digit(const char c){
	return (c >= '0' && c <= '9') || (c >= 'A' && c <= 'F') || (c >= 'a' && c <= 'f');
}

// out must be the same size as in.  in_size includes null termination
void ProbeInstanceASD::MajorMinor(const char *in, unsigned int in_size, char *out, bool right) {
	char c;
	unsigned int points = 0;

	if (in_size == 0 ||
		in == nullptr ||
		out == nullptr)
		return;

	out[0] = 0;

	if (right) {
		// From the right
		unsigned int i = in_size - 1;
		unsigned int previousPoint = 0;
		unsigned int actualPoint = i;
		while (i) {
			c = in[--i];
			if (is_char_a_version_number_digit(c) || c == '.') {
				if (c == '.') {
					points++;
					if (points > 1) {
						actualPoint = previousPoint;
					}
					previousPoint = i;
				}
			} else
				break;
		}
		if (i < in_size - 1) {
			if(i > 0)
				i++; // skip the break on character
			unsigned int x = 0;
			for (; i < in_size - 1; x++) {
				if (i < actualPoint) {
					out[x] = in[i];
				}
				i++;
			}
			out[x] = 0;
		}
	} else {
		// From the left
		unsigned int i = 0;
		if (strncmp(in, "dev", 3) != 0) {
			for (; i < in_size - 1; i++) {
				c = in[i];
				if (c == '.') {
					if (points > 0)
						break;
					out[i] = c;
					points++;
				}
				if (is_char_a_version_number_digit(c)) {
					out[i] = c;
				}
			}
			out[i] = 0;
		}
		else {
			strcpy_s(out, in_size, "dev");
		}
	}
}

uint32_t ProbeInstanceASD::GetNextJtagInterfaceId()
{
	uint32_t max = 0;
	JtagChainParameters params;
	JtagChainInfo jtagSettings;
	InterfacePadding padding = { 0 };
	params.InterfacePaddingSettings = &padding;
	params.JtagChainSettings = &jtagSettings;
	std::vector<std::shared_ptr<InterfaceInstanceJtagASD>>::iterator iter;

	if (jtagInterfaces.empty())
		return 0;

	for (iter = jtagInterfaces.begin(); iter < jtagInterfaces.end(); iter++)
	{
		(*iter)->ObtainJtagChainParameters(&params);
		if (params.JtagChainSettings->JtagChainProtocolId >= max)
		{
			max = params.JtagChainSettings->JtagChainProtocolId + 1;
		}
	}
	return max;
}

std::shared_ptr<InterfaceInstanceJtagASD> ProbeInstanceASD::GetJtagInterface(uint32_t jtagProtocolId)
{
	std::shared_ptr<InterfaceInstanceJtagASD> result = nullptr;
	JtagChainParameters params;
	JtagChainInfo jtagSettings;
	InterfacePadding padding = { 0 };
	params.InterfacePaddingSettings = &padding;
	params.JtagChainSettings = &jtagSettings;

	std::vector<std::shared_ptr<InterfaceInstanceJtagASD>>::iterator iter;
	for (iter = jtagInterfaces.begin(); iter < jtagInterfaces.end(); iter++)
	{
		(*iter)->ObtainJtagChainParameters(&params);
		if (params.JtagChainSettings->JtagChainProtocolId == jtagProtocolId)
		{
			result = (*iter);
			break;
		}
	}

	return result;
}

OpenIPC_Error ProbeInstanceASD::AddJtagInterface(std::shared_ptr<InterfaceInstanceJtagASD> jtagInterface)
{
	OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
	JtagChainParameters params;
	JtagChainInfo jtagSettings;
	InterfacePadding padding = { 0 };
	params.InterfacePaddingSettings = &padding;
	params.JtagChainSettings = &jtagSettings;

	jtagInterface->ObtainJtagChainParameters(&params);
	if (GetJtagInterface(params.JtagChainSettings->JtagChainProtocolId) != nullptr) {
		openIPCError = OpenIPC_Error_Interface_Already_Initialized;
	} else {
		jtagInterfaces.push_back(jtagInterface);
		AddInterface(jtagInterface);
	}

	return openIPCError;
}

void ProbeInstanceASD::JtagDeinitialize()
{
	jtagInterfaces.clear();
}

uint32_t ProbeInstanceASD::GetI2cInterfaceBusCount()
{
	uint32_t count = 0;
	std::vector<std::shared_ptr<InterfaceInstanceI2cASD>>::iterator iter;

	if (i2cInterfaces.empty())
		return 0;

	for (iter = i2cInterfaces.begin(); iter < i2cInterfaces.end(); iter++)
	{
		count++;
	}

	return count;
}


OpenIPC_Error ProbeInstanceASD::AddI2CInterface(std::shared_ptr<InterfaceInstanceI2cASD> i2cInterface)
{
	OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

	i2cInterfaces.push_back(i2cInterface);
	AddInterface(i2cInterface);

	return openIPCError;
}

void ProbeInstanceASD::I2CDeinitialize()
{
	i2cInterfaces.clear();
}


OpenIPC_Error ProbeInstanceASD::ProbeInitializeHwConfig(DataElement* probeChild, uint8_t &hardware_gpio_config, JtagChainParameters *params)
{
	OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
	DataElement* elementChild;

	params->JtagDriverMode = JTAG_DRIVER_MODE_SOFTWARE;
	params->JtagChainSelectMode = JTAG_CHAIN_SELECT_MODE_SINGLE;
	params->JtagSyncTimeoutValue = DEFAULT_JTAG_SYNC_TIMEOUT_VALUE;
	params->JtagSyncDelayValue = DEFAULT_JTAG_SYNC_DELAY_VALUE;

	elementChild = probeChild->FirstChild();
	while (elementChild)
	{
		DataAttribute* configAttribute = nullptr;
		if (elementChild->Name() == "Gpio")
		{
			if ((configAttribute = elementChild->FindAttribute("Config")) == nullptr) {
				elementChild = elementChild->NextSibling();
				continue;
			}
			PluginNotifications::GetInstance().Notify(
				deviceID,
				OpenIPC_Error_No_Error,
				PPI_infoNotification,
				[&](std::basic_ostream<char>& stream)
			{
				stream << "Hardware GPIO config = " << configAttribute->Value().c_str();
			});
			hardware_gpio_config = (uint8_t)strtoul(configAttribute->Value().c_str(), NULL, 0);
		}
		else if (elementChild->Name() == "Jtag")
		{
			if ((configAttribute = elementChild->FindAttribute("Mode")) != nullptr) {
				std::string data = configAttribute->Value();
				transform(data.begin(), data.end(), data.begin(), ::tolower);

				if (data.compare("hardware") == 0 || data.compare("hw") == 0)
					params->JtagDriverMode = JTAG_DRIVER_MODE_HARDWARE;

				PluginNotifications::GetInstance().Notify(
					deviceID,
					OpenIPC_Error_No_Error,
					PPI_infoNotification,
					[&](std::basic_ostream<char>& stream)
				{
					stream << "JTAG Mode = " << (params->JtagDriverMode == JTAG_DRIVER_MODE_HARDWARE ? "HARDWARE" : "SOFTWARE");
				});
			}

			if ((configAttribute = elementChild->FindAttribute("Chain_Select_Mode")) != nullptr) {
				std::string data = configAttribute->Value();
				transform(data.begin(), data.end(), data.begin(), ::tolower);
				if (data.compare("multi") == 0) {
					params->JtagChainSelectMode = JTAG_CHAIN_SELECT_MODE_MULTI;
				}
				PluginNotifications::GetInstance().Notify(
					deviceID,
					OpenIPC_Error_No_Error,
					PPI_infoNotification,
					[&](std::basic_ostream<char>& stream)
				{
					stream << "JTAG Chain Select Mode = " << (params->JtagChainSelectMode == JTAG_CHAIN_SELECT_MODE_SINGLE ? "SINGLE" : "MULTI");
				});
			}
		} else if (elementChild->Name() == "Sync") {
			if ((configAttribute = elementChild->FindAttribute("SyncTimeout")) != nullptr) {
				params->JtagSyncTimeoutValue = configAttribute->ToUInt16();
				PluginNotifications::GetInstance().Notify(
					deviceID,
					OpenIPC_Error_No_Error,
					PPI_infoNotification,
					[&](std::basic_ostream<char>& stream)
				{
					stream << "JTAG Sync Timeout = " << params->JtagSyncTimeoutValue;
				});
			}

			if ((configAttribute = elementChild->FindAttribute("SyncDelay")) != nullptr) {
				params->JtagSyncDelayValue = configAttribute->ToUInt16();
				PluginNotifications::GetInstance().Notify(
					deviceID,
					OpenIPC_Error_No_Error,
					PPI_infoNotification,
					[&](std::basic_ostream<char>& stream)
				{
					stream << "JTAG Sync Delay = " << params->JtagSyncDelayValue;
				});
			}
		}
		elementChild = elementChild->NextSibling();
	}

	return openIPCError;
}

#include <boost/filesystem.hpp>
#include <boost/foreach.hpp>
#include <regex>

OpenIPC_Error ProbeInstanceASD::ProbeInitializeJtagDevHandle(DataElement* probeChild, int &cnt, std::shared_ptr<ConnectionParameters> connectionParameters)
{
	OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
	if (!probeChild)
		return OpenIPC_Error_Null_Pointer;
	DataAttribute *fileName = nullptr;
	uint32_t protocolId = GetNextJtagInterfaceId();
	std::string fileNames = "asd";
	if ((fileName = probeChild->FindAttribute("fileName")) != nullptr)
		fileNames.append(fileName->Value());
	else
		return OpenIPC_Error_Invalid_Argument;

	boost::filesystem::path devDir("/dev");
	boost::filesystem::directory_iterator iter(devDir), end;
	BOOST_FOREACH(const boost::filesystem::path& devPath, std::make_pair(iter, end))
	{
		if (boost::filesystem::is_regular_file(devPath))
		{
			if (std::regex_match(devPath.stem().string(), std::regex(fileNames)))
			{
				DataElement* newChain = probeChild->CreateChild("Interface.Jtag");
				newChain->CreateAttribute("Protocol_Value", std::to_string(protocolId++));
				newChain->CreateAttribute("FileName", devPath.stem().string());
				openIPCError = ProbeInitializeJtag(newChain, cnt);
				if (!OpenIPC_PASS(openIPCError))
					goto end;
				connectionParameters->transportParameters->fileNames.push_back(devPath.string());
			}
		}
	}
end:
	return openIPCError;
}

OpenIPC_Error ProbeInstanceASD::ProbeInitializeJtag(DataElement* probeChild, int &cnt)
{
	OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
	std::string finalJtagSpeed("1000000"); //default is 1MHz
	PPI_InterfaceJTAGCapabilities jtagCapabilities = { 1, 1, 0 };
	JtagChainParameters params;
	JtagChainInfo jtagSettings;
	InterfacePadding padding = { 0 };
	DataAttribute* jtagIdAttr = nullptr;
	bool foundJtagId = false;
	uint32_t jtagProtocolId = ASD_JTAG_PROTOCOL::DEFAULT;
	DataAttribute* jtagSpeed = nullptr;
	DataAttribute* divisorAttr = nullptr;
	uint8_t jtag_prescaler = DEFAULT_JTAG_PRESCALER;
	uint8_t jtag_divisor = DEFAULT_JTAG_DIVISOR;

	jtagSettings.JtagChainProtocolId = ASD_JTAG_PROTOCOL::DEFAULT;
	jtagSettings.JtagPrescaler = jtag_prescaler;
	jtagSettings.JtagDivisor = jtag_divisor;
	jtagSettings.SendWaitSync = false;
	params.InterfacePaddingSettings = &padding;
	params.JtagChainSettings = &jtagSettings;
	params.JtagSyncDelayValue = DEFAULT_JTAG_SYNC_DELAY_VALUE;
	params.JtagSyncTimeoutValue = DEFAULT_JTAG_SYNC_TIMEOUT_VALUE;

	if ((jtagSpeed = probeChild->FindAttribute("JTAG_Speed")) != nullptr) {
		std::string jtagSpeedStr = jtagSpeed->Value();
		if (jtagSpeedStr.find("MHZ") != std::string::npos || jtagSpeedStr.find("Mhz") != std::string::npos || jtagSpeedStr.find("mhz") != std::string::npos || jtagSpeedStr.find("MHz") != std::string::npos) {
			finalJtagSpeed = jtagSpeedStr.substr(0, (jtagSpeedStr.length() - 3));
			finalJtagSpeed.append("000000");
		}
		else finalJtagSpeed = jtagSpeedStr;
	}

	if ((jtagIdAttr = probeChild->FindAttribute("Protocol_Value")) != nullptr) {
		jtagProtocolId = strtoul(jtagIdAttr->Value().c_str(), NULL, 0);
		// search for previous entry for this protocol value jtagProtocolId
		foundJtagId = (GetJtagInterface(jtagProtocolId) != nullptr);
	}

	if ((divisorAttr = probeChild->FindAttribute("Divisor")) != nullptr) {
		uint16_t divisor = (uint16_t)strtoul(divisorAttr->Value().c_str(), NULL, 0);
		if (divisor == 0 || divisor > 512) {
			PluginNotifications::GetInstance().Notify(
				deviceID,
				OpenIPC_Error_No_Error,
				PPI_warningNotification,
				[&](std::basic_ostream<char>& stream)
			{
				stream << "JTAG Chain " << std::to_string(jtagProtocolId) << " Frequency Divisor must be greater than 0 and less than or equal to 512: " << divisorAttr->Value().c_str();
			});
			divisor = (divisor == 0) ? 1 : 512;
		}
		openIPCError = DetermineProtocolPrescaleAndDivisor(divisor, jtag_prescaler, jtag_divisor);

		PluginNotifications::GetInstance().Notify(
			deviceID,
			OpenIPC_Error_No_Error,
			PPI_infoNotification,
			[&](std::basic_ostream<char>& stream)
		{
			stream << "Chain " << std::to_string(jtagProtocolId) << ": (Adjusted) JTAG Frequency Divisor: " << std::to_string(divisor);
		});
	}

	// be sure not to add a duplicate jtag protocol
	if (!foundJtagId) {
		jtagSettings.JtagChainProtocolId = jtagProtocolId;
		jtagSettings.JtagPrescaler = jtag_prescaler;
		jtagSettings.JtagDivisor = jtag_divisor;
		auto jtagInterface = std::make_shared<InterfaceInstanceJtagASD>(refid | cnt++, PPI_interfaceTypeJtag, jtagCapabilities, jtagSettings);
		jtagInterface->settings.Set(std::string("Jtag.TclkRate"), finalJtagSpeed);
		clockCycleTimeinUs = 1000000.0 / (std::stol(finalJtagSpeed));
		AddJtagInterface(jtagInterface);
	}
	else {
		PrintNotification(PPI_warningNotification, "Duplicated jtag protocol!");
		openIPCError = Initialize_Error;
	}

	return openIPCError;
}

OpenIPC_Error ProbeInstanceASD::ProbeInitializeI2C(DataElement* probeChild, int &cnt)
{
	OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
	DataAttribute* busSelectAttribute = nullptr;
	DataAttribute* speedAttribute = nullptr;
	std::stringstream msgStream;
	PPI_InterfaceI2CCapabilities caps;
	I2cConfiguration config = {};
	std::shared_ptr<InterfaceInstanceI2cASD> i2cInterface = nullptr;
	int busSpeed = 0; //TBD: caps.minimumtimeGraularityInMicroSeconds vs busSpeed

					  // default i2c interface required parameter
	if ((speedAttribute = probeChild->FindAttribute("Bus_Speed")) != nullptr) {
		busSpeed = atoi(speedAttribute->Value().c_str());
		if (IsParamInRange(I2C_MIN_BUS_SPEED, I2C_MAX_BUS_SPEED, busSpeed)) {
			caps.minimumtimeGraularityInMicroSeconds = busSpeed;
		}
		else {
			msgStream << "ASD I2C Bus_Speed: " << busSpeed << " is invalid. Bus_Speed must be between ";
			msgStream << I2C_MIN_BUS_SPEED << " and " << I2C_MAX_BUS_SPEED;
			PrintNotification(PPI_errorNotification, msgStream.str());
			openIPCError = OpenIPC_Error_Null_Pointer;
			goto end;
		}
	}
	i2cInterface = std::make_shared<InterfaceInstanceI2cASD>(refid | cnt++, PPI_interfaceTypeI2c, caps);

	// custom parameter(s)
	if ((busSelectAttribute = probeChild->FindAttribute("Bus_Select")) != nullptr) {
		config.busSelect = atoi(busSelectAttribute->Value().c_str());
		if (!IsParamInRange(I2C_MIN_BUS_SELECT, I2C_MAX_BUS_SELECT, config.busSelect)) {
			msgStream << "ASD I2C Bus_Select: " << config.busSelect << " is invalid. Bus_Select must be between ";
			msgStream << I2C_MIN_BUS_SELECT << " and " << I2C_MAX_BUS_SELECT;
			PrintNotification(PPI_errorNotification, msgStream.str());
			goto end;
		}
	}
	config.busSpeed = busSpeed;
	msgStream << "ASD I2C Bus_Select: " << config.busSelect << " ";
	msgStream << "Bus_Speed: " << busSpeed;
	PrintNotification(PPI_infoNotification, msgStream.str());
	i2cInterface->SetConfig(config);
	AddI2CInterface(i2cInterface);
end:
	return openIPCError;
}

OpenIPC_Error ProbeInstanceASD::ProbeInitializePins(std::shared_ptr<InterfaceInstanceRemotePins> &pinInterface, DataElement* probeChild, int &cnt)
{
	OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
	PPI_InterfacePinsCapabilities pinCapabilities;
	DataElement* elementChild;
	if (probeChild == nullptr)
		return OpenIPC_Error_Null_Pointer;
	elementChild = probeChild->FirstChild();
	while (elementChild)
	{
		DataAttribute* OpenIPC_Value = nullptr;
		DataAttribute* Protocol_Value = nullptr;
		DataAttribute* ReadWrite = nullptr;
		uint32_t readWrite = 0;
		uint32_t ocValue = 0;
		uint32_t pValue = 0;
		if (elementChild->Name() == "Pin")
		{
			if ((OpenIPC_Value = elementChild->FindAttribute("OpenIPC_Value")) == nullptr ||
				(Protocol_Value = elementChild->FindAttribute("Protocol_Value")) == nullptr ||
				(ReadWrite = elementChild->FindAttribute("ReadWrite")) == nullptr) {
				elementChild = elementChild->NextSibling();
				continue;
			}
			if (!pinInterface)
			{
				std::shared_ptr<PPI_InterfaceTriggerResponseBreakAll> breakAll(new PPI_InterfaceTriggerResponseBreakAll((uint32_t)(1), false, PPI_PINS_Prdy_N, false, PPI_PINS_Preq_N, false, this_shared));
				std::shared_ptr<PPI_InterfaceTriggerResponseResetBreak> resetBreak(new PPI_InterfaceTriggerResponseResetBreak((uint32_t)(2), true, PPI_PINS_SystemReset, false, PPI_PINS_Preq_N, false, this_shared));
				pinInterface = std::make_shared<InterfaceInstanceRemotePins>(refid | cnt++, PPI_interfaceTypePins, pinCapabilities);
				if (!pinInterface) {
					openIPCError = OpenIPC_Error_Null_Pointer;
					goto end;
				}
				pinInterface->settings.Set(std::string("Pins.PrdyBankAAssert"), std::string("true"));
				pinInterface->settings.Set(std::string("Pins.PrdyBankBAssert"), std::string("false"));
				pinInterface->settings.Set(std::string("Pins.PrdyBankCAssert"), std::string("false"));
				pinInterface->settings.Set(std::string("Pins.PrdyBankDAssert"), std::string("false"));
				pinInterface->AddTriggerResponse(breakAll);
				pinInterface->AddTriggerResponse(resetBreak);
			}

			if (ReadWrite->Value() == "Read")
				readWrite = InterfaceInstanceRemotePins::READ;
			else if (ReadWrite->Value() == "Write")
				readWrite = InterfaceInstanceRemotePins::WRITE;
			else
				readWrite = InterfaceInstanceRemotePins::READWRITE;
			ocValue = strtoul(OpenIPC_Value->Value().c_str(), NULL, 0);
			pValue = strtoul(Protocol_Value->Value().c_str(), NULL, 0);
			pinInterface->AddPin(ocValue, pValue, readWrite);
		}
		elementChild = elementChild->NextSibling();
	}
end:
	return openIPCError;
}

std::string ProbeInstanceASD::CheckAttributeValue(DataElement* probeChild, std::string attribName, std::string defaultValue)
{
	DataAttribute* temp = nullptr;
	if (probeChild == nullptr || (temp = probeChild->FindAttribute(attribName)) == nullptr)
	{
		return defaultValue;
	}
	return temp->Value();
}

OpenIPC_Error ProbeInstanceASD::ProbeInitializeConnectionParams(DataElement* probeChild, int &cnt)
{
	OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
	DataAttribute* securityProtocolAttribute = nullptr;
	DataAttribute* serverCertificateAttribute = nullptr;
	DataAttribute* serverCertificateVerificationAttribute = nullptr;
	DataAttribute* passphraseAttribute = nullptr;
	DataElement* localChild;
	std::string securityProtocol;
	std::stringstream msgStream;

	if (probeChild == nullptr) {
		openIPCError = OpenIPC_Error_Null_Pointer;
		goto end;
	}

	connectionParameters->transportParameters = std::make_shared<TransportParameters>();
	connectionParameters->protocolParameters = std::make_shared<ProtocolParameters>();
	connectionParameters->connectionInitialized = false;

	localChild = probeChild->FirstChild();
	connectionParameters->transportParameters->ipAddress = CheckAttributeValue(probeChild, "ipAddress", DEFAULT_IP_ADDR).c_str();
	connectionParameters->transportParameters->ipPort = CheckAttributeValue(probeChild, "portNum", DEFAULT_PORT).c_str();

	connectionParameters->protocolParameters->timeout = atoi(CheckAttributeValue(probeChild, "timeout", WAIT_TIME).c_str());
	if (connectionParameters->protocolParameters->timeout == 0)
		connectionParameters->protocolParameters->timeout = atoi(WAIT_TIME);

	connectionParameters->protocolParameters->payloadSize = atoi(CheckAttributeValue(probeChild, "payloadSize", std::to_string(DEFAULT_PAYLOAD_SIZE)).c_str());
	if (connectionParameters->protocolParameters->payloadSize == 0)
		connectionParameters->protocolParameters->payloadSize = DEFAULT_PAYLOAD_SIZE;

	connectionParameters->protocolParameters->maxNumBuffers = atoi(CheckAttributeValue(probeChild, "maxNumBuffers", std::to_string(DEFAULT_NUMBER_BUFFERS)).c_str());
	if (connectionParameters->protocolParameters->maxNumBuffers == 0)
		connectionParameters->protocolParameters->maxNumBuffers = DEFAULT_NUMBER_BUFFERS;

	if (!IsParamInRange(1, DEFAULT_PAYLOAD_SIZE, connectionParameters->protocolParameters->payloadSize)) {
		msgStream << "payloadSize: " << connectionParameters->protocolParameters->payloadSize <<
			" must be between 1 and " << DEFAULT_PAYLOAD_SIZE <<
			", resetting to default value " << DEFAULT_PAYLOAD_SIZE << "\n";
		connectionParameters->protocolParameters->payloadSize = DEFAULT_PAYLOAD_SIZE;
		PrintNotification(PPI_warningNotification, msgStream.str());
		msgStream.str("");
	}

	if (!IsParamInRange(1, DEFAULT_NUMBER_BUFFERS, connectionParameters->protocolParameters->maxNumBuffers)) {
		msgStream << "maxBufferNum: " << connectionParameters->protocolParameters->maxNumBuffers <<
			" must be between 1 and " << DEFAULT_NUMBER_BUFFERS <<
			", resetting to default value " << DEFAULT_NUMBER_BUFFERS << "\n";
		connectionParameters->protocolParameters->maxNumBuffers = DEFAULT_NUMBER_BUFFERS;
		PrintNotification(PPI_warningNotification, msgStream.str());
		msgStream.str("");
	}

	connectionParameters->transportParameters->socketType = SSL_SOCKET;
	if ((securityProtocolAttribute = probeChild->FindAttribute("securityProtocol")) != nullptr) {
		securityProtocol = securityProtocolAttribute->Value();
		transform(securityProtocol.begin(), securityProtocol.end(), securityProtocol.begin(), ::tolower);
		if (IsParamValid({ "tls_auth", "none" }, securityProtocol)) {
			if (securityProtocol == "none") {
				connectionParameters->transportParameters->socketType = BASIC_SOCKET;
			}
		}
		else {
			PluginNotifications::GetInstance().Notify(
				deviceID,
				OpenIPC_Error_No_Error,
				PPI_errorNotification,
				[&](std::basic_ostream<char>& stream)
			{
				stream << "Parameter securityProtocol=\"" << securityProtocol << "\"" << " is invalid, please use \"tls_auth\" or \"none\" \n";
			});
			openIPCError = Initialize_Error;
			goto end;
		}
	}
	if ((serverCertificateAttribute = probeChild->FindAttribute("serverCertificate")) != nullptr) {
		connectionParameters->transportParameters->targetChainFile = serverCertificateAttribute->Value();
	}
	if ((passphraseAttribute = probeChild->FindAttribute("passphrase")) != nullptr) {
		connectionParameters->transportParameters->passphrase = passphraseAttribute->Value();
	}
	if ((serverCertificateVerificationAttribute = probeChild->FindAttribute("serverCertificateVerification")) != nullptr) {
		connectionParameters->transportParameters->chainFileVerification = serverCertificateVerificationAttribute->Value();
		transform(connectionParameters->transportParameters->chainFileVerification.begin(), connectionParameters->transportParameters->chainFileVerification.end(), connectionParameters->transportParameters->chainFileVerification.begin(), ::tolower);
		if (!IsParamValid({ "abort", "warning" }, connectionParameters->transportParameters->chainFileVerification)) {
			PluginNotifications::GetInstance().Notify(
				deviceID,
				OpenIPC_Error_No_Error,
				PPI_errorNotification,
				[&](std::basic_ostream<char>& stream)
			{
				stream << "Parameter serverCertificateVerification=\"" << connectionParameters->transportParameters->chainFileVerification << "\"" << " is invalid, please use \"abort\" or \"warning\" \n";
			});
			openIPCError = Initialize_Error;
			goto end;
		}
	}

	while (localChild != nullptr)
	{
		const std::string& elementName = probeChild->Name();
		if (elementName == "Interface.JtagDevHandle")
		{
			openIPCError = ProbeInitializeJtagDevHandle(probeChild, cnt, connectionParameters);
			if (!OpenIPC_PASS(openIPCError))
				goto end;
		}
		localChild = localChild->NextSibling();
	}
	connectionParameters->connectionInitialized = true;

end:
	return  openIPCError;
}

void ProbeInstanceASD::PrintNotification(uint32_t type, std::string msg)
{
	PluginNotifications::GetInstance().Notify(
		this->deviceID,
		OpenIPC_Error_No_Error,
		type,
		[&](std::basic_ostream<char>& stream)
	{
		stream << msg;
	});
}

bool ProbeInstanceASD::IsParamValid(std::vector<std::string> param_list, std::string param)
{
	for (auto const& value : param_list) {
		if (value == param)
			return true;
	}
	return false;
}

bool ProbeInstanceASD::IsParamInRange(int min, int max, int param)
{
	return (param >= min && param <= max);
}


OpenIPC_Error ProbeInstanceASD::DetermineProtocolPrescaleAndDivisor(uint16_t& value, uint8_t& pre, uint8_t& div) {
	uint16_t temp = 0;

	if (value == 0 || value > 512)
		return OpenIPC_Error_Invalid_Argument;

	// Find pre value by checking if divisible by 1, 2, 4, 8
	for (pre = 1; pre <= 8; pre *= 2) {
		if (value / pre > 0) {
			// Find div value
			temp = value / pre;
			if (temp <= 64) {
				div = (uint8_t)temp;
				// If remainder is present, then exact value requested cannot be sent via this protocol.
				// round up div so that final JTAG frequency is slower rather than faster than requested.
				if (value % pre > 0)
					div++;
				value = div * pre;
				break;
			}
		}
	}

	// Per protocol, if div == 64, it is sent as 0
	if (div == 64)
		div = 0;

	// Map prescaler to protocol values
	switch (pre) {
	case 1:
		pre = 0;
		break;
	case 2:
		pre = 1;
		break;
	case 4:
		pre = 2;
		break;
	case 8:
		pre = 3;
		break;
	}

	return OpenIPC_Error_No_Error;
}
