/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////

#include "Frequency.h"
#include <cmath>

namespace
{
    const std::regex FREQUENCY_RE{ R"re((\d+(?:\.\d+)?)([GMk]?Hz)?)re" };
}

Frequency::Frequency():
    _hertz(0)
{
}

Frequency::Frequency(uint64_t hertz):
    _hertz(hertz)
{
}

Frequency::Frequency(std::string frequency)
{
    std::smatch match;
    std::regex_search(frequency, match, FREQUENCY_RE);

    if (match.empty())
    {
        throw std::invalid_argument("frequency passed in is in an incorrect format.");
    }

    const double freq = std::stod(match.str(1));
    const std::string unit = match[2].matched ? match.str(2) : "Hz";

    if (unit == "Hz")
    {
        _hertz = std::llround(freq);
    }
    else if (unit == "kHz")
    {
        _hertz = std::llround(freq * 1000);
    }
    else if (unit == "MHz")
    {
        _hertz = std::llround(freq * 1000 * 1000);
    }
    else if (unit == "GHz")
    {
        _hertz = std::llround(freq * 1000 * 1000 * 1000);
    }
    else
    {
        _hertz = std::llround(freq);
    }
}

uint64_t Frequency::GetHertz() const
{
    return _hertz;
}

bool Frequency::IsFrequency(std::string frequency)
{
    std::smatch match;
    std::regex_search(frequency, match, FREQUENCY_RE);
    return !match.empty();
}

bool operator==(const Frequency& lhs, const Frequency& rhs)
{
    return lhs._hertz == rhs._hertz;
}

bool operator!=(const Frequency& lhs, const Frequency& rhs)
{
    return !(lhs == rhs);
}

bool operator<(const Frequency& lhs, const Frequency& rhs)
{
    return lhs._hertz < rhs._hertz;
}

bool operator<=(const Frequency& lhs, const Frequency& rhs)
{
    return !(rhs < lhs);
}

bool operator>(const Frequency& lhs, const Frequency& rhs)
{
    return rhs < lhs;
}

bool operator>=(const Frequency& lhs, const Frequency& rhs)
{
    return !(lhs < rhs);
}

std::ostream& operator<<(std::ostream& os, const Frequency& obj)
{
    return os << obj._hertz << "Hz";
}
