/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#include "HardwareBreakpointProtect.h"

#ifdef HOST_WINDOWS

void HardwareBreakpointProtect::_SetHardwareBreakpoint()
{
    std::thread thread(&HardwareBreakpointProtect::_InstallHardwareBreakpointThread, this);
    thread.join();
}

void HardwareBreakpointProtect::_ClearHardwareBreakpoint()
{
    if (!_isInstalled)
    {
        return;
    }
    std::thread thread(&HardwareBreakpointProtect::_UninstallHardwareBreakpointThread, this);
    thread.join();
}

void HardwareBreakpointProtect::_InstallHardwareBreakpointThread()
{
    std::lock_guard<std::mutex> lock(_lock);
    SuspendThread(_targetThread); // must suspend the thread to get it's context

    CONTEXT context{};
    context.ContextFlags = CONTEXT_DEBUG_REGISTERS;
    GetThreadContext(_targetThread, &context);

    const bool dr0InUse = (context.Dr7 & (1 << 0)) != 0;
    const bool dr1InUse = (context.Dr7 & (1 << 2)) != 0;
    const bool dr2InUse = (context.Dr7 & (1 << 4)) != 0;
    const bool dr3InUse = (context.Dr7 & (1 << 6)) != 0;

    if (!dr0InUse)
    {
        _registerUsed = 0;
        context.Dr0 = _targetAddress;
    }
    else if (!dr1InUse)
    {
        _registerUsed = 1;
        context.Dr1 = _targetAddress;
    }
    else if (!dr2InUse)
    {
        _registerUsed = 2;
        context.Dr2 = _targetAddress;
    }
    else if (!dr3InUse)
    {
        _registerUsed = 3;
        context.Dr3 = _targetAddress;
    }
    else
    {
        // No available hardware registers. Just don't install it
        ResumeThread(_targetThread);
        return;
    }

    context.Dr6 = 0;
    context.Dr7 = SetBits(context.Dr7, 16 + _registerUsed * 4, 2, static_cast<int>(_type));
    context.Dr7 = SetBits(context.Dr7, 18 + _registerUsed * 4, 2, static_cast<int>(_size));
    context.Dr7 = SetBits(context.Dr7, _registerUsed * 2, 1, 1);

    context.ContextFlags = CONTEXT_DEBUG_REGISTERS;
    SetThreadContext(_targetThread, &context);
    _isInstalled = true;
    ResumeThread(_targetThread);
}

void HardwareBreakpointProtect::_UninstallHardwareBreakpointThread()
{
    std::lock_guard<std::mutex> lock(_lock);
    if (!_isInstalled)
    {
        return;
    }

    SuspendThread(_targetThread); // must suspend the thread to get it's context

    CONTEXT context{};
    context.ContextFlags = CONTEXT_DEBUG_REGISTERS;
    GetThreadContext(_targetThread, &context);

    int d7EnableBitOffset;
    if (_registerUsed == 0)
    {
        d7EnableBitOffset = 0;
        context.Dr0 = 0;
    }
    else if (_registerUsed == 0)
    {
        d7EnableBitOffset = 2;
        context.Dr1 = 0;
    }
    else if (_registerUsed == 0)
    {
        d7EnableBitOffset = 4;
        context.Dr2 = 0;
    }
    else
    {
        d7EnableBitOffset = 6;
        context.Dr3 = 0;
    }

    // clear the enable bit
    context.Dr7 &= ~(1 << d7EnableBitOffset);

    context.ContextFlags = CONTEXT_DEBUG_REGISTERS;
    SetThreadContext(_targetThread, &context);
    _isInstalled = false;
    ResumeThread(_targetThread);
}

#endif
