/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#include "stdafx.h"

#include "MemoryOperations.h"
#include "ManipulatePath.h"

#include "PushNewOverride.h"
    #include <boost/filesystem.hpp>
#include "PopNewOverride.h"

namespace CommonUtils
{
    OpenIPC_Error SaveMemory(
        const std::string& fileName,
        bool overwriteFile,
        std::string fileType,
        std::string saveOptions,
        uint64_t offset,
        uint32_t byteCount,
        const std::function<OpenIPC_Error(uint64_t offset, uint32_t byteCount, uint8_t* buffer, uint32_t& bytesRead)>& readMemCallback,
        uint32_t& numBytesRead)
    {
        OpenIPC_Error error = OpenIPC_Error_No_Error;

        numBytesRead = 0;
        std::transform(fileType.begin(), fileType.end(), fileType.begin(), [](unsigned char c) {return static_cast<char>(::tolower(c)); });
        std::transform(saveOptions.begin(), saveOptions.end(), saveOptions.begin(), [](unsigned char c) {return static_cast<char>(::tolower(c)); });

        if (fileType == "binary")
        {
            boost::filesystem::path filePath(fileName);
            error = SanitizePath(filePath);
            if (!OpenIPC_PASS(error))
            {
                POST_ERROR_MESSAGE(error, "Failed to sanitize the file path: '" << fileName << "'");
            }

            if (OpenIPC_PASS(error) && boost::filesystem::exists(filePath) && !overwriteFile)
            {
                error = OpenIPC_Error_File_Already_Exists;
                POST_ERROR_MESSAGE(error, "Cannot save memory to file '" << filePath.string() << "' because the file already exists and 'overwriteFile' was false");
            }

            boost::filesystem::ofstream file;
            if (OpenIPC_PASS(error))
            {
                file.open(filePath, std::ios::binary | std::ios::trunc);
                if (!file)
                {
                    error = OpenIPC_Error_Cannot_Open_File;
                    POST_ERROR_MESSAGE(error, "Failed to open the file '" << filePath.string() << "' for writing");
                }
            }

            if (OpenIPC_PASS(error))
            {
                //To avoid loading the entire memory contents into the host system's memory, we
                //read in chunks of maximum size 64MB
                std::vector<uint8_t> buffer(std::min<uint32_t>(byteCount, 64 * 1024 * 1024));

                while (OpenIPC_PASS(error) && (byteCount != 0))
                {
                    uint32_t chunkSize = std::min(byteCount, static_cast<uint32_t>(buffer.size()));
                    uint32_t temp = 0;
                    error = readMemCallback(offset + numBytesRead, chunkSize, buffer.data(), temp);

                    byteCount -= temp;
                    numBytesRead += temp;

                    if (temp != 0)
                    {
                        file.write(reinterpret_cast<const char*>(buffer.data()), temp);
                    }
                }
            }
        }
        else
        {
            error = OpenIPC_Error_Not_Supported;
            POST_ERROR_MESSAGE(error, "Saving memory to file type '" << fileType << "' is not supported");
        }

        return error;
    }

    OpenIPC_Error LoadMemory(
        const std::string& fileName,
        std::string fileType,
        std::string loadOptions,
        uint64_t offset,
        uint32_t byteCount,
        const std::function<OpenIPC_Error(uint64_t offset, uint32_t byteCount, uint8_t* buffer, uint32_t& bytesWritten)>& writeMemCallback,
        uint32_t& numBytesWritten)
    {
        OpenIPC_Error error = OpenIPC_Error_No_Error;

        numBytesWritten = 0;
        std::transform(fileType.begin(), fileType.end(), fileType.begin(), [](unsigned char c) {return static_cast<char>(::tolower(c)); });
        std::transform(loadOptions.begin(), loadOptions.end(), loadOptions.begin(), [](unsigned char c) {return static_cast<char>(::tolower(c)); });

        if (fileType == "binary")
        {
            boost::filesystem::path filePath(fileName);
            error = SanitizePath(filePath);
            if (!OpenIPC_PASS(error))
            {
                POST_ERROR_MESSAGE(error, "Failed to sanitize the file path: '" << fileName << "'");
            }

            boost::filesystem::ifstream file;
            if (OpenIPC_PASS(error))
            {
                file.open(filePath, std::ios::binary | std::ios::ate);
            }

            if (OpenIPC_PASS(error) && !file)
            {
                error = OpenIPC_Error_Cannot_Open_File;
                POST_ERROR_MESSAGE(error, "Failed to open the file '" << filePath.string() << "' for reading");
            }

            if (OpenIPC_PASS(error))
            {
                std::streamsize fileSize = file.tellg();
                file.seekg(0, file.beg);

                if (fileSize < 0)
                {
                    error = OpenIPC_Error_Internal_Error;
                    POST_ERROR_MESSAGE(error, "Failed to get the size of the file '" << filePath.string() << "'");
                }
                else if (fileSize < byteCount)
                {
                    //We can safely cast because 'fileSize' is less than 'byteCount', which is a uint32_t
                    byteCount = static_cast<uint32_t>(fileSize);
                }
            }

            if (OpenIPC_PASS(error))
            {
                //To avoid loading the entire file contents into the host system's memory,
                //we will write in chunks of maximum size 64MB
                std::vector<uint8_t> buffer(std::min<uint32_t>(byteCount, 64 * 1024 * 1024));

                while (OpenIPC_PASS(error) && (byteCount != 0))
                {
                    uint32_t chunkSize = std::min(byteCount, static_cast<uint32_t>(buffer.size()));
                    file.read(reinterpret_cast<char*>(buffer.data()), chunkSize);
                    uint32_t bytesReadFromFile = static_cast<uint32_t>(file.gcount()); //We can cast because the maximum value is chunkSize (uint32_t)

                    uint32_t temp = 0;
                    error = writeMemCallback(offset + numBytesWritten, bytesReadFromFile, buffer.data(), temp);

                    byteCount -= temp;
                    numBytesWritten += temp;

                    if (OpenIPC_PASS(error) && !file)
                    {
                        //There was an error while reading from the file. Perhaps the file is no longer accessible
                        error = OpenIPC_Error_Internal_Error;
                        POST_ERROR_MESSAGE(error, "Access to the file '" << filePath.string() << "' was lost");
                    }
                }
            }
        }
        else
        {
            error = OpenIPC_Error_Not_Supported;
            POST_ERROR_MESSAGE(error, "Loading memory from file type '" << fileType << "' is not supported");
        }

        return error;
    }
}
