/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
///  @file
///
///  @brief This file implements functions used for loading DLLs and finding
///         the functions they expose.
//////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "ModuleAccess.h"

#include "PushNewOverride.h"
#include <boost/filesystem.hpp>
#include "PopNewOverride.h"

#include <mutex>

#if defined(HOST_WINDOWS)
    #include <codecvt>
#else
    #include <InternalUtils/accessdll.h>
#endif

extern void InitializeBoostFilesystem();

OpenIPC_Error CommonUtils::GetCurrentModulePath(std::string& path)
{
    OpenIPC_Error error = OpenIPC_Error_No_Error;

#ifdef HOST_WINDOWS
    HMODULE moduleHandle = NULL;

    // Note: this will get the path of the DLL that links to this function.
    // This is normally correct.  If the calling code expects the path of
    // the executable, they won't get it for a NULL handle.
    DWORD flags = GET_MODULE_HANDLE_EX_FLAG_FROM_ADDRESS | GET_MODULE_HANDLE_EX_FLAG_UNCHANGED_REFCOUNT;
    BOOL status = ::GetModuleHandleEx(flags, (LPCTSTR)GetCurrentModulePath, &moduleHandle);

    if (status != TRUE)
    {
        error = OpenIPC_Error_Internal_Error;
    }

    if (OpenIPC_PASS(error))
    {
        DWORD bufferSize = 0;
        DWORD numCharsCopied = 0;
        std::vector<wchar_t> buffer;
        while (numCharsCopied == bufferSize)
        {
            bufferSize += 1024;
            buffer.resize(bufferSize, L'\0');
            numCharsCopied = ::GetModuleFileNameW(moduleHandle, buffer.data(), bufferSize);
            if (numCharsCopied == 0)
            {
                //An error occured...
                error = OpenIPC_Error_Internal_Error;
                break;
            }
        }

        if (OpenIPC_PASS(error))
        {
            buffer[numCharsCopied] = L'\0';

            std::wstring_convert<std::codecvt_utf8_utf16<wchar_t>, wchar_t> converter;
            path = converter.to_bytes(buffer.data());
        }
    }
#else
    //On UNIX dladdr on InternalUtils::getcurrentmodulepath would return IPC libraries path instead of OpenIPC.
    Dl_info dlInfo = {};
    dladdr((void *)GetCurrentModulePath, &dlInfo);
    if (dlInfo.dli_fname != NULL)
    {
        path = dlInfo.dli_fname;
    }
#endif

    return error;
}

OpenIPC_Error CommonUtils::LoadModule(
        IN  const std::string&		path,
        OUT MODULE_ACCESS_HANDLE*	moduleHandle)
{
	OpenIPC_Error error = OpenIPC_Error_No_Error;

	if (moduleHandle == nullptr)
	{
		POST_ERROR_MESSAGE(OpenIPC_Error_Null_Pointer, "CommonUtils::LoadModule handle cannot be null");
		error = OpenIPC_Error_Null_Pointer;
	}

    if (OpenIPC_PASS(error))
    {
        InitializeBoostFilesystem();

#ifdef HOST_WINDOWS
        // Tell Windows to search the DLL's path for dependencies the DLL
        // might have (by default, Windows will not search the same folder
        // as the DLL itself for dependencies).

        std::string modulePathString;
        error = GetCurrentModulePath(modulePathString);

        if (OpenIPC_PASS(error))
        {
            boost::filesystem::path modulePath = boost::filesystem::path(modulePathString).parent_path();
            boost::filesystem::path basePath = boost::filesystem::canonical(modulePath.parent_path());

            boost::filesystem::path fullPath(path);

            if (fullPath.is_relative())
            {
                fullPath = boost::filesystem::canonical(basePath / path);
            }

            std::string folder = boost::filesystem::path(path).parent_path().string();
            std::wstring_convert<std::codecvt_utf8_utf16<wchar_t>, wchar_t> converter;

            ::SetDllDirectoryW(converter.from_bytes(folder).c_str());

            *moduleHandle = ::LoadLibraryW(converter.from_bytes(fullPath.string()).c_str());
        }
#else
		*moduleHandle = InternalUtils::loaddll(path);
#endif
	}

	return error;
}

OpenIPC_Error CommonUtils::UnloadModule(
        IN MODULE_ACCESS_HANDLE	moduleHandle)
{
	OpenIPC_Error error = OpenIPC_Error_No_Error;

    if (moduleHandle != 0)
    {
#ifdef HOST_WINDOWS

        if (!::FreeLibrary(static_cast<HMODULE>(moduleHandle)))
        {
			error = OpenIPC_Error_Internal_Error;
			POST_ERROR_MESSAGE(error, "CommonUtils::FreeModule FreeLibrary failed");
        }

#else
        if (InternalUtils::freedll(moduleHandle))
        {
			error = OpenIPC_Error_Internal_Error;
			POST_ERROR_MESSAGE(error, "CommonUtils::FreeModule dlclose failed");
        }
#endif
    }
	else
	{
		error = OpenIPC_Error_Internal_Error;
		POST_ERROR_MESSAGE(error, "CommonUtils::FreeModule handle is not assigned");
	}

    return error;
}

OpenIPC_Error CommonUtils::GetFunctionFromModule(void** functionPointer,
        IN  MODULE_ACCESS_HANDLE	moduleHandle,
        IN  const std::string&		functionName)
{
	OpenIPC_Error error = OpenIPC_Error_No_Error;

	if (functionPointer == nullptr)
	{
		error = OpenIPC_Error_Null_Pointer;
		POST_ERROR_MESSAGE(error, "CommonUtils::GetFunctionFromModule functionPointer cannot be null");
	}

	if (OpenIPC_PASS(error))
	{
		*functionPointer = nullptr;

		if (moduleHandle == 0)
		{
			error = OpenIPC_Error_Internal_Error;
			POST_ERROR_MESSAGE(error, "CommonUtils::GetFunctionFromModule handle not assigned");
		}

		if (OpenIPC_PASS(error))
		{
			if (functionName.empty())
			{
				error = OpenIPC_Error_Internal_Error;
				POST_ERROR_MESSAGE(error, "CommonUtils::GetFunctionFromModule function name not set");
			}

			if (OpenIPC_PASS(error))
			{
#ifdef HOST_WINDOWS
				*functionPointer = ::GetProcAddress(static_cast<HMODULE>(moduleHandle), functionName.c_str());
#else
				*functionPointer = InternalUtils::getprocedurefromdll(moduleHandle, functionName.c_str());
#endif
				if (*functionPointer == nullptr)
				{
					error = OpenIPC_Error_Internal_Error;
					POST_ERROR_MESSAGE(error, "CommonUtils::GetFunctionFromModule unable to find function: " << functionName);
				}
			}
		}
	}

    return error;
}
