/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#include "stdafx.h"

#include <system_error>

#include "PollingThreadPool.h"

using namespace CommonUtils;

PollingThreadPool::PollingThreadPool() :
    _nextHandle(1),
    _handleToData()
{
}

PollingThreadPool::~PollingThreadPool()
{
    KillAll();
}

PollingThreadHandle PollingThreadPool::Spawn(const PollingThreadDescriptor& descriptor)
{
    PollingThreadHandle handle = _nextHandle++;

    std::unique_ptr<ThreadData>& threadData = _handleToData[handle];
    threadData.reset(new ThreadData(descriptor));
    threadData->thread = std::thread([this, &threadData]
                                     {
                                         _PollAction(*threadData);
                                     });

    return handle;
}

void PollingThreadPool::Kill(PollingThreadHandle handle)
{
    ThreadData* threadData = _LookUpThreadData(handle);
    if (threadData)
    {
        {
            std::unique_lock<std::mutex> lock(threadData->mutex);
            threadData->stop    = true;
            threadData->suspend = false;
        }
        threadData->stopOrSuspendCondition.notify_one();

        try
        {
            threadData->thread.join();
        }
        catch (const std::system_error&)
        {
            // Failed to join
        }

        _handleToData.erase(handle);
    }
}

void PollingThreadPool::KillAll()
{
    std::vector<PollingThreadHandle> handles;
    for (auto& pair : _handleToData)
    {
        handles.push_back(pair.first);
    }

    for (PollingThreadHandle handle : handles)
    {
        Kill(handle);
    }
}

void PollingThreadPool::Suspend(PollingThreadHandle handle)
{
    ThreadData* threadData = _LookUpThreadData(handle);
    if (threadData)
    {
        std::unique_lock<std::mutex> lock(threadData->mutex);
        threadData->suspend = true;
    }
}

void PollingThreadPool::Resume(PollingThreadHandle handle)
{
    ThreadData* threadData = _LookUpThreadData(handle);
    if (threadData)
    {
        {
            std::unique_lock<std::mutex> lock(threadData->mutex);
            threadData->suspend = false;
        }
        threadData->stopOrSuspendCondition.notify_one();
    }
}

PollingThreadPool::ThreadData* PollingThreadPool::_LookUpThreadData(PollingThreadHandle handle)
{
    ThreadData* threadData = nullptr;

    auto it = _handleToData.find(handle);
    if (it != _handleToData.end())
    {
        threadData = it->second.get();
    }

    return threadData;
}

// This function will execute the action every startToStartInterval, if at least endToStartInterval has elapsed and we are not suspended or stopped.
// The purpose of the endToStartInterval is so that we don't run the action back to back in the case where the action takes a long time to execute.
void PollingThreadPool::_PollAction(ThreadData& threadData)
{
    auto nextPollTime = std::chrono::high_resolution_clock::now() + threadData.startToStartInterval;

    while (true)
    {
        {
            // Need to wait of either a stop, or for our next polling interval while not suspended
            std::unique_lock<std::mutex> lock(threadData.mutex);

            // First we wait until we have exceeded our nextPollTime or are stopped
            threadData.stopOrSuspendCondition.wait_until(lock, nextPollTime, [&]
                                                         {
                                                             return threadData.stop;
                                                         });
            // now lets wait until we are stopped or not suspended (if already true, wait won't block)
            threadData.stopOrSuspendCondition.wait(lock, [&]
                                                   {
                                                       return threadData.stop || !threadData.suspend;
                                                   });
            if (threadData.stop)
            {
                return;
            }

        }

        auto lastPollStarted = std::chrono::high_resolution_clock::now();
        threadData.action();

        // TODO: TimeScale the intervals here
        auto startToStartNextPollTime = lastPollStarted + threadData.startToStartInterval;
        auto endToStartNextPollTime   = std::chrono::high_resolution_clock::now() + threadData.endToStartInterval;
        nextPollTime = std::max(startToStartNextPollTime, endToStartNextPollTime);
    }
}
