/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
///  @file
///
///  @brief
//////////////////////////////////////////////////////////////////////////////

#pragma once

#include <string>
#include <unordered_map>
#include <map>

#include "Name.h"

namespace CommonUtils
{
    // A case-insensitive string hash function
    struct CaseInsensitiveStringHash
    {
        size_t operator()(const std::string& key) const
        {
            size_t hash = 2166136261;
            for (const char c : key)
            {
                hash *= 16777619;
                hash ^= static_cast<size_t>(::tolower(c));
            }
            return hash;
        }
    };

    // A case-insensitive name hash function
    struct CaseInsensitiveNameHash
    {
        size_t operator()(Name key) const
        {
            const std::string& string = key.AsString();
            return CaseInsensitiveStringHash()(string);
        }
    };

    // A case-insensitive string compare function
    struct CaseInsensitiveStringEqual
    {
        bool operator()(const std::string& left, const std::string& right) const
        {
            const size_t length = left.length();

            if (length != right.length())
            {
                return false;
            }
            else
            {
                for (size_t i = 0; i < length; ++i)
                {
                    if (::tolower(left[i]) != ::tolower(right[i]))
                    {
                        return false;
                    }
                }
            }

            return true;
        }
    };

    // A case-insensitive name compare function
    struct CaseInsensitiveNameEqual
    {
        bool operator()(Name left, Name right) const
        {
            const std::string& leftString = left.AsString();
            const std::string& rightString = right.AsString();
            return CaseInsensitiveStringEqual()(leftString, rightString);
        }
    };

    // A case-insensitive string compare function
    struct CaseInsensitiveStringLess
    {
        bool operator()(const std::string& left, const std::string& right) const;
    };

    // A case-insensitive name compare function
    struct CaseInsensitiveNameLess
    {
        bool operator()(Name left, Name right) const
        {
            const std::string& leftString = left.AsString();
            const std::string& rightString = right.AsString();
            return CaseInsensitiveStringLess()(leftString, rightString);
        }
    };

    // An unordered map keyed by case-insensitive strings
    template <typename T>
    struct CaseInsensitiveMap
    {
        typedef std::unordered_map<std::string, T, CaseInsensitiveStringHash, CaseInsensitiveStringEqual> Type;
    };

    // An unordered map keyed by case-insensitive names
    template <typename T>
    struct CaseInsensitiveNameMap
    {
        typedef std::unordered_map<Name, T, CaseInsensitiveNameHash, CaseInsensitiveNameEqual> Type;
    };

    // An ordered map keyed by case-insensitive strings
    template <typename T>
    struct CaseInsensitiveOrderedMap
    {
        typedef std::map<std::string, T, CaseInsensitiveStringLess> Type;
    };
}
