/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#pragma once

#include <functional>
#include <stdexcept>

#include <Foundation/Error/Error.h>

class OpenIPC_Error_Exception : public std::exception
{
    OpenIPC_Error _errorCode;
public:
    OpenIPC_Error_Exception(OpenIPC_Error errorCode) :
        std::exception(),
             _errorCode(errorCode)
    {

    }

    OpenIPC_Error GetErrorCode() const { return _errorCode; }
};

template<typename Callable>
OpenIPC_Error CallAndTranslateError(Callable&& function) noexcept
{
    try
    {
        function();
        return OpenIPC_Error_No_Error;
    }
    catch (const OpenIPC_Error_Exception& ex)
    {
        return ex.GetErrorCode();
    }
    catch (std::bad_alloc const&)
    {
        POST_ERROR_MESSAGE(OpenIPC_Error_Internal_Error, "Out of memory");
        return OpenIPC_Error_Internal_Error;
    }
    catch (std::exception const& ex)
    {
        POST_ERROR_MESSAGE(OpenIPC_Error_Internal_Error, ex.what());
        return OpenIPC_Error_Internal_Error;
    }
    catch (...)
    {
        POST_ERROR_MESSAGE(OpenIPC_Error_Internal_Error, "Unexpected exception.");
        return OpenIPC_Error_Internal_Error;
    }
}

inline void ThrowIfError(OpenIPC_Error errorCode)
{
    if (!OpenIPC_PASS(errorCode))
    {
        throw OpenIPC_Error_Exception(errorCode);
    }
}

template<class T>
class UniquePtrToOutputPointer
{
    typedef typename T::pointer TP;
public:
    UniquePtrToOutputPointer(T& Ptr) :
        _safePtr{Ptr} {}
    ~UniquePtrToOutputPointer() { _safePtr.reset(_rawPointer); }

    UniquePtrToOutputPointer(const UniquePtrToOutputPointer& other) = delete;
    UniquePtrToOutputPointer& operator=(const UniquePtrToOutputPointer& other) = delete;

    UniquePtrToOutputPointer(UniquePtrToOutputPointer&& other) noexcept = default;
    UniquePtrToOutputPointer& operator=(UniquePtrToOutputPointer&& other) noexcept = default;

    operator TP*() { return &_rawPointer; }

private:
    T& _safePtr;
    TP _rawPointer{};
};

// Allows a unique_ptr<T> to be passed to a function that takes T** as an output pointer
template<class T>
UniquePtrToOutputPointer<T> AsOutput(T& smartPointer) noexcept
{
    return UniquePtrToOutputPointer<T>(smartPointer);
}

class OpenIPC_UniqueHandle
{
    OpenIPC_Error (* _deleter)(OpenIPC_Handle);
    OpenIPC_Handle _handle;
public:
    OpenIPC_UniqueHandle(OpenIPC_Error (*deleter)(OpenIPC_Handle))
        : _deleter(deleter),
        _handle(OpenIPC_INVALID_HANDLE)
    {

    }

    OpenIPC_UniqueHandle(const OpenIPC_UniqueHandle& other) = delete;
    OpenIPC_UniqueHandle& operator=(const OpenIPC_UniqueHandle& other) = delete;

    OpenIPC_UniqueHandle(OpenIPC_UniqueHandle&& other) noexcept
        : _deleter(other._deleter),
        _handle(other._handle)
    {
        other._handle = OpenIPC_INVALID_HANDLE;
    }

    OpenIPC_UniqueHandle& operator=(OpenIPC_UniqueHandle&& other) noexcept
    {
        if (this == &other)
        {
            return *this;
        }
        _deleter = other._deleter;
        _handle  = other._handle;
        other._handle = OpenIPC_INVALID_HANDLE;
        return *this;
    }

    ~OpenIPC_UniqueHandle()
    {
        if (_handle != OpenIPC_INVALID_HANDLE && _deleter != nullptr)
        {
            _deleter(_handle);
            _handle = OpenIPC_INVALID_HANDLE;
        }
    }

    OpenIPC_Handle& Get()
    {
        return _handle;
    }

};

class RollbackGuard
{
    bool _committed;
    std::function<void()> _rollback;

public:
    RollbackGuard(std::function<void()>&&rollback)
        : _committed(false),
        _rollback(std::move(rollback))
    {
    }

    RollbackGuard(const RollbackGuard& other)     = delete;
    RollbackGuard(RollbackGuard&& other) noexcept = delete;
    RollbackGuard& operator=(const RollbackGuard& other)     = delete;
    RollbackGuard& operator=(RollbackGuard&& other) noexcept = delete;

    ~RollbackGuard()
    {
        if (!_committed)
        {
            _rollback();
        }
    }

    void Commit() noexcept { _committed = true; }

};

class OpenIpcResource
{
    std::function<OpenIPC_Error()> _release;

public:
    OpenIpcResource(OpenIPC_Error aquireResult, std::function<OpenIPC_Error()>&&release)
        : _release(std::move(release))
    {
        ThrowIfError(aquireResult);
    }

    OpenIpcResource(const OpenIpcResource& other) = delete;
    OpenIpcResource(OpenIpcResource&& other) noexcept = delete;
    OpenIpcResource& operator=(const OpenIpcResource& other) = delete;
    OpenIpcResource& operator=(RollbackGuard&& other) noexcept = delete;

    ~OpenIpcResource()
    {
        auto releaseResult = _release();
        if (!std::uncaught_exception())
        {
            ThrowIfError(releaseResult);
        }
    }
};
