/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#pragma once

#include <cstdint>
#include <string>

namespace CommonUtils
{

///
/// @brief A reference to a static string in a string pool.
///
/// @note The underlying string pool is static and may differ from library to
///       library.  For this reason, names should not cross shared library
///       boundaries as they may resolve to different or invalid strings.
class Name
{
public:

    ///
    /// @brief A blank name.
    static const Name Unnamed;

    ///
    /// @brief An index into a static string pool.
    typedef uint32_t Index;

    ///
    /// @brief Constructs an empty name.
    Name();

    ///
    /// @brief Constructs a name from a NULL-terminated string.
    ///
    /// @param name
    ///     The string value of the name.
    Name(const char* name);

    ///
    /// @brief Constructs a name from a string.
    ///
    /// @param name
    ///     The string value of the name.
    Name(const std::string& name);

    ///
    /// @brief Returns a reference to the corresponding string of the name.
    const std::string& AsString() const;

    ///
    /// @brief Returns a pointer to the corresponding NULL-terminated string
    ///        of the name.
    const char* GetData() const;

    ///
    /// @brief Returns the index into the static string pool of the name.
    Index GetIndex() const;

    ///
    /// @brief Returns whether the name is empty.
    bool IsEmpty() const;

    bool operator<(Name name) const;

    bool operator==(Name name) const;
    bool operator!=(Name name) const;

private:
    static Index _LookUpIndex(const std::string& string);

    Index _index;
};

}

namespace std
{

template <>
struct hash<CommonUtils::Name>
{
    std::size_t operator()(const CommonUtils::Name& name) const;
};

}
