/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////

#pragma once
#include <functional>
#include <chrono>
#include <thread>
#include <condition_variable>

class TimeScalerBase
{
public:
    TimeScalerBase() = default;
    virtual ~TimeScalerBase() = default;
    TimeScalerBase(const TimeScalerBase& other)     = default;
    TimeScalerBase(TimeScalerBase&& other) noexcept = default;
    TimeScalerBase& operator=(const TimeScalerBase& other)     = default;
    TimeScalerBase& operator=(TimeScalerBase&& other) noexcept = default;

private:
    template <typename T>
    T ScaleHelper(std::true_type /*isSignedTag*/, const T& value) const
    {
        auto factor = GetFactorPercentage();
        auto numerator = value * factor;
        // If we are overflowing, just use the value as is
        if (value > 0 && numerator < value)
        {
            return value;
        }
        if (value < 0 && numerator > value)
        {
            return value;
        }
        return  numerator / 100;
    }
    template <typename T>
    T ScaleHelper(std::false_type /*isSignedTag*/, const T& value) const
    {
        auto factor = GetFactorPercentage();
        auto numerator = value * factor;
        // If we are overflowing, just use the value as is
        if (numerator < value)
        {
            return value;
        }
        return numerator / 100;
    }
public:

    template<typename T>
    T Scale(const T& value) const
    {
        return ScaleHelper<T>(typename std::is_signed<T>::type(), value);
    }

    bool DoUntil(const std::function<bool()>& action, std::chrono::nanoseconds interval, std::chrono::nanoseconds duration) const
    {
        const auto scaledInterval = Scale(interval);
        const auto scaledDuration = Scale(duration);
        const auto start = std::chrono::high_resolution_clock::now();
        const auto end   = start + scaledDuration;

        if (action())
        {
            return true;
        }

        while (std::chrono::high_resolution_clock::now() < end)
        {
            if (interval != std::chrono::nanoseconds(0))
            {
                std::this_thread::sleep_for(scaledInterval);
            }
            if (action())
            {
                return true;
            }
        }
        return false;
    }

    bool DoUntil(const std::function<bool()>& action, std::chrono::nanoseconds duration) const
    {
        return DoUntil(action, std::chrono::nanoseconds(0), duration);
    }

    auto UntilTimeout(std::chrono::nanoseconds duration) const
    {
        return TimeoutActionBuilder(duration, *this);
    }

    bool WaitFor(std::condition_variable& cv, std::unique_lock<std::mutex>& lock, std::chrono::nanoseconds duration, std::function<bool()> predicate) const
    {
        return cv.wait_for(lock, Scale(duration), std::move(predicate));
    }

    void SleepFor(std::chrono::nanoseconds duration) const
    {
        std::this_thread::sleep_for(Scale(duration));
    }

private:
    class TimeoutActionBuilder
    {
        std::chrono::nanoseconds _duration;
        std::chrono::nanoseconds _interval;
        const TimeScalerBase& _base;
    public:
        TimeoutActionBuilder(std::chrono::nanoseconds duration, const TimeScalerBase& base)
            : _duration(duration),
            _interval(0),
            _base(base)
        {
        }

        TimeoutActionBuilder& Every(std::chrono::nanoseconds interval)
        {
            _interval = interval;
            return *this;
        }

        bool Run(const std::function<bool()>& action) const
        {
            return _base.DoUntil(action, _interval, _duration);
        }

    };

    virtual uint32_t GetFactorPercentage() const = 0;

};
