﻿/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#include "stdafx.h"
#include "ErrorContextImpl.h"

#include <SafeString.h>
#include <Foundation/Logging/OpenIPC_Logger.h>
#include <Foundation/Error/OpenIPC_ErrorContext.h>

#include <sstream>
#include <algorithm>

namespace
{

static std::unique_ptr<OpenIPC_Logger> _logger(OpenIPC_Logger::GetLogger(OpenIPC_LoggingLoggerId_Error));
static std::map<OpenIPC_Error, const char*> _defaultMessages;

}

ErrorImpl::ErrorImpl() :
    _errorCode(OpenIPC_Error_No_Error),
    _message(""),
    _sourceFile(""),
    _sourceLineNumber(0)
{
}

ErrorImpl::ErrorImpl(OpenIPC_Error errorCode, const char* message, const char* sourceFile, unsigned sourceLineNumber) :
    _errorCode(errorCode),
    _message(message ? message : ""),
    _sourceFile(sourceFile),
    _sourceLineNumber(sourceLineNumber)
{
}

OpenIPC_Error ErrorImpl::GetErrorCode() const
{
    return _errorCode;
}

std::string ErrorImpl::GetErrorMessage() const
{
    std::stringstream ss;

    // Get either the default message or the specific message
    if (_message.empty())
    {
        ss << ErrorMessages::GetDefaultMessage(_errorCode);
    }
    else
    {
        ss << _message;
    }

    return ss.str();
}

bool ErrorImpl::HasSourceContext() const
{
    return _sourceFile != "";
}

const std::string& ErrorImpl::GetSourceFile() const
{
    return _sourceFile;
}

unsigned ErrorImpl::GetSourceLineNumber() const
{
    return _sourceLineNumber;
}

void ErrorMessages::SetDefaultMessage(OpenIPC_Error error, const char* message)
{
    _defaultMessages[error] = message;
}

const char* ErrorMessages::GetDefaultMessage(OpenIPC_Error error)
{
    auto it = _defaultMessages.find(error);

    if (it == _defaultMessages.end())
    {
        return "No error message defined";
    }

    return it->second;
}

static boost::thread_specific_ptr<std::vector<ErrorContextImpl*>> _contexts;

ErrorContextImpl::ErrorContextImpl()
{
    if (!_contexts.get())
    {
        _contexts.reset(new std::vector<ErrorContextImpl*>());
    }
    _contexts->push_back(this);
}

ErrorContextImpl::~ErrorContextImpl()
{
    // If Trace32 is connected to the TargSim Probe Plugin,
    // the pop_back() will cause a segmentation fault when exiting on Linux.
    if (_contexts.get())
    {
        _contexts->pop_back();
    }
}

void ErrorContextImpl::PostError(OpenIPC_Error errorCode, const char* message, const char* sourceFile, unsigned sourceLineNumber)
{
    if (errorCode)
    {
        ErrorImpl error(errorCode, message, sourceFile, sourceLineNumber);

        if (_contexts.get())
        {
            for (ErrorContextImpl* context : *_contexts)
            {
                context->AddError(error);
            }
        }

		OpenIPC_LOG_USAGE(OpenIPC_LoggingLoggerId_Error, "PostError", LOG_UINT32_HEX(error.GetErrorCode()) << " " << error.GetErrorMessage() << " (" << error.GetSourceFile() << ":" << LOG_INT64_DEC(error.GetSourceLineNumber()) << ")");

		if (_logger->IsLogEnabled(OpenIPC_LoggingSeverityLevel_Trace))
        {
            OpenIPC_LOG(_logger, OpenIPC_LoggingSeverityLevel_Trace, error.GetErrorMessage() << " (" << error.GetSourceFile() << ":" << LOG_INT64_DEC(error.GetSourceLineNumber()) << ")");
        }
        else if (_logger->IsLogEnabled(OpenIPC_LoggingSeverityLevel_Error))
        {
            OpenIPC_LOG(_logger, OpenIPC_LoggingSeverityLevel_Error, error.GetErrorMessage());
        }
    }
}

std::string ErrorContextImpl::GetErrorMessage() const
{
    std::stringstream ss;

    if (!_errors.empty())
    {
        for (int i = static_cast<int>(_errors.size() - 1); i >= 0; --i)
        {
            ss << _errors[i].GetErrorMessage();
            if (i != 0)
            {
                ss << ".  ";
            }
        }
    }

    return ss.str();
}

void ErrorContextImpl::AddError(const ErrorImpl& error)
{
    _errors.push_back(error);
}

const std::vector<ErrorImpl>& ErrorContextImpl::GetErrors() const
{
    return _errors;
}

ErrorImpl ErrorContextImpl::GetLastError() const
{
    static ErrorImpl noError(OpenIPC_Error_No_Error, "", "", 0);
    if (_errors.empty())
    {
        return noError;
    }
    else
    {
        return _errors.back();
    }
}

OpenIPC_ErrorToken OpenIPC_ErrorEnterContex()
{
	ErrorContextImpl* context = new ErrorContextImpl();

	return reinterpret_cast<OpenIPC_ErrorToken>(context);
}

void OpenIPC_ErrorExitContex(OpenIPC_ErrorToken token)
{
	delete reinterpret_cast<ErrorContextImpl*>(token);
}

void OpenIPC_ErrorGetErrorMessageSize(OpenIPC_ErrorToken token, uint32_t* messageSize)
{
	ErrorContextImpl* context = reinterpret_cast<ErrorContextImpl*>(token);

	std::string internalMessage = context->GetErrorMessage();

	*messageSize = static_cast<uint32_t>(internalMessage.size()) + 1;
}

OpenIPC_Bool OpenIPC_ErrorOccurred(OpenIPC_ErrorToken token, OpenIPC_Error errorCode)
{
    bool errorOccurred = false;
	ErrorContextImpl* context = reinterpret_cast<ErrorContextImpl*>(token);
    std::vector<ErrorImpl> errors = context->GetErrors();
    for (auto it = errors.begin(); it != errors.end() && !errorOccurred; it++)
    {
        if (errorCode == it->GetErrorCode())
        {
            errorOccurred = true;
        }
    }
    return errorOccurred;
}

void OpenIPC_ErrorGetErrorMessage(OpenIPC_ErrorToken token, char* message, uint32_t messageSize)
{
	ErrorContextImpl* context = reinterpret_cast<ErrorContextImpl*>(token);

	const std::string internalMessage = context->GetErrorMessage();
    CommonUtils::SafeStringCopy(message, messageSize, internalMessage);
}

void OpenIPC_ErrorGetLastErrorMessageSize(OpenIPC_ErrorToken token, uint32_t* messageSize)
{
	ErrorContextImpl* context = reinterpret_cast<ErrorContextImpl*>(token);

	const std::string internalMessage = context->GetLastError().GetErrorMessage();

	*messageSize = static_cast<uint32_t>(internalMessage.size()) + 1;
}

void OpenIPC_ErrorGetLastErrorMessage(OpenIPC_ErrorToken token, char* message, uint32_t messageSize)
{
	ErrorContextImpl* context = reinterpret_cast<ErrorContextImpl*>(token);

	const std::string internalMessage = context->GetLastError().GetErrorMessage();
    CommonUtils::SafeStringCopy(message, messageSize, internalMessage);
}

OpenIPC_Error OpenIPC_ErrorGetErrorCode(OpenIPC_ErrorToken token)
{
	ErrorContextImpl* context = reinterpret_cast<ErrorContextImpl*>(token);

	return context->GetLastError().GetErrorCode();
}

void OpenIPC_ErrorPostError(OpenIPC_Error errorCode, const char* message, const char* sourceFile, unsigned sourceLineNumber)
{
	ErrorContextImpl::PostError(errorCode, message, sourceFile, sourceLineNumber);
}

void OpenIPC_ErrorSetDefaultMessage(OpenIPC_Error error, const char* message)
{
	ErrorMessages::SetDefaultMessage(error, message);
}

const char* OpenIPC_ErrorGetDefaultMessage(OpenIPC_Error error)
{
	return ErrorMessages::GetDefaultMessage(error);
}
