##############################################################################
# createdefifnewer
# Started: Oct. 20, 2014
# By: Stephen P. Lepisto
# 
# This python script generates OpenIPC_IndexList.def from IndexList.h by searching for
# functions that start with "IndexList_" in IndexList.h.
# 
# The functionality is triggered only if OpenIPC_IndexList.def does not exist or
# IndexList.h is newer than OpenIPC_IndexList.def.
# 
# Requires Python 2.7 or 3.x.
# 
# In Visual Studio, this is executed in a Pre-Build Event script in the
# OpenIPC_IndexList project.
##############################################################################

import os
import re
import sys

inputFilename = "../../Public_Include/Foundation/IndexList/IndexList.h"      # Source file to get functions from
outputFilename = "IndexList.def"   # File to write functions names to
outputDir = "./"                   # Directory where to store IndexList.def


def writeDefFile(names, defFileName):
    """
    Write the given names to a Windows DLL .def file.

    Arguments:
      names       - List of function names to write to the .def file
      defFileName - Name of the .def file to write

    Returns:
      True if the write was successful; otherwise, returns False.
      The reason for the failure has already been displayed.
    """

    writeSuccessful = True
    try:
        with open(defFileName, "wt") as file:
            file.write(
"""; These functions are implemented in the IndexList_Static library and declared in IndexList.h.
; This definition file is automatically generated from IndexList.h.

""")
            file.write("EXPORTS\n")
            for name in names:
                file.write("%s\n"%name)
    except Exception as e:
        print("Error writing file '{0}': {1}".format(defFileName, e))
        writeSuccessful = False
    return writeSuccessful


def gatherFunctionNames(fileContents):
    """
    Search the given text block for function names that start with 'IndexList_'.
    The text block is assumed to be the contents of a C/C++ header file.

    Arguments:
      fileContents - The contents of a header file with function declarations.

    Returns:
      A list of function names found in the text block.  Only the names are
      provided.  If the list is empty then no functions that start with
      'IndexList_' were found.
    """

    names = []
    pattern = "IndexList_[a-zA-Z0-9]+"
    names = re.findall(pattern, fileContents)
    return names


def loadFile(filename):
    """
    Load the specified file in its entirety.
    Assumed to be a text file that is less than 32k  bytes in size.

    Arguments:
      filename - The name of the file to read

    Returns:
      A string containing up to 32768 characters from the file.
    """

    fileSizeLimit = 32768
    file = None
    fileContents = None
    try:
        with open(filename) as file:
            fileContents = file.read(fileSizeLimit)
    except Exception as e:
        print("Error reading file '{0}': {1}".format(filename, e))
    return fileContents


def isNewer(firstFilename, secondFilename):
    """
    Determine if the first file is newer than the second file.
    
    If the first file exists and the second file does not, this function
    returns 1, indicating the first file is newer.

    Arguments:
      firstFilename  - The name of the first file to check.
      secondFilename - The name of the second file to check.

    Returns:
      1 if the first file is newer than the second file.
      0 if the first file is the same age or older than the second file.
    """

    if not os.path.exists(firstFilename):
        return 0
 
    if not os.path.exists(secondFilename):
        return 1 # The second file does not exist so treat first file as newer

    firstTimestamp = os.path.getmtime(firstFilename)
    secondTimestamp = os.path.getmtime(secondFilename)
    return 1 if firstTimestamp > secondTimestamp else 0

    
def main():
    """
    main entry point to this application.

    Returns:
      0 if successful
      1 if failed (reason for failure has been sent to standard out)
    """

    # Allow the caller to override the output directory
    if len(sys.argv) > 1:
        global outputDir
        outputDir = sys.argv[1]

    creationSuccessful = True
    outputFullPath = outputDir + "/" + outputFilename
    if isNewer(inputFilename, outputFullPath):
        creationSuccessful = False
        fileContents = loadFile(inputFilename)
        if fileContents:
            print("Processing {0} ==> {1}...".format(inputFilename, outputFullPath))
            functionNames = gatherFunctionNames(fileContents)
            creationSuccessful = writeDefFile(functionNames, outputFullPath)

    exit(0 if creationSuccessful else 1)

if __name__ == "__main__":
    main()
