﻿/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////

#include <stdafx.h>
#include "JtagStub.h"

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
//	JtagStubTapRegister members
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
JtagStubTapRegister::JtagStubTapRegister(const char* regname, BitData* initvalue, bool rdonly, bool rdwrsame) :
    _name(regname)
{
    SetAttributes((initvalue != nullptr) ? (uint32_t)initvalue->bitsize : 0, initvalue, rdonly, rdwrsame);
}

void JtagStubTapRegister::SetAttributes(uint32_t length, BitData* initvalue, bool rdonly, bool rdwrsame)
{
    if (initvalue == nullptr)
    {
        _captureValue = BitData_CreateManaged(length);
    }
    else
    {
        _captureValue = BitData_CreateManagedFromBitData(initvalue);
        BitData_Resize(_captureValue, length);
    }

    if (rdonly)
    {
        _updateValue = nullptr;
    }
    else if (rdwrsame)
    {
        _updateValue = _captureValue;
    }
    else
    {
        _updateValue = BitData_CreateManaged(_captureValue->bitsize);
    }
}

void JtagStubTapRegister::GetCapture(BitData* dest, uint32_t& offset)
{
    BitData_Copy(_captureValue, 0, dest, offset, _captureValue->bitsize);
    offset += (uint32_t)_captureValue->bitsize;
}

void JtagStubTapRegister::GetUpdate(BitData* dest, uint32_t& offset)
{
    if (_updateValue != nullptr)
    {
        BitData_Copy(_updateValue, 0, dest, offset, _updateValue->bitsize);
        offset += (uint32_t)_updateValue->bitsize;
    }
}

void JtagStubTapRegister::SetUpdate(const BitData* src, uint32_t& offset)
{
    if (_updateValue != nullptr)
    {
        BitData_Copy(src, offset, _updateValue, 0, _updateValue->bitsize);
        offset += (uint32_t)_updateValue->bitsize;
    }
}

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
//	JtagStubTapController members
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
JtagStubTapController::JtagStubTapController(const char* name, uint32_t irlen, uint32_t ircaptureval, uint32_t tapresetir) :
    _irTapReset(tapresetir & (0xffffffff >> (32 - irlen))),
    _instructionReg(nullptr),
    _tapRegisters(),
    _tapRegistersReadonly(),
    _tapRegisterNames(),
    _name(name)
{
    BitData irCaptureValueBd = BitData_CreateLocalFromBuffer(irlen, 32, &ircaptureval);
    _instructionReg = new JtagStubTapRegister("INSTRUCTION", &irCaptureValueBd, false, false);
    uint32_t bypassCaptureValue   = 0;
    BitData  bypassCaptureValueBd = BitData_CreateLocalFromBuffer(1, 32, &bypassCaptureValue);
    AddInstruction(0xffffffff >> (32 - irlen), nullptr, new JtagStubTapRegister("BYPASS", &bypassCaptureValueBd, true), true);
    TapReset();
}

JtagStubTapController::~JtagStubTapController()
{
    for (auto it = this->_tapRegisters.begin(); it != this->_tapRegisters.end(); ++it)
    {
        delete it->second;
    }
    this->_tapRegisters.clear();
    delete this->_instructionReg;
}

JtagStubTapRegister* JtagStubTapController::GetDr()
{
    uint32_t irValue;
    BitData  irValueBd = BitData_CreateLocalFromBuffer(32, 32, &irValue);
    uint32_t offset    = 0;
    _instructionReg->GetUpdate(&irValueBd, offset);
    uint32_t irReg =  irValue & (0xffffffff >> (32 - _instructionReg->GetLength()));
    return _tapRegisters[irReg];
}

void JtagStubTapController::AddInstruction(uint32_t irValue, const char* instructionName, JtagStubTapRegister* reg, bool readonly)
{
    _tapRegisters[irValue] = reg;

    if (readonly)
    {
        _tapRegistersReadonly[irValue] = true;
    }

    if (instructionName != nullptr)
    {
        _tapRegisterNames[irValue] = instructionName;
    }
}

void JtagStubTapController::TapReset()
{
    BitData  irValueBd = BitData_CreateLocalFromBuffer(32, 32, (uint8_t*)&_irTapReset);
    uint32_t offset    = 0;
    _instructionReg->SetUpdate(&irValueBd, offset);
}

uint32_t JtagStubTapController::GetShiftLength(bool bIrScan)
{
    if (bIrScan)
    {
        return _instructionReg->GetLength();
    }
    else
    {
        JtagStubTapRegister* reg = GetDr();
        // register wasn't defined... not really an error, and definitely not an internal error, but either a gap in the TAP defs, or unintentional client activity.
        // regardless, we'll treate this as a open chain, which is how it would likely look on real Si, if the register didn't exist
        // in the future we should have a opt-in facility that can somehow flag unintended IR accesses, or suspicious shift lengths.
        // assert(reg != NULL);
        return (reg != nullptr) ? reg->GetLength() : (uint32_t)-1;
    }
}

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
//	JtagStubTapChain members
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
JtagStubTapChain::JtagStubTapChain() :
    _tapStateMachine(TapStateMachine::JtagTLR),
    _virtualIr(static_cast<uint64_t>(0)),
    _virtualDr(static_cast<uint64_t>(0)),
    _shiftIrClks(0),
    _shiftDrClks(0)
{
}

void JtagStubTapChain::AddTapController(JtagStubTapController* tapcontroller, uint32_t position)
{
    if (uint32_t(-1) == position)
    {
        _tapControllers.push_back(tapcontroller);
    }
    else
    {
        _tapControllers.insert(_tapControllers.begin() + position, tapcontroller);
    }

    _virtualIr.Resize(GetTotalShiftLength(true));
}

void JtagStubTapChain::RemoveTapControllers(uint32_t position, uint32_t count)
{
    uint32_t numTapControllers = static_cast<uint32_t>(_tapControllers.size());
    if (position == uint32_t(-1))
    {
        if (count == uint32_t(-1))
        {
            position = 0;
        }
        else
        {
            position = numTapControllers - count;
        }
    }

    if (count == uint32_t(-1))
    {
        count = numTapControllers - position;
    }
    else if ((count + position) > numTapControllers)
    {
        count = 0;
    }

    if ((count > 0) && (position < numTapControllers))
    {
        _tapControllers.erase(_tapControllers.begin() + position, _tapControllers.end() + position + count);
        _virtualIr.Resize(GetTotalShiftLength(true));
    }
}

void JtagStubTapChain::TapReset()
{
    for (size_t tapCtlr = 0; tapCtlr < _tapControllers.size(); tapCtlr++)
    {
        _tapControllers[tapCtlr]->TapReset();
    }
    _shiftIrClks = 0;
    _shiftDrClks = 0;
}

void JtagStubTapChain::GotoState(TapStateMachine::JtagStateEncode state, uint32_t count)
{
    unsigned char tmsbits;
    unsigned char tmscount;
    _tapStateMachine.GetTmsCycles(state, tmsbits, tmscount);
    DrivePins(tmscount, &tmsbits, 0, 0);
    if (count > 1)
    {
        TapStateMachine::JtagStateEncode currentState = _tapStateMachine.GetState();
        if ((currentState == TapStateMachine::JtagRTI) || (currentState == TapStateMachine::JtagPauDR) || (currentState == TapStateMachine::JtagPauIR))
        {
            DrivePins(count, 0, 0, 0);
        }
        else if (currentState == TapStateMachine::JtagTLR)
        {
            uint32_t numbytes  = BitsToUint8s(count);
            uint8_t* tmsbuffer = new uint8_t[numbytes];
            memset(tmsbuffer, -1, numbytes);
            DrivePins(count, tmsbuffer, 0, 0);
            delete [] tmsbuffer;
        }
        else
        {
            assert(false);
        }
    }
}

void JtagStubTapChain::Shift(TapStateMachine::JtagStateEncode state, uint32_t count, const uint8_t* writebuffer, uint8_t* readbuffer)
{
    assert((state == TapStateMachine::JtagShfDR) || (state == TapStateMachine::JtagShfIR));

    unsigned char tmsbits;
    unsigned char tmscount;
    _tapStateMachine.GetTmsCycles(state, tmsbits, tmscount);
    DrivePins(tmscount, &tmsbits, 0, 0);

    uint32_t numBytes = BitsToUint8s(count);
    uint8_t* tms = new uint8_t[numBytes];

    memset(tms, 0, numBytes);
    tms[numBytes - 1] |= 1 << ((count - 1) & 0x7);
    DrivePins(count, tms, writebuffer, readbuffer);
    delete[] tms;
}

void JtagStubTapChain::DrivePins(uint32_t count, const uint8_t* tms, const uint8_t* tdi, uint8_t* tdo)
{
    for (uint32_t tck = 0; tck < count; tck++)
    {
        TapStateMachine::JtagStateEncode currentState = _tapStateMachine.GetState();
        // note: the 'shift' (transfering TDI into, and TDO out of, the virtual shift register) actually happens each clock after one is in the shift state (each time there is a choice to either stay in shift or move to exit1)
        // however, for performance, we'd prefer not to shift 1 bit at a time, so only do the actual shifting when absolutely necessary (i.e., when the call is complete, or we hit the exit1 state)
        // so here we just track the number of times we were in a shift state, before we transition to the next state
        if (currentState == TapStateMachine::JtagShfIR)
        {
            _shiftIrClks++;
        }
        else if (currentState == TapStateMachine::JtagShfDR)
        {
            _shiftDrClks++;
        }

        // transition to the next state
        _tapStateMachine.DriveTmsCycle((tms != nullptr) ? ((tms[tck >> 3] >> (tck & 0x7)) & 1) : 0);
        currentState = _tapStateMachine.GetState();

        // perform side-effects to entering certain states
        switch (currentState)
        {
        case TapStateMachine::JtagCapIR:
        {
            uint32_t offset = 0;
            GetTotalCapture(true, _virtualIr.GetCStruct(), offset);
            break;
        }
        case TapStateMachine::JtagCapDR:
        {
            uint32_t offset = 0;
            GetTotalCapture(false, _virtualDr.GetCStruct(), offset);
            break;
        }
        case TapStateMachine::JtagEx1IR:
        {
            if (_shiftIrClks > 0)
            {
                // perform shift
                JtagShift(_virtualIr.GetCStruct(), _shiftIrClks, 1 + tck - _shiftIrClks, tdi, tdo);
                _shiftIrClks = 0;
            }
            break;
        }
        case TapStateMachine::JtagEx1DR:
        {
            if (_shiftDrClks > 0)
            {
                // perform shift
                JtagShift(_virtualDr.GetCStruct(), _shiftDrClks, 1 + tck - _shiftDrClks, tdi, tdo);
                _shiftDrClks = 0;
            }
            break;
        }
        case TapStateMachine::JtagUpdIR:
        {
            uint32_t offset = 0;
            SetTotalUpdate(true, _virtualIr.GetCStruct(), offset);
            break;
        }
        case TapStateMachine::JtagUpdDR:
        {
            uint32_t offset = 0;
            SetTotalUpdate(false, _virtualDr.GetCStruct(), offset);
            break;
        }
        case TapStateMachine::JtagTLR:
        {
            TapReset();
            break;
        }
        default:
            break;
        }
    }

    // if the call has finished, and we have outstanding shifts (have not passed through update) that need to be reconciled, do so now
    if (_shiftIrClks > 0)
    {
        // perform shift
        JtagShift(_virtualIr.GetCStruct(), _shiftIrClks, count - _shiftIrClks, tdi, tdo);
        _shiftIrClks = 0;
    }
    if (_shiftDrClks > 0)
    {
        // perform shift
        JtagShift(_virtualDr.GetCStruct(), _shiftDrClks, count - _shiftDrClks, tdi, tdo);
        _shiftDrClks = 0;
    }
}

uint32_t JtagStubTapChain::GetTotalShiftLength(bool bIrScan)
{
    uint32_t bitsize = 0;
    for (auto tapCtrl: _tapControllers)
    {
        uint32_t tapreglen = tapCtrl->GetShiftLength(bIrScan);
        if (tapreglen == (uint32_t)-1)
        {
            bitsize = (uint32_t)-1;
            break;
        }

        bitsize += tapreglen;
    }
    return bitsize;
}

void JtagStubTapChain::GetTotalCapture(bool bIrScan, BitData* dest, uint32_t& offset)
{
    // TBD: technically, if one of the registers in the chain doesn't have continuity, we should capture at least all the regs that do have continuity to TDO
    uint32_t totalshiftlen = GetTotalShiftLength(bIrScan);
    if (totalshiftlen != (uint32_t)-1)
    {
        BitData_Resize(dest, totalshiftlen);

        for (auto tapCtrl = _tapControllers.rbegin(); tapCtrl != _tapControllers.rend(); tapCtrl++)
        {
            JtagStubTapRegister* reg = bIrScan ? (*tapCtrl)->GetIr() : (*tapCtrl)->GetDr();
            reg->GetCapture(dest, offset);
        }
    }
    else
    {
        BitData_Resize(dest, 0);
    }
}

void JtagStubTapChain::SetTotalUpdate(bool bIrScan, const BitData* src, uint32_t& offset)
{
    // TBD: technically, if one of the registers in the chain doesn't have continuity, we should update at least all the regs that do have continuity from TDI
    for (auto tapCtrl = _tapControllers.rbegin(); tapCtrl != _tapControllers.rend(); tapCtrl++)
    {
        JtagStubTapRegister* reg = bIrScan ? (*tapCtrl)->GetIr() : (*tapCtrl)->GetDr();
        if (reg != nullptr)
        {
            reg->SetUpdate(src, offset);
        }
    }
}

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
//	JtagStubTapChainTapLink members
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

JtagStubTapChainTapLink::JtagStubTapChainTapLink(JtagStubTapController* parentTapController, uint32_t irRegIr, uint32_t drRegIr, uint32_t irLowDeracers, uint32_t irHighDeracers, uint32_t drLowDeracers, uint32_t drHighDeracers)
    : _irLowDeracers(irLowDeracers),
    _irHighDeracers(irHighDeracers),
    _drLowDeracers(drLowDeracers),
    _drHighDeracers(drHighDeracers)
{
    parentTapController->AddInstruction(irRegIr, nullptr, new JtagStubTapRegisterTapLink("TAPLINK_IR", true, this));
    parentTapController->AddInstruction(drRegIr, nullptr, new JtagStubTapRegisterTapLink("TAPLINK_DR", false, this));
}

uint32_t JtagStubTapChainTapLink::GetTotalShiftLength(bool bIrScan)
{
    uint32_t totalshiftlen = JtagStubTapChain::GetTotalShiftLength(bIrScan);

    if (totalshiftlen != uint32_t(-1))
    {
        if (bIrScan)
        {
            totalshiftlen += _irLowDeracers + _irHighDeracers;
        }
        else
        {
            totalshiftlen += _drLowDeracers + _drHighDeracers;
        }
    }

    return totalshiftlen;
}

void JtagStubTapChainTapLink::GetTotalCapture(bool bIrScan, BitData* dest, uint32_t& offset)
{
    offset += (bIrScan ? _irLowDeracers : _drLowDeracers);
    JtagStubTapChain::GetTotalCapture(bIrScan, dest, offset);
    // TBD: leave leading/trailing bits uninitialized?
}

void JtagStubTapChainTapLink::SetTotalUpdate(bool bIrScan, const BitData* src, uint32_t& offset)
{
    offset += (bIrScan ? _irLowDeracers : _drLowDeracers);
    JtagStubTapChain::SetTotalUpdate(bIrScan, src, offset);
}

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
//	JtagStubTapRegisterTapLink members
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

uint32_t JtagStubTapRegisterTapLink::GetLength()
{
    return _tapLinkChain->GetTotalShiftLength(_isIrRegister);
}

void JtagStubTapRegisterTapLink::GetCapture(BitData* dest, uint32_t& offset)
{
    _tapLinkChain->GetTotalCapture(_isIrRegister, dest, offset);
}

void JtagStubTapRegisterTapLink::SetUpdate(const BitData* src, uint32_t& offset)
{
    _tapLinkChain->SetTotalUpdate(_isIrRegister, src, offset);
}
