/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
///  @file
///
///  @brief This file implement functions used for logging
//////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"

#include <Foundation/Logging/Logging.h>
#include <Foundation/Version.h>
#include <ExecutionEnvironment.h>
#include <FormatTime.h>
#include <ManipulatePath.h>

#include "OpenIPCLogger.h"

#include <cstddef>
#include <string>
#include <ostream>
#include <iomanip>
#include <sstream>
#include <atomic>
#include <mutex>
#include <stack>
#include <set>

#define ZIP_STATIC
#include <zip.h>

#include <PushNewOverride.h>
#include <boost/log/core.hpp>
#include <boost/log/sources/severity_channel_logger.hpp>
#include <boost/log/sources/record_ostream.hpp>
#include <boost/log/expressions.hpp>
#include <boost/log/sinks/sync_frontend.hpp>
#include <boost/log/sinks/text_ostream_backend.hpp>
#include <boost/log/sinks/text_file_backend.hpp>
#include <boost/log/sinks/syslog_backend.hpp>
#include <boost/log/utility/setup/common_attributes.hpp>
#include <boost/log/sources/record_ostream.hpp>
#include <boost/log/support/date_time.hpp>
#include <boost/algorithm/string/predicate.hpp>
#include <boost/core/null_deleter.hpp>
#include <boost/phoenix/bind.hpp>
#include <boost/filesystem.hpp>
#include <boost/locale.hpp>
#include <boost/shared_ptr.hpp>
#include <PopNewOverride.h>

#include <SafeString.h>

#if defined(HOST_WINDOWS)
    #include <codecvt>
#elif defined(HOST_DARWIN) || defined(HOST_LINUX)
    #include <accessdll.h>
#endif

//#define UNIQUE_LOG_NAME

#define OpenIPC_QUOTE(name) #name
#define OpenIPC_STR(macro) OpenIPC_QUOTE(macro)

#define PRODUCTID "OpenIPC_" OpenIPC_BITNESS "_" OpenIPC_STR(OpenIPC_VERSION_MAJOR) "." OpenIPC_STR(OpenIPC_VERSION_MINOR) "." OpenIPC_STR(OpenIPC_VERSION_BUILD) "." OpenIPC_STR(OpenIPC_VERSION_CLASSIFICATION)

BOOST_LOG_ATTRIBUTE_KEYWORD(line_id, "LineID", unsigned int)
BOOST_LOG_ATTRIBUTE_KEYWORD(thread_id, "ThreadID", boost::log::attributes::current_thread_id::value_type)
BOOST_LOG_ATTRIBUTE_KEYWORD(timestamp, "TimeStamp", boost::posix_time::ptime)
BOOST_LOG_ATTRIBUTE_KEYWORD(severity, "Severity", OpenIPC_LoggingSeverityLevel)
BOOST_LOG_ATTRIBUTE_KEYWORD(channel, "Channel", std::string)
BOOST_LOG_ATTRIBUTE_KEYWORD(targets, "Targets", std::vector<std::string>)

static std::mutex _initializeGuard;
static std::string _logFilenameSuffix = "Client";
static bool _initialized = false;
thread_local std::string _lastErrorMessage;

static void *_usageDllhandle = nullptr;

typedef int (*RMAppIndividualEvent_Func)(const char* productId, const char* eventId, const char* componentId, const char* message, const int force);
static RMAppIndividualEvent_Func RMAppIndividualEvent = nullptr;

static boost::thread_specific_ptr<bool> _loggingSuppressed;

static std::map<std::string, std::map<OpenIPC_LoggingLoggerId, std::pair<OpenIPC_LoggingSeverityLevel, std::vector<std::string>>>> _loggingPresets;
static std::vector<std::string> _loggingPresetNames;
static std::string _lastLoadedLoggingPreset;
static std::stack<std::map<OpenIPC_LoggingLoggerId, OpenIPC_LoggingSeverityLevel>> _logSeverityLevelStack;
static std::set<boost::filesystem::path> _logDirectories;
static std::vector<OpenIPC_LoggingOnPresetChangedCallback> _onPresetChangedCallbacks;

static std::recursive_mutex _loggingMutex;

static boost::log::formatter _regularFormatter;
static boost::log::formatter _shortFormatter;

#define OPENIPC_LOGGER_LIST_NEXT
#define OPENIPC_LOGGER_ID(NAME) OpenIPCLogger Logger_##NAME(OpenIPC_LoggingLoggerId_##NAME, #NAME, OpenIPC_LoggingSeverityLevel_Off);

	OPENIPC_LOGGER_IDS

#undef OPENIPC_LOGGER_ID
#undef OPENIPC_LOGGER_LIST_NEXT

OpenIPCLogger* GetLoggerInstance(OpenIPC_LoggingLoggerId loggerId)
{
	switch (loggerId)
	{
		#define OPENIPC_LOGGER_LIST_NEXT
		#define OPENIPC_LOGGER_ID(NAME) case OpenIPC_LoggingLoggerId_##NAME: \
			return &Logger_##NAME; \

		OPENIPC_LOGGER_IDS

		#undef OPENIPC_LOGGER_ID
		#undef OPENIPC_LOGGER_LIST_NEXT

		case OpenIPC_LoggingLoggerId_Unknown:
			break;
	}

	return NULL;
}

std::ostream& operator<< (std::ostream& stream, OpenIPC_LoggingSeverityLevel level)
{
    static const char* strings[] =
    {
        "T",
        "D",
        "I",
		"W",
		"E",
		"O"
    };

    stream << strings[level];

    return stream;
}

OpenIPC_LoggingToken OpenIPC_LoggingGetLogger(OpenIPC_LoggingLoggerId loggerId)
{
	OpenIPCLogger* logger = GetLoggerInstance(loggerId);

	return reinterpret_cast<OpenIPC_LoggingToken>(logger);
}

bool OpenIPC_LoggingIsLogEnabled(OpenIPC_LoggingToken token, OpenIPC_LoggingSeverityLevel desiredSeverity)
{
	OpenIPCLogger* logger = reinterpret_cast<OpenIPCLogger*>(token);
	if (logger)
		return logger->IsLogEnabled(desiredSeverity);

	return false;
}

void OpenIPC_LoggingSuppressLogging(bool suppressed)
{
    if (!_loggingSuppressed.get())
    {
        _loggingSuppressed.reset(new bool);
    }

    *_loggingSuppressed = suppressed;
}

bool OpenIPC_LoggingIsLoggingSuppressed()
{
    if (!_loggingSuppressed.get())
    {
        return false;
    }

    return *_loggingSuppressed;
}

OpenIPC_LoggingSeverityLevel OpenIPC_LoggingGetLogSeverityLevel(OpenIPC_LoggingToken token)
{
	OpenIPCLogger* logger = reinterpret_cast<OpenIPCLogger*>(token);

	return logger->GetLogSeverityLevel();
}

void OpenIPC_LoggingSetLogSeverityLevel(OpenIPC_LoggingToken token, OpenIPC_LoggingSeverityLevel value)
{
	OpenIPCLogger* logger = reinterpret_cast<OpenIPCLogger*>(token);

	logger->SetLogSeverityLevel(value);
}

void OpenIPC_LoggingGetLogSinkTargets(OpenIPC_LoggingToken token, char(*sinkTargets)[OpenIPC_LOGGING_MAX_SINK_TARGET_NAME_LENGTH], uint32_t maxSinkTargets, uint32_t* numSinkTargets)
{
	OpenIPCLogger* logger = reinterpret_cast<OpenIPCLogger*>(token);
	auto sinks = logger->GetSinkTargets();
	if (numSinkTargets)
	{
		*numSinkTargets = static_cast<uint32_t>(sinks.size());
	}
	if (sinkTargets)
	{
		for (size_t i = 0; i < maxSinkTargets && i < sinks.size(); ++i)
		{
			CommonUtils::SafeStringCopy(sinkTargets[i], OpenIPC_LOGGING_MAX_SINK_TARGET_NAME_LENGTH, sinks[i]);
		}
	}
}

void OpenIPC_LoggingSetSinkTargets(OpenIPC_LoggingToken token, const char** sinkTargets, uint32_t numSinkTargets)
{
	OpenIPCLogger* logger = reinterpret_cast<OpenIPCLogger*>(token);
	std::vector<std::string> sinks;
	if (sinkTargets)
	{
		for (size_t i = 0; i < numSinkTargets; ++i)
		{
			sinks.push_back(std::string(sinkTargets[i]));
		}
		logger->SetSinkTargets(sinks);
	}
}

void OpenIPC_LoggingPushLogSeverityLevels()
{
    std::map<OpenIPC_LoggingLoggerId, OpenIPC_LoggingSeverityLevel> logSeverityLevels;

    #define OPENIPC_LOGGER_LIST_NEXT
    #define OPENIPC_LOGGER_ID(NAME) logSeverityLevels[Logger_##NAME.GetId()] = Logger_##NAME.GetLogSeverityLevel(); \

    OPENIPC_LOGGER_IDS

    #undef OPENIPC_LOGGER_ID
    #undef OPENIPC_LOGGER_LIST_NEXT

    _logSeverityLevelStack.push(logSeverityLevels);
}

void OpenIPC_LoggingPopLogSeverityLevels()
{
    std::map<OpenIPC_LoggingLoggerId, OpenIPC_LoggingSeverityLevel> logSeverityLevels = _logSeverityLevelStack.top();
    _logSeverityLevelStack.pop();

    #define OPENIPC_LOGGER_LIST_NEXT
    #define OPENIPC_LOGGER_ID(NAME) Logger_##NAME.SetLogSeverityLevel(logSeverityLevels[Logger_##NAME.GetId()]); \

    OPENIPC_LOGGER_IDS

    #undef OPENIPC_LOGGER_ID
    #undef OPENIPC_LOGGER_LIST_NEXT
}

void OpenIPC_LoggingLog(OpenIPC_LoggingToken token, OpenIPC_LoggingSeverityLevel desiredSeverity, const char* message, int indent)
{
	std::lock_guard<std::recursive_mutex> lock(_loggingMutex);
	OpenIPCLogger* logger = reinterpret_cast<OpenIPCLogger*>(token);

	logger->Log(desiredSeverity, std::string(message), indent);
}

bool OpenIPC_LoggingIsLoggingUsage()
{
	return RMAppIndividualEvent != nullptr;
}

void OpenIPC_LoggingLogUsage(const char* componentName, const char* eventName, const char* message)
{
	if (RMAppIndividualEvent != nullptr)
	{
		RMAppIndividualEvent(PRODUCTID, eventName, componentName, message, 0);
	}
}

void OpenIPC_LoggingCreateLoggingPreset(const char* presetName)
{
	std::lock_guard<std::mutex> lock(_initializeGuard);

    if (std::find(_loggingPresetNames.begin(), _loggingPresetNames.end(), presetName) == _loggingPresetNames.end())
    {
        _loggingPresetNames.push_back(presetName);
    }

    _loggingPresets[presetName] = std::map<OpenIPC_LoggingLoggerId, std::pair<OpenIPC_LoggingSeverityLevel, std::vector<std::string>>>();
}

std::vector<std::string> GetLoggingPresetNames()
{
	std::lock_guard<std::mutex> lock(_initializeGuard);

    return _loggingPresetNames;
}

void OpenIPC_LoggingGetLoggingPresetNamesCount(uint32_t* numPresetNames)
{
	std::lock_guard<std::mutex> lock(_initializeGuard);

    if (numPresetNames != nullptr)
    {
        *numPresetNames = static_cast<uint32_t>(_loggingPresetNames.size());
    }
}

void OpenIPC_LoggingGetLoggingPresetNames(char (*presetNames)[OpenIPC_LOGGING_MAX_PRESET_NAME_LENGTH], uint32_t numPresetNames)
{
	std::lock_guard<std::mutex> lock(_initializeGuard);

    if (presetNames != nullptr)
    {
        for (uint32_t i = 0; i < (numPresetNames > _loggingPresetNames.size() ? _loggingPresetNames.size() : numPresetNames); ++i)
        {
            CommonUtils::SafeStringCopy(presetNames[i], OpenIPC_LOGGING_MAX_PRESET_NAME_LENGTH, _loggingPresetNames[i]);
        }
    }
}

bool OpenIPC_LoggingLoadLoggingPreset(const char* presetName)
{
	std::unique_lock<std::mutex> lock(_initializeGuard);

    // If the preset exists
    auto it = _loggingPresets.find(presetName);
    if (it != _loggingPresets.end())
    {
        // Initially set all logger levels to off
		#define OPENIPC_LOGGER_LIST_NEXT
		#define OPENIPC_LOGGER_ID(NAME) Logger_##NAME.SetLogSeverityLevel(OpenIPC_LoggingSeverityLevel_Off); \

		OPENIPC_LOGGER_IDS

		#undef OPENIPC_LOGGER_ID
		#undef OPENIPC_LOGGER_LIST_NEXT

        // Set each logger level to the preset level
        for (auto& pair : it->second)
        {
            OpenIPCLogger* logger = GetLoggerInstance(pair.first);
            if (logger)
            {
                logger->SetLogSeverityLevel(pair.second.first);
				logger->SetSinkTargets(pair.second.second);
            }
        }

        _lastLoadedLoggingPreset = presetName;

        // Copy the list of preset change callbacks
        auto onPresetChangedCallbacks = _onPresetChangedCallbacks;

        // Unlock the mutex and invoke the callbacks
        lock.unlock();
		for (auto& callback : onPresetChangedCallbacks)
		{
			callback(presetName);
		}

        return true;
    }

    return false;
}

void OpenIPC_LoggingGetLoadedLoggingPreset(char presetName[OpenIPC_LOGGING_MAX_PRESET_NAME_LENGTH])
{
	std::lock_guard<std::mutex> lock(_initializeGuard);
    CommonUtils::SafeStringCopy(presetName, OpenIPC_LOGGING_MAX_PRESET_NAME_LENGTH, _lastLoadedLoggingPreset);
}

bool OpenIPC_LoggingAddOnPresetChangedCallback(OpenIPC_LoggingOnPresetChangedCallback callback)
{
	std::lock_guard<std::mutex> lock(_initializeGuard);
	if (callback)
	{
		auto find = std::find(_onPresetChangedCallbacks.begin(), _onPresetChangedCallbacks.end(), callback);
		if (find == _onPresetChangedCallbacks.end())
		{
			_onPresetChangedCallbacks.push_back(callback);
			return true;
		}
	}
	return false;
}

bool OpenIPC_LoggingRemoveOnPresetChangedCallback(OpenIPC_LoggingOnPresetChangedCallback callback)
{
	std::lock_guard<std::mutex> lock(_initializeGuard);
	if (callback)
	{
		auto find = std::find(_onPresetChangedCallbacks.begin(), _onPresetChangedCallbacks.end(), callback);
		if (find != _onPresetChangedCallbacks.end())
		{
			_onPresetChangedCallbacks.erase(find);
			return true;
		}
	}
	return false;
}

bool OpenIPC_LoggingSetPresetLoggerLevel(const char* presetName, const char* loggerName, OpenIPC_LoggingSeverityLevel desiredSeverity, const char** sinkTargets, uint32_t numSinkTargets)
{
	std::lock_guard<std::mutex> lock(_initializeGuard);

    // If the preset exists
    auto it = _loggingPresets.find(presetName);
    if (it != _loggingPresets.end())
    {
        OpenIPC_LoggingLoggerId loggerId = OpenIPC_LoggingLoggerId_Unknown;

        // Get the logger ID for the logger of the specified name
#define OPENIPC_LOGGER_LIST_NEXT
#define OPENIPC_LOGGER_ID(NAME) if (boost::iequals(loggerName, #NAME)) { loggerId = Logger_##NAME.GetId(); }

OPENIPC_LOGGER_IDS

#undef OPENIPC_LOGGER_ID
#undef OPENIPC_LOGGER_LIST_NEXT

        // If the logger exists
        if (loggerId != OpenIPC_LoggingLoggerId_Unknown)
        {
			std::vector<std::string> sinks;
			for (size_t i = 0; i < numSinkTargets; ++i)
			{
				sinks.push_back(std::string(sinkTargets[i]));
			}
            // Remember the severity and target
            it->second[loggerId] = std::pair<OpenIPC_LoggingSeverityLevel, std::vector<std::string>>(desiredSeverity, sinks);

            return true;
        }
    }

    return false;
}

bool OpenIPC_LoggingInitializeLogging(const char* suffix)
{
	// Lock the current section so that only one thread can initialize
	std::lock_guard<std::mutex> lock(_initializeGuard);

	if (!_initialized)
	{
		if (suffix)
		{
			_logFilenameSuffix = suffix;
		}

        _logDirectories.clear();
        _lastLoadedLoggingPreset.clear();

#if defined(HOST_WINDOWS)
        //We can't rely on the global locale here because this code can also run on the client side,
        //where we can't touch the global locale. So we need to use a local one
        // Create a UTF-8 locale
        std::locale loc = boost::locale::generator().generate("");
        // Make boost.filesystem use it
        boost::filesystem::path::imbue(loc);
#else
        std::locale loc = std::locale();
#endif
        boost::filesystem::path::imbue(loc);
		// Setup the common formatter for all sinks
		_regularFormatter = boost::log::expressions::stream
			<< "L:" << std::setw(6) << std::setfill('0') << line_id << std::setfill(' ') << " "
			<< "T:" << thread_id << " "
			<< boost::log::expressions::format_date_time(timestamp, "%H:%M:%S.%f") << " "
			<< "[" << std::setw(20) << channel << "] "
			<< severity << "> "
			<< boost::log::expressions::smessage;

		_shortFormatter = boost::log::expressions::stream
			<< "# "
			<< boost::log::expressions::format_date_time(timestamp, "%H:%M:%S.%f")
			<< std::endl
			<< boost::log::expressions::smessage;

        boost::log::core::get()->set_logging_enabled(false);

	    // Add common attributes
	    boost::log::add_common_attributes();

#define DISABLE_INTELREMOTEMON

#if defined(HOST_WINDOWS)
// *********** WINDOWS-SPECIFIC CODE ***********
#if defined(HOST_X64)
	const std::string usageDllPath = "C:\\Program Files (x86)\\Common Files\\Intel\\Intel Software Manager\\intel64\\intelremotemon.dll";
#else
	const std::string usageDllPath = "C:\\Program Files (x86)\\Common Files\\Intel\\Intel Software Manager\\intelremotemon.dll";
#endif

#ifndef DISABLE_INTELREMOTEMON
    std::wstring_convert<std::codecvt_utf8_utf16<wchar_t>, wchar_t> converter;
    _usageDllhandle = ::LoadLibraryW(converter.from_bytes(usageDllPath).c_str());
#endif

    if (_usageDllhandle)
    {
        RMAppIndividualEvent = (RMAppIndividualEvent_Func)::GetProcAddress((HMODULE)_usageDllhandle, "RMAppIndividualEvent");
    }

#else
    #if defined(HOST_LINUX)
        std::string homeDirectory;
        CommonUtils::GetUserHomeDirectory(homeDirectory);

        #if defined(HOST_X64)
            const std::string usageDllPath = homeDirectory + "/intel/ism/bin/intel64/libintelremotemon.so";
        #else
            const std::string usageDllPath = homeDirectory + "/intel/ism/bin/ia32/libintelremotemon.so";
        #endif
    #elif defined(HOST_DARWIN)
        const std::string usageDllPath = "/Applications/Intel(R) Software Manager.app/Contents/Resources/bin/universal_mac10/libintelremotemon.dylib";
    #endif

#ifndef DISABLE_INTELREMOTEMON
     _usageDllhandle = InternalUtils::loaddll(usageDllPath.c_str());
#endif

    if(_usageDllhandle)
    {
        RMAppIndividualEvent = (RMAppIndividualEvent_Func)InternalUtils::getprocedurefromdll(_usageDllhandle, "RMAppIndividualEvent");
    }
#endif
	    _initialized = true;

		return true;
	}

	return false;
}

bool OpenIPC_LoggingUninitializeLogging()
{
    if (_initialized)
    {
        boost::log::core::get()->set_logging_enabled(false);
        boost::log::core::get()->remove_all_sinks();
        RMAppIndividualEvent = nullptr;
        if (_usageDllhandle)
        {
#if defined(HOST_WINDOWS)
            ::FreeLibrary((HMODULE)_usageDllhandle);
#else
            InternalUtils::freedll(_usageDllhandle);
#endif
            _usageDllhandle = nullptr;
        }
        _initialized = false;

        return true;
    }

    return false;
}

bool FilterForSinkTargets(std::shared_ptr<std::vector<std::string>> sinkTargets, boost::log::value_ref<std::vector<std::string>, tag::targets> const& desiredTargets)
{
	for (std::string const& sinkTarget : *sinkTargets)
	{
		for(std::string const& target : desiredTargets.get())
		{
			if (target == sinkTarget)
			{
				return true;
			}
		}
	}

    return false;
}

void OpenIPC_LoggingAddSyslogSink(const char* address, const uint16_t port, const char** sinkTargets, uint32_t numberOfSinkTargets, bool shortFormatter)
{
    std::shared_ptr<std::vector<std::string>> localSinkTargets(new std::vector<std::string>());
    for (uint32_t i = 0; i < numberOfSinkTargets; ++i)
    {
        localSinkTargets->push_back(std::string(sinkTargets[i]));
    }

    typedef boost::log::sinks::synchronous_sink<boost::log::sinks::syslog_backend> sink_t;
    boost::shared_ptr<sink_t> sink = nullptr;

    // Create a backend
    boost::shared_ptr<boost::log::sinks::syslog_backend> backend(new boost::log::sinks::syslog_backend(
        boost::log::keywords::facility = boost::log::sinks::syslog::local0,
        boost::log::keywords::use_impl = boost::log::sinks::syslog::udp_socket_based
    ));

    // Configure the backend
    backend->set_target_address(address, port);

    // Create a severity mapping to map our custom levels to syslog levels
    boost::log::sinks::syslog::custom_severity_mapping<std::string> mapping("mapping");

    mapping["trace"] = boost::log::sinks::syslog::debug;
    mapping["debug"] = boost::log::sinks::syslog::info;
    mapping["info"] = boost::log::sinks::syslog::notice;
    mapping["warning"] = boost::log::sinks::syslog::warning;
    mapping["error"] = boost::log::sinks::syslog::error;

    backend->set_severity_mapper(mapping);

#if defined(HOST_WINDOWS)
    //We can't rely on the global locale here because this code can also run on the client side,
    //where we can't touch the global locale. So we need to use a local one
    // Create a UTF-8 locale
    std::locale loc = boost::locale::generator().generate("");
#else
    std::locale loc = std::locale();
#endif

    sink = boost::shared_ptr<sink_t>(new sink_t(backend));

    if (shortFormatter)
    {
        sink->set_formatter(_shortFormatter);
    }
    else
    {
        sink->set_formatter(_regularFormatter);
    }

    sink->set_filter(boost::phoenix::bind(&FilterForSinkTargets, localSinkTargets, targets.or_none()));
    sink->imbue(loc);

    boost::log::core::get()->add_sink(sink);
    boost::log::core::get()->set_logging_enabled(true);
}

void OpenIPC_LoggingAddConsoleSink(const char** sinkTargets, uint32_t numberOfSinkTargets)
{
	std::lock_guard<std::mutex> lock(_initializeGuard);

#if defined(HOST_WINDOWS)
    //We can't rely on the global locale here because this code can also run on the client side,
    //where we can't touch the global locale. So we need to use a local one
    // Create a UTF-8 locale
    std::locale loc = boost::locale::generator().generate("");
#else
    std::locale loc = std::locale();
#endif

	std::shared_ptr<std::vector<std::string>> localSinkTargets(new std::vector<std::string>());
	for (uint32_t i = 0; i < numberOfSinkTargets; ++i)
	{
		localSinkTargets->push_back(std::string(sinkTargets[i]));
	}

    // Add a console sink
	typedef boost::log::sinks::synchronous_sink<boost::log::sinks::text_ostream_backend> console_sink;

	boost::shared_ptr<console_sink> consoleSink = boost::make_shared<console_sink>();
	boost::shared_ptr<std::ostream> stream(&std::cout, boost::null_deleter());

	consoleSink->locked_backend()->add_stream(stream);

	consoleSink->set_filter(boost::phoenix::bind(&FilterForSinkTargets, localSinkTargets, targets.or_none()));
	consoleSink->set_formatter(_shortFormatter);
    consoleSink->imbue(loc);
	boost::log::core::get()->add_sink(consoleSink);
    boost::log::core::get()->set_logging_enabled(true);
}

void OpenIPC_LoggingAddFileSink(const char* filePath, const char* fileName, uint32_t rolloverSize, uint32_t maxSize, const char** sinkTargets, uint32_t numberOfSinkTargets, bool shortFormatter)
{
    if (!fileName || !sinkTargets)
    {
        return;
    }

    boost::filesystem::path fileLocation = ".";
    if (filePath)
    {
        fileLocation = filePath;
    }

    if (!OpenIPC_PASS(CommonUtils::SanitizePath(fileLocation)))
    {
        return;
    }

	std::lock_guard<std::mutex> lock(_initializeGuard);

    const boost::filesystem::path logDirectory = fileLocation / "Log";
    _logDirectories.insert(logDirectory);

	std::shared_ptr<std::vector<std::string>> localSinkTargets(new std::vector<std::string>());
	for (uint32_t i = 0; i < numberOfSinkTargets; ++i)
	{
		localSinkTargets->push_back(std::string(sinkTargets[i]));
	}

    std::string suffix;
    if(_logFilenameSuffix.empty() == false)
    {
        suffix = "_" + _logFilenameSuffix;
    }

#if defined(HOST_WINDOWS)
    //We can't rely on the global locale here because this code can also run on the client side,
    //where we can't touch the global locale. So we need to use a local one
    // Create a UTF-8 locale
    std::locale loc = boost::locale::generator().generate("");
#else
    std::locale loc = std::locale();
#endif

    std::string unique;

#ifdef UNIQUE_LOG_NAME
		auto now  = boost::posix_time::second_clock::local_time();
		unique = boost::posix_time::to_iso_string(now);
#endif

	if (rolloverSize)
	{
		typedef boost::log::sinks::synchronous_sink<boost::log::sinks::text_file_backend> sink_t;
        const std::string filePattern = std::string(fileName) + suffix + unique + "_%5N.log";

		boost::shared_ptr<boost::log::sinks::text_file_backend> backend =
			boost::make_shared<boost::log::sinks::text_file_backend>(
				boost::log::keywords::auto_flush = true,
				boost::log::keywords::file_name = fileLocation / filePattern,
				boost::log::keywords::rotation_size = rolloverSize * 1024 * 1024); // Convert from MB

		auto sink = boost::shared_ptr<sink_t>(new sink_t(backend));

		if (shortFormatter)
		{
			sink->set_formatter(_shortFormatter);
		}
		else
		{
			sink->set_formatter(_regularFormatter);
		}

		sink->set_filter(boost::phoenix::bind(&FilterForSinkTargets, localSinkTargets, targets.or_none()));
        sink->imbue(loc);

		if (maxSize)
		{
            auto collector = boost::log::sinks::file::make_collector
            (
                boost::log::keywords::target = logDirectory, // Collect log files to the "Log" folder.
                boost::log::keywords::max_size = maxSize * 1024 * 1024 // Convert from MB
            );
            sink->locked_backend()->set_file_collector(collector);

            sink->locked_backend()->scan_for_files(boost::log::sinks::file::scan_matching, true);
		}

		boost::log::core::get()->add_sink(sink);
	}
	else
	{
		typedef boost::log::sinks::synchronous_sink<boost::log::sinks::text_ostream_backend> text_sink;
		boost::shared_ptr<text_sink> sink = boost::make_shared<text_sink>();

        //Make sure the log directory exists. The Boost text stream backend will not create it for us
        boost::filesystem::create_directories(logDirectory);

		boost::filesystem::path fullpath = logDirectory / (std::string(fileName) + suffix + unique + ".log");

		sink->locked_backend()->add_stream(boost::make_shared<boost::filesystem::ofstream>(fullpath));
		sink->locked_backend()->auto_flush(true);

		if (shortFormatter)
		{
			sink->set_formatter(_shortFormatter);
		}
		else
		{
			sink->set_formatter(_regularFormatter);
		}

		sink->set_filter(boost::phoenix::bind(&FilterForSinkTargets, localSinkTargets, targets.or_none()));
        sink->imbue(loc);

		boost::log::core::get()->add_sink(sink);
	}

    boost::log::core::get()->set_logging_enabled(true);
}

void OpenIPC_LoggingSetLoggerLevel(const char* name, OpenIPC_LoggingSeverityLevel desiredSeverity)
{
	std::lock_guard<std::mutex> lock(_initializeGuard);

#define OPENIPC_LOGGER_LIST_NEXT
#define OPENIPC_LOGGER_ID(NAME) if (boost::iequals(name, #NAME)) { Logger_##NAME.SetLogSeverityLevel(desiredSeverity); }

OPENIPC_LOGGER_IDS

#undef OPENIPC_LOGGER_ID
#undef OPENIPC_LOGGER_LIST_NEXT
}

void OpenIPC_LoggingSetLoggerSinkTargets(const char* name, const char** sinkTargets, uint32_t numSinkTargets)
{
	std::lock_guard<std::mutex> lock(_initializeGuard);
	std::vector<std::string> sinks;
	for (size_t i = 0; i < numSinkTargets; ++i)
	{
		sinks.push_back(std::string(sinkTargets[i]));
	}

#define OPENIPC_LOGGER_LIST_NEXT
#define OPENIPC_LOGGER_ID(NAME) if (boost::iequals(name, #NAME)) { Logger_##NAME.SetSinkTargets(sinks); }

OPENIPC_LOGGER_IDS

#undef OPENIPC_LOGGER_ID
#undef OPENIPC_LOGGER_LIST_NEXT
}

void OpenIPC_LoggingAddListener(OpenIPC_LoggingToken token, void* data, OpenIPC_LoggingReceiveMessageCallback receiveMessageCallback, OpenIPC_LoggingSeverityLevelChangedCallback severityLevelChangedCallback)
{
	OpenIPCLogger* logger = reinterpret_cast<OpenIPCLogger*>(token);

	logger->AddListener(data, receiveMessageCallback, severityLevelChangedCallback);
}

void OpenIPC_LoggingRemoveListener(OpenIPC_LoggingToken token, void* data, OpenIPC_LoggingReceiveMessageCallback receiveMessageCallback, OpenIPC_LoggingSeverityLevelChangedCallback severityLevelChangedCallback)
{
	OpenIPCLogger* logger = reinterpret_cast<OpenIPCLogger*>(token);

	logger->RemoveListener(data, receiveMessageCallback, severityLevelChangedCallback);
}

void OpenIPC_LoggingLock()
{
    _loggingMutex.lock();
}

void OpenIPC_LoggingUnlock()
{
    _loggingMutex.unlock();
}

// Returns the file paths of each log file in the specified directory
std::vector<boost::filesystem::path> _GetLogFilePaths(const boost::filesystem::path& logDirectory)
{
    static std::set<std::string> _logFileExtensions{ ".log", ".dmp" };

    std::vector<boost::filesystem::path> logFilePaths;

    // For each file in the directory
    for (boost::filesystem::directory_iterator iter(logDirectory); iter != boost::filesystem::directory_iterator(); ++iter)
    {
        auto path = iter->path();
        auto extension = path.extension();

        // If it is a log file
        if (boost::filesystem::is_regular_file(path) && _logFileExtensions.count(extension.string()) > 0)
        {
            logFilePaths.push_back(path);
        }
    }

    return logFilePaths;
}

bool OpenIPC_LoggingClearLogs()
{
	std::lock_guard<std::mutex> lock(_initializeGuard);

    try
    {
        // For each log directory
        for (auto& logDirectory : _logDirectories)
        {
            // For each log file in the directory
            const std::vector<boost::filesystem::path> logFilePaths = _GetLogFilePaths(logDirectory);
            for (auto& logFilePath : logFilePaths)
            {
                // Delete the log file
                boost::filesystem::remove(logFilePath);
            }
        }
    }
    catch (const std::exception& exception)
    {
        _lastErrorMessage = exception.what();
        return false;
    }

    return true;
}

// Generates a timestamped name of an archive containing OpenIPC logs
std::string _GenerateLogArchiveName()
{
    uint32_t milliseconds = 0;
    std::string firstChunk = CommonUtils::FormatTime("%Y-%m-%d_%H-%M-%S", &milliseconds);

    std::stringstream ss;
    ss << "OpenIPC_Log_Archive_" << firstChunk << "-" << std::setfill('0') << std::setw(3) << milliseconds << ".zip";
    return ss.str();
}

// Adds a file to the specified zip archive; creates the archive if it doesn't
// already exist
bool _AddFileToArchive(const boost::filesystem::path& archivePath, const boost::filesystem::path& filePath)
{
    // Create the zip archive
    std::string archivePathString = archivePath.string();
    zip_t* zipArchive = zip_open(archivePathString.c_str(), ZIP_CREATE, NULL);
    if (!zipArchive)
    {
        std::stringstream ss;
        ss << "Failed to create zip archive '" << archivePath << "': " << zip_strerror(zipArchive);
        _lastErrorMessage = zip_strerror(zipArchive);
        return false;
    }

    // Open the file for reading and scan the size
    boost::filesystem::ifstream fileStream(filePath, std::ios::binary | std::ios::ate);
    std::streamsize size = fileStream.tellg();
    fileStream.seekg(0, std::ios::beg);

    // Load the file to a buffer in memory
    std::vector<char> buffer(static_cast<std::size_t>(size));
    fileStream.read(buffer.data(), size);
    if (fileStream.bad())
    {
        zip_close(zipArchive);

        std::stringstream ss;
        ss << "Failed to read file '" << filePath << "'";
        _lastErrorMessage = ss.str();
        return false;
    }

    fileStream.close();

    const std::string fileName = filePath.filename().string();

    // Create a source buffer from the file buffer
    zip_source_t* source = zip_source_buffer(zipArchive, buffer.data(), buffer.size(), 0);
    if (!source)
    {
        zip_close(zipArchive);

        std::stringstream ss;
        ss << "Failed to add file '" << fileName << "' to zip archive: " << zip_strerror(zipArchive);
        _lastErrorMessage = ss.str();
        return false;
    }

    // Add the file to the zip archive
    if (zip_file_add(zipArchive, fileName.c_str(), source, ZIP_FL_ENC_UTF_8) < 0)
    {
        zip_source_free(source);
        zip_close(zipArchive);

        std::stringstream ss;
        ss << "Failed to add file '" << fileName << "' to zip archive: " << zip_strerror(zipArchive);
        _lastErrorMessage = ss.str();
        return false;
    }

    // Close the zip archive
    if (zip_close(zipArchive) < 0)
    {
        std::stringstream ss;
        ss << "Failed to close zip archive '" << archivePath << "': " << zip_strerror(zipArchive);
        _lastErrorMessage = ss.str();
        return false;
    }

    return true;
}

bool OpenIPC_LoggingArchiveLogs()
{
	std::lock_guard<std::mutex> lock(_initializeGuard);

    try
    {
        // For each log directory
        for (auto& logDirectory : _logDirectories)
        {
            const boost::filesystem::path archiveFilePath = logDirectory / ".." / _GenerateLogArchiveName();

            // For each log file in the directory
            const std::vector<boost::filesystem::path> logFilePaths = _GetLogFilePaths(logDirectory);
            for (auto& logFilePath : logFilePaths)
            {
                // Add the log file to the archive
                if (!_AddFileToArchive(archiveFilePath, logFilePath))
                {
                    return false;
                }

                // Remove the log file
                boost::filesystem::remove(logFilePath);
            }
        }
    }
    catch (const std::exception& exception)
    {
        _lastErrorMessage = exception.what();
        return false;
    }

    return true;
}

const char* OpenIPC_LoggingGetLastErrorMessage()
{
    return _lastErrorMessage.c_str();
}
