﻿/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////

#include "stdafx.h"
#include "OpenIPCLogger.h"

#include <boost/log/attributes/mutable_constant.hpp>

OpenIPCLogger::OpenIPCLogger(OpenIPC_LoggingLoggerId id, std::string name, OpenIPC_LoggingSeverityLevel loggerLevel) :
    _loggerId(id),
	_loggerLevel(loggerLevel),
    _name(name),
    _listeners(),
	_sinkTargets(std::vector<std::string>({ "Default" })),
	_sinkTargetsAttribute(_sinkTargets)
{
	auto logger = boost::log::sources::severity_channel_logger_mt<OpenIPC_LoggingSeverityLevel>(boost::log::keywords::channel = name);

	logger.add_attribute("Targets", _sinkTargetsAttribute);

	_boostLogger = std::move(logger);
}

void OpenIPCLogger::Log(OpenIPC_LoggingSeverityLevel severity, const std::string& message, int indent)
{
    static int totalIndent = 0;
    if (indent < 0)
    {
        totalIndent += indent;
    }
	for (::boost::log::record _boost_log_record = _boostLogger.open_record(::boost::log::keywords::severity = severity); !!_boost_log_record;)
	{
		auto pump = ::boost::log::aux::make_record_pump(_boostLogger, _boost_log_record);
		boost::log::basic_formatting_ostream<char>& booststream = pump.stream();
        if (totalIndent > 0)
        {
            std::string indentSpace(totalIndent * 2, ' ');
            booststream << indentSpace.c_str() << message.c_str();
        }
        else
        {
            booststream << message.c_str();
        }
	}
    if (indent > 0)
    {
        totalIndent += indent;
    }
    // Notify all registered listeners
    if (!_listeners.empty())
    {
        std::unique_lock<std::timed_mutex> lock(_listenersMutex, std::defer_lock);
        if (lock.try_lock_for(std::chrono::seconds(2)))
        {
            for (std::tuple<void*, OpenIPC_LoggingReceiveMessageCallback, OpenIPC_LoggingSeverityLevelChangedCallback> listener : _listeners)
            {
                std::get<1>(listener)(std::get<0>(listener), severity, message.c_str());
            }
        }
        // Release lock if aquired
    }
}

void OpenIPCLogger::SetLogSeverityLevel(OpenIPC_LoggingSeverityLevel value)
{
    static const char* strings[] =
    {
        "Trace",
        "Debug",
        "Info",
		"Warning",
		"Error",
		"Off"
    };

	std::size_t level = static_cast<std::size_t>(value);

    if (level != _loggerLevel)
    {
        // Temporarily set the level to "Info" and log a message indicating that
        // log severity level has changed
		std::stringstream stream;
		stream << "Log severity level set to '" << strings[level] << "'";

        _loggerLevel = OpenIPC_LoggingSeverityLevel_Info;
		Log(OpenIPC_LoggingSeverityLevel_Info, stream.str());

        // Set the logger level
        _loggerLevel = level;
    }

    // Notify all registered listeners
    if (!_listeners.empty())
    {
        std::lock_guard<std::timed_mutex> lock(_listenersMutex);
        for (std::tuple<void*, OpenIPC_LoggingReceiveMessageCallback, OpenIPC_LoggingSeverityLevelChangedCallback> listener : _listeners)
        {
			std::get<2>(listener)(std::get<0>(listener), value);
        }
    }
}

void OpenIPCLogger::SetSinkTargets(const std::vector<std::string>& sinkTargets)
{
	_sinkTargets = sinkTargets;
	_sinkTargetsAttribute.set(sinkTargets);
}

void OpenIPCLogger::AddListener(void* data, OpenIPC_LoggingReceiveMessageCallback receiveMessage, OpenIPC_LoggingSeverityLevelChangedCallback severityLevelChanged)
{
    std::lock_guard<std::timed_mutex> lock(_listenersMutex);

	std::tuple<void*, OpenIPC_LoggingReceiveMessageCallback, OpenIPC_LoggingSeverityLevelChangedCallback> item(data, receiveMessage, severityLevelChanged);

    auto it = std::find(_listeners.begin(), _listeners.end(), item);
    if (it == _listeners.end())
    {
        _listeners.push_back(item);
    }
}

void OpenIPCLogger::RemoveListener(void* data, OpenIPC_LoggingReceiveMessageCallback receiveMessage, OpenIPC_LoggingSeverityLevelChangedCallback severityLevelChanged)
{
    std::lock_guard<std::timed_mutex> lock(_listenersMutex);

	std::tuple<void*, OpenIPC_LoggingReceiveMessageCallback, OpenIPC_LoggingSeverityLevelChangedCallback> item(data, receiveMessage, severityLevelChanged);

    auto it = std::find(_listeners.begin(), _listeners.end(), item);
    if (it != _listeners.end())
    {
        _listeners.erase(it);
    }
}
