﻿/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#pragma once
#include <StructuredData.h>

#ifdef _MSC_VER
// Disable the DLL interface warning for STL usage
#pragma warning( push )
#pragma warning( disable : 4251 )
#endif

#include "DataIndexNode.h"
#include "DataQuery.h"

///
/// @brief Provides access to a store of data within a directory.
class STRUCTUREDDATA_API DataStore
{
    friend class DataIndexNode;
public:

    ///
    /// @brief Gets the global data store instance.
    static DataStore* Get();

    DataStore();

    ///
    /// @brief Includes a directory to the data store.
    ///
    /// @note If any malformed XML is encountered the file is ignored and the
    /// error is logged.
    ///
    /// @param directory
    ///     The directory path.
    /// @param useIndexFiles
    ///     Whether the index tree should be built using any encountered index
    ///     index files, allowing files to only be loaded as needed.
    /// @param recursive
    ///     If true, any directories that exist under the given directory will also be included
    void IncludeDirectory(const std::string& directory, bool useIndexFiles, bool recursive);

    ///
    /// @brief Includes a directory to the data store.
    ///
    /// @param directory
    ///     The directory path.
    /// @param useIndexFiles
    ///     Whether the index tree should be built using any encountered index
    ///     index files, allowing files to only be loaded as needed.
    /// @param recursive
    ///     If true, any directories that exist under the given directory will also be included
    /// @param exceptions
    ///     The list of exceptions to add to if exceptions are encountered.
    void IncludeDirectory(const std::string& directory, bool useIndexFiles, bool recursive, std::vector<DataStoreException>& exceptions);

    // Expands the index tree to include an XML document
    ///
    /// @param filePath
    ///     The path to the XML file to load
    /// @param decrypt
    ///     Whether the file should be decrypted or not (if decryption is supported)
    /// @param skipIfIndexed
    ///     Ignore the file if it has already been indexed by an index file
    /// @param exceptions
    ///     The list of exceptions to add to if exceptions are encountered.
    void AddDocument(const std::string& filePath, bool decrypt, bool skipIfIndexed, std::vector<DataStoreException>& exceptions);

    // Expands the index tree to include an XML document
    ///
    /// @param filePath
    ///     The path to the XML file to load
    /// @param decrypt
    ///     Whether the file should be decrypted or not (if decryption is supported)
    /// @param skipIfIndexed
    ///     Ignore the file if it has already been indexed by an index file
    /// @param schemaFilePath
    ///     The schema file to use when validating the given XML file. This is not used
    ///     if the DataStore was told to disable schema validation
    /// @param exceptions
    ///     The list of exceptions to add to if exceptions are encountered.
    ///
    /// @sa SetSchemaValidationEnabled()
    void AddDocument(const std::string& filePath, bool decrypt, bool skipIfIndexed, const std::string& schemaFilePath, std::vector<DataStoreException>& exceptions);

    ///
    /// @brief Finds the data elements in the data store based on the criteria
    /// in a query.
    ///
    /// @note A query may cause XML to be loaded if needed.  No exceptions are
    /// thrown if malformed XML is encountered.  Instead, the errors are
    /// accessible from the query result.
    ///
    /// @param query
    ///     The data query to perform.
    ///
    /// @returns The result of the data query.
    DataQueryResult Find(const DataQuery& query);

    ///
    /// @brief Returns whether schema validation is enabled.
    bool IsSchemaValidationEnabled() const;

    ///
    /// @brief Sets whether schema validation is enabled.
    ///
    /// @param enabled
    ///     Whether to enable or disable schema validation.
    void SetSchemaValidationEnabled(bool enabled);

    typedef std::function<void(const std::string&)> NotificationCallback;

    ///
    /// @brief Sets a function to be called when DataStore has
    /// a notification for the user.
    ///
    /// @param callback
    ///     The callback to be called with the notification to the user.
    void SetNotificationCallback(NotificationCallback&& callback);
private:
    // Files where we don't care if they've been changed since install.
    static const std::set<std::string> _willNotNotifyIfChanged;

	std::shared_ptr<DataDocument> _LoadXmlDocument(const std::string& filePath, bool decrypt, bool useSchema, const std::string& schemaFilePath, bool alreadyIndexed);

    // Expands the index tree to include the contents of an index XML file
    void _AddIndex(const std::string& filePath, bool decrypt);

    std::recursive_mutex _mutex;

    // Directories which have been included using IncludeDirectory()
    std::set<std::string> _directories;

    // Full paths to the documents that have been indexed in an index file but
    // not necessarily loaded into the index tree
    std::set<std::string> _indexedDocuments;

    // Full paths to the documents that have been loaded into the index tree
    std::set<std::string> _loadedDocuments;

    // The documents that have been loaded into the index tree
    std::vector<std::shared_ptr<DataDocument>> _documents;

    // The root index node of the index tree
    DataIndexNode _rootIndexNode;

    NotificationCallback _notificationCallback;

    // Whether schema validation is enabled when loading XML
    bool _schemaValidationEnabled;
};

#ifdef _MSC_VER
#pragma warning( pop )
#endif
