/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#include "stdafx.h"
#include "StructuredData.h"

DataElement* DataElement::Parent() const
{
    return _parent;
}

DataAttribute* DataElement::CreateAttribute(const std::string& name, const std::string& value)
{
    DataAttribute* attribute = FindAttribute(name);
    if (attribute)
    {
        // An attribute with the same name already exists, so just change the
        // value of the existing attribute
        attribute->SetValue(value);
    }
    else
    {
        // Create a new attribute
        attribute = new DataAttribute(name, value, this, _attributes.size());

        // Add it as a child node (handles deallocation)
        _children.push_back(std::shared_ptr<DataNode>(attribute));

        // Add it to the attribute list and map
        _attributes.push_back(attribute);
        _attributeMap[name] = attribute;
    }

    return attribute;
}

DataElement* DataElement::CreateChild(const std::string& name, const std::string& nameSpaceUri)
{
    // Create a new child element
    DataElement* child = new DataElement(name, nameSpaceUri, this, _elements.size(), _document);

    // Add it as a child node (handles deallocation)
    _children.push_back(std::shared_ptr<DataNode>(child));

    // Add it to the child element list
    _elements.push_back(child);

    return child;
}

DataAttribute* DataElement::FirstAttribute() const
{
    if (_attributes.empty())
    {
        return nullptr;
    }
    else
    {
        return _attributes.front();
    }
}

std::vector<DataAttribute*> DataElement::GetAttributes() const
{
    return _attributes;
}

DataAttribute* DataElement::FindAttribute(const std::string& name) const
{
    // Look up the attribute by name
    auto it = _attributeMap.find(name);
    if (it == _attributeMap.end())
    {
        // Did not find it
        return nullptr;
    }
    else
    {
        // Found it
        return (*it).second;
    }
}

bool DataElement::HasAttribute(const std::string& name, const std::string& value) const
{
    DataAttribute* attribute = FindAttribute(name);
    return attribute && attribute->Value() == value;
}

DataElement* DataElement::FirstChild() const
{
    if (_elements.empty())
    {
        return nullptr;
    }
    else
    {
        return _elements.front();
    }
}

DataElement* DataElement::NextSibling() const
{
    if (!_parent)
    {
        // This is a root element so it cannot have a sibling
        return nullptr;
    }
    else
    {
        if (_position + 1 >= _parent->_elements.size())
        {
            // This is the last child of the parent element
            return nullptr;
        }
        else
        {
            // Return the next child of the parent element
            return _parent->_elements[_position + 1];
        }
    }
}

DataElement* DataElement::FindChildByName(const std::string& name) const
{
    return FindChild([&](const DataElement* e) { return e->Name() == name; });
}

DataElement* DataElement::FindChildByAttribute(const std::string& name, const std::string& value) const
{
    return FindChild([&](const DataElement* e)
    {
        return e->HasAttribute(name, value);
    });
}

DataElement::Array DataElement::FindChildrenByName(const std::string& name) const
{
    return FindChildren([&](const DataElement* e) { return e->Name() == name; });
}

DataElement::Array DataElement::FindChildrenByAttribute(const std::string& name, const std::string& value) const
{
    return FindChildren([&](const DataElement* e)
    {
        return e->HasAttribute(name, value);
    });
}

DataElement* DataElement::FindChild(DataElement::Predicate predicate) const
{
    DataElement* child = FirstChild();
    while (child)
    {
        if (predicate(child))
        {
            break;
        }
        child = child->NextSibling();
    }
    return child;
}

DataElement::Array DataElement::FindChildren(DataElement::Predicate predicate) const
{
    DataElement::Array results;

    DataElement* child = FirstChild();
    while (child)
    {
        if (predicate(child))
        {
            results.push_back(child);
        }
        child = child->NextSibling();
    }
    return results;
}

DataElement* DataElement::FindDescendantByName(const std::string& name) const
{
    return FindDescendant([&](const DataElement* e) { return e->Name() == name; });
}

DataElement* DataElement::FindDescendantByAttribute(const std::string& name, const std::string& value) const
{
    return FindDescendant([&](const DataElement* e)
    {
        return e->HasAttribute(name, value);
    });
}

DataElement::Array DataElement::FindDescendantsByName(const std::string& name) const
{
    return FindDescendants([&](const DataElement* e) { return e->Name() == name; });
}

DataElement::Array DataElement::FindDescendantsByAttribute(const std::string& name, const std::string& value) const
{
    return FindDescendants([&](const DataElement* e)
    {
        return e->HasAttribute(name, value);
    });
}

DataElement* DataElement::FindDescendant(DataElement::Predicate predicate) const
{
    DataElement* child = FirstChild();
    while (child)
    {
        if (predicate(child))
        {
            return child;
        }
        else
        {
            DataElement* result = child->FindDescendant(predicate);
            if (result)
            {
                return result;
            }
        }
        child = child->NextSibling();
    }

    return nullptr; // Did not find it
}

DataElement::Array DataElement::FindDescendants(DataElement::Predicate predicate) const
{
    DataElement::Array results;

    DataElement* child = FirstChild();
    while (child)
    {
        if (predicate(child))
        {
            results.push_back(child);
        }

        // Search children of this child
        for (DataElement* element : child->FindDescendants(predicate))
        {
            results.push_back(element);
        }

        child = child->NextSibling();
    }

    return results;
}

DataElement* DataElement::FindAncestorByName(const std::string& name) const
{
    return FindAncestor([&](const DataElement* e)
    {
        return e->Name() == name;
    });
}

DataElement* DataElement::FindAncestorByAttribute(const std::string& name, const std::string& value) const
{
    return FindAncestor([&](const DataElement* e)
    {
        return e->HasAttribute(name, value);
    });
}

DataElement* DataElement::FindAncestor(DataElement::Predicate predicate) const
{
    DataElement* ancestor = _parent;
    while (ancestor)
    {
        if (predicate(ancestor))
        {
            break;
        }
        ancestor = ancestor->_parent;
    }
    return ancestor;
}

const std::string& DataElement::Text() const
{
    return _text;
}

void DataElement::SetText(const std::string& text)
{
    _text = text;
}

void DataElement::SetName(const std::string& name)
{
    _name = name;
}

size_t DataElement::Position() const
{
    return _position;
}

DataDocument* DataElement::Document() const
{
	return _document;
}

unsigned DataElement::LineNumber() const
{
	return _lineNumber;
}

void DataElement::SetLineNumber(unsigned lineNumber)
{
	_lineNumber = lineNumber;
}

const std::string& DataElement::NameSpaceUri() const
{
    return _nameSpaceUri;
}

void DataElement::Copy(const DataElement* element)
{
    // Clear all attributes/children
    _attributes.clear();
    _attributeMap.clear();
    _elements.clear();
    _children.clear();
	_document = element->_document;
	_lineNumber = element->_lineNumber;

    SetName(element->Name());
    SetText(element->Text());

    // Re-create all attributes from the source element
    DataAttribute* attribute = element->FirstAttribute();
    while (attribute)
    {
        CreateAttribute(attribute->Name(), attribute->Value());
        attribute = attribute->NextSibling();
    }

    // Iterate over all children of the source element and recursively copy
    DataElement* child = element->FirstChild();
    while (child)
    {
        DataElement* newChild = CreateChild(child->Name());
        newChild->Copy(child);

        child = child->NextSibling();
    }
}

void DataElement::ResolveAttributeParameters(const DataAttribute::ParameterValues& parameterValues)
{
    // For each parameter
    for (auto& parameter : parameterValues)
    {
        const std::string& name = parameter.first;
        const std::string& value = parameter.second;

        // Resolve parameter for each attribute
        for (DataAttribute* attribute : _attributes)
        {
            attribute->ResolveParameter(name, value);
        }
    }

    // Recursively resolve parameters for each child element
	DataElement* child = FirstChild();
    while (child)
    {
        child->ResolveAttributeParameters(parameterValues);
        child = child->NextSibling();
    }
}

std::string DataElement::ToXmlString() const
{
    DataDocument document("Temporary");
    document.RootElement()->Copy(this);
    const std::string xmlString = document.SaveXmlString();
    return xmlString;
}

DataElement& DataElement::operator=(const DataElement& element)
{
    Copy(&element);
    return *this;
}

void DataElement::_SetDocument(DataDocument* document)
{
	_document = document;

	// Set the document for all children
	DataElement* child = FirstChild();
    while (child)
    {
		child->_SetDocument(document);
        child = child->NextSibling();
    }
}

DataElement::DataElement(const std::string& name, const std::string& nameSpaceUri, DataElement* parent, size_t position, DataDocument* document) :
    DataNode(name),
	_document(document),
	_lineNumber(0),
    _parent(parent),
    _position(position),
    _nameSpaceUri(nameSpaceUri),
    _attributeMap()
{
}

