/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#pragma once

class XmlParser;

///
/// @brief A document.
class STRUCTUREDDATA_API DataDocument
{
public:

    ///
    /// @brief Constructs an empty document without a root element.
    DataDocument();

    ///
    /// @brief Constructs a document with a root element.
    ///
    /// @param rootElementName
    ///     The name of the root element.
    /// @param nameSpaceUri
    ///     The namespace URI of the root element.
    DataDocument(const std::string& rootElementName, const std::string& nameSpaceUri = std::string());

    ///
    /// @brief Constructs a document a root element.
    ///
    /// @param rootElement
    ///     The root element to copy as the document's element tree.
    DataDocument(const DataElement& rootElement);

    ///
    /// @brief Copy constructor.
    ///
    /// @note Copies the entire node tree.
    DataDocument(const DataDocument& document);

    ///
    /// @brief Move constructor
    ///
    /// @note Does not copy the node tree.
    DataDocument(DataDocument&& document);

    ///
    /// @brief Destructor
    ~DataDocument();

    ///
    /// @brief Gets the root element of the document.
    ///
    /// @returns The root element; null if the document does not have a root
    /// element.
    DataElement* RootElement();

    ///
    /// @copydoc DataDocument::RootElement()
    const DataElement* RootElement() const;

    ///
    /// @brief Loads the document from an XML file.
    ///
    /// @note If the document already had a root element then it is first
    /// removed.
    ///
    /// @param filePath
    ///     The path to the XML file.
    /// @param enableSchemaValidation
    ///     Whether to enable schema validation.
    /// @param newToIndex
    ///     If this file is new. Default true.
    void LoadXmlFile(const std::string& filePath, bool enableSchemaValidation = false, bool newToIndex = true);

    ///
    /// @brief Loads the document from an XML file.
    ///
    /// @note If the document already had a root element then it is first
    /// removed.
    ///
    /// @note No external schemas will be loaded. Only the given schema file
    /// will be used.
    ///
    /// @param filePath
    ///     The path to the XML file.
    /// @param schemaFilePath
    ///     The schema file to be used for schema validation.
    /// @param newToIndex
    ///     If this file is new. Default true.
    void LoadXmlFile(const std::string& filePath, const std::string& schemaFilePath, bool newToIndex = true);

    ///
    /// @brief Loads the document from an XML string.
    ///
    /// @note If the document already had a root element then it is first
    /// removed.
    ///
    /// @param xml
    ///     The XML string.
    /// @param newToIndex
    ///     If this file is new. Default true.
    void LoadXmlString(const std::string& xml, bool newToIndex = true);

    ///
    /// @brief Saves the document to an XML file.
    ///
    /// @param filePath
    ///     The path to the XML file.
    void SaveXmlFile(const std::string& filePath) const;

    ///
    /// @brief Saves the document to an XML string.
    ///
    /// @returns The resulting XML string.
    std::string SaveXmlString() const;

    ///
    /// @brief Loads the document from a bin stream.
    ///
    /// @note If the document already had a root element then it is first
    /// removed.
    ///
    /// @param stream
    ///     The stream containing the binary file.
    /// @param newToIndex
    ///     If this file is new. Default true.
    void LoadBinStream(std::basic_istream<char>& stream, bool newToIndex = true);

	///
    /// @brief Loads the document from a bin file.
    ///
    /// @note If the document already had a root element then it is first
    /// removed.
    ///
    /// @param filePath
    ///     The path to the bin file.
    /// @param newToIndex
    ///     If this file is new. Default true.
	void LoadBinFile(const std::string& filePath, bool newToIndex = true);

    ///
    /// @brief Saves the document to an bin file.
    ///
    /// @param filePath
    ///     The path to the bin file.
	void SaveBinFile(const std::string& filePath);

    ///
    /// @brief Saves the document to a stream.
    ///
    /// @param stream
    ///     The stream to save to.
	void SaveBinStream(std::basic_ostream<char>& stream) const;

	///
	/// @brief Gets the file path that the document was loaded from.
	const std::string& FilePath() const;

    ///
    /// @brief Assignment operator.
    ///
    /// @note Copies the entire node tree.
    DataDocument& operator=(const DataDocument& document);

    ///
    /// @brief Move operator.
    ///
    /// @note Does not copy the node tree.
    DataDocument& operator=(DataDocument&& document);

    ///
    /// @brief Creates a processing instruction.
	DataProcessingInstruction* CreateDataProcessingInstruction(const std::string& target, const std::string& data);

    ///
    /// @brief Gets the first processing instruction.
    ///
    /// @returns The first processing instruction; null if the document has
    /// no processing instruction.
    DataProcessingInstruction* FirstProcessingInstruction() const;

	///
    /// @brief Accesses the processing instruction collection.
    const std::vector<std::shared_ptr<DataProcessingInstruction>>& GetProcessingInstructions() const;

    ///
    /// @brief Returns true if this document is new to the index..
    bool NewToIndex() const;

    ///
    /// @brief Sets this document as no longer new to the index.
    void MarkAsOld();

private:
    void _CreateRootElement(const std::string& name, const std::string& nameSpaceUri);
    void _Copy(const DataDocument& document);

	static uint32_t _GetStringTableIndex(std::vector<std::string>& stringTable, std::string value);
	std::string _GetStringTableValue(std::vector<std::string>& stringTable, uint32_t index);
	void _SaveBinElement(const DataElement* element, std::stringstream& stream, std::vector<std::string>& stringTable) const;
	void _LoadBinElement(DataElement* parent, std::basic_istream<char>& stream, std::vector<std::string>& stringTable);

    static const uint64_t BIN_FILE_SIGNATURE = 0x4E6F742044414C21;

	std::vector<std::shared_ptr<DataProcessingInstruction>> _processingInstructions;
    std::shared_ptr<DataElement> _rootElement;
	std::string _filePath;
    std::unique_ptr<XmlParser> _xmlParser;
    bool _newToIndex;
};
