/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#include "stdafx.h"
#include "Xerces.h"

#include <memory>
#include <sstream>

#include <PushNewOverride.h>
#include <xercesc/parsers/XercesDOMParser.hpp>
#include <xercesc/parsers/SAXParser.hpp>
#include <xercesc/sax/HandlerBase.hpp>
#include <xercesc/sax/Locator.hpp>
#include <xercesc/dom/DOM.hpp>
#include <xercesc/framework/XMLGrammarPoolImpl.hpp>
#include <xercesc/framework/LocalFileInputSource.hpp>
#include <xercesc/framework/LocalFileFormatTarget.hpp>
#include <xercesc/framework/MemBufInputSource.hpp>
#include <xercesc/framework/MemBufFormatTarget.hpp>
#include <xercesc/util/XMLString.hpp>
#include <xercesc/util/TransService.hpp>
#include <xercesc/util/PlatformUtils.hpp>

#include <boost/filesystem.hpp>
#include <PopNewOverride.h>

using namespace xercesc;

// For initializing/deinitializing Xerces.
class XercesInitializer
{
public:
    XercesInitializer()
    {
        XMLPlatformUtils::Initialize();
    }

    ~XercesInitializer()
    {
        XMLPlatformUtils::Terminate();
    }
} initializer;

// Converts from a Xerces string to an STD string.
std::string FromXercesString(const XMLCh* string)
{
    return std::string(reinterpret_cast<const char*>(TranscodeToStr(string, "UTF-8").str()));
}

// Converts from an STD string to a Xerces string.
std::unique_ptr<TranscodeFromStr> ToXercesString(const std::string& string)
{
    return std::unique_ptr<TranscodeFromStr>(new TranscodeFromStr(reinterpret_cast<const XMLByte*>(string.c_str()), string.size(), "UTF-8"));
}

class ParserHandler :
	public HandlerBase
{
	bool _atRoot;
	DataElement* _element;
	const Locator* _locator;
	std::stringstream _textStream;

public:
    ParserHandler(DataElement* rootElement) :
        _atRoot(true),
		_element(rootElement),
		_locator(nullptr)
	{
	}

	void startElement(const XMLCh* const name, AttributeList& attributes)
	{
		if (!_atRoot)
		{
            // Create a new element
            _element = _element->CreateChild(FromXercesString(name));
		}
		else
		{
			// If we are at the root then we need to operate on the current element
			_element->SetName(FromXercesString(name));
			_atRoot = false;
		}

		// Set the line number that the element was loaded from
		if (_locator)
		{
			_element->SetLineNumber((unsigned)_locator->getLineNumber());
		}

		// For each element attribute
		for (XMLSize_t i = 0; i < attributes.getLength(); ++i)
		{
			// Transcode the attribute name/value
			std::string n = FromXercesString(attributes.getName(i));
			std::string v = FromXercesString(attributes.getValue(i));

			// Create the attribute
			_element->CreateAttribute(n, v);
		}
	}

	void endElement(const XMLCh* const /*name*/)
	{
		// Set the element text and clear the stream
		_element->SetText(_textStream.str());
		_textStream.str(std::string());
		_textStream.clear();

		// Move up to the parent
		_element = _element->Parent();
	}

	void processingInstruction(const XMLCh* const target, const XMLCh* const data)
	{
		std::string targetText = FromXercesString(target);
		std::string dataText = FromXercesString(data);

		DataDocument* doc = _element->Document();
		DataProcessingInstruction* instruction = doc->CreateDataProcessingInstruction(targetText, dataText);

		if (_locator)
		{
			instruction->SetLineNumber((unsigned)_locator->getLineNumber());
		}
	}

	void characters(const XMLCh* const chars, const XMLSize_t length)
	{
		// Transcode the characters
		std::vector<char> buffer(length+1);
		XMLString::transcode(chars, &buffer[0], length);

		// Append the text
		_textStream << &buffer[0];
	}

	void setDocumentLocator(const Locator* const locator)
	{
		_locator = locator;
	}

	void fatalError(const SAXParseException& e)
    {
        throw e; // Re-throw all exceptions
    }

	void error(const SAXParseException& e)
    {
        throw e; // Re-throw all exceptions
    }

    void warning(const SAXParseException& e)
    {
        throw e; // Re-throw all exceptions
    }
};

// Converts to a Xerces element recursively.
void XercesParser::_ConvertToXerces(const DataElement* element, DOMDocument* doc, DOMElement* xercesElement) const
{
    // Create the attributes
    const DataAttribute* attribute = element->FirstAttribute();
    while (attribute)
    {
        auto name = ToXercesString(attribute->Name());
        auto value = ToXercesString(attribute->Value());

        DOMAttr* xercesAttribute = doc->createAttribute(name->str());
        xercesAttribute->setValue(value->str());
        xercesElement->setAttributeNode(xercesAttribute);

        attribute = attribute->NextSibling();
    }

    // Set text
    auto text = ToXercesString(element->Text());
    xercesElement->setTextContent(text->str());

    // Convert all children recursively
    const DataElement* child = element->FirstChild();
    while (child)
    {
        auto name = ToXercesString(child->Name());
        DOMElement* xercesChild;

        if (child->NameSpaceUri().empty())
        {
            xercesChild = doc->createElement(name->str());
        }
        else
        {
            auto nameSpaceUri = ToXercesString(child->NameSpaceUri());
            xercesChild = doc->createElementNS(nameSpaceUri->str(), name->str());
        }
        _ConvertToXerces(child, doc, xercesChild);
        xercesElement->appendChild(xercesChild);

        child = child->NextSibling();
    }
}

// Parses XML from an input source returning the root data node.
void XercesParser::_LoadXml(DataElement* element, const std::string& filePath, const InputSource& source, bool enableSchemaValidation, XMLGrammarPool* grammarPool) const
{
    try
    {
        // Create the Xerces parser
        SAXParser parser(nullptr, XMLPlatformUtils::fgMemoryManager, grammarPool);

        // Attach our error handler to the parser
        ParserHandler handler(element);
        parser.setDocumentHandler(&handler);
		parser.setErrorHandler(&handler);

        // Configure the parser for schema validation if needed
        if (enableSchemaValidation)
        {
            parser.setDoNamespaces(true);
            parser.setDoSchema(true);
            parser.setValidationScheme(SAXParser::Val_Always);
            parser.setValidationConstraintFatal(true);
            parser.setValidationSchemaFullChecking(true);

            if (grammarPool)
            {
                //Do not load any external schemas
                parser.setLoadSchema(false);
                parser.useCachedGrammarInParse(true);
            }
        }

        // Try parsing the XML
        parser.parse(source);
    }
    catch (const SAXParseException& e)
    {
		std::string path = (e.getSystemId() && *(e.getSystemId())) ? (FromXercesString(e.getSystemId())) : (filePath);
        std::string message = FromXercesString(e.getMessage());
        throw XmlLoadException(message, path, (size_t)e.getLineNumber());
    }
    catch (const SAXException& e)
    {
        std::string message = FromXercesString(e.getMessage());
        throw XmlLoadException(message, filePath, 0);
    }
    catch (const XMLException& e)
    {
		std::string path = (e.getSrcFile()) ? (e.getSrcFile()) : (filePath);
        std::string message = FromXercesString(e.getMessage());
        throw XmlLoadException(message, path, (size_t)e.getSrcLine());
    }
    catch (const DOMException& e)
    {
        std::string message = FromXercesString(e.getMessage());
        throw XmlLoadException(message, filePath, 0);
    }
    catch (...)
    {
        throw XmlLoadException("An unknown error occurred");
    }
}

void XercesParser::_SaveXml(const DataElement* element, XMLFormatTarget& target) const
{
    try
    {
        // Create the DOM implementation
        XMLCh tempStr[3] = { chLatin_L, chLatin_S, chNull };
        DOMImplementation* impl = DOMImplementationRegistry::getDOMImplementation(tempStr);

        // Create the document to build on
        auto nameSpaceUri = ToXercesString(element->NameSpaceUri());
        auto rootElementName = ToXercesString(element->Name());
        std::unique_ptr<DOMDocument> doc(impl->createDocument(nameSpaceUri->str(), rootElementName->str(), nullptr));

        // Convert to Xerces
        _ConvertToXerces(element, doc.get(), doc->getDocumentElement());

        // Create and configure the serializer
        std::unique_ptr<DOMLSSerializer> serializer(((DOMImplementationLS*)impl)->createLSSerializer());
        DOMConfiguration* serializerConfig = serializer->getDomConfig();
        if (serializerConfig->canSetParameter(XMLUni::fgDOMWRTFormatPrettyPrint, true))
        {
            serializerConfig->setParameter(XMLUni::fgDOMWRTFormatPrettyPrint, true);
        }

        // Create and configure the output
        std::unique_ptr<DOMLSOutput> output(((DOMImplementationLS*)impl)->createLSOutput());
        auto encoding = ToXercesString("utf-8");
        output->setEncoding(encoding->str());
        output->setByteStream(&target);

        // Write the XML
        serializer->write(doc.get(), output.get());
    }
    catch (const SAXParseException& e)
    {
        std::string message = FromXercesString(e.getMessage());
        throw XmlSaveException(message);
    }
    catch (const SAXException& e)
    {
        std::string message = FromXercesString(e.getMessage());
        throw XmlSaveException(message);
    }
    catch (const XMLException& e)
    {
        std::string message = FromXercesString(e.getMessage());
        throw XmlSaveException(message);
    }
    catch (const DOMException& e)
    {
        std::string message = FromXercesString(e.getMessage());
        throw XmlSaveException(message);
    }
    catch (...)
    {
        throw XmlSaveException("An unknown error occurred");
    }
}

// xerces overrides the 'new' operator for XMLGrammarPoolImpl (using XMemory::operator new),
// which does not support our override that detects memory leaks
#include <PushNewOverride.h>
XercesGrammarPool::XercesGrammarPool()
    : GrammarPool()
    , xercesPool(new XMLGrammarPoolImpl())
{
    //Keep the pool locked unless we're modifying it
    xercesPool->lockPool();
}
#include <PopNewOverride.h>

XercesGrammarPool::~XercesGrammarPool()
{
}

void XercesGrammarPool::LoadSchemaDirectory(const std::string& directoryPath)
{
    std::vector<std::string> filePaths;

    using boost::filesystem::recursive_directory_iterator;
    for (auto it = recursive_directory_iterator(directoryPath); it != recursive_directory_iterator(); ++it)
    {
        const boost::filesystem::path& path = it->path();

        if (!boost::filesystem::is_regular_file(path))
        {
            continue;
        }

        filePaths.push_back(path.string());
    }

    //Make sure to unlock the pool just long enough to cache the grammar files
    PoolUnlocker lock(GetXercesGrammarPool());
    ParseGrammarContent(filePaths);
}

void XercesGrammarPool::LoadSchemaFile(const std::string& filePath)
{
    std::vector<std::string> filePaths = { filePath };

    //Make sure to unlock the pool just long enough to cache the grammar file
    PoolUnlocker lock(GetXercesGrammarPool());
    ParseGrammarContent(filePaths);
}

void XercesGrammarPool::ParseGrammarContent(const std::vector<std::string>& filePaths)
{
    std::string currentFilePath;
    try
    {
        // Create the Xerces parser
        SAXParser parser(nullptr, XMLPlatformUtils::fgMemoryManager, GetXercesGrammarPool());

        // Attach our error handler to the parser
        ParserHandler handler(nullptr);
        parser.setErrorHandler(&handler);

        // Configure the parser for schema validation
        parser.setDoNamespaces(true);
        parser.setDoSchema(true);
        parser.setValidationScheme(SAXParser::Val_Always);
        parser.setValidationConstraintFatal(true);
        parser.setValidationSchemaFullChecking(true);
        parser.setHandleMultipleImports(true);

        // Try parsing the schema files
        for (const auto& file : filePaths)
        {
            currentFilePath = file;
            auto path = ToXercesString(currentFilePath);
            LocalFileInputSource source(path->str());
            parser.loadGrammar(source, Grammar::GrammarType::SchemaGrammarType, true);
        }
    }
    catch (const SAXParseException& e)
    {
        std::string message = FromXercesString(e.getMessage());
        throw XmlLoadException(message, currentFilePath, (size_t)e.getLineNumber());
    }
    catch (const SAXException& e)
    {
        std::string message = FromXercesString(e.getMessage());
        throw XmlLoadException(message, currentFilePath, 0);
    }
    catch (const XMLException& e)
    {
        std::string message = FromXercesString(e.getMessage());
        throw XmlLoadException(message, currentFilePath, (size_t)e.getSrcLine());
    }
    catch (const DOMException& e)
    {
        std::string message = FromXercesString(e.getMessage());
        throw XmlLoadException(message, currentFilePath, 0);
    }
    catch (...)
    {
        throw XmlLoadException("An unknown error occurred");
    }
}

void XercesParser::LoadXmlFile(DataDocument* doc, const std::string& filePath, bool enableSchemaValidation) const
{
    auto path = ToXercesString(filePath);
    LocalFileInputSource source(path->str());
    _LoadXml(doc->RootElement(), filePath, source, enableSchemaValidation, nullptr);
}

void XercesParser::LoadXmlFile(DataDocument* doc, const std::string& filePath, const GrammarPool& grammarPool) const
{
    auto path = ToXercesString(filePath);
    LocalFileInputSource source(path->str());

    try
    {
        const XercesGrammarPool& pool = dynamic_cast<const XercesGrammarPool&>(grammarPool);
        _LoadXml(doc->RootElement(), filePath, source, true, pool.GetXercesGrammarPool());
    }
    catch (std::bad_cast&)
    {
        throw XmlLoadException("The grammar pool is of the wrong type. This is a bug.");
    }
}

void XercesParser::LoadXmlString(DataDocument* doc, const std::string& xml) const
{
    MemBufInputSource source((XMLByte*)xml.c_str(), xml.size(), "");
    _LoadXml(doc->RootElement(), "", source, false, nullptr);
}

void XercesParser::SaveXmlFile(const DataDocument* doc, const std::string& filePath) const
{
    LocalFileFormatTarget target(filePath.c_str());
    _SaveXml(doc->RootElement(), target);
}

std::string XercesParser::SaveXmlString(const DataDocument* doc) const
{
    MemBufFormatTarget target;
    _SaveXml(doc->RootElement(), target);
    return std::string((const char*)target.getRawBuffer());
}
