/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#include "CProbePlugin.hpp"

#include "JtagEncoding.h"
#include "I2cEncoding.h"
#include "MetaEncoding.h"

#include "SimpleTapStateMachine.h"
#include "ProbePlugin.h"
#include "InterfaceScanBulkOperations.h"
#include "TraceOperations.h"
#include "BundleOperations.h"
#include "ProbePluginErrors.h"
#include "PluginLogger.h"

#include <Components/Common/Common.h>
#include <Foundation/BitData/BitData.h>
#include <ExceptionBridge.h>

#include <MemoryCopy.h>

#include <thread>
#include <iostream>
#include <vector>
#include <unordered_map>
#include <algorithm>
#include <stringcopy.h> // For InternalUtils::stringcopy
#include <cstring>

// Error contexts arranged by thread id.
thread_local ErrorContext* _threadErrorContext = nullptr;

// Last error message by thread id. Note this list will leak slightly as the number of threads grows.
thread_local std::pair<OpenIPC_Error, std::string> _threadLastErrorMessages;

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
// PPI_TriggerResponse Methods
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

OpenIPC_Error PPI_InterfaceTriggerResponse::GetObservable(bool& value) const
{
    value = this->observable;
    return OpenIPC_Error_No_Error;
}

OpenIPC_Error PPI_InterfaceTriggerResponse::GetObserved(bool& value)  const
{
    value = this->observed;
    return OpenIPC_Error_No_Error;
}

OpenIPC_Error PPI_InterfaceTriggerResponse::SetObserved(bool value)
{
    this->observed = value;
    return OpenIPC_Error_No_Error;
}

OpenIPC_Error PPI_InterfaceTriggerResponse::GetIsEnabled(bool& enabled) const
{
    enabled = this->isSet;
    return OpenIPC_Error_No_Error;
}

OpenIPC_Error PPI_InterfaceTriggerResponse::SetIsEnabled(bool enabled)
{
    this->isSet = enabled;
    return OpenIPC_Error_No_Error;
}

OpenIPC_Error PPI_InterfaceTriggerResponse::GetTriggerInfo(PPI_Pins_TypeEncode& pin, bool& triggerOnAssertValue) const
{
    pin = this->trigger;
    triggerOnAssertValue = this->triggerOnAssert;
    return OpenIPC_Error_No_Error;
}

OpenIPC_Error PPI_InterfaceTriggerResponse::GetResponseInfo(PPI_Pins_TypeEncode& pin, bool& respondWithAssertValue) const
{
    pin = this->response;
    respondWithAssertValue = this->respondWithAssert;
    return OpenIPC_Error_No_Error;
}

OpenIPC_Error PPI_InterfaceTriggerResponse::GetIdentifier(uint32_t& ident) const
{
    ident = this->identifier;
    return OpenIPC_Error_No_Error;
}

void PPI_Action::_ParseParameter(std::string input, size_t count, std::string& param)
{
    param = input;
}

void PPI_Action::_ParseParameter(std::string input, size_t count, uint32_t& param)
{
    try
    {
        param = DeviceSettings::ParseUint32(input);
    }
    catch (std::out_of_range& e)
    {
        PPI_EXCEPT_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, OpenIPC_Error_Invalid_Argument, "Error parsing parameter " << count << ": \"" << e.what() << "\".");
    }
    catch (std::invalid_argument& e)
    {
        PPI_EXCEPT_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, OpenIPC_Error_Invalid_Argument, "Error parsing parameter " << count << ": \"" << e.what() << "\".");
    }
}

PPI_Action::PPI_Action(const std::string& name, const std::string& description, bool isVisible)
    : _name(name)
    , _description(description)
    , _input("")
    , _isVisible(isVisible)
{
}
const std::string& PPI_Action::Name() const
{
    return this->_name;
}

const std::string& PPI_Action::GetDescription() const
{
    return this->_description;
}

std::string PPI_Action::Run() const
{
    std::string result("Action has done nothing; it needs to be subclassed for anything interesting to happen! ");
    result += this->_input;
    return result;
}

void PPI_Action::SetParameter(const std::string& input)
{
    this->_input = input;
}

bool PPI_Action::IsVisible() const
{
    return this->_isVisible;
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
// IntefaceSettings functions
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////


const std::string DeviceSettings::EmptyValue;
const std::unordered_set<std::string> DeviceSetting::BoolOptions{ "true", "false" };


bool DeviceSettings::ParseBool(std::string const& value)
{
    if (!value.empty())
    {
        if (value == "true" || value == "True" || value == "TRUE" || value == "1")
        {
            return true;
        }
        else if (value == "false" || value == "False" || value == "FALSE" || value == "0")
        {
            return false;
        }
    }
    throw std::invalid_argument("Not a valid boolean value.  Valid values are \"true\", \"True\", \"TRUE\", \"1\", \"false\", \"False\", \"FALSE\", and \"0\".");
}
double DeviceSettings::ParseDouble(std::string const& value)
{
    return std::stod(value);
}
int DeviceSettings::ParseInt(std::string const& value)
{
    size_t numCharsProcessed = 0;
    int tempVal = 0;
    try
    {
        tempVal = std::stoi(value, &numCharsProcessed, 10);
        if (numCharsProcessed == value.length())
        {
            return tempVal;
        }
    }
    catch (std::invalid_argument)
    {
    }
    try
    {
        tempVal = std::stoi(value, &numCharsProcessed, 16);
        if (numCharsProcessed == value.length())
        {
            return tempVal;
        }
    }
    catch (std::invalid_argument)
    {
    }
    throw std::invalid_argument("Could not interpret value has decimal or hexidecimal integer.");
}
uint32_t DeviceSettings::ParseUint32(std::string const& value)
{
    size_t numCharsProcessed = 0;
    unsigned long tempVal = 0;
    bool success = false;

    // We don't let std::stoul auto-detect the base because we don't
    // want to support octal, as it could cause confusion to users (e.g. 0255 would be interpreted as 173)

    // First try to parse it as a decimal
    try
    {
        tempVal = std::stoul(value, &numCharsProcessed, 10);

        if (numCharsProcessed == value.length())
        {
            success = true;
        }
    }
    catch (std::invalid_argument)
    {
    }

    if (!success)
    {
        // Now try to parse it as hex
        try
        {
            numCharsProcessed = 0;
            tempVal = std::stoul(value, &numCharsProcessed, 16);

            if (numCharsProcessed == value.length())
            {
                success = true;
            }
        }
        catch (std::invalid_argument)
        {
        }
    }

    if (!success)
    {
        throw std::invalid_argument("Could not interpret value has decimal or hexidecimal integer.");
    }

    if (tempVal > std::numeric_limits<uint32_t>::max() && tempVal < std::numeric_limits<decltype(tempVal)>::max())
    {
        throw std::out_of_range("Value is larger than allowed (max = 2^32-1).");
    }

    return static_cast<uint32_t>(tempVal);
}
std::vector<std::string> DeviceSettings::ParseList(std::string const& value)
{
    std::vector<std::string> list;
    std::stringstream stream(value);
    std::string item;
    while (stream >> item)
    {
        if (!item.empty())
        {
            list.emplace_back(item);
        }
    }
    return list;
}
bool DeviceSettings::TryParseBool(std::string const& value, bool& dest)
{
    return TryParseT<bool>(&ParseBool, value, dest);
}
bool DeviceSettings::TryParseDouble(std::string const& value, double& dest)
{
    return TryParseT<double>(&ParseDouble, value, dest);
}
bool DeviceSettings::TryParseInt(std::string const& value, int& dest)
{
    return TryParseT<int>(&ParseInt, value, dest);
}
bool DeviceSettings::TryParseUint32(std::string const& value, uint32_t& dest)
{
    return TryParseT<uint32_t>(&ParseUint32, value, dest);
}
bool DeviceSettings::TryParseList(std::string const& value, std::vector<std::string>& dest)
{
    return TryParseT<std::vector<std::string>>(&ParseList, value, dest);
}

DeviceSettings::DeviceSettings()
{
}

void DeviceSettings::Load(Map::const_iterator const& begin, Map::const_iterator const& end)
{
    std::lock_guard<std::recursive_mutex> _(lock);
    for(auto iter = begin; iter != end; ++iter)
    {
        AddKeyIfNotExist(iter->first);
        _map[iter->first] = iter->second;
    }
}

void DeviceSettings::Set(std::string const& key, std::string value)
{
    std::lock_guard<std::recursive_mutex> _(lock);
    AddKeyIfNotExist(key);

    auto it = _map.find(key);
    if(it == _map.end())
    {
        // New value, just add it for backwards compatibility. But callers should use Add to add new settings
        _map[key] = DeviceSetting(std::move(value));
        return;
    }

    auto& item = it->second;
    if (!item.Options.empty())
    {
        if (item.Options.find(value) == item.Options.end())
        {
            // Item was not in the options, so don't set it
            return;
        }
    }
    if (item.OnSet)
    {
        if(!item.OnSet(item, value))
        {
            // Set was rejected by callback
            return;
        }
    }
    item.Value = std::move(value);
}

void DeviceSettings::Add(std::string const& key, DeviceSetting value)
{
    std::lock_guard<std::recursive_mutex> _(lock);
    AddKeyIfNotExist(key);
    _map[key] = std::move(value);
}

void DeviceSettings::Remove(std::string const& key)
{
    std::lock_guard<std::recursive_mutex> _(lock);
    auto iter = _map.find(key);
    if (iter != _map.end())
    {
        _map.erase(iter);
    }
    RemoveKeyIfExists(key);
}

bool DeviceSettings::TryGetString(std::string const& key, std::string& dest) const
{
    std::lock_guard<std::recursive_mutex> _(lock);
    auto found = _map.find(key);
    if (found == _map.end())
    {
        return false;
    }
    else
    {
        dest = found->second.Value;
        return true;
    }
}
bool DeviceSettings::TryGetBool(std::string const& key, bool& dest) const
{
    std::lock_guard<std::recursive_mutex> _(lock);
    std::string const& value = GetString(key);
    return TryParseT<bool>(&ParseBool,value, dest);
}
bool DeviceSettings::TryGetDouble(std::string const& key, double& dest) const
{
    std::lock_guard<std::recursive_mutex> _(lock);
    std::string const& value = GetString(key);
    return TryParseT<double>(&ParseDouble, value, dest);
}

bool DeviceSettings::TryGetInt(std::string const& key, int& dest) const
{
    std::lock_guard<std::recursive_mutex> _(lock);
    std::string const& value = GetString(key);
    return TryParseT<int>(&ParseInt, value, dest);
}
bool DeviceSettings::TryGetUint32(std::string const& key, uint32_t& dest) const
{
    std::lock_guard<std::recursive_mutex> _(lock);
    std::string const& value = GetString(key);
    return TryParseT<uint32_t>(&ParseUint32, value, dest);
}

bool DeviceSettings::TryGetStringList(std::string const& key, std::vector<std::string>& dest) const
{
    std::lock_guard<std::recursive_mutex> _(lock);
    std::string const& value = GetString(key);
    return TryParseT<std::vector<std::string>>(&ParseList, value, dest);
}
bool DeviceSettings::TryGetBoolList(std::string const& key, std::vector<bool>& dest) const
{
    std::lock_guard<std::recursive_mutex> _(lock);
    std::vector<std::string> values = GetStringList(key);
    std::vector<bool> temp(values.size());
    for (size_t i = 0; i < values.size(); ++i)
    {
        //Self documenting code!
        bool stupidVecorBoolSpecializationDoesNotLetYouTakeAReference = false;
        if (!TryParseT<bool>(&ParseBool, values[i], stupidVecorBoolSpecializationDoesNotLetYouTakeAReference))
        {
            return false;
        }
        temp[i] = stupidVecorBoolSpecializationDoesNotLetYouTakeAReference;
    }
    dest = std::move(temp);
    return true;
}
bool DeviceSettings::TryGetDoubleList(std::string const& key, std::vector<double>& dest) const
{
    std::lock_guard<std::recursive_mutex> _(lock);
    return TryParseListT<double>(&ParseDouble, GetString(key), dest);
}
bool DeviceSettings::TryGetIntList(std::string const& key, std::vector<int>& dest) const
{
    std::lock_guard<std::recursive_mutex> _(lock);
    return TryParseListT<int>(&ParseInt, GetString(key), dest);
}
bool DeviceSettings::TryGetUint32List(std::string const& key, std::vector<uint32_t>& dest) const
{
    std::lock_guard<std::recursive_mutex> _(lock);
    return TryParseListT<uint32_t>(&ParseUint32, GetString(key), dest);
}

const std::string& DeviceSettings::GetString(std::string const& key) const
{
    static const std::string emptyString;
    std::lock_guard<std::recursive_mutex> _(lock);
    auto found = _map.find(key);
    if (found == _map.end())
    {
        return emptyString;
    }
    else
    {
        return found->second.Value;
    }
}

bool DeviceSettings::GetBool(std::string const& key, bool defaultValue) const
{
    TryGetBool(key, defaultValue);
    return defaultValue;
}

double DeviceSettings::GetDouble(std::string const& key,double defaultValue) const
{
    TryGetDouble(key, defaultValue);
    return defaultValue;
}

int32_t DeviceSettings::GetInt(std::string const& key, int defaultValue) const
{
    TryGetInt(key, defaultValue);
    return defaultValue;
}

uint32_t DeviceSettings::GetUint32(std::string const& key, uint32_t defaultValue) const
{
    TryGetUint32(key, defaultValue);
    return defaultValue;
}
std::vector<std::string> DeviceSettings::GetStringList(std::string const& key, std::vector<std::string> defaultValue) const
{
    TryGetStringList(key, defaultValue);
    return defaultValue;
}
std::vector<bool> DeviceSettings::GetBoolList(std::string const& key, std::vector<bool> defaultValue) const
{
    TryGetBoolList(key, defaultValue);
    return defaultValue;
}

std::vector<double> DeviceSettings::GetDoubleList(std::string const& key, std::vector<double> defaultValue) const
{
    TryGetDoubleList(key, defaultValue);
    return defaultValue;
}

std::vector<int32_t> DeviceSettings::GetIntList(std::string const& key, std::vector<int> defaultValue) const
{
    TryGetIntList(key, defaultValue);
    return defaultValue;
}

std::vector<uint32_t> DeviceSettings::GetUint32List(std::string const& key, std::vector<uint32_t> defaultValue) const
{
    TryGetUint32List(key, defaultValue);
    return defaultValue;
}

void DeviceSettings::RemoveKeyIfExists(const std::string& key)
{
    for (auto i = this->keys.begin(); i != this->keys.end(); ++i)
    {
        if (**i == key)
        {
            keys.erase(i);
            break;
        }
    }
}

void DeviceSettings::AddKeyIfNotExist(const std::string& key)
{
    bool found = false;
    for (auto i = this->keys.begin(); i != this->keys.end() && !found; ++i)
    {
        if (**i == key)
        {
            found = true;
        }
    }
    if (!found)
    {
        keys.push_back(std::unique_ptr<std::string>(new std::string(key)));
    }
}

std::string& DeviceSettings::operator[](std::string const& key)
{
    std::lock_guard<std::recursive_mutex> _(lock);
    AddKeyIfNotExist(key);
    return _map[key].Value;
}

DeviceSettings::Map::iterator DeviceSettings::find(std::string const& key)
{
    std::lock_guard<std::recursive_mutex> _(lock);
    return _map.find(key);
}

void DeviceSettings::erase(std::string const& key)
{
    std::lock_guard<std::recursive_mutex> _(lock);
    _map.erase(key);
    RemoveKeyIfExists(key);
}

void DeviceSettings::clear()
{
    std::lock_guard<std::recursive_mutex> _(lock);
    _map.clear();
    keys.clear();
}

size_t DeviceSettings::size()
{
    std::lock_guard<std::recursive_mutex> _(lock);
    return _map.size();
}

DeviceSettings::Map::iterator DeviceSettings::begin()
{
    std::lock_guard<std::recursive_mutex> _(lock);
    return _map.begin();
}

DeviceSettings::Map::iterator DeviceSettings::end()
{
    std::lock_guard<std::recursive_mutex> _(lock);
    return _map.end();
}

DeviceSettings::Map::const_iterator DeviceSettings::begin() const
{
    std::lock_guard<std::recursive_mutex> _(lock);
    return _map.begin();
}
DeviceSettings::Map::const_iterator DeviceSettings::cbegin() const
{
    std::lock_guard<std::recursive_mutex> _(lock);
    return _map.cbegin();
}

DeviceSettings::Map::const_iterator DeviceSettings::end() const
{
    std::lock_guard<std::recursive_mutex> _(lock);
    return _map.end();
}

DeviceSettings::Map::const_iterator DeviceSettings::cend() const
{
    std::lock_guard<std::recursive_mutex> _(lock);
    return _map.cend();
}


DeviceSettings::Keys::iterator DeviceSettings::keyBegin()
{
    return this->keys.begin();
}

/**
* Calls end on the underlying map.
*/
DeviceSettings::Keys::iterator DeviceSettings::keyEnd()
{
    return this->keys.end();
}

/**
* Calls begin on the underlying map.
*/
DeviceSettings::Keys::const_iterator DeviceSettings::keyBegin() const
{
    return this->keys.begin();
}

/**
* Calls cbegin on the underlying map.
*/
DeviceSettings::Keys::const_iterator DeviceSettings::keyCbegin() const
{
    return this->keys.cbegin();
}

/**
* Calls end on the underlying map.
*/
DeviceSettings::Keys::const_iterator DeviceSettings::keyEnd() const
{
    return this->keys.end();
}

/**
* Calls cend on the underlying map.
*/
DeviceSettings::Keys::const_iterator DeviceSettings::keyCend() const
{
    return this->keys.cend();
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
// IInterfaceBundle functions
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

IInterfaceBundle::IInterfaceBundle(OpenIPC_DeviceId interfaceDeviceId)
    :interfaceDeviceId(interfaceDeviceId)
{
}

IInterfaceBundle::~IInterfaceBundle()
{
}

bool IInterfaceBundle::operator==(IInterfaceBundle const& rhs)
{
    return this == &rhs;
}
bool IInterfaceBundle::operator!=(IInterfaceBundle const& rhs)
{
    return this != &rhs;
}

class SpecialInterfaceBundle : public IInterfaceBundle
{
public:
    SpecialInterfaceBundle()
        :IInterfaceBundle(OpenIPC_INVALID_DEVICE_ID)
    {
    }

    virtual ~SpecialInterfaceBundle()
    {
    }

    OpenIPC_Error Append(IInterfaceBundle& source, uint8_t* outputBuffer, uint32_t outputBufferSize) override
    {
        return OpenIPC_Error_Not_Implemented;
    }


    static SpecialInterfaceBundle Default;
    static SpecialInterfaceBundle ExecuteImmediately;

};

SpecialInterfaceBundle SpecialInterfaceBundle::Default;
SpecialInterfaceBundle SpecialInterfaceBundle::ExecuteImmediately;

IInterfaceBundle& IInterfaceBundle::Default(SpecialInterfaceBundle::Default);
IInterfaceBundle& IInterfaceBundle::ExecuteImmediately(SpecialInterfaceBundle::ExecuteImmediately);


//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
// ProbePluginBundle functions
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

ProbePluginBundle::ProbePluginBundle()
    :_interfaceBundle(&IInterfaceBundle::Default)
{
}

ProbePluginBundle::~ProbePluginBundle()
{
    Clear();
    _interfaceBundle.release();
}

OpenIPC_Error ProbePluginBundle::Clear()
{
    if (_interfaceBundle.get() != &IInterfaceBundle::Default)
    {
        _interfaceBundle.reset(&IInterfaceBundle::Default);
    }
    return OpenIPC_Error_No_Error;
}

OpenIPC_Error ProbePluginBundle::Append(ProbePluginBundle& source, uint8_t* outputBuffer, uint32_t outputBufferSize)
{
    if (_interfaceBundle.get() == &IInterfaceBundle::Default)
    {
        return OpenIPC_Error_Invalid_Device_ID;
    }
    else
    {
        return _interfaceBundle->Append(source.InterfaceBundle(), outputBuffer, outputBufferSize);
    }
}

IInterfaceBundle& ProbePluginBundle::InterfaceBundle()
{
    return *_interfaceBundle;
}

void ProbePluginBundle::InterfaceBundle(std::unique_ptr<IInterfaceBundle>&& bundle)
{
    if (bundle.get() == &IInterfaceBundle::ExecuteImmediately)
    {
        Clear();
    }
    else
    {
        if (_interfaceBundle.get() == &IInterfaceBundle::Default)
        {
            _interfaceBundle.release();
        }
        _interfaceBundle = std::move(bundle);
    }
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
// Helper class functions
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

ProbeInstance::ProbeInstance(uint32_t probe_refid, std::string typeString, uint32_t typeID)
    : refid(probe_refid), typid(typeID), typestring(typeString), deviceID(OpenIPC_INVALID_DEVICE_ID), initializationState(InitializationState::Uninitialized)
{
}


OpenIPC_Error ProbeInstance::AddInterface(std::weak_ptr<InterfaceInstance> probeinterface)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    std::lock_guard<std::mutex> lock(_refid2interfaceMutex);

    std::shared_ptr<InterfaceInstance> strongRef = probeinterface.lock();
    strongRef->probe = this;
    _refid2interface[strongRef->refid] = probeinterface;

    return openIPCError;
}

OpenIPC_Error ProbeInstance::GetInterfaces(std::vector<std::weak_ptr<InterfaceInstance>>& interfaces)
{
    return OpenIPC_Error_Not_Implemented;
}

OpenIPC_Error ProbeInstance::GetInterfaceRefids(std::vector<uint32_t>& refids)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    std::lock_guard<std::mutex> lock(_refid2interfaceMutex);

    refids.clear();
    for(auto iter = _refid2interface.cbegin(); iter != _refid2interface.cend(); iter++)
    {
        refids.push_back((*iter).first);
    }

    return openIPCError;
}

OpenIPC_Error ProbeInstance::GetInterfaceFromRefid(uint32_t interfacerefid, std::weak_ptr<InterfaceInstance>& probeinterface)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    std::lock_guard<std::mutex> lock(_refid2interfaceMutex);

    const auto iter =  _refid2interface.find(interfacerefid);
    if (iter != _refid2interface.end())
    {
        probeinterface = (*iter).second;
    }
    else
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
    }

    return openIPCError;
}

InterfaceInstance::InterfaceInstance(uint32_t interface_refid, PPI_EInterfaceType interface_type, uint32_t instance_id)
    : refid(interface_refid), interfaceType(interface_type), instanceid(instance_id), deviceID(OpenIPC_INVALID_DEVICE_ID),lock(false), probe(nullptr), initializationState(InitializationState::Uninitialized)
{
}

InterfaceInstance::~InterfaceInstance()
{
}

OpenIPC_Error InterfaceInstance::OperationCancel()
{
    return OpenIPC_Error_Operation_Not_Supported;
}

std::unique_ptr<IInterfaceBundle> InterfaceInstance::BundleAllocate()
{
    return std::unique_ptr<IInterfaceBundle>();
}

OpenIPC_Error InterfaceInstance::BundleExecute(IInterfaceBundle& bundle, bool keepLock)
{
    return OpenIPC_Error_Operation_Not_Supported;
}

void InterfaceInstanceJtag::SettingsUpdated()
{
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
// base class public virtual functions
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

OpenIPC_Error CProbePlugin::PostError(OpenIPC_Error errorNumber, const char* errorMessage){
    PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, errorNumber, errorMessage);

    return errorNumber;
}

const char* CProbePlugin::GetLastErrorMessage(OpenIPC_Error errorNumber)
{
    return _threadLastErrorMessages.second.c_str();
}

OpenIPC_Error CProbePlugin::PluginSetNotificationRequiredForLogCallBack(PPI_NotificationLevel severity)
{
    PluginLogger::GetInstance().Level(severity);

    return OpenIPC_Error_No_Error;
}

OpenIPC_Error CProbePlugin::PluginSetNotificationRequiredForNotificationCallBack(PPI_NotificationLevel severity)
{
    PluginNotifications::GetInstance().Level(severity);

    return OpenIPC_Error_No_Error;
}

//////////////////////////////////////////////////////////////////////////////
// InterfaceGetRefIds
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::InterfaceGetRefIds(
    OpenIPC_DeviceId probeID,
    uint8_t maxIds,
    PPI_RefId* refIds,
    uint32_t* interfacecount
    )
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
    *interfacecount = 0;

    std::lock_guard<std::recursive_mutex> lock(_pluginMutex);

    std::shared_ptr<ProbeInstance> probe;
    openIPCError = _LookupProbe(probeID, probe);
    if (OpenIPC_PASS(openIPCError))
    {
        std::vector<uint32_t> interfacerefids;
        openIPCError = probe->GetInterfaceRefids(interfacerefids);
        for (uint32_t i = 0; (i < maxIds) && (i < interfacerefids.size()); i++)
        {
            refIds[i] = interfacerefids[i];
        }
        *interfacecount = (uint32_t)interfacerefids.size();
    }
    else
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
    }

    return openIPCError;
}

//////////////////////////////////////////////////////////////////////////////
// InterfaceGetInfo
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::InterfaceGetType(
    OpenIPC_DeviceId probeID,
    PPI_RefId interface_refid,
    PPI_EInterfaceType* interface_type
    )
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    std::lock_guard<std::recursive_mutex> lock(_pluginMutex);

    std::shared_ptr<ProbeInstance> probe;
    openIPCError = _LookupProbe(probeID, probe);
    if (OpenIPC_PASS(openIPCError))
    {
        std::weak_ptr<InterfaceInstance> probeinterface_ref;
        openIPCError = probe->GetInterfaceFromRefid(interface_refid, probeinterface_ref);
        auto probeinterface = probeinterface_ref.lock();
        if (OpenIPC_PASS(openIPCError) && (probeinterface))
        {
            *interface_type = probeinterface->interfaceType;
        }
    }
    else
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
    }

    return openIPCError;
}

OpenIPC_Error CProbePlugin::InterfaceGetInfoJTAG(OpenIPC_DeviceId probeID, PPI_RefId interface_refid, PPI_InterfaceJTAGCapabilities* capabilities)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    std::lock_guard<std::recursive_mutex> lock(_pluginMutex);
    std::shared_ptr<ProbeInstance> probe;
    openIPCError = _LookupProbe(probeID, probe);
    if (OpenIPC_PASS(openIPCError))
    {
        std::weak_ptr<InterfaceInstance> probeinterface_ref;
        openIPCError = probe->GetInterfaceFromRefid(interface_refid, probeinterface_ref);
        // probeinterface will be an empty shared_ptr if either
        // 1. the probeinterface_ref weak_ptr points to nothign
        // 2. the dynamic_pointer_cast downcast fails.
        auto probeinterface = std::dynamic_pointer_cast<InterfaceInstanceJtag>(probeinterface_ref.lock());
        if (OpenIPC_PASS(openIPCError) && probeinterface)
        {
            capabilities->supportPauDRInRegisterInterface = probeinterface->capabilities.supportPauDRInRegisterInterface;
            capabilities->supportTLR = probeinterface->capabilities.supportTLR;
            capabilities->supportTRST = probeinterface->capabilities.supportTRST;
        }
        else if (!probeinterface)
        {
            openIPCError = OpenIPC_Error_Invalid_Device_ID;
            PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
        }
    }
    else
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
    }

    return openIPCError;
}

OpenIPC_Error CProbePlugin::InterfaceGetInfoPins(OpenIPC_DeviceId probeID, PPI_RefId interface_refid, PPI_InterfacePinsCapabilities* capabilities)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    std::lock_guard<std::recursive_mutex> lock(_pluginMutex);
    std::shared_ptr<ProbeInstance> probe;
    openIPCError = _LookupProbe(probeID, probe);
    if (OpenIPC_PASS(openIPCError))
    {
        std::weak_ptr<InterfaceInstance> probeinterface_ref;
        openIPCError = probe->GetInterfaceFromRefid(interface_refid, probeinterface_ref);
        auto probeinterface = std::dynamic_pointer_cast<InterfaceInstancePins>(probeinterface_ref.lock());
        if (OpenIPC_PASS(openIPCError) && (probeinterface))
        {
            // Nothing to copy
        }
        else if (probeinterface == nullptr)
        {
            openIPCError = OpenIPC_Error_Invalid_Device_ID;
            PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
        }
    }
    else
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
    }

    return openIPCError;
}
OpenIPC_Error CProbePlugin::InterfaceGetInfoI2C(OpenIPC_DeviceId probeID, PPI_RefId interface_refid, PPI_InterfaceI2CCapabilities* capabilities)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    std::lock_guard<std::recursive_mutex> lock(_pluginMutex);
    std::shared_ptr<ProbeInstance> probe;
    openIPCError = _LookupProbe(probeID, probe);
    if (OpenIPC_PASS(openIPCError))
    {
        std::weak_ptr<InterfaceInstance> probeinterface_ref;
        openIPCError = probe->GetInterfaceFromRefid(interface_refid, probeinterface_ref);
        auto probeinterface = std::dynamic_pointer_cast<InterfaceInstanceI2c>(probeinterface_ref.lock());
        if (OpenIPC_PASS(openIPCError) && probeinterface)
        {
            capabilities->minimumtimeGraularityInMicroSeconds = probeinterface->capabilities.minimumtimeGraularityInMicroSeconds;
        }
        else if (probeinterface == nullptr)
        {
            openIPCError = OpenIPC_Error_Invalid_Device_ID;
            PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
        }
    }
    else
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
    }

    return openIPCError;
}

OpenIPC_Error CProbePlugin::InterfaceBeginInitialization(OpenIPC_DeviceId probeID, PPI_RefId interface_refid, OpenIPC_DeviceId interfaceID)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    std::lock_guard<std::recursive_mutex> lock(_pluginMutex);

    std::shared_ptr<ProbeInstance> probe;
    openIPCError = _LookupProbe(probeID, probe);
    if (!(OpenIPC_PASS(openIPCError)))
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
        return openIPCError;
    }

    std::weak_ptr<InterfaceInstance> probeinterface_ref;
    openIPCError = probe->GetInterfaceFromRefid(interface_refid, probeinterface_ref);
    auto probeinterface = probeinterface_ref.lock();
    if (!(OpenIPC_PASS(openIPCError)) || !probeinterface)
    {
        return openIPCError;
    }

    if (probeinterface->initializationState != InitializationState::Uninitialized)
    {
        openIPCError = OpenIPC_Error_Interface_Already_Initialized;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
        return openIPCError;
    }

    probeinterface->deviceID = interfaceID;
    _deviceid2probeinterface[interfaceID] = probeinterface;
    probeinterface->initializationState = InitializationState::Initalizing;
    return openIPCError;
}

OpenIPC_Error CProbePlugin::InterfaceSpecifyVendorInitString(OpenIPC_DeviceId interfaceID, const PPI_char* vendorinit)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    std::lock_guard<std::recursive_mutex> lock(_pluginMutex);

    std::shared_ptr<InterfaceInstance> probeinterface;
    openIPCError = _LookupInterface(interfaceID, probeinterface);
    if (!(OpenIPC_PASS(openIPCError)) || !probeinterface)
    {
        return openIPCError;
    }

    if (probeinterface->initializationState == InitializationState::Uninitialized)
    {
        openIPCError = OpenIPC_Error_Interface_Not_Initialized;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
        return openIPCError;
    }
    if (probeinterface->initializationState == InitializationState::Initialized)
    {
        openIPCError = OpenIPC_Error_Interface_Already_Initialized;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
        return openIPCError;
    }
    return openIPCError;
}

OpenIPC_Error CProbePlugin::InterfaceFinishInitialization(OpenIPC_DeviceId interfaceID)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    std::lock_guard<std::recursive_mutex> lock(_pluginMutex);

    std::shared_ptr<InterfaceInstance> probeinterface;
    openIPCError = _LookupInterface(interfaceID, probeinterface);
    if (!(OpenIPC_PASS(openIPCError)) || !probeinterface)
    {
        return openIPCError;
    }

    if (probeinterface->initializationState != InitializationState::Initalizing)
    {
        openIPCError = OpenIPC_Error_Interface_Not_Initialized;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
        return openIPCError;
    }
    probeinterface->initializationState = InitializationState::Initialized;
    return openIPCError;
}

//////////////////////////////////////////////////////////////////////////////
// InterfaceInitialize
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::InterfaceInitialize(
    OpenIPC_DeviceId probeID,
    PPI_RefId interface_refid,
    OpenIPC_DeviceId interfaceID,
    const char* /*vendorinit*/
    )
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    std::lock_guard<std::recursive_mutex> lock(_pluginMutex);

    std::shared_ptr<ProbeInstance> probe;
    openIPCError = _LookupProbe(probeID, probe);
    if (OpenIPC_PASS(openIPCError))
    {
        std::weak_ptr<InterfaceInstance> probeinterface_ref;
        openIPCError = probe->GetInterfaceFromRefid(interface_refid, probeinterface_ref);
        auto probeinterface = probeinterface_ref.lock();
        if (OpenIPC_PASS(openIPCError) && (probeinterface))
        {
            // Check to see if the INTERFACE is ALREADY INITIALIZED
            if (probeinterface->deviceID == -1)
            {
                probeinterface->deviceID = interfaceID;
                _deviceid2probeinterface[interfaceID] = probeinterface;
                probeinterface->initializationState = InitializationState::Initialized;
            }
            else
            {
                //openIPCError = OpenIPC_Error_Interface_Already_Initialized;
                PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
            }
        }
    }
    else
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
    }

    return openIPCError;
}

//////////////////////////////////////////////////////////////////////////////
// InterfaceDeInitialize
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::InterfaceDeInitialize(
    OpenIPC_DeviceId interfaceID
    )
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    std::lock_guard<std::recursive_mutex> lock(_pluginMutex);
    std::shared_ptr<InterfaceInstance> probeinterface;
    openIPCError = _LookupInterface(interfaceID, probeinterface);
    if (OpenIPC_PASS(openIPCError))
    {
        // TBD: clean up management structures
        // If the deviceID is already -1, then we are an invalid Device ID
        if (probeinterface->deviceID == -1)
        {
            openIPCError = OpenIPC_Error_Invalid_Device_ID;
            PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
        }
        // Show we are not initialized by setting deviceID to -1
        probeinterface->deviceID = OpenIPC_INVALID_DEVICE_ID;
        probeinterface->initializationState = InitializationState::Uninitialized;
    }
    else
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
    }

    return openIPCError;
}

//////////////////////////////////////////////////////////////////////////////
// LockTargetInterface
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::LockTargetInterface(OpenIPC_DeviceId deviceInterface)
{
    std::lock_guard<std::recursive_mutex> lock(_pluginMutex);
    this->_currentLockedInterface = deviceInterface;
    this->_inLock = true;
    this->_lockedInterfaces.insert(deviceInterface);
    return OpenIPC_Error_No_Error;
}

//////////////////////////////////////////////////////////////////////////////
// ListLockedInterfaces
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::ListLockedInterfaces(uint32_t maxNumberOfInterfaces, OpenIPC_DeviceId* interfaces, uint32_t* numberOfInterfaces)
{
    std::lock_guard<std::recursive_mutex> lock(_pluginMutex);
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (OpenIPC_PASS(result) && interfaces == nullptr && maxNumberOfInterfaces != 0)
    {
        result = OpenIPC_Error_Null_Pointer;
    }
    if (OpenIPC_PASS(result) && numberOfInterfaces == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
    }
    if (OpenIPC_PASS(result))
    {
        *numberOfInterfaces = (uint32_t) this->_lockedInterfaces.size();
        uint32_t count = 0;
        for (auto& item : this->_lockedInterfaces)
        {
            if (count < maxNumberOfInterfaces)
            {
                interfaces[count] = item;
            }
        }
    }
    return result;
}

OpenIPC_Error CProbePlugin::PPI_JTAG_GetCurrentBundlePadding(PPI_ProbeBundleHandle bundle, int32_t* irPaddingNearTDI, int32_t* irPaddingNearTDO, int32_t* drPaddingNearTDI,
    int32_t* drPaddingNearTDO, PPI_bool* drValueConstantOne)
{
    auto result = OpenIPC_Error_No_Error;
    if (irPaddingNearTDI == nullptr ||
        irPaddingNearTDO == nullptr ||
        drPaddingNearTDI == nullptr ||
        drPaddingNearTDO == nullptr ||
        drValueConstantOne == nullptr)
    {
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result = OpenIPC_Error_Null_Pointer, "PPI_JTAG_GetCurrentBundlePadding has at least one parameter that is null.");
    }
    else
    {
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result = OpenIPC_Error_Not_Implemented, "PPI_JTAG_GetCurrentBundlePadding must be overridden by the plugin: " << this->_pluginname);
    }
    return result;
}

OpenIPC_Error CProbePlugin::PPI_JTAG_UpdateBundlePadding(PPI_ProbeBundleHandle bundle, int32_t irPaddingNearTDI, int32_t irPaddingNearTDO, int32_t drPaddingNearTDI,
    int32_t drPaddingNearTDO, PPI_bool drValueConstantOne)
{
    auto result = OpenIPC_Error_No_Error;
    PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result = OpenIPC_Error_Not_Implemented, "PPI_JTAG_UpdateBundlePadding must be overridden by the plugin: " << this->_pluginname);
    return result;
}

PPI_ProbeBundleHandle CProbePlugin::BundleAllocate()
{
    if (_supportBundles)
    {
        return new ProbePluginBundle();
    }
    else
    {
        return PPI_PROBE_LOCK_RELEASE;
    }
}
OpenIPC_Error CProbePlugin::BundleFree(PPI_ProbeBundleHandle* handle)
{
    if (handle == nullptr)
    {
        return OpenIPC_Error_Null_Pointer;
    }
    else
    {
        if (*handle == PPI_PROBE_LOCK_RELEASE || *handle == PPI_PROBE_LOCK_HOLD)
        {
            return OpenIPC_Error_No_Error;
        }
        delete reinterpret_cast<ProbePluginBundle*>(*handle);
        return OpenIPC_Error_No_Error;
    }
}
OpenIPC_Error CProbePlugin::BundleExecute(PPI_ProbeBundleHandle handle, OpenIPC_DeviceId deviceInterface, PPI_bool keepLock)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
    if (handle == PPI_PROBE_LOCK_HOLD || (keepLock && handle != PPI_PROBE_LOCK_RELEASE))
    {
        openIPCError = _ProbeLockHold(deviceInterface);
    }
    else if (handle == PPI_PROBE_LOCK_RELEASE)
    {
        return _ProbeLockRelease(deviceInterface);
    }

    if (OpenIPC_PASS(openIPCError) && handle != PPI_PROBE_LOCK_RELEASE && handle != PPI_PROBE_LOCK_HOLD)
    {
		auto probePluginBundle = reinterpret_cast<ProbePluginBundle*>(handle);
        std::shared_ptr<InterfaceInstance> probeinterface;
        openIPCError = _LookupInterface(deviceInterface, probeinterface);
        auto potentialSpecialBundle = dynamic_cast<SpecialInterfaceBundle*>(&probePluginBundle->InterfaceBundle());
        if (OpenIPC_PASS(openIPCError) && potentialSpecialBundle == nullptr)
        {
            openIPCError = probeinterface->BundleExecute(probePluginBundle->InterfaceBundle(), (keepLock == 0) ? (false) : (true));
        }
    }
    return openIPCError;
}
OpenIPC_Error CProbePlugin::BundleExecuteMultiChain(PPI_ProbeBundleHandle handle, OpenIPC_DeviceId* deviceInterfaces, uint32_t deviceInterfacesLength, PPI_bool keepLock)
{
    auto result = OpenIPC_Error_No_Error;
    if (OpenIPC_PASS(result) && (deviceInterfacesLength == 0 || deviceInterfaces == nullptr))
    {
        result = OpenIPC_Error_Probe_Invalid_Parameter;
        POST_ERROR_MESSAGE(result, "Multi device bundle execute received either a null deviceInterfaces parameter or a list of 0 devices to execute on.");
    }
    // For the first 2 cases, we can provide a default implementation (we likely don't care when exactly we lock the interfaces, just that they are all locked)
    if (OpenIPC_PASS(result) && (handle == PPI_PROBE_LOCK_HOLD || (keepLock && handle != PPI_PROBE_LOCK_RELEASE)))
    {
        for (uint32_t i = 0; OpenIPC_PASS(result) && i < deviceInterfacesLength; ++i)
        {
            result = _ProbeLockHold(deviceInterfaces[i]);
        }
    }
    else if (OpenIPC_PASS(result) && handle == PPI_PROBE_LOCK_RELEASE)
    {
        for (uint32_t i = 0; OpenIPC_PASS(result) && i < deviceInterfacesLength; ++i)
        {
            result = _ProbeLockRelease(deviceInterfaces[i]);
        }
    }
    if (OpenIPC_PASS(result) && handle != PPI_PROBE_LOCK_RELEASE && handle != PPI_PROBE_LOCK_HOLD)
    {
        auto probePluginBundle = reinterpret_cast<ProbePluginBundle*>(handle);
        auto potentialSpecialBundle = dynamic_cast<SpecialInterfaceBundle*>(&probePluginBundle->InterfaceBundle());
        std::vector<std::shared_ptr<InterfaceInstance>> interfaces;
        PPI_EInterfaceType interfaceType = PPI_interfaceMAX;
        for (uint32_t i = 0; OpenIPC_PASS(result) && i < deviceInterfacesLength; ++i)
        {
            std::shared_ptr<InterfaceInstance> probeinterface;
            result = _LookupInterface(deviceInterfaces[i], probeinterface);
            if (interfaces.empty())
            {
                interfaceType = probeinterface->interfaceType;
            }
            else if (OpenIPC_PASS(result) && interfaces[0]->interfaceType != interfaceType)
            {
                result = OpenIPC_Error_Probe_Invalid_JTAG_Device;
                POST_ERROR_MESSAGE(result, "Attempting to run a bundle on 2 types of interfaces in BundleExecuteMultiChain. Interfacs types " << interfaces[0]->interfaceType << " and " << interfaceType << " device id is" << deviceInterfaces[i]);
            }

            interfaces.push_back(std::move(probeinterface));
        }
        // At this point, all of the inputs have been sanity checked and we call into the probe plugin specific code
        if (OpenIPC_PASS(result) && potentialSpecialBundle == nullptr)
        {
            result = this->_BundleExecuteMultiChain(probePluginBundle, interfaces, keepLock);
        }
    }

    return result;
}
OpenIPC_Error CProbePlugin::BundleClear(PPI_ProbeBundleHandle handle)
{
    if (handle == PPI_PROBE_LOCK_RELEASE || handle == PPI_PROBE_LOCK_HOLD)
    {
        return OpenIPC_Error_No_Error;
    }
    return reinterpret_cast<ProbePluginBundle*>(handle)->Clear();
}
OpenIPC_Error CProbePlugin::BundleAppend(PPI_ProbeBundleHandle destHandle, PPI_ProbeBundleHandle sourceHandle, uint8_t* outputBuffer, uint32_t outputBufferSize)
{
    if (destHandle == PPI_PROBE_LOCK_RELEASE || destHandle == PPI_PROBE_LOCK_HOLD || sourceHandle == PPI_PROBE_LOCK_RELEASE || sourceHandle == PPI_PROBE_LOCK_HOLD)
    {
        return OpenIPC_Error_No_Error;
    }
    else
    {
        return reinterpret_cast<ProbePluginBundle*>(destHandle)->Append(*reinterpret_cast<ProbePluginBundle*>(sourceHandle), outputBuffer, outputBufferSize);
    }
}

//////////////////////////////////////////////////////////////////////////////
// InterfaceListLockInterfacePeers
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::InterfaceListLockInterfacePeers( OpenIPC_DeviceId interfaceID, uint32_t peerInterfacesLength,OpenIPC_DeviceId * peerInterfaces, uint32_t* numberOfPeerInterfaces){
    // Default implementation assumes that *every interface in the plugin* is a peer
    std::lock_guard<std::recursive_mutex> lock(_pluginMutex);
    *numberOfPeerInterfaces = (uint32_t)  this->_deviceid2probeinterface.size(); // the number of interfaces we have in the plugin
    uint32_t count = 0;
    for(auto it = this->_deviceid2probeinterface.begin(); it != this->_deviceid2probeinterface.end() && count < peerInterfacesLength;++it, count++)
    {
        auto probeinterface = it->second.lock();
        if(probeinterface)
        {
            peerInterfaces[count] = probeinterface->deviceID;
        }
    }
    return OpenIPC_Error_No_Error;
}

//////////////////////////////////////////////////////////////////////////////
// InterfaceScan
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::InterfaceScan(
    OpenIPC_DeviceId interfaceID,
    const uint32_t* input,
    uint32_t inputdwords,
    uint32_t* output,
    uint32_t maxoutputdwords,
    uint32_t* outputdwords
    )
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
    std::shared_ptr<InterfaceInstance> probeinterface;
    openIPCError = _LookupInterface(interfaceID, probeinterface);
    if (OpenIPC_PASS(openIPCError))
    {
        switch(probeinterface->interfaceType)
        {
        case PPI_interfaceTypeJtag:
            {
                std::lock_guard<std::recursive_mutex> _(probeinterface->_instanceMutex);
                openIPCError = _JtagScan(*probeinterface, input, inputdwords, output, maxoutputdwords, outputdwords);
            break;
            }
        case PPI_interfaceTypeI2c:
            {
                std::lock_guard<std::recursive_mutex> _(probeinterface->_instanceMutex);
                openIPCError = _I2cScan(*probeinterface, input, inputdwords, output, maxoutputdwords, outputdwords);
            break;
            }
        case PPI_interfaceTypePins:
            {
                std::lock_guard<std::recursive_mutex> _(probeinterface->_instanceMutex);
                openIPCError = _PinsScan(*probeinterface, input, inputdwords, output, maxoutputdwords, outputdwords);
            break;
            }
        default:
            openIPCError = OpenIPC_Error_Interface_Not_Supported;
            PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "Unsupported interface type requested (" << LOG_UINT32_DEC(probeinterface->interfaceType) << ")");
        }
    }
    else
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
    }
    return openIPCError;
}

std::unique_ptr<IInterfaceBundle> InterfaceInstanceJtag::GetNewBundle(void)
{
	return std::unique_ptr<IInterfaceBundle>(nullptr);
}

OpenIPC_Error InterfaceInstancePins::ListSupportedTriggerResponses(uint32_t length, PPI_InterfaceTriggerResponse** responses, uint32_t* triggerResponsesPopulated) const
{
    OpenIPC_Error openIpcError = OpenIPC_Error_No_Error;
    std::lock_guard<std::recursive_mutex> _(_instanceMutex);
    if (OpenIPC_PASS(openIpcError) && triggerResponsesPopulated == nullptr)
    {
        openIpcError = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID,openIpcError, "triggerResponsesPopulated is null in PPI_ListSupportedTriggerResponses.");
    }

    if (OpenIPC_PASS(openIpcError) && length == 0)
    {
        // This is ok, we want to get the size of the list (we don't care about the responses buffer)
        *triggerResponsesPopulated = static_cast<uint32_t>(this->_triggerResponsesSupported.size());
    }
    else if (OpenIPC_PASS(openIpcError) && responses == nullptr)
    {
        openIpcError = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID,openIpcError, "responses is null in PPI_ListSupportedTriggerResponses.");
    }
    else if (OpenIPC_PASS(openIpcError))
    {
        // length > 0, responses is not null and triggerResponsesPopulated is not null!
        uint32_t count = 0;
        for (auto& triggerResponse : this->_triggerResponsesSupported)
        {
            // Actionable means that we can set it
            if (count < length)
            {
                responses[count] = triggerResponse.get();
                count++;
            }
            else
            {
                break;
            }
        }
        *triggerResponsesPopulated = count;
    }

    return openIpcError;
}

OpenIPC_Error InterfaceInstancePins::ListObservableTriggerResponses(uint32_t length, PPI_InterfaceTriggerResponse** responses, uint32_t* triggerResponsesPopulated) const
{
    OpenIPC_Error openIpcError = OpenIPC_Error_No_Error;
    std::lock_guard<std::recursive_mutex> _(_instanceMutex);
    if (OpenIPC_PASS(openIpcError) && triggerResponsesPopulated == nullptr)
    {
        openIpcError = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID,openIpcError, "triggerResponsesPopulated is null in PPI_ListObservableTriggerResponses.");
    }

    if (OpenIPC_PASS(openIpcError) && length == 0)
    {
        uint32_t count = 0;
        for (auto& triggerResponse : this->_triggerResponsesSupported)
        {
            bool temp = false;
            openIpcError = triggerResponse->GetObservable(temp);
            if (!OpenIPC_PASS(openIpcError))
            {
                // bail
                break;
            }
            if (temp)
            {
                count++;
            }
        }

        // This is ok, we want to get the size of the list (we don't care about the responses buffer)
        *triggerResponsesPopulated = count;
    }
    else if (OpenIPC_PASS(openIpcError) && responses == nullptr)
    {
        openIpcError = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID,openIpcError, "responses is null in PPI_ListObservableTriggerResponses.");
    }
    else if (OpenIPC_PASS(openIpcError))
    {
        // length >0, responses is not null and triggerResponsesPopulated is not null!
        uint32_t& count = *triggerResponsesPopulated; // To avoid my own errors
        for (auto& triggerResponse : this->_triggerResponsesSupported)
        {
            bool temp = false;
            openIpcError = triggerResponse->GetObservable(temp);
            if (!OpenIPC_PASS(openIpcError) || count >= length)
            {
                // Bail!
                break;
            }
            if (temp)
            {
                responses[count] = triggerResponse.get();
                count++;
            }
        }
    }

    return openIpcError;
}

OpenIPC_Error InterfaceInstancePins::TriggerResponseIsSet(const PPI_InterfaceTriggerResponse* triggerResponse, PPI_bool* triggerResponseSet, PPI_TriggerResponseSetOptions* optionsSet) const
{
    OpenIPC_Error openIpcError = OpenIPC_Error_No_Error;
    std::lock_guard<std::recursive_mutex> _(this->_instanceMutex);
    if (OpenIPC_PASS(openIpcError) && triggerResponse == nullptr)
    {
        openIpcError = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID,openIpcError, "triggerResponse is null in PPI_TriggerResponseIsSet.");
    }

    if (OpenIPC_PASS(openIpcError) && triggerResponseSet == nullptr)
    {
        openIpcError = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID,openIpcError, "triggerResponseSet is null in PPI_TriggerResponseIsSet.");
    }

    if (OpenIPC_PASS(openIpcError))
    {
        bool temp;
        openIpcError = triggerResponse->GetIsEnabled(temp);
        *triggerResponseSet = temp ? 1 : 0;
    }
    return openIpcError;
}

OpenIPC_Error InterfaceInstancePins::TriggerResponseSet(PPI_InterfaceTriggerResponse* triggerResponse, PPI_bool triggerResponseSet, const PPI_TriggerResponseSetOptions* optionsSet)
{
    OpenIPC_Error openIpcError = OpenIPC_Error_No_Error;
    std::lock_guard<std::recursive_mutex> _(this->_instanceMutex);
    if (OpenIPC_PASS(openIpcError) && triggerResponse == nullptr)
    {
        openIpcError = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID,openIpcError, "triggerResponse is null in PPI_TriggerResponseSet.");
    }

    if (OpenIPC_PASS(openIpcError))
    {
        bool temp = triggerResponseSet != 0;
        triggerResponse->SetIsEnabled(temp);
    }
    return openIpcError;
}

InterfaceInstanceStatePort::InterfaceInstanceStatePort(uint32_t interface_refid, uint32_t instance_id, std::vector<PPI_InterfaceStatePortDefinition const*>&& definitions) :
    InterfaceInstance(interface_refid, PPI_interfaceTypeStatePort, instance_id),
    definitions(std::move(definitions))
{
}

InterfaceInstanceStatePort::~InterfaceInstanceStatePort()
{
}

OpenIPC_Error InterfaceInstanceStatePort::GetDefinitions(const PPI_InterfaceStatePortDefinition** stateportDefinitions, uint32_t definitionsSize, uint32_t* numberOfDefinitions)
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (OpenIPC_PASS(result) && stateportDefinitions == nullptr && definitionsSize > 0)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: definitions is NULL, but definitionsSize is non-zero");
    }
    if (OpenIPC_PASS(result) && numberOfDefinitions != nullptr)
    {
        *numberOfDefinitions = static_cast<uint32_t>(this->definitions.size());
    }
    if (OpenIPC_PASS(result) && definitionsSize > 0)
    {
        const uint32_t copysize = std::min(static_cast<uint32_t>(this->definitions.size()), definitionsSize) * sizeof(const PPI_InterfaceStatePortDefinition*);

        CommonUtils::MemoryCopy(stateportDefinitions, definitionsSize * sizeof(const PPI_InterfaceStatePortDefinition*), this->definitions.data(), copysize);
    }
    return result;
}

OpenIPC_Error CProbePlugin::PPI_TriggerGetTrigger(const PPI_InterfaceTriggerResponse* triggerResponse, PPI_Pins_TypeEncode* pin, PPI_bool* onAssert, uint32_t* identifier) const
{
    OpenIPC_Error openIpcError = OpenIPC_Error_No_Error;

    if (OpenIPC_PASS(openIpcError) && triggerResponse == nullptr)
    {
        openIpcError = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID,openIpcError, "triggerResponse is null in PPI_TriggerGetTrigger.");
    }
    if (OpenIPC_PASS(openIpcError) && pin == nullptr)
    {
        openIpcError = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID,openIpcError, "pin is null in PPI_TriggerGetTrigger.");
    }
    if (OpenIPC_PASS(openIpcError) && onAssert == nullptr)
    {
        openIpcError = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID,openIpcError, "onAssert is null in PPI_TriggerGetTrigger.");
    }
    if (OpenIPC_PASS(openIpcError) && identifier == nullptr)
    {
        openIpcError = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID,openIpcError, "identifier is null in PPI_TriggerGetTrigger.");
    }

    if (OpenIPC_PASS(openIpcError))
    {
        bool temp;
        openIpcError = triggerResponse->GetTriggerInfo(*pin, temp);
        *onAssert = temp ? 1 : 0;
    }
    if (OpenIPC_PASS(openIpcError))
    {
        openIpcError = triggerResponse->GetIdentifier(*identifier);
    }
    return openIpcError;
}

OpenIPC_Error CProbePlugin::PPI_TriggerGetResponse(const PPI_InterfaceTriggerResponse* triggerResponse, PPI_Pins_TypeEncode* pin, PPI_bool* respondAsAssert, uint32_t* identifier) const
{
    OpenIPC_Error openIpcError = OpenIPC_Error_No_Error;

    if (OpenIPC_PASS(openIpcError) && triggerResponse == nullptr)
    {
        openIpcError = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID,openIpcError, "triggerResponse is null in PPI_TriggerGetTrigger.");
    }
    if (OpenIPC_PASS(openIpcError) && pin == nullptr)
    {
        openIpcError = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID,openIpcError, "pin is null in PPI_TriggerGetTrigger.");
    }
    if (OpenIPC_PASS(openIpcError) && respondAsAssert == nullptr)
    {
        openIpcError = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID,openIpcError, "onAssert is null in PPI_TriggerGetTrigger.");
    }

    if (OpenIPC_PASS(openIpcError))
    {
        bool temp;
        openIpcError = triggerResponse->GetResponseInfo(*pin,temp);
        *respondAsAssert = temp ? 1 : 0;
    }
    if (OpenIPC_PASS(openIpcError))
    {
        openIpcError = triggerResponse->GetIdentifier(*identifier);
    }
    return openIpcError;
}

OpenIPC_Error CProbePlugin::PPI_InterfaceListTriggerResponses(OpenIPC_DeviceId device, uint32_t triggerResponseSupportedLength, PPI_InterfaceTriggerResponse** triggerResponseSupported, uint32_t* triggerResponsePopulated) const
{
    OpenIPC_Error openIpcError = OpenIPC_Error_No_Error;
    std::shared_ptr<InterfaceInstancePins> probeinterface;
    openIpcError = _LookupInterface(device, probeinterface);
    if (OpenIPC_PASS(openIpcError))
    {
        openIpcError = probeinterface->ListSupportedTriggerResponses(triggerResponseSupportedLength, triggerResponseSupported,triggerResponsePopulated);
    }
    return openIpcError;
}

OpenIPC_Error CProbePlugin::PPI_InterfaceListObservableTriggerResponses(OpenIPC_DeviceId device, uint32_t triggerResponseSupportedLength, PPI_InterfaceTriggerResponse** triggerResponseSupported, uint32_t* triggerResponsePopulated) const
{
    OpenIPC_Error openIpcError = OpenIPC_Error_No_Error;
    std::shared_ptr<InterfaceInstancePins> probeinterface;
    openIpcError = _LookupInterface(device, probeinterface);
    if (OpenIPC_PASS(openIpcError))
    {
        openIpcError = probeinterface->ListObservableTriggerResponses(triggerResponseSupportedLength, triggerResponseSupported,triggerResponsePopulated);
    }
    return openIpcError;
}

OpenIPC_Error CProbePlugin::PPI_InterfaceTriggerResponseIsSet(OpenIPC_DeviceId device, const PPI_InterfaceTriggerResponse* triggerResponse, PPI_bool* triggerResponseSet, PPI_TriggerResponseSetOptions* optionsSet)
{
    OpenIPC_Error openIpcError = OpenIPC_Error_No_Error;
    std::shared_ptr<InterfaceInstancePins> probeinterface;
    openIpcError = _LookupInterface(device, probeinterface);
    if (OpenIPC_PASS(openIpcError))
    {
        openIpcError = probeinterface->TriggerResponseIsSet(triggerResponse,triggerResponseSet, optionsSet);
    }
    return openIpcError;
}

OpenIPC_Error CProbePlugin::PPI_InterfaceTriggerResponseSet(OpenIPC_DeviceId device, PPI_InterfaceTriggerResponse* triggerResponse, PPI_bool triggerResponseSet, const PPI_TriggerResponseSetOptions* optionsSet) const
{
    OpenIPC_Error openIpcError = OpenIPC_Error_No_Error;
    std::shared_ptr<InterfaceInstancePins> probeinterface;
    openIpcError = _LookupInterface(device, probeinterface);
    if (OpenIPC_PASS(openIpcError))
    {
        openIpcError = probeinterface->TriggerResponseSet(triggerResponse,triggerResponseSet, optionsSet);
    }
    return openIpcError;
}

//////////////////////////////////////////////////////////////////////////////
// ProbeGetRefIds
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::ProbeGetRefIds(
    uint32_t maxIds,
    PPI_RefId* refIds,
    uint32_t* probecount
)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    std::lock_guard<std::recursive_mutex> lock(_pluginMutex);
    if (probecount == nullptr)
    {
        openIPCError = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "probe count is null");
    }
    else
    {
        *probecount = 0;
    }
    if (OpenIPC_PASS(openIPCError) && _plugin_refid == -1)
    {
        openIPCError = OpenIPC_Error_Plugin_Is_Not_Initialized;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
    }
    if (OpenIPC_PASS(openIPCError))
    {
        if (refIds == nullptr)
        {
            *probecount = static_cast<uint32_t>(_refid2ProbeInstance.size());
        }
        else
        {
            uint32_t count = 0;
            for (auto iter = _refid2ProbeInstance.begin(); (iter != _refid2ProbeInstance.end()) && (maxIds > count); iter++)
            {
                refIds[count++] = (*iter).first;
            }
            *probecount = count;
        }
    }

    return openIPCError;
}

//////////////////////////////////////////////////////////////////////////////
// ProbeGetInfo
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::ProbeGetInfo(
    PPI_RefId refId,
    PPI_ProbeInfo* info)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    std::lock_guard<std::recursive_mutex> lock(_pluginMutex);

    std::shared_ptr<ProbeInstance> probe;
    auto iter = _refid2ProbeInstance.find(refId);
    if (iter != _refid2ProbeInstance.end()
        && (probe = iter->second.lock()))
    {
        InternalUtils::stringcopy(info->type, PPI_MAX_INFO_LEN, probe->typestring.c_str());
        InternalUtils::stringcopy(info->uniqueIdentifier, PPI_MAX_INFO_LEN, "0"); // Only 1 stub probe!
        InternalUtils::stringcopy(info->probeInfo, PPI_MAX_INFO_LEN, "0");
    }
    else
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
    }

    return openIPCError;
}

OpenIPC_Error CProbePlugin::ProbeBeginInitialization(PPI_RefId probe_refid, OpenIPC_DeviceId probeID)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_Not_Implemented;
    PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
    return openIPCError;
}

OpenIPC_Error CProbePlugin::ProbeSpecifyVendorInitString(OpenIPC_DeviceId probeID, const PPI_char* vendorinit)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_Not_Implemented;
    PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
    return openIPCError;
}

OpenIPC_Error CProbePlugin::ProbeFinishInitialization(OpenIPC_DeviceId probeID)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_Not_Implemented;
    PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
    return openIPCError;
}

//////////////////////////////////////////////////////////////////////////////
// ProbeInitialize
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::ProbeInitialize(
    PPI_RefId probe_refid,
    OpenIPC_DeviceId probeID,
    const char* vendorinit)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    std::lock_guard<std::recursive_mutex> lock(_pluginMutex);

    std::shared_ptr<ProbeInstance> probe;
    auto iter = _refid2ProbeInstance.find(probe_refid);
    if (iter != _refid2ProbeInstance.end()
        && (probe = iter->second.lock()))
    {
        // Check to see if the PROBE IS ALREADY INITIALIZED
        if (probe->deviceID == -1)
        {
            _deviceid2ProbeInstance[probeID] = probe;
            probe->deviceID = probeID;
            probe->initializationState = InitializationState::Initialized;

            std::vector<std::weak_ptr<InterfaceInstance>> probeinterfaces;
            openIPCError = _GetProbeInterfaces(probe, probeinterfaces);
            for (auto& probeInterface_ref : probeinterfaces)
            {
                auto probeInterface = probeInterface_ref.lock();
                if (!probeInterface)
                {
                    openIPCError = OpenIPC_Error_Invalid_Device;
                }
                else
                {
                    openIPCError = probe->AddInterface(probeInterface);
                }
                if (!OpenIPC_PASS(openIPCError))
                {
                    break;
                }
            }
        }
        else
        {
            //openIPCError = OpenIPC_Error_Probe_Already_Initialized;
            PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
        }
    }
    else
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
    }

    return openIPCError;
}

//////////////////////////////////////////////////////////////////////////////
// ProbeDeInitialize
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::ProbeDeInitialize(
    OpenIPC_DeviceId probeID
    )
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
    std::lock_guard<std::recursive_mutex> lock(_pluginMutex);
    std::shared_ptr<ProbeInstance> probe;
    openIPCError = _LookupProbe(probeID, probe);
    if (OpenIPC_PASS(openIPCError))
    {
        // If the deviceID is already -1, then we are an invalid Device ID
        if (probe->deviceID == OpenIPC_INVALID_DEVICE_ID)
        {
            openIPCError = OpenIPC_Error_Invalid_Device_ID;
            PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
        }
        else
        {
            for (const auto& kvPair : _deviceid2probeinterface)
            {
                auto interfaceID = kvPair.first;
                openIPCError = InterfaceDeInitialize(interfaceID);
                if (openIPCError != OpenIPC_Error_No_Error)
                {
                    PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError,
                        "Uninitialized interface 0x" << LOG_UINT32_HEX(interfaceID));
                }
            }
        }

        // Show we are not initialized by setting deviceID to -1
        probe->deviceID = OpenIPC_INVALID_DEVICE_ID;
        probe->initializationState = InitializationState::Uninitialized;
    }
    else
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
    }

    return openIPCError;
}

//////////////////////////////////////////////////////////////////////////////
// PluginGetInfo
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::PluginGetInfo(
    PPI_PluginApiVersion clientInterfaceVersion,
    PPI_PluginInfo *info
    )
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
    if (info == nullptr)
    {
        openIPCError = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID,openIPCError, "Info passed into PPI_PluginGetInfo is null");
    }
    if (OpenIPC_PASS(openIPCError))
    {
        info->clientInterfaceSupported = clientInterfaceVersion == PPI_PLUGIN_API_VERSION;
        info->pluginInterfaceVersion = PPI_PLUGIN_API_VERSION;
        info->requiresLockNotifications = 1;
        InternalUtils::stringcopy(info->pluginName, PPI_MAX_INFO_LEN, _pluginname);
    }
    return openIPCError;
}

//////////////////////////////////////////////////////////////////////////////
// PluginInitialize
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::PluginInitialize(
    PPI_RefId pluginid,
    const char* vendorinit
    )
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    std::lock_guard<std::recursive_mutex> lock(_pluginMutex);

    // Check to see if the PLUGIN is ALREADY INITIALIZED
    if (_plugin_refid == -1)
    {
        // set the refid base for this plugin
        _plugin_refid = pluginid << 24;

        std::vector<std::weak_ptr<ProbeInstance>> probes;
        openIPCError = _GetProbes(probes, vendorinit);
        for (uint8_t probe_idx = 0; probe_idx < probes.size(); probe_idx++)
        {
            auto probe = probes[probe_idx].lock();
            if (probe)
            {
                _refid2ProbeInstance[probe->refid] = probes[probe_idx];
            }
        }
    }
    else
    {
        //openIPCError = OpenIPC_Error_Plugin_Already_Initialized;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
    }

    return openIPCError;
}

//////////////////////////////////////////////////////////////////////////////
// PluginDeinitialize
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::PluginDeinitialize(void)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    std::lock_guard<std::recursive_mutex> lock(_pluginMutex);

    // set the refid base for this plugin to the uninitialized value
    _plugin_refid = static_cast<uint32_t>(-1);

    // For each probe that we are managing, deinitialize it (which should de-initialize all of the interfaces as well)
    for (const auto& kvPair : _deviceid2ProbeInstance)
    {
        openIPCError = ProbeDeInitialize(kvPair.first);
    }

    // At this point, all probes and interfaces have been deinitialized
    _deviceid2ProbeInstance.clear();
    _refid2ProbeInstance.clear();
    _deviceid2probeinterface.clear();

    // Disable logging, since we don't actually care about getting data now!
	for (uint32_t stream = 0; stream < PPI_LAST_STREAM; ++stream)
	{
		PluginLogger::GetInstance().Level(PPI_noNotification, stream);
	}
    PluginNotifications::GetInstance().Level(PPI_noNotification);

    PluginLogger::GetInstance().Clear();
    PluginNotifications::GetInstance().Clear();

    _eventHandlerFunction = nullptr;

    return openIPCError;
}


//////////////////////////////////////////////////////////////////////////////
// PluginGetProbeTypes
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::PluginGetProbeTypes(uint32_t maxProbeTypes, PPI_char const** probeTypes, uint32_t* probeTypeCount)
{
	return OpenIPC_Error_Not_Implemented;
}

//////////////////////////////////////////////////////////////////////////////
// PluginCreateStaticProbe
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::PluginCreateStaticProbe(const PPI_char probeType[PPI_MAX_PROBE_TYPE_LEN], PPI_RefId* refId)
{
	return OpenIPC_Error_Not_Implemented;
}

OpenIPC_Error CProbePlugin::InterfaceListConfigString(OpenIPC_DeviceId interfaceID,
                                                      uint32_t configTypeLength,
                                                      char const ** configType,
                                                      uint32_t* numberOfConfigTypes)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
    std::shared_ptr<InterfaceInstance> probeinterface;
    openIPCError = _LookupInterface(interfaceID, probeinterface);
    if (OpenIPC_PASS(openIPCError))
    {
        std::lock_guard<std::recursive_mutex> _(probeinterface->_instanceMutex);
        uint32_t configTypeIdx = 0;
        // Write out the number of entries
        *numberOfConfigTypes = (uint32_t) (probeinterface->settings.size());
		if (configType != nullptr)
		{
        for(auto iter = probeinterface->settings.keyBegin();iter != probeinterface->settings.keyEnd();++iter)
        {
            if (configTypeIdx >= configTypeLength)
            {
					PPI_LOG(OpenIPC_INVALID_DEVICE_ID, PPI_warningNotification, "Insufficient configType buffer. Truncating output.");
                break;
            }
            else
            {
                configType[configTypeIdx++] = iter->get()->c_str();
            }
        }
    }
    }
    else
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "Invalid device ID requested (" << LOG_UINT32_HEX((uint32_t)interfaceID) << ")");
    }

    return openIPCError;
}

//////////////////////////////////////////////////////////////////////////////
// InterfaceOperationCancel
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::InterfaceOperationCancel(
    OpenIPC_DeviceId interfaceID
)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    std::shared_ptr<InterfaceInstance> probeinterface;
    openIPCError = _LookupInterface(interfaceID, probeinterface);
    if (OpenIPC_PASS(openIPCError))
    {
        return probeinterface->OperationCancel();
    }
    else
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "Invalid device ID requested (" << LOG_UINT32_HEX((uint32_t)interfaceID) << ")");
    }

    return openIPCError;
}
//////////////////////////////////////////////////////////////////////////////
// InterfaceSetConfig
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::InterfaceSetConfig(
    OpenIPC_DeviceId interfaceID,
    const char* configType,
    const PPI_char value[PPI_MAX_INFO_LEN]
)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    std::shared_ptr<InterfaceInstance> probeinterface;
    openIPCError = _LookupInterface(interfaceID, probeinterface);
    if (OpenIPC_PASS(openIPCError))
    {
        PPI_LOG(interfaceID, PPI_debugNotification, "InterfaceSetConfig " << configType << " = " << value);
        std::lock_guard<std::recursive_mutex> _(probeinterface->_instanceMutex);
        std::string strValue(value);
        // This is here for paranoia
        if (strValue.size() > PPI_MAX_INFO_LEN)
        {
            strValue.resize(PPI_MAX_INFO_LEN);
        }
        probeinterface->settings.Set(configType, std::move(strValue));
        probeinterface->SettingsUpdated();
    }
    else
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "Invalid device ID requested (" << LOG_UINT32_HEX((uint32_t)interfaceID) << ")");
    }

    return openIPCError;
}


OpenIPC_Error CProbePlugin::PPI_InterfacePinsQueryAliases(OpenIPC_DeviceId device, PPI_Pins_TypeEncode pin,
                                                          uint32_t pinListLength, PPI_Pins_TypeEncode* pinList, uint32_t* numberOfPinsWritten) const
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;

    std::vector<PPI_Pins_TypeEncode> pinAliases;
    if (OpenIPC_PASS(result) && numberOfPinsWritten == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        POST_ERROR_MESSAGE(result, "numberOfPinsWritten is null.");
    }
    if (OpenIPC_PASS(result))
    {
        *numberOfPinsWritten = 0;
    }

    if (OpenIPC_PASS(result) && pinListLength == 0)
    {

        result = this->_LookupPinAliases(pin,pinAliases);
        *numberOfPinsWritten = (uint32_t)(pinAliases.size());
    }

    if (OpenIPC_PASS(result) && pinListLength != 0 && pinList == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        POST_ERROR_MESSAGE(result, "pinList is null.");
    }

    for (uint32_t i = 0;OpenIPC_PASS(result) && i<pinListLength && i < pinAliases.size();i++)
    {
        pinList[i] = pinAliases[i];
        (*numberOfPinsWritten)++;
    }
    return result;
}

OpenIPC_Error CProbePlugin::PPI_InterfacePinsListDrivablePins(
    OpenIPC_DeviceId device, uint32_t pinsListLength,
    PPI_Pins_TypeEncode* pinsList, uint32_t* numberOfPinsPopulated) const
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    std::vector<PPI_Pins_TypeEncode> drivablePins;
    if (OpenIPC_PASS(result) && numberOfPinsPopulated == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        POST_ERROR_MESSAGE(result, "numberOfPinsPopulated is null.");
    }

    if (OpenIPC_PASS(result))
    {
        *numberOfPinsPopulated = 0;
        result = this->_GetDrivablePins(drivablePins);
    }

    if (OpenIPC_PASS(result) && pinsListLength == 0)
    {
        *numberOfPinsPopulated = (uint32_t)(drivablePins.size());
    }

    if (OpenIPC_PASS(result) && pinsListLength != 0 && pinsList == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        POST_ERROR_MESSAGE(result, "pinsList is null.");
    }

    if (OpenIPC_PASS(result) && pinsListLength != 0)// && numberOfPinsPopulated != nullptr && pinsList != nullptr)
    {
        uint32_t count = 0;
        for (uint32_t i = 0; OpenIPC_PASS(result) && i < pinsListLength && i < drivablePins.size(); i++)
        {
            pinsList[i] = drivablePins[i];
            count++;
        }
        *numberOfPinsPopulated = count;
    }
    return result;
}

OpenIPC_Error CProbePlugin::PPI_InterfacePinsListReadablePins(
    OpenIPC_DeviceId device, uint32_t pinsListLength,
    PPI_Pins_TypeEncode* pinsList, uint32_t* numberOfPinsPopulated) const
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    std::vector<PPI_Pins_TypeEncode> readablePins;
    if (OpenIPC_PASS(result) && numberOfPinsPopulated == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        POST_ERROR_MESSAGE(result, "numberOfPinsPopulated is null.");
    }

    if (OpenIPC_PASS(result) && numberOfPinsPopulated != nullptr)
    {
        *numberOfPinsPopulated = 0;
        result = this->_GetReadablePins(readablePins);
    }

    if (OpenIPC_PASS(result) && pinsListLength == 0 && numberOfPinsPopulated != nullptr)
    {
        *numberOfPinsPopulated = 0;
        *numberOfPinsPopulated = (uint32_t)(readablePins.size());
    }

    if (OpenIPC_PASS(result) && pinsListLength != 0 && pinsList == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        POST_ERROR_MESSAGE(result, "pinsList is null.");
    }

    if (OpenIPC_PASS(result) && pinsListLength != 0 && numberOfPinsPopulated != nullptr && pinsList != nullptr)
    {
        uint32_t count = 0;
        for (uint32_t i = 0;OpenIPC_PASS(result) && i<pinsListLength && i < readablePins.size();i++)
        {
            pinsList[i] = readablePins[i];
            count++;
        }
        *numberOfPinsPopulated = count;
    }
    return result;
}


OpenIPC_Error CProbePlugin::_LookupPinAliases(PPI_Pins_TypeEncode pin,std::vector<PPI_Pins_TypeEncode>& pinAliases) const
{
    pinAliases.clear();
    pinAliases.push_back(pin);
    return OpenIPC_Error_No_Error;
}

OpenIPC_Error CProbePlugin::_GetDrivablePins(std::vector<PPI_Pins_TypeEncode>& pins) const
{
    static std::vector<uint32_t> staticValues = {
            PPI_PINS_hook0,
            PPI_PINS_hook1,
            PPI_PINS_hook2,
            PPI_PINS_hook3,
            PPI_PINS_hook4,
            PPI_PINS_hook5,
            PPI_PINS_hook6,
            PPI_PINS_hook7,
            PPI_PINS_hook8,
            PPI_PINS_hook9,
            PPI_PINS_Pod_Prsnt1_N,
            PPI_PINS_Pod_Prsnt2_N,
            PPI_PINS_Preq_N,
            PPI_PINS_Prdy_N,
            PPI_PINS_TRST,
            PPI_PINS_TMS_bit,
            PPI_PINS_TDI_bit,
            PPI_PINS_TDO_bit,
            PPI_PINS_TCK_bit,
            PPI_PINS_PowerButton,
            PPI_PINS_ResetButton,
            PPI_PINS_SystemTapPowerGood,
            PPI_PINS_CoreTapPowerGood,
            PPI_PINS_SystemReset,
            PPI_PINS_CoreReset,
            PPI_PINS_CorePowerGood,
            PPI_PINS_TapReady,
            PPI_PINS_PowerBreak1,
            PPI_PINS_PowerBreak2,
            PPI_PINS_PowerBreak3,
            PPI_PINS_JtagDisable,
            PPI_PINS_I2CDisable,
            PPI_PINS_SystemPowerGood,
            PPI_PINS_SystemBootStall,
            PPI_PINS_GlobalResetEntryStall,
            PPI_PINS_SxEntryStall,
            PPI_PINS_SxExitStall,

            PPI_PINS_obsA_func0,
            PPI_PINS_obsA_func1,
            PPI_PINS_obsB_func0,
            PPI_PINS_obsB_func1,
            PPI_PINS_obsC_func0,
            PPI_PINS_obsC_func1,
            PPI_PINS_obsD_func0,
            PPI_PINS_obsD_func1,
            PPI_PINS_obsA_data0,
            PPI_PINS_obsA_data1,
            PPI_PINS_obsA_data2,
            PPI_PINS_obsA_data3,
            PPI_PINS_obsB_data0,
            PPI_PINS_obsB_data1,
            PPI_PINS_obsB_data2,
            PPI_PINS_obsB_data3,
            PPI_PINS_obsC_data0,
            PPI_PINS_obsC_data1,
            PPI_PINS_obsC_data2,
            PPI_PINS_obsC_data3,
            PPI_PINS_obsD_data0,
            PPI_PINS_obsD_data1,
            PPI_PINS_obsD_data2,
            PPI_PINS_obsD_data3,
            PPI_PINS_NO_ACTION};

    pins = staticValues;

    return OpenIPC_Error_No_Error;
}

OpenIPC_Error CProbePlugin::_GetReadablePins(std::vector<PPI_Pins_TypeEncode>& pins) const
{
    static std::vector<uint32_t> staticValues = {
            PPI_PINS_hook0,
            PPI_PINS_hook1,
            PPI_PINS_hook2,
            PPI_PINS_hook3,
            PPI_PINS_hook4,
            PPI_PINS_hook5,
            PPI_PINS_hook6,
            PPI_PINS_hook7,
            PPI_PINS_hook8,
            PPI_PINS_hook9,
            PPI_PINS_Pod_Prsnt1_N,
            PPI_PINS_Pod_Prsnt2_N,
            PPI_PINS_Preq_N,
            PPI_PINS_Prdy_N,
            PPI_PINS_TRST,
            PPI_PINS_TMS_bit,
            PPI_PINS_TDI_bit,
            PPI_PINS_TDO_bit,
            PPI_PINS_TCK_bit,
            PPI_PINS_PowerButton,
            PPI_PINS_ResetButton,
            PPI_PINS_SystemTapPowerGood,
            PPI_PINS_CoreTapPowerGood,
            PPI_PINS_SystemReset,
            PPI_PINS_CoreReset,
            PPI_PINS_CorePowerGood,
            PPI_PINS_TapReady,
            PPI_PINS_PowerBreak1,
            PPI_PINS_PowerBreak2,
            PPI_PINS_PowerBreak3,
            PPI_PINS_JtagDisable,
            PPI_PINS_I2CDisable,
            PPI_PINS_SystemPowerGood,
            PPI_PINS_SystemBootStall,
            PPI_PINS_GlobalResetEntryStall,
            PPI_PINS_SxEntryStall,
            PPI_PINS_SxExitStall,

            PPI_PINS_obsA_func0,
            PPI_PINS_obsA_func1,
            PPI_PINS_obsB_func0,
            PPI_PINS_obsB_func1,
            PPI_PINS_obsC_func0,
            PPI_PINS_obsC_func1,
            PPI_PINS_obsD_func0,
            PPI_PINS_obsD_func1,
            PPI_PINS_obsA_data0,
            PPI_PINS_obsA_data1,
            PPI_PINS_obsA_data2,
            PPI_PINS_obsA_data3,
            PPI_PINS_obsB_data0,
            PPI_PINS_obsB_data1,
            PPI_PINS_obsB_data2,
            PPI_PINS_obsB_data3,
            PPI_PINS_obsC_data0,
            PPI_PINS_obsC_data1,
            PPI_PINS_obsC_data2,
            PPI_PINS_obsC_data3,
            PPI_PINS_obsD_data0,
            PPI_PINS_obsD_data1,
            PPI_PINS_obsD_data2,
            PPI_PINS_obsD_data3,
            PPI_PINS_NO_ACTION};

    pins = staticValues;

    return OpenIPC_Error_No_Error;
}

OpenIPC_Error CProbePlugin::PPI_DeviceSetConfig(OpenIPC_DeviceId deviceID,const PPI_char* configType,const PPI_char value[PPI_MAX_INFO_LEN])
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    if (value == NULL)
    {
        openIPCError = OpenIPC_Error_Probe_Invalid_Parameter;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "Invalid parameter: value is NULL");
    }
    else if (deviceID == OpenIPC_INVALID_DEVICE_ID || deviceID == 0 || deviceID == (_plugin_refid >> 24))
    {
        std::lock_guard<std::recursive_mutex> _(this->_pluginMutex);
        std::string strValue(value);
        // This is here for paranoia
        if (strValue.size() > PPI_MAX_INFO_LEN)
        {
            strValue.resize(PPI_MAX_INFO_LEN);
        }
        this->_settings.Set(configType, std::move(strValue));
        this->SettingsUpdated();
    }
    else if (this->_deviceid2ProbeInstance.find(deviceID) != this->_deviceid2ProbeInstance.end())
    {
        std::shared_ptr<ProbeInstance> probeinterface;
        openIPCError = _LookupProbe(deviceID, probeinterface);
        if (OpenIPC_PASS(openIPCError))
        {
            std::string strValue(value);
            // This is here for paranoia
            if (strValue.size() > PPI_MAX_INFO_LEN)
            {
                strValue.resize(PPI_MAX_INFO_LEN);
            }
            probeinterface->settings.Set(configType, std::move(strValue));
            probeinterface->SettingsUpdated();
        }
    }
    else if (this->_deviceid2probeinterface.find(deviceID) != this->_deviceid2probeinterface.end())
    {
        openIPCError = PPI_InterfaceSetConfig(deviceID, configType, value);
    }
    else
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "Invalid device ID requested (" << LOG_UINT32_HEX((uint32_t)deviceID) << ")");
    }

    return openIPCError;
}

OpenIPC_Error CProbePlugin::_GetDeviceSettings(OpenIPC_DeviceId deviceID, const PPI_char* configType, DeviceSetting*& deviceSettings)
{
    deviceSettings = nullptr;
    if (deviceID == OpenIPC_INVALID_DEVICE_ID)
    {
        auto it = this->_settings.find(configType);
        if (it != this->_settings.end())
        {
            deviceSettings = &it->second;
        }
        return OpenIPC_Error_No_Error;
    }

    if (this->_deviceid2ProbeInstance.find(deviceID) != this->_deviceid2ProbeInstance.end())
    {
        std::shared_ptr<ProbeInstance> probeInstance;
        OpenIPC_Error openIPCError = _LookupProbe(deviceID, probeInstance);
        if (OpenIPC_PASS(openIPCError))
        {
            auto it = probeInstance->settings.find(configType);
            if (it != probeInstance->settings.end())
            {
                deviceSettings = &it->second;
            }
        }
        return openIPCError;
    }

    if (this->_deviceid2probeinterface.find(deviceID) != this->_deviceid2probeinterface.end())
    {
        std::shared_ptr<InterfaceInstance> interfaceInstance;
        OpenIPC_Error openIPCError = _LookupInterface(deviceID, interfaceInstance);
        if (OpenIPC_PASS(openIPCError))
        {
            auto it = interfaceInstance->settings.find(configType);
            if (it != interfaceInstance->settings.end())
            {
                deviceSettings = &it->second;
            }
        }
        return openIPCError;
    }

    OpenIPC_Error openIPCError = OpenIPC_Error_Invalid_Device_ID;
    PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "Invalid device ID requested (" << LOG_UINT32_HEX((uint32_t)deviceID) << ")");
    return openIPCError;
}

OpenIPC_Error CProbePlugin::PPI_DeviceGetConfigDescription(OpenIPC_DeviceId deviceID, const PPI_char* configType, PPI_char description[PPI_MAX_INFO_LEN])
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
    std::lock_guard<std::recursive_mutex> _(this->_pluginMutex);

    if (description == nullptr)
    {
        openIPCError = OpenIPC_Error_Probe_Invalid_Parameter;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "Invalid parameter named description: value is NULL");
        return openIPCError;
    }
    if (configType == nullptr)
    {
        openIPCError = OpenIPC_Error_Probe_Invalid_Parameter;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "Invalid parameter named configType: value is NULL");
        return openIPCError;
    }

    DeviceSetting* deviceSetting;
    openIPCError = _GetDeviceSettings(deviceID, configType, deviceSetting);
    if (OpenIPC_PASS(openIPCError) && deviceSetting != nullptr)
    {
        InternalUtils::stringcopy(description, PPI_MAX_INFO_LEN, deviceSetting->Description.c_str());
    }

    return openIPCError;
}

OpenIPC_Error CProbePlugin::PPI_DeviceListConfigValues(OpenIPC_DeviceId deviceID, const PPI_char* configType, uint32_t valuesLength, PPI_char const** values, uint32_t* numberOfValues)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
    std::lock_guard<std::recursive_mutex> _(this->_pluginMutex);

    if (values == nullptr)
    {
        openIPCError = OpenIPC_Error_Probe_Invalid_Parameter;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "Invalid parameter named values: value is NULL");
        return openIPCError;
    }
    if (configType == nullptr)
    {
        openIPCError = OpenIPC_Error_Probe_Invalid_Parameter;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "Invalid parameter named configType: value is NULL");
        return openIPCError;
    }
    *numberOfValues = 0;

    DeviceSetting* deviceSetting;
    openIPCError = _GetDeviceSettings(deviceID, configType, deviceSetting);
    if (OpenIPC_PASS(openIPCError) && deviceSetting != nullptr)
    {
        *numberOfValues = static_cast<uint32_t>(deviceSetting->Options.size());

        auto it = deviceSetting->Options.begin();
        for (uint32_t i = 0; i < valuesLength && i < *numberOfValues; i++)
        {
            values[i] = it->c_str();
            std::advance(it, 1);
        }
    }

    return openIPCError;
}


OpenIPC_Error CProbePlugin::PPI_DeviceGetConfig(OpenIPC_DeviceId deviceID,const PPI_char* configType,PPI_char value[PPI_MAX_INFO_LEN])
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
    std::lock_guard<std::recursive_mutex> _(this->_pluginMutex);

    if (value == NULL)
    {
        openIPCError = OpenIPC_Error_Probe_Invalid_Parameter;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "Invalid parameter named value: value is NULL");
    }
    else if (configType == NULL)
    {
        openIPCError = OpenIPC_Error_Probe_Invalid_Parameter;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "Invalid parameter named configType: value is NULL");
    }
    else if (deviceID == OpenIPC_INVALID_DEVICE_ID || deviceID == (_plugin_refid >> 24))
    {
        InternalUtils::stringcopy(value,PPI_MAX_INFO_LEN,this->_settings.GetString(configType).c_str());
    }
    else if (this->_deviceid2ProbeInstance.find(deviceID) != this->_deviceid2ProbeInstance.end())
    {
        std::shared_ptr<ProbeInstance> probeInstance;
        openIPCError = _LookupProbe(deviceID, probeInstance);
        if (OpenIPC_PASS(openIPCError))
        {
            InternalUtils::stringcopy(value, PPI_MAX_INFO_LEN, probeInstance->settings.GetString(configType).c_str());
        }
    }
    else if (this->_deviceid2probeinterface.find(deviceID) != this->_deviceid2probeinterface.end())
    {
        openIPCError = PPI_InterfaceGetConfig(deviceID, configType, value); // Question: Can we get rid of InterfaceSetConfig from CProbePlugin? It is an SDK breaking change, but not a PPI breaking change
    }
    else
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "Invalid device ID requested (" << LOG_UINT32_HEX((uint32_t)deviceID) << ")");
    }

    return openIPCError;
}

OpenIPC_Error CProbePlugin::PPI_DeviceListConfigString(OpenIPC_DeviceId deviceID,uint32_t configTypeLength,PPI_char const ** configType, uint32_t* numberOfConfigTypes)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    if (numberOfConfigTypes == NULL)
    {
        openIPCError = OpenIPC_Error_Probe_Invalid_Parameter;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "Invalid parameter: value is NULL");
    }
    else if (configType == NULL && configTypeLength > 0)
    {
        openIPCError = OpenIPC_Error_Probe_Invalid_Parameter;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "Invalid parameter: configType is NULL");
    }
    else if (deviceID == OpenIPC_INVALID_DEVICE_ID || deviceID == (_plugin_refid >> 24))
    {
        std::lock_guard<std::recursive_mutex> _(this->_pluginMutex);
        uint32_t configTypeIdx = 0;
        // Write out the number of entries
        *numberOfConfigTypes = (uint32_t) (this->_settings.size());
        for(auto iter = this->_settings.keyBegin(); iter != this->_settings.keyEnd(); ++ iter)
        {
            if (configTypeIdx >= configTypeLength)
            {
                if (configType != nullptr)
                {
                    PPI_LOG(OpenIPC_INVALID_DEVICE_ID, PPI_warningNotification, "Insufficient configType buffer. Truncating output.");
                }
                break;
            }
            else
            {
                configType[configTypeIdx++] = iter->get()->c_str();
            }
        }
    }
    else if (this->_deviceid2ProbeInstance.find(deviceID) != this->_deviceid2ProbeInstance.end())
    {
        std::shared_ptr<ProbeInstance> probeinterface;
        openIPCError = _LookupProbe(deviceID, probeinterface);
        uint32_t configTypeIdx = 0;
        // Write out the number of entries
        *numberOfConfigTypes = (uint32_t) (probeinterface->settings.size());
        for(auto iter = probeinterface->settings.keyBegin(); iter != probeinterface->settings.keyEnd(); ++ iter)
        {
            if (configTypeIdx >= configTypeLength)
            {
                if (configType != nullptr)
                {
                    PPI_LOG(OpenIPC_INVALID_DEVICE_ID, PPI_warningNotification, "Insufficient configType buffer. Truncating output.");
                }
                break;
            }
            else
            {
                configType[configTypeIdx++] = iter->get()->c_str();
            }
        }
    }
    else if (this->_deviceid2probeinterface.find(deviceID) != this->_deviceid2probeinterface.end())
    {
        openIPCError = PPI_InterfaceListConfigString(deviceID, configTypeLength, configType, numberOfConfigTypes);
    }
    else
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "Invalid device ID requested (" << LOG_UINT32_HEX((uint32_t)deviceID) << ")");
    }

    return openIPCError;
}

OpenIPC_Error CProbePlugin::PPI_DeviceListActions(OpenIPC_DeviceId deviceID, uint32_t actionsLength, PPI_Action** actions, uint32_t* numberOfActions)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    if (numberOfActions == NULL)
    {
        openIPCError = OpenIPC_Error_Probe_Invalid_Parameter;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "Invalid parameter: numberOfActions is NULL");
    }
    else if (actions == NULL && actionsLength > 0)
    {
        openIPCError = OpenIPC_Error_Probe_Invalid_Parameter;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "Invalid parameter: configType is NULL");
    }
    else if (deviceID == OpenIPC_INVALID_DEVICE_ID)
    {
        std::lock_guard<std::recursive_mutex> _(this->_pluginMutex);
        for (uint32_t i = 0;(i<this->_actions.size()) && (i < actionsLength);++i)
        {
            actions[i] = this->_actions[i].get();
        }
        *numberOfActions = static_cast<uint32_t>(this->_actions.size());
    }
    else if (this->_deviceid2ProbeInstance.find(deviceID) != this->_deviceid2ProbeInstance.end())
    {
        std::shared_ptr<ProbeInstance> probeinterface;
        openIPCError = _LookupProbe(deviceID, probeinterface);
        if (OpenIPC_PASS(openIPCError))
        {
            for (uint32_t i = 0;(i<probeinterface->actions.size()) && (i < actionsLength);++i)
            {
                actions[i] = probeinterface->actions[i].get();
            }
            *numberOfActions = static_cast<uint32_t>(probeinterface->actions.size());
        }
    }
    else if (this->_deviceid2probeinterface.find(deviceID) != this->_deviceid2probeinterface.end())
    {
        std::shared_ptr<InterfaceInstance> probeinterface;
        openIPCError = _LookupInterface(deviceID, probeinterface);
        if (OpenIPC_PASS(openIPCError))
        {
            std::lock_guard<std::recursive_mutex> _(probeinterface->_instanceMutex);
            for (uint32_t i = 0;(i<probeinterface->actions.size()) && (i < actionsLength);++i)
            {
                actions[i] = probeinterface->actions[i].get();
            }
            *numberOfActions = static_cast<uint32_t>(probeinterface->actions.size());
        }
    }
    else
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "Invalid device ID requested (" << LOG_UINT32_HEX((uint32_t)deviceID) << ")");
    }

    return openIPCError;
}

OpenIPC_Error CProbePlugin::PPI_ActionGetName (const PPI_Action* action,uint32_t valueLength, PPI_char* value)
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (action == nullptr)
    {
        result = OpenIPC_Error_Probe_Invalid_Parameter;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Invalid parameter: action is NULL");
    }
    if (OpenIPC_PASS(result) && value == nullptr)
    {
        result = OpenIPC_Error_Probe_Invalid_Parameter;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Invalid parameter: value is NULL");
    }
    if (OpenIPC_PASS(result))
    {
        auto description = action->Name();
        InternalUtils::stringcopy(value,valueLength,description.c_str());
    }
    return result;
}

OpenIPC_Error CProbePlugin::PPI_ActionGetDescription(const PPI_Action* action, uint32_t valueLength, PPI_char* value)
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (action == nullptr)
    {
        result = OpenIPC_Error_Probe_Invalid_Parameter;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Invalid parameter: action is NULL");
    }
    if (OpenIPC_PASS(result) && value == nullptr)
    {
        result = OpenIPC_Error_Probe_Invalid_Parameter;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Invalid parameter: value is NULL");
    }
    if (OpenIPC_PASS(result))
    {
        auto description = action->GetDescription();
        InternalUtils::stringcopy(value,valueLength,description.c_str());
    }

    return result;
}

OpenIPC_Error CProbePlugin::PPI_DeviceActionRun(OpenIPC_DeviceId deviceID,const PPI_Action* action, uint32_t messageLength, PPI_char* message)
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (action == nullptr)
    {
        result = OpenIPC_Error_Probe_Invalid_Parameter;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Invalid parameter: action is NULL");
    }
    if (OpenIPC_PASS(result) && message == nullptr)
    {
        result = OpenIPC_Error_Probe_Invalid_Parameter;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Invalid parameter: message is NULL");
    }
    if (OpenIPC_PASS(result))
    {
        result = CallAndTranslateError([&]()
        {
            auto info = action->Run();
            InternalUtils::stringcopy(message,messageLength,info.c_str());
        });
    }

    return result;
}

OpenIPC_Error CProbePlugin::PPI_ActionSetArgument(PPI_Action* action, const PPI_char* value)
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (action == nullptr)
    {
        result = OpenIPC_Error_Probe_Invalid_Parameter;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Invalid parameter: action is NULL");
    }
    if (OpenIPC_PASS(result) && value == nullptr)
    {
        result = OpenIPC_Error_Probe_Invalid_Parameter;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Invalid parameter: message is NULL");
    }
    if (OpenIPC_PASS(result))
    {
        action->SetParameter(value);
    }

    return result;
}

OpenIPC_Error CProbePlugin::PPI_ActionIsVisible(const PPI_Action* action, PPI_bool* isVisible) const
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (action == nullptr)
    {
        result = OpenIPC_Error_Probe_Invalid_Parameter;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Invalid parameter: action is NULL");
    }
    if (OpenIPC_PASS(result) && isVisible == nullptr)
    {
        result = OpenIPC_Error_Probe_Invalid_Parameter;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Invalid parameter: isVisible is NULL");
    }
    if (OpenIPC_PASS(result))
    {
        *isVisible = action->IsVisible() ?  1 : 0;
    }

    return result;
}

OpenIPC_Error CProbePlugin::PPI_StatePortAllocateOperation(OpenIPC_DeviceId deviceId, const PPI_InterfaceStatePortDefinition* definition, PPI_StatePort_OPERATION_TYPE_ET operationType, PPI_InterfaceStatePortOperation** operation, const PPI_StatePortAllocateOperation_Options* options)
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (OpenIPC_PASS(result) && operationType != StatePort_OPERATION_READ && operationType != StatePort_OPERATION_WRITE)
    {
        result = OpenIPC_Error_Probe_Invalid_Parameter;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Invalid parameter: operationType must be StatePort_OPERATION_READ or StatePort_OPERATION_WRITE");
    }
    if (OpenIPC_PASS(result) && definition == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: definition is NULL");
    }
    if (OpenIPC_PASS(result) && operation == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: operation is NULL");
    }
    if (OpenIPC_PASS(result) && (definition->capabilities & operationType) == StatePort_OPERATION_NONE)
    {
        result = OpenIPC_Error_Invalid_Argument;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Invalid Argument: operationType: definition \"" << definition->name << "\" doesn't support that operation type.");
    }
    std::shared_ptr<InterfaceInstanceStatePort> probeinterface;
    if (OpenIPC_PASS(result))
    {
        result = _LookupInterface(deviceId, probeinterface);
    }
    if (OpenIPC_PASS(result))
    {
        result = probeinterface->AllocateOperation(definition, operationType, operation, options);
    }

    return result;
}
OpenIPC_Error CProbePlugin::PPI_StatePortFreeOperation(OpenIPC_DeviceId deviceId, PPI_InterfaceStatePortOperation** operation)
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (OpenIPC_PASS(result) && operation == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: operation is NULL");
    }
    std::shared_ptr<InterfaceInstanceStatePort> probeinterface;
    if (OpenIPC_PASS(result))
    {
        result = _LookupInterface(deviceId, probeinterface);
    }
    if (OpenIPC_PASS(result))
    {
        result = probeinterface->FreeOperation(operation);
    }

    return result;
}
OpenIPC_Error CProbePlugin::PPI_StatePortAppendOperation(OpenIPC_DeviceId deviceId, PPI_ProbeBundleHandle bundle, PPI_InterfaceStatePortOperation** operationPointer, const PPI_StatePortAppendOperation_Options* options)
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (OpenIPC_PASS(result) && operationPointer == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: operationPointer is NULL");
    }
    if (OpenIPC_PASS(result) && *operationPointer == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: *operationPointer is NULL");
    }
    std::shared_ptr<InterfaceInstanceStatePort> probeinterface;
    if (OpenIPC_PASS(result))
    {
        result = _LookupInterface(deviceId, probeinterface);
    }
    IInterfaceBundle* interfaceBundle = &IInterfaceBundle::Default;
    if (OpenIPC_PASS(result))
    {
        if (bundle == PPI_PROBE_LOCK_RELEASE || bundle == PPI_PROBE_LOCK_HOLD)
        {
            interfaceBundle = &IInterfaceBundle::ExecuteImmediately;
        }
        else
        {
            ProbePluginBundle& probeBundle = *reinterpret_cast<ProbePluginBundle*>(bundle);
            interfaceBundle = &probeBundle.InterfaceBundle();
            if (probeBundle.InterfaceBundle() == IInterfaceBundle::Default)
            {
                std::unique_ptr<IInterfaceBundle> newInterfaceBundle = probeinterface->BundleAllocate();
                if (newInterfaceBundle != nullptr)
                {
                    interfaceBundle = newInterfaceBundle.get();
                    probeBundle.InterfaceBundle(std::move(newInterfaceBundle));
                }
                else
                {
                    result = OpenIPC_Error_Internal_Error;
                    PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Internal Error: interface could not allocate new bundle contents");
                }
            }
            else if (interfaceBundle->interfaceDeviceId != probeinterface->deviceID)
            {
                result = OpenIPC_Error_Invalid_Device_ID;
                PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Invalid Device ID: This bundle is already being used by another interface");
            }
        }
    }
    if (OpenIPC_PASS(result))
    {
        result = probeinterface->AppendOperation(*interfaceBundle, operationPointer, options);
    }
    if (OpenIPC_PASS(result))
    {
        *operationPointer = nullptr;
    }

    return result;
}
OpenIPC_Error CProbePlugin::PPI_StatePortOperationSetParameter(PPI_InterfaceStatePortOperation* operation, const PPI_InterfaceStatePortParameter* parameter, const uint8_t* value, uint32_t size)
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (OpenIPC_PASS(result) && operation == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: operation is NULL");
    }
    if (OpenIPC_PASS(result) && parameter == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: parameter is NULL");
    }
    if (OpenIPC_PASS(result) && value == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: value is NULL");
    }
    if (OpenIPC_PASS(result) && size < ((parameter->size + 7) / 8))
    {
        result = OpenIPC_Error_Invalid_Argument;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Invalid Argument: size too small to hold the parameter");
    }
    if (OpenIPC_PASS(result))
    {
        result = operation->SetParameter(parameter, value, size);
    }

    return result;
}
OpenIPC_Error CProbePlugin::PPI_StatePortOperationSetWriteValue(PPI_InterfaceStatePortOperation* operation, const uint8_t* buffer, uint32_t size)
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (OpenIPC_PASS(result) && operation == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: operation is NULL");
    }
    if (OpenIPC_PASS(result) && buffer == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: buffer is NULL");
    }
    if (OpenIPC_PASS(result))
    {
        result = operation->SetWriteValue(buffer, size);
    }

    return result;
}
OpenIPC_Error CProbePlugin::PPI_StatePortOperationSetReadBuffer(PPI_InterfaceStatePortOperation* operation, uint8_t* buffer, uint32_t size)
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (OpenIPC_PASS(result) && operation == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: operation is NULL");
    }
    if (OpenIPC_PASS(result) && buffer == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: buffer is NULL");
    }
    if (OpenIPC_PASS(result))
    {
        result = operation->SetReadBuffer(buffer, size);
    }

    return result;
}
OpenIPC_Error CProbePlugin::PPI_StatePortOperationSetErrorBuffer(PPI_InterfaceStatePortOperation* operation, ProbeStatePortError* errorBuffer)
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (OpenIPC_PASS(result) && operation == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: operation is NULL");
    }
    if (OpenIPC_PASS(result) && errorBuffer == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: errorBuffer is NULL");
    }
    if (OpenIPC_PASS(result))
    {
        result = operation->SetErrorBuffer(errorBuffer);
    }

    return result;
}
OpenIPC_Error CProbePlugin::PPI_StatePortGetDefinitions(OpenIPC_DeviceId deviceId, const PPI_InterfaceStatePortDefinition** definitions, uint32_t definitionsSize, uint32_t* numberOfDefinitions)
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (OpenIPC_PASS(result) && definitions == nullptr && definitionsSize > 0)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: definitions is NULL, but definitionsSize is non-zero");
    }
    std::shared_ptr<InterfaceInstanceStatePort> probeinterface;
    if (OpenIPC_PASS(result))
    {
        result = _LookupInterface(deviceId, probeinterface);
    }
    if (OpenIPC_PASS(result))
    {
        result = probeinterface->GetDefinitions(definitions, definitionsSize, numberOfDefinitions);
    }

    return result;
}
OpenIPC_Error CProbePlugin::PPI_StatePortGetName(const PPI_InterfaceStatePortDefinition* definition, char* buffer, uint32_t bufferSize, uint32_t* realSize)
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (OpenIPC_PASS(result) && definition == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: definition is NULL");
    }
    if (OpenIPC_PASS(result) && buffer == nullptr && bufferSize > 0)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: buffer is NULL, but bufferSize is non-zero");
    }

    if (OpenIPC_PASS(result))
    {
        std::string const& name = definition->name;
        uint32_t size = static_cast<uint32_t>(name.length()) + 1;
        if (realSize != nullptr)
        {
            *realSize = size;
        }

        if (size > bufferSize)
        {
            size = bufferSize;
        }

        if (buffer != nullptr && bufferSize > 0)
        {
            InternalUtils::stringcopy(buffer, size, name.c_str());
        }
    }

    return result;
}
OpenIPC_Error CProbePlugin::PPI_StatePortGetDescription(const PPI_InterfaceStatePortDefinition* definition, char* buffer, uint32_t bufferSize, uint32_t* realSize)
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (OpenIPC_PASS(result) && definition == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: definition is NULL");
    }
    if (OpenIPC_PASS(result) && buffer == nullptr && bufferSize > 0)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: buffer is NULL, but bufferSize is non-zero");
    }
    if (OpenIPC_PASS(result))
    {
        std::string const& description = definition->description;
        uint32_t size = static_cast<uint32_t>(description.length()) + 1;
        if (realSize != nullptr)
        {
            *realSize = size;
        }

        if (size > bufferSize)
        {
            size = bufferSize;
        }

        if (buffer != nullptr && bufferSize > 0)
        {
            InternalUtils::stringcopy(buffer, size, description.c_str());
        }
    }

    return result;
}
OpenIPC_Error CProbePlugin::PPI_StatePortGetCapabilites(const PPI_InterfaceStatePortDefinition* definition, PPI_StatePort_OPERATION_TYPE_ET* capabilities)
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (OpenIPC_PASS(result) && definition == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: definition is NULL");
    }
    if (OpenIPC_PASS(result) && capabilities == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: capabilities is NULL");
    }
    if (OpenIPC_PASS(result))
    {
        *capabilities = definition->capabilities;
    }

    return result;
}
OpenIPC_Error CProbePlugin::PPI_StatePortGetAccessSize(const PPI_InterfaceStatePortDefinition* definition, uint32_t* size)
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (OpenIPC_PASS(result) && definition == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: definition is NULL");
    }
    if (OpenIPC_PASS(result) && size == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: size is NULL");
    }
    if (OpenIPC_PASS(result))
    {
        *size = definition->accessSize;
    }

    return result;
}
OpenIPC_Error CProbePlugin::PPI_StatePortGetParameters(const PPI_InterfaceStatePortDefinition* definition, const PPI_InterfaceStatePortParameter** parameters, uint32_t length, uint32_t* realLength)
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (OpenIPC_PASS(result) && definition == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: definition is NULL");
    }
    if (OpenIPC_PASS(result) && parameters == nullptr && length > 0)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: parameters is NULL, but length is non-zero");
    }
    if (OpenIPC_PASS(result) && realLength != nullptr)
    {
        *realLength = static_cast<uint32_t>(definition->parameters.size());
    }
    if (OpenIPC_PASS(result) && length > 0)
    {
        const uint32_t copylength = std::min(static_cast<uint32_t>(definition->parameters.size()), length) * sizeof(const PPI_InterfaceStatePortParameter*);

        CommonUtils::MemoryCopy(parameters, length * sizeof(const PPI_InterfaceStatePortParameter*), &(definition->parameters[0]), copylength);
    }

    return result;
}
OpenIPC_Error CProbePlugin::PPI_StatePortParameterGetName(const PPI_InterfaceStatePortParameter* parameter, char* buffer, uint32_t bufferSize, uint32_t* realSize)
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (OpenIPC_PASS(result) && parameter == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: parameter is NULL");
    }
    if (OpenIPC_PASS(result) && buffer == nullptr && bufferSize > 0)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: buffer is NULL, but bufferSize is non-zero");
    }
    if (OpenIPC_PASS(result))
    {
        std::string const& name = parameter->name;
        uint32_t size = static_cast<uint32_t>(name.length()) + 1;
        if (realSize != nullptr)
        {
            *realSize = size;
        }

        if (size > bufferSize)
        {
            size = bufferSize;
        }

        if (buffer != nullptr && bufferSize > 0)
        {
            InternalUtils::stringcopy(buffer, size, name.c_str());
        }
    }

    return result;
}
OpenIPC_Error CProbePlugin::PPI_StatePortParameterGetDescription(const PPI_InterfaceStatePortParameter* parameter, char* buffer, uint32_t bufferSize, uint32_t* realSize)
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (OpenIPC_PASS(result) && parameter == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: parameter is NULL");
    }
    if (OpenIPC_PASS(result) && buffer == nullptr && bufferSize > 0)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: buffer is NULL, but bufferSize is non-zero");
    }
    if (OpenIPC_PASS(result))
    {
        std::string const& description = parameter->description;
        uint32_t size = static_cast<uint32_t>(description.length()) + 1;
        if (realSize != nullptr)
        {
            *realSize = size;
        }

        if (size > bufferSize)
        {
            size = bufferSize;
        }

        if (buffer != nullptr && bufferSize > 0)
        {
            InternalUtils::stringcopy(buffer, size, description.c_str());
        }
    }

    return result;
}
OpenIPC_Error CProbePlugin::PPI_StatePortParameterGetSize(const PPI_InterfaceStatePortParameter* parameter, uint32_t* size)
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (OpenIPC_PASS(result) && parameter == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: parameter is NULL");
    }
    if (OpenIPC_PASS(result) && size == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: size is NULL");
    }
    if (OpenIPC_PASS(result))
    {
        *size = parameter->size;
    }

    return result;
}
OpenIPC_Error CProbePlugin::PPI_StatePortParameterGetDefaultValue(const PPI_InterfaceStatePortParameter* parameter, uint8_t* buffer, uint32_t size)
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (OpenIPC_PASS(result) && parameter == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: parameter is NULL");
    }
    if (OpenIPC_PASS(result) && !(parameter->optional))
    {
        result = OpenIPC_Error_Not_Supported;
    }
    if (OpenIPC_PASS(result) && buffer == nullptr && size > 0)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: buffer is NULL, but size is non-zero");
    }
    if (OpenIPC_PASS(result) && size > 0)
    {
        uint32_t parameterByteSize = (parameter->size + 7) / 8;
        if (size >= parameterByteSize)
        {
            CommonUtils::MemoryCopy(buffer, size, parameter->pDefaultValue, parameterByteSize);
        }
        else
        {
            result = OpenIPC_Error_Invalid_Argument;
            PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Invalid Argument: size too small to hold the parameter");
        }
    }

    return result;
}
OpenIPC_Error CProbePlugin::PPI_StatePortErrorGetName(const PPI_InterfaceStatePortDefinition* definition, ProbeStatePortError error, char* buffer, uint32_t bufferSize, uint32_t* realSize)
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (OpenIPC_PASS(result) && definition == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: definition is NULL");
    }
    if (OpenIPC_PASS(result) && buffer == nullptr && bufferSize > 0)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: buffer is NULL, but bufferSize is non-zero");
    }
    if (OpenIPC_PASS(result))
    {
        definition->ErrorGetName(error, buffer, bufferSize, realSize);
    }

    return result;
}
OpenIPC_Error CProbePlugin::PPI_StatePortErrorGetDescription(const PPI_InterfaceStatePortDefinition* definition, ProbeStatePortError error, char* buffer, uint32_t bufferSize, uint32_t* realSize)
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (OpenIPC_PASS(result) && definition == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: definition is NULL");
    }
    if (OpenIPC_PASS(result) && buffer == nullptr && bufferSize > 0)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: buffer is NULL, but bufferSize is non-zero");
    }
    if (OpenIPC_PASS(result))
    {
        definition->ErrorGetDescription(error, buffer, bufferSize, realSize);
    }

    return result;
}
OpenIPC_Error CProbePlugin::PPI_StatePortErrorGetOpenIpcError(const PPI_InterfaceStatePortDefinition* definition, ProbeStatePortError error, OpenIPC_Error* openIpcError)
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (OpenIPC_PASS(result) && definition == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: definition is NULL");
    }
    if (OpenIPC_PASS(result) && openIpcError == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: openIpcError is NULL");
    }
    if (OpenIPC_PASS(result))
    {
        definition->ErrorGetOpenIpcError(error, openIpcError);
    }

    return result;
}
OpenIPC_Error CProbePlugin::PPI_StatePortEventGetValueChangedEventDescription(OpenIPC_DeviceId deviceId, PPI_InterfaceStatePortValueChangedEventDescription const** eventDescriptionPtr)
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (OpenIPC_PASS(result) && eventDescriptionPtr == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Null Pointer: eventDescriptionPtr is NULL");
    }
    std::shared_ptr<InterfaceInstanceStatePort> probeinterface;
    if (OpenIPC_PASS(result))
    {
        result = _LookupInterface(deviceId, probeinterface);
    }
    if (OpenIPC_PASS(result))
    {
        result = probeinterface->GetValueChangedEventDescription(eventDescriptionPtr);
    }

    return result;
}

//////////////////////////////////////////////////////////////////////////////
// InterfaceGetConfig
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::InterfaceGetConfig(
    OpenIPC_DeviceId interfaceID,
    const char* configType,
    PPI_char value[PPI_MAX_INFO_LEN]
)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    if (value == NULL){
        openIPCError = OpenIPC_Error_Probe_Invalid_Parameter;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "Invalid parameter: value is NULL");
        return openIPCError;
    }

    std::shared_ptr<InterfaceInstance> probeinterface;
    openIPCError = _LookupInterface(interfaceID, probeinterface);
    if (OpenIPC_PASS(openIPCError))
    {
        std::lock_guard<std::recursive_mutex> _(probeinterface->_instanceMutex);
        InternalUtils::stringcopy(value,PPI_MAX_INFO_LEN,probeinterface->settings.GetString(configType).c_str());
    }
    else
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "Invalid device ID requested (" << LOG_UINT32_HEX((uint32_t)interfaceID) << ")");
    }

    return openIPCError;
}

//////////////////////////////////////////////////////////////////////////////
// InterfacePortRead
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::InterfacePortRead(OpenIPC_DeviceId interfaceID, uint8_t* output, uint32_t maxoutputbytes, uint32_t* outputbytes)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
    std::shared_ptr<InterfaceInstance> probeinterface;
    openIPCError = _LookupInterface(interfaceID, probeinterface);

    if (OpenIPC_PASS(openIPCError))
    {
        std::lock_guard<std::recursive_mutex> _(probeinterface->_instanceMutex);
        openIPCError = _PortRead(*probeinterface, output, maxoutputbytes, outputbytes);
    }
    else
    {
        std::stringstream ss;
        ss << "Invalid device ID requested (" << LOG_UINT32_HEX((uint32_t)interfaceID) << ")";
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, ss.str());
    }

    return openIPCError;
}

//////////////////////////////////////////////////////////////////////////////
// InterfacePortWrite
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::InterfacePortWrite(OpenIPC_DeviceId interfaceID, const uint8_t* input, uint32_t maxinputbytes, uint32_t* inputbytes)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    std::shared_ptr<InterfaceInstance> probeinterface;
    openIPCError = _LookupInterface(interfaceID, probeinterface);
    if (OpenIPC_PASS(openIPCError))
    {
        std::lock_guard<std::recursive_mutex> _(probeinterface->_instanceMutex);
        openIPCError = _PortWrite(*probeinterface, input, maxinputbytes, inputbytes);
    }
    else
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "Invalid device ID requested (" << LOG_UINT32_HEX((uint32_t)interfaceID) << ")");
    }

    return openIPCError;
}

//////////////////////////////////////////////////////////////////////////////
// InterfacePortOpenWindow
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::InterfacePortOpenWindow(OpenIPC_DeviceId interfaceID, PPI_Trace_PortAccessMode accessMode)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    std::shared_ptr<InterfaceInstance> probeinterface;
    openIPCError = _LookupInterface(interfaceID, probeinterface);
    if (OpenIPC_PASS(openIPCError))
    {
        std::lock_guard<std::recursive_mutex> _(probeinterface->_instanceMutex);
        openIPCError = _PortOpenWindow(*probeinterface, accessMode);
    }
    else
    {
        std::stringstream ss;
        ss << "Invalid device ID requested (" << LOG_UINT32_HEX((uint32_t)interfaceID) << ")";
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, ss.str());
    }

    return openIPCError;
}

//////////////////////////////////////////////////////////////////////////////
// InterfacePortCloseWindow
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::InterfacePortCloseWindow(OpenIPC_DeviceId interfaceID)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    std::shared_ptr<InterfaceInstance> probeinterface;
    openIPCError = _LookupInterface(interfaceID, probeinterface);
    if (OpenIPC_PASS(openIPCError))
    {
        std::lock_guard<std::recursive_mutex> _(probeinterface->_instanceMutex);
        openIPCError = _PortCloseWindow(*probeinterface);
    }
    else
    {
        std::stringstream ss;
        ss << "Invalid device ID requested (" << LOG_UINT32_HEX((uint32_t)interfaceID) << ")";
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, ss.str());
    }

    return openIPCError;
}

//////////////////////////////////////////////////////////////////////////////
// InterfacePortIsReadDataAvailable
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::InterfacePortIsReadDataAvailable(OpenIPC_DeviceId interfaceID, bool* isDataAvailable)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    std::shared_ptr<InterfaceInstance> probeinterface;
    openIPCError = _LookupInterface(interfaceID, probeinterface);
    if (OpenIPC_PASS(openIPCError))
    {
        std::lock_guard<std::recursive_mutex> _(probeinterface->_instanceMutex);
        openIPCError = _PortIsReadDataAvailable(*probeinterface, isDataAvailable);
    }
    else
    {
        std::stringstream ss;
        ss << "Invalid device ID requested (" << LOG_UINT32_HEX((uint32_t)interfaceID) << ")";
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, ss.str());
    }

    return openIPCError;
}

//////////////////////////////////////////////////////////////////////////////
// InterfacePortIsWindowOpen
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::InterfacePortIsWindowOpen(OpenIPC_DeviceId interfaceID, bool* isWindowOpen)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    std::shared_ptr<InterfaceInstance> probeinterface;
    openIPCError = _LookupInterface(interfaceID, probeinterface);
    if (OpenIPC_PASS(openIPCError))
    {
        std::lock_guard<std::recursive_mutex> _(probeinterface->_instanceMutex);
        openIPCError = _PortIsWindowOpen(*probeinterface, isWindowOpen);
    }
    else
    {
        std::stringstream ss;
        ss << "Invalid device ID requested (" << LOG_UINT32_HEX((uint32_t)interfaceID) << ")";

        openIPCError = OpenIPC_Error_Invalid_Device_ID;

        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, ss.str());
    }

    return openIPCError;
}


//////////////////////////////////////////////////////////////////////////////
// PluginRegisterEventHandler
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::PluginRegisterEventHandler(
    PluginEventCallbackHandler eventHandlerFunction
    )
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
    std::lock_guard<std::recursive_mutex> _(_pluginMutex);
    if (_eventHandlerFunction == nullptr)
    {
        _eventHandlerFunction = eventHandlerFunction;
    }
    else
    {
        openIPCError = OpenIPC_Error_Callback_Already_Initialized;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
    }

    return openIPCError;
}

bool CProbePlugin::_eventCanBeReliedOn(const std::shared_ptr<ProbeInstance>& probe, PPI_PluginEvent pluginEvent) const
{
    PPI_LOG(OpenIPC_INVALID_DEVICE_ID, PPI_traceNotification,"Note: base class method called on _eventCanBeReliedOn, so no plugin event can be assumed to be relied on. Most plugins will likely wish to override this!");
    // Making true until probe plugins are updated to override this.
    return true;
}

OpenIPC_Error CProbePlugin::PPI_PluginEventGenerated(OpenIPC_DeviceId probe, PPI_PluginEvent pluginEvent, PPI_bool* pluginGeneratesEvent) const
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;
    if (OpenIPC_PASS(result) && pluginGeneratesEvent == nullptr)
    {
        result = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "pluginGeneratesEvent must not be NULL in PPI_PluginEventGenerated");
    }
    std::shared_ptr<ProbeInstance> _probe;
    if (OpenIPC_PASS(result))
    {
        result = _LookupProbe(probe,_probe);
    }
    if (OpenIPC_PASS(result))
    {
        *pluginGeneratesEvent = _eventCanBeReliedOn(_probe, pluginEvent) ? 1 : 0;
    }

    return result;
}

//////////////////////////////////////////////////////////////////////////////
// PluginRegisterNotificationHandler
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::PluginRegisterNotificationHandler(PluginNotificationCallbackHandler notificationHandlerFunction)
{
    std::lock_guard<std::recursive_mutex> _(_pluginMutex);
    return PluginNotifications::GetInstance().Handler(notificationHandlerFunction);
}

OpenIPC_Error CProbePlugin::PluginSetLogEventHandler(PluginLogCallbackHandler logHandlerFunction)
{
    std::lock_guard<std::recursive_mutex> _(_pluginMutex);
    return PluginLogger::GetInstance().Handler(logHandlerFunction);
}

 OpenIPC_Error CProbePlugin::PPI_PluginSetStreamLogEventHandler(PluginStreamLogCallbackHandler logCallBackFunction)
 {
    std::lock_guard<std::recursive_mutex> _(_pluginMutex);
    return PluginLogger::GetInstance().Handler(logCallBackFunction);
 }

OpenIPC_Error CProbePlugin::PPI_PluginUsesLoggerStream( PPI_Stream stream, PPI_bool* usesLoggerStream)
{
    std::lock_guard<std::recursive_mutex> _(_pluginMutex);
    if (usesLoggerStream != nullptr)
    {
        *usesLoggerStream =  stream == PPI_INTERNAL_STREAM;
        return OpenIPC_Error_No_Error;
    }
    PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID,OpenIPC_Error_Probe_Invalid_Parameter,"usesLoggerStream is null");
    return OpenIPC_Error_Probe_Invalid_Parameter;
}

OpenIPC_Error CProbePlugin::PPI_PluginSetNotificationRequiredForStreamLogCallBack( PPI_Stream stream, PPI_NotificationLevel level)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
    std::lock_guard<std::recursive_mutex> _(_pluginMutex);
    PluginLogger::GetInstance().Level(level,stream);
    return openIPCError;
}


//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
// internal functions
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// _SendEvent
//////////////////////////////////////////////////////////////////////////////
void CProbePlugin::_SendEvent(
    OpenIPC_DeviceId deviceId,
    PPI_PluginEvent pluginEvent,
    uint64_t value
    )
{
    // if an event handler is registered
    if (_eventHandlerFunction != 0)
    {
        _eventHandlerFunction(deviceId, pluginEvent, value);
    }
    // else drop it (maybe log for posterity?)
    // TBD: should we pend events (up to a certain count) that happen in the window between plugin init and the time an event handler is registered?
}

//////////////////////////////////////////////////////////////////////////////
// _SendNotification
//////////////////////////////////////////////////////////////////////////////
void CProbePlugin::_SendNotification(
    OpenIPC_DeviceId deviceId,
    PPI_NotificationLevel level,
    OpenIPC_Error error,
    const char* message
    )
{
    PluginNotifications::GetInstance().Notify(deviceId, error, level, [&](std::basic_ostream<char>& stream) { stream << message; });
}

//////////////////////////////////////////////////////////////////////////////
// _GetProbes
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::_GetProbes(
    std::vector<std::weak_ptr<ProbeInstance>>& probes,
    const char* vendorinit
    )
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    // return an empty list for the base implementation (perhaps this should be a pure virtual function, and derivatives must override?)
    probes.clear();

    return openIPCError;
}

//////////////////////////////////////////////////////////////////////////////
// _GetProbeInterfaces
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::_GetProbeInterfaces(
    std::shared_ptr<ProbeInstance> probe,
    std::vector<std::weak_ptr<InterfaceInstance>>& probeinterfaces)
{
    return probe->GetInterfaces(probeinterfaces);
}

//////////////////////////////////////////////////////////////////////////////
// _JtagScan
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::_JtagScan(
    InterfaceInstance& /*interfaceInstance*/,
    const uint32_t* /*input*/,
    uint32_t /*inputdwords*/,
    uint32_t* /*output*/,
    uint32_t /*maxoutputdwords*/,
    uint32_t* /*outputdwords*/
    )
{
    auto result = OpenIPC_Error_Not_Implemented;
    POST_ERROR_MESSAGE(result, "_JtagScan is unimplemented in the current plugin.");
    return result;
}


OpenIPC_Error CProbePlugin::_BundleExecuteMultiChain(ProbePluginBundle* bundle, std::vector<std::shared_ptr<InterfaceInstance>>& interfaces, PPI_bool keepLock)
{
    (void)bundle;
    (void)interfaces;
    (void)keepLock;
    auto result = OpenIPC_Error_Not_Implemented;
    POST_ERROR_MESSAGE(result, "_BundleExecuteMultiChain is unimplemented in the current plugin.");
    return result;
}

//////////////////////////////////////////////////////////////////////////////
// _I2cScan
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::_I2cScan(
    InterfaceInstance& /*interfaceInstance*/,
    const uint32_t* /*input*/,
    uint32_t /*inputdwords*/,
    uint32_t* /*output*/,
    uint32_t /*maxoutputdwords*/,
    uint32_t* /*outputdwords*/
    )
{
    auto result = OpenIPC_Error_Not_Implemented;
    POST_ERROR_MESSAGE(result, "_I2cScan is unimplemented in the current plugin.");
    return result;
}

//////////////////////////////////////////////////////////////////////////////
// _PinsScan
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::_PinsScan(
    InterfaceInstance& /*interfaceInstance*/,
    const uint32_t* /*input*/,
    uint32_t /*inputdwords*/,
    uint32_t* /*output*/,
    uint32_t /*maxoutputdwords*/,
    uint32_t* /*outputdwords*/
    )
{
    auto result = OpenIPC_Error_Not_Implemented;
    POST_ERROR_MESSAGE(result, "_PinsScan is unimplemented in the current plugin.");
    return result;
}

//////////////////////////////////////////////////////////////////////////////
// _PortRead
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::_PortRead(
    InterfaceInstance& /*interfaceInstance*/,
    uint8_t* /*output*/,
    uint32_t /*maxoutputbytes*/,
    uint32_t* /*outputbytes*/
    )
{
    return OpenIPC_Error_Not_Implemented;
}

//////////////////////////////////////////////////////////////////////////////
// _PortWrite
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::_PortWrite(
    InterfaceInstance& /*interfaceInstance*/,
    const uint8_t* /*input*/,
    uint32_t /*maxinputbytes*/,
    uint32_t* /*inputbytes*/
    )
{
    return OpenIPC_Error_Not_Implemented;
}

//////////////////////////////////////////////////////////////////////////////
// _PortOpenWindow
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::_PortOpenWindow(
    InterfaceInstance& /*interfaceInstance*/,
    PPI_Trace_PortAccessMode /*accessMode*/
    )
{
    return OpenIPC_Error_Not_Implemented;
}

//////////////////////////////////////////////////////////////////////////////
// _PortCloseWindow
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::_PortCloseWindow(
    InterfaceInstance& /*interfaceInstance*/
    )
{
    return OpenIPC_Error_Not_Implemented;
}

//////////////////////////////////////////////////////////////////////////////
// _PortIsReadDataAvailable
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::_PortIsReadDataAvailable(
    InterfaceInstance& /*interfaceInstance*/,
    bool* /*isDataAvailable*/
    )
{
    return OpenIPC_Error_Not_Implemented;
}

//////////////////////////////////////////////////////////////////////////////
// _PortIsWindowOpen
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error CProbePlugin::_PortIsWindowOpen(
    InterfaceInstance& /*interfaceInstance*/,
    bool* /*isWindowOpen*/
    )
{
    return OpenIPC_Error_Not_Implemented;
}

OpenIPC_Error CProbePlugin::_ProbeLockHold(
    OpenIPC_DeviceId deviceInterface
    )
{
    std::lock_guard<std::recursive_mutex> lock(_pluginMutex);
    auto it = this->_lockedInterfaces.find(deviceInterface);
    if (it == this->_lockedInterfaces.end())
    {
        this->_lockedInterfaces.insert(deviceInterface);
    }
    this->_inLock = true;
    return OpenIPC_Error_No_Error;
}
OpenIPC_Error CProbePlugin::_ProbeLockRelease(
    OpenIPC_DeviceId deviceInterface
    )
{
    std::lock_guard<std::recursive_mutex> lock(_pluginMutex);
    auto it = this->_lockedInterfaces.find(deviceInterface);

    if (it != this->_lockedInterfaces.end())
    {
        this->_lockedInterfaces.erase(deviceInterface);
    }
    else
    {
        return this->PostError(OpenIPC_Error_Invalid_Device_ID, "Invalid probe handle passed to BundleExecute Framework method.");
    }

    // set false when there are no more locked interfaces?
    if (this->_lockedInterfaces.size() == 0)
    {
        this->_inLock = false;
    }

    return OpenIPC_Error_No_Error;
}

//////////////////////////////////////////////////////////////////////////////
// CProbePlugin constructor
//////////////////////////////////////////////////////////////////////////////
CProbePlugin::CProbePlugin(const char* plugin_name, bool supportBundles) :
    _pluginname(plugin_name),
    _supportBundles(supportBundles),
    _plugin_refid(static_cast<uint32_t>(-1)),
    _eventHandlerFunction(nullptr),
    _currentLockedInterface(OpenIPC_INVALID_DEVICE_ID),
    _inLock(false)
{
}

std::unique_ptr<IInterfaceBundle> CProbePlugin::CreateInterfaceBundle(PPI_EInterfaceType interfaceType)
{
    // This method should be overriden with what is needed
    return std::unique_ptr<IInterfaceBundle>(nullptr);
}

OpenIPC_Error CProbePlugin::GetInterfaceBundle(PPI_EInterfaceType interfaceType, PPI_ProbeBundleHandle handle, IInterfaceBundle*& interfaceBundle)
{
    OpenIPC_Error result = OpenIPC_Error_No_Error;

    interfaceBundle = &IInterfaceBundle::Default;
    if (OpenIPC_PASS(result))
    {
        if (handle == PPI_PROBE_LOCK_RELEASE || handle == PPI_PROBE_LOCK_HOLD)
        {
            interfaceBundle = &IInterfaceBundle::ExecuteImmediately;
        }
        else
        {
            auto probeBundle = reinterpret_cast<ProbePluginBundle*>(handle);
            interfaceBundle = &(probeBundle->InterfaceBundle());
            if (*interfaceBundle == IInterfaceBundle::Default)
            {
                std::unique_ptr<IInterfaceBundle> newInterfaceBundle = this->CreateInterfaceBundle(interfaceType);
                if (newInterfaceBundle)
                {
                    probeBundle->InterfaceBundle(std::move(newInterfaceBundle));
                    interfaceBundle = &probeBundle->InterfaceBundle();
                }
                else
                {
                    result = OpenIPC_Error_Internal_Error;
                    PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "Internal Error: interface could not allocate new bundle contents");
                }
            }
        }
    }
    return result;
}

OpenIPC_Error CProbePlugin::_ProbeBeginInitialization(PPI_RefId probe_refid, OpenIPC_DeviceId probeID)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    std::lock_guard<std::recursive_mutex> lock(_pluginMutex);

    auto iter = _refid2ProbeInstance.find(probe_refid);
    if (iter == _refid2ProbeInstance.end())
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
        return openIPCError;
    }
    auto probe = iter->second.lock();
    if (!probe)
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
        return openIPCError;
    }
    if (probe->initializationState != InitializationState::Uninitialized)
    {
        openIPCError = OpenIPC_Error_Probe_Already_Initialized;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
        return openIPCError;
    }

    _deviceid2ProbeInstance[probeID] = probe;
    probe->deviceID = probeID;
    probe->initializationState = InitializationState::Initalizing;

    return openIPCError;
}

OpenIPC_Error CProbePlugin::_ProbeSpecifyVendorInitString(OpenIPC_DeviceId probeID, const PPI_char * vendorinit)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    std::lock_guard<std::recursive_mutex> lock(_pluginMutex);

    auto iter = _deviceid2ProbeInstance.find(probeID);
    if (iter == _deviceid2ProbeInstance.end())
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
        return openIPCError;
    }
    auto probe = iter->second.lock();
    if (!probe)
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
        return openIPCError;
    }
    if (probe->initializationState == InitializationState::Uninitialized)
    {
        openIPCError = OpenIPC_Error_Probe_Not_Initialized;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
        return openIPCError;
    }
    if (probe->initializationState == InitializationState::Initialized)
    {
        openIPCError = OpenIPC_Error_Probe_Already_Initialized;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
        return openIPCError;
    }

    return openIPCError;
}

OpenIPC_Error CProbePlugin::_ProbeFinishInitialization(OpenIPC_DeviceId probeID)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    std::lock_guard<std::recursive_mutex> lock(_pluginMutex);

    auto iter = _deviceid2ProbeInstance.find(probeID);
    if (iter == _deviceid2ProbeInstance.end())
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
        return openIPCError;
    }
    auto probe = iter->second.lock();
    if (!probe)
    {
        openIPCError = OpenIPC_Error_Invalid_Device_ID;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
        return openIPCError;
    }
    if (probe->initializationState == InitializationState::Initialized)
    {
        openIPCError = OpenIPC_Error_Probe_Already_Initialized;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
        return openIPCError;
    }

    probe->initializationState = InitializationState::Initialized;
    std::vector<std::weak_ptr<InterfaceInstance>> probeinterfaces;
    openIPCError = _GetProbeInterfaces(probe, probeinterfaces);
    for (auto& probeInterface_ref : probeinterfaces)
    {
        auto probeInterface = probeInterface_ref.lock();
        if (!probeInterface)
        {
            openIPCError = OpenIPC_Error_Invalid_Device;
        }
        else
        {
            openIPCError = probe->AddInterface(probeInterface);
        }
        if (!OpenIPC_PASS(openIPCError))
        {
            break;
        }
    }

    return openIPCError;
}

OpenIPC_Error CProbePlugin::PPI_JTAG_PinsShift(PPI_ProbeBundleHandle handle,
    uint32_t shiftLengthBits, const uint8_t* const tdi,
    const uint8_t* const tms, uint8_t* tdo,
    const PPI_JTAG_PinsOptions* const options)
{
    IInterfaceBundle* interfaceBundle = &IInterfaceBundle::Default;
    OpenIPC_Error result = this->GetInterfaceBundle(PPI_interfaceTypeJtag, handle, interfaceBundle);

    if (OpenIPC_PASS(result))
    {
        result = this->_AppendJtagPinsOperationToBundle(*interfaceBundle, shiftLengthBits, tdi, tms, tdo, options);
    }

    return result;
}

OpenIPC_Error CProbePlugin::_AppendJtagPinsOperationToBundle(IInterfaceBundle& handle,
    uint32_t shiftLengthBits, const uint8_t* const tdi,
    const uint8_t* const tms, uint8_t* tdo,
    const PPI_JTAG_PinsOptions* const options)
{
    auto result = OpenIPC_Error_Not_Implemented;
    PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "_AppendJtagPinsOperationToBundle is unimplemented and must be overridden in the current class");
    return result;
}

OpenIPC_Error CProbePlugin::PPI_JTAG_GoToState(PPI_ProbeBundleHandle handle, JtagStateEncode gotoState,
    uint32_t numberOfClocksInState, const PPI_JTAG_StateGotoOptions* const options)
{
    IInterfaceBundle* interfaceBundle = &IInterfaceBundle::Default;
    OpenIPC_Error result = this->GetInterfaceBundle(PPI_interfaceTypeJtag, handle, interfaceBundle);

    if (OpenIPC_PASS(result))
    {
        result = this->_AppendJtagGoToStateOperationToBundle(*interfaceBundle, gotoState, numberOfClocksInState, options);
    }

    return result;
}

OpenIPC_Error CProbePlugin::_AppendJtagGoToStateOperationToBundle(IInterfaceBundle& handle,
    JtagStateEncode gotoState,
    uint32_t numberOfClocksInState,
    const PPI_JTAG_StateGotoOptions* const options)
{
    auto result = OpenIPC_Error_Not_Implemented;
    PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "_AppendJtagGoToStateOperationToBundle is unimplemented and must be overridden in the current class");
    return result;
}

OpenIPC_Error CProbePlugin::PPI_JTAG_StateIRShift(PPI_ProbeBundleHandle handle, uint32_t shiftLengthBits, const uint8_t* const inBits, uint8_t* outBits, const PPI_JTAG_StateShiftOptions* const options)
{
    IInterfaceBundle* interfaceBundle = &IInterfaceBundle::Default;
    OpenIPC_Error result = this->GetInterfaceBundle(PPI_interfaceTypeJtag, handle, interfaceBundle);

    if (OpenIPC_PASS(result))
    {
        result = this->_AppendJtagStateIRShiftOperationToBundle(*interfaceBundle, shiftLengthBits, inBits, outBits, options);
    }

    return result;
}

OpenIPC_Error CProbePlugin::_AppendJtagStateIRShiftOperationToBundle(IInterfaceBundle& handle,
    uint32_t shiftLengthBits,
    const uint8_t* const inBits,
    uint8_t* outBits,
    const PPI_JTAG_StateShiftOptions* const options)
{
    auto result = OpenIPC_Error_Not_Implemented;
    PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "_AppendJtagStateIRShiftOperationToBundle is unimplemented and must be overridden in the current class");
    return result;
}

OpenIPC_Error CProbePlugin::PPI_JTAG_StateDRShift(PPI_ProbeBundleHandle handle, uint32_t shiftLengthBits, const uint8_t* const inBits, uint8_t* outBits, const PPI_JTAG_StateShiftOptions* const options)
{
    IInterfaceBundle* interfaceBundle = &IInterfaceBundle::Default;
    OpenIPC_Error result = this->GetInterfaceBundle(PPI_interfaceTypeJtag, handle, interfaceBundle);

    if (OpenIPC_PASS(result))
    {
        result = this->_AppendJtagStateDRShiftOperationToBundle(*interfaceBundle, shiftLengthBits, inBits, outBits, options);
    }

    return result;
}

OpenIPC_Error CProbePlugin::PPI_Bundle_InterfaceSetConfig(PPI_ProbeBundleHandle handle,
    const char* configType,
    const char* value)
{
    IInterfaceBundle* interfaceBundle = &IInterfaceBundle::Default;
    OpenIPC_Error result = this->GetInterfaceBundle(PPI_interfaceTypeJtag, handle, interfaceBundle);

    if (OpenIPC_PASS(result))
    {
        result = this->_AppendJtagSetConfigToBundle(*interfaceBundle, configType, value);
    }

    return result;
}

OpenIPC_Error CProbePlugin::PPI_MemoryRead(PPI_ProbeBundleHandle bundle, uint64_t address, uint8_t* buffer, uint32_t* size)
{
    return OpenIPC_Error_Operation_Not_Supported;
}

OpenIPC_Error CProbePlugin::PPI_MemoryWrite(PPI_ProbeBundleHandle bundle, uint64_t address, const uint8_t* buffer, uint32_t* size)
{
    return OpenIPC_Error_Operation_Not_Supported;
}

OpenIPC_Error CProbePlugin::_AppendJtagStateDRShiftOperationToBundle(IInterfaceBundle& handle,
    uint32_t shiftLengthBits,
    const uint8_t* const inBits,
    uint8_t* outBits,
    const PPI_JTAG_StateShiftOptions* const options)
{
    auto result = OpenIPC_Error_Not_Implemented;
    PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result, "_AppendJtagStateDRShiftOperationToBundle is unimplemented and must be overridden in the current class");
    return result;
}

OpenIPC_Error CProbePlugin::_AppendJtagSetConfigToBundle(IInterfaceBundle& interfaceBundle,
    const char* configType, const char* value)
{
    return OpenIPC_Error_Operation_Not_Supported;
}

OpenIPC_Error CProbePlugin::GetInterfaceJTAGPadding(OpenIPC_DeviceId device,
                                                         uint32_t* irPaddingNearTDI,
                                                         uint32_t* irPaddingNearTDO,
                                                         uint32_t* drPaddingNearTDI,
                                                         uint32_t* drPaddingNearTDO,
                                                         PPI_bool* drValueConstantOne
                                                         )
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    if (irPaddingNearTDI == nullptr ||
        irPaddingNearTDO == nullptr ||
        drPaddingNearTDI == nullptr ||
        drPaddingNearTDO == nullptr ||
        drValueConstantOne == nullptr)
    {
        openIPCError = OpenIPC_Error_Null_Pointer;
        PPI_ERROR_WITH_MESSAGE(device, openIPCError, "Invalid pointer passed in to PPI_JTAG_GetInterfacePadding.");
    }
    std::shared_ptr<InterfaceInstanceJtag> probeinterface;
    if (OpenIPC_PASS(openIPCError))
    {
        openIPCError = _LookupInterface(device, probeinterface);
    }

    if (OpenIPC_PASS(openIPCError))
    {
        std::lock_guard<std::recursive_mutex> _(probeinterface->_instanceMutex);
        *irPaddingNearTDI = probeinterface->irPaddingNearTDI;
        *irPaddingNearTDO = probeinterface->irPaddingNearTDO;
        *drPaddingNearTDI = probeinterface->drPaddingNearTDI;
        *drPaddingNearTDO = probeinterface->drPaddingNearTDO;
        *drValueConstantOne = probeinterface->drValueConstantOne;
    }
    return openIPCError;
}
OpenIPC_Error CProbePlugin::SetInterfaceJTAGPadding( OpenIPC_DeviceId device,
                                                         uint32_t irPaddingNearTDI,
                                                         uint32_t irPaddingNearTDO,
                                                         uint32_t drPaddingNearTDI,
                                                         uint32_t drPaddingNearTDO,
                                                         PPI_bool drValueConstantOne
                                                         )
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
    std::shared_ptr<InterfaceInstanceJtag> probeinterface;
    openIPCError = _LookupInterface(device, probeinterface);
    if (OpenIPC_PASS(openIPCError))
    {
         std::lock_guard<std::recursive_mutex> _(probeinterface->_instanceMutex);
         probeinterface->irPaddingNearTDI = irPaddingNearTDI;
         probeinterface->irPaddingNearTDO = irPaddingNearTDO;
         probeinterface->drPaddingNearTDI = drPaddingNearTDI;
         probeinterface->drPaddingNearTDO = drPaddingNearTDO;
         probeinterface->drValueConstantOne = drValueConstantOne;
    }
    return openIPCError;
}


ErrorContext::ErrorContext()
{
    _threadErrorContext = this;
}

ErrorContext::~ErrorContext()
{
    OpenIPC_Error error = OpenIPC_Error_No_Error;
    if (!_errorStack.empty())
    {
        error = _errorStack.back().first;
    }

    if (error != OpenIPC_Error_No_Error)
    {
        _threadLastErrorMessages = std::make_pair(error, GetErrorMessage());
    }
    else
    {
        _threadLastErrorMessages.first = OpenIPC_Error_No_Error;
        _threadLastErrorMessages.second.clear();
    }

    // Remove this error context from the stack
    if (_threadErrorContext == this)
    {
        _threadErrorContext = nullptr; // set to null instead of erase to save time on repeated alloc/dealloc
    }
}

// Note: A raw pointer is guaranteed to be safe here since the ErrorContext exists in a higher stack
// frame on the same thread. i.e. The pointer is registered in _contexts in its constructor and removed
// in its destructor, so if its in this list it definately exists.
ErrorContext* ErrorContext::GetErrorContext()
{
    return _threadErrorContext;
}

void ErrorContext::_PostInternal(OpenIPC_Error errorCode, std::string message)
{
    if (errorCode != OpenIPC_Error_No_Error)
    {
        _errorStack.push_back(std::pair<OpenIPC_Error, std::string>(errorCode, message));
    }
}

std::string ErrorContext::GetErrorMessage() const
{
    std::stringstream ss;

    if (!_errorStack.empty())
    {
        for (int i = static_cast<int>(_errorStack.size() - 1); i >= 0; --i)
        {
            if (!_errorStack[i].second.empty())
            {
                ss << _errorStack[i].second;
                if (i != 0)
                {
                    ss << ".  ";
                }
            }
        }
    }

    return ss.str();
}

OpenIPC_Error PluginNotifications::Handler(PluginNotificationCallbackHandler notificationHandlerFunction)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    if (_notificationHandlerFunction == nullptr)
    {
        _notificationHandlerFunction = notificationHandlerFunction;
    }
    else
    {
        openIPCError = OpenIPC_Error_Callback_Already_Initialized;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "A handler is already installed for this logger.");
    }

    return openIPCError;
}

OpenIPC_Error PluginLogger::Handler(PluginStreamLogCallbackHandler logHandlerFunction)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    if (_logStreamHandlerFunction == nullptr)
    {
        _logStreamHandlerFunction = logHandlerFunction;
    }
    else
    {
        openIPCError = OpenIPC_Error_Callback_Already_Initialized;
        PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "A handler is already installed for this logger.");
    }

    return openIPCError;
}

OpenIPC_Error PluginLogger::Handler(PluginLogCallbackHandler logHandlerFunction)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    if (_logHandlerFunction == nullptr)
    {
        _logHandlerFunction = logHandlerFunction;
    }
    else
    {
        openIPCError = OpenIPC_Error_Callback_Already_Initialized;
        PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
    }

    return openIPCError;
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
// C conversion functions for standard plugin API
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

extern CProbePlugin* GetProbePluginInstance(); // this needs to be defined somewhere in the plugin-specific code for the plugin to build

OpenIPC_Error PPI_PluginGetInfo(PPI_PluginApiVersion clientInterfaceVersion, PPI_PluginInfo *info){ErrorContext context; return GetProbePluginInstance()->PluginGetInfo(clientInterfaceVersion, info);}
OpenIPC_Error PPI_PluginInitialize(PPI_RefId pluginid, const PPI_char* vendorinit){ErrorContext context; return GetProbePluginInstance()->PluginInitialize(pluginid, (const char *) vendorinit);}
OpenIPC_Error PPI_PluginDeinitialize(void){ErrorContext context; return GetProbePluginInstance()->PluginDeinitialize();}

OpenIPC_Error PPI_ProbeGetRefIds(uint32_t maxIds, PPI_RefId* refIds, uint32_t* probecount){ErrorContext context; return GetProbePluginInstance()->ProbeGetRefIds(maxIds, refIds, probecount);}
OpenIPC_Error PPI_ProbeGetInfo(PPI_RefId refId, PPI_ProbeInfo* info){ErrorContext context; return GetProbePluginInstance()->ProbeGetInfo(refId, info);}
OpenIPC_Error PPI_ProbeBeginInitialization(PPI_RefId probe_refid, OpenIPC_DeviceId probeID) { ErrorContext context; return GetProbePluginInstance()->ProbeBeginInitialization(probe_refid, probeID); }
OpenIPC_Error PPI_ProbeSpecifyVendorInitString(OpenIPC_DeviceId probeID, const PPI_char* vendorinit) { ErrorContext context; return GetProbePluginInstance()->ProbeSpecifyVendorInitString(probeID, vendorinit); }
OpenIPC_Error PPI_ProbeFinishInitialization(OpenIPC_DeviceId probeID) { ErrorContext context; return GetProbePluginInstance()->ProbeFinishInitialization(probeID); }
OpenIPC_Error PPI_ProbeInitialize(PPI_RefId probe_refid, OpenIPC_DeviceId probeID, const PPI_char* vendorinit){ErrorContext context; return GetProbePluginInstance()->ProbeInitialize(probe_refid, probeID, (const char *)vendorinit);}
OpenIPC_Error PPI_ProbeDeInitialize(OpenIPC_DeviceId probeID){ErrorContext context; return GetProbePluginInstance()->ProbeDeInitialize(probeID);}

OpenIPC_Error PPI_InterfaceGetRefIds(OpenIPC_DeviceId probeID, uint8_t maxIds, PPI_RefId* refIds, uint32_t* interfacecount){ErrorContext context; return GetProbePluginInstance()->InterfaceGetRefIds(probeID, maxIds, refIds, interfacecount);}
OpenIPC_Error PPI_InterfaceGetType(OpenIPC_DeviceId probeID, PPI_RefId interface_refid, PPI_EInterfaceType* interface_type){ErrorContext context; return GetProbePluginInstance()->InterfaceGetType(probeID, interface_refid, interface_type);}
OpenIPC_Error PPI_InterfaceGetInfoJTAG(OpenIPC_DeviceId probeID, PPI_RefId interface_refid, PPI_InterfaceJTAGCapabilities* capabilities){ErrorContext context; return GetProbePluginInstance()->InterfaceGetInfoJTAG(probeID, interface_refid, capabilities);}
OpenIPC_Error PPI_InterfaceGetInfoPins(OpenIPC_DeviceId probeID, PPI_RefId interface_refid, PPI_InterfacePinsCapabilities* capabilities){ErrorContext context; return GetProbePluginInstance()->InterfaceGetInfoPins(probeID, interface_refid, capabilities);}
OpenIPC_Error PPI_InterfaceGetInfoI2C(OpenIPC_DeviceId probeID, PPI_RefId interface_refid, PPI_InterfaceI2CCapabilities* capabilities){ErrorContext context; return GetProbePluginInstance()->InterfaceGetInfoI2C(probeID, interface_refid, capabilities);}
OpenIPC_Error PPI_InterfaceBeginInitialization(OpenIPC_DeviceId probeID, PPI_RefId interface_refid, OpenIPC_DeviceId interfaceID) { ErrorContext context; return GetProbePluginInstance()->InterfaceBeginInitialization(probeID, interface_refid, interfaceID); }
OpenIPC_Error PPI_InterfaceSpecifyVendorInitString(OpenIPC_DeviceId interfaceID, const PPI_char* vendorinit) { ErrorContext context; return GetProbePluginInstance()->InterfaceSpecifyVendorInitString(interfaceID, vendorinit); }
OpenIPC_Error PPI_InterfaceFinishInitialization(OpenIPC_DeviceId interfaceID) { ErrorContext context; return GetProbePluginInstance()->InterfaceFinishInitialization(interfaceID); }
OpenIPC_Error PPI_InterfaceInitialize(OpenIPC_DeviceId probeID, uint32_t interface_refid, OpenIPC_DeviceId interfaceID, const PPI_char* vendorinit){ErrorContext context; return GetProbePluginInstance()->InterfaceInitialize(probeID, interface_refid, interfaceID, (const char*) vendorinit);}
OpenIPC_Error PPI_InterfaceDeInitialize(OpenIPC_DeviceId interfaceID){ErrorContext context; return GetProbePluginInstance()->InterfaceDeInitialize(interfaceID);}
OpenIPC_Error PPI_InterfaceScan(OpenIPC_DeviceId interfaceID, const uint32_t* input, uint32_t inputdwords, uint32_t* output, uint32_t maxoutputdwords, uint32_t* outputdwords){ErrorContext context; return GetProbePluginInstance()->InterfaceScan(interfaceID, input, inputdwords, output, maxoutputdwords, outputdwords);}
OpenIPC_Error PPI_InterfaceListConfigString(OpenIPC_DeviceId interfaceID,uint32_t configTypeLength,PPI_char const ** configType, uint32_t* numberOfConfigTypes){ErrorContext context; return GetProbePluginInstance()->InterfaceListConfigString(interfaceID, configTypeLength, configType,numberOfConfigTypes);}
OpenIPC_Error PPI_InterfaceOperationCancel(OpenIPC_DeviceId interfaceID){ErrorContext context; return GetProbePluginInstance()->InterfaceOperationCancel(interfaceID);}
OpenIPC_Error PPI_InterfaceSetConfig(OpenIPC_DeviceId interfaceID, const PPI_char* configType, const PPI_char value[PPI_MAX_INFO_LEN]){ErrorContext context; return GetProbePluginInstance()->InterfaceSetConfig(interfaceID, (const char *)configType, value);}
OpenIPC_Error PPI_InterfaceGetConfig(OpenIPC_DeviceId interfaceID, const PPI_char* configType, PPI_char value[PPI_MAX_INFO_LEN]){ErrorContext context; return GetProbePluginInstance()->InterfaceGetConfig(interfaceID, (const char *)configType, value);}
OpenIPC_Error PPI_InterfaceTracePortRead(OpenIPC_DeviceId interfaceID, uint8_t* output, uint32_t maxoutputbytes, uint32_t* outputbytes){ErrorContext context; return GetProbePluginInstance()->InterfacePortRead(interfaceID, output, maxoutputbytes, outputbytes);}
OpenIPC_Error PPI_InterfaceTracePortWrite(OpenIPC_DeviceId interfaceID, const uint8_t* input, uint32_t maxinputbytes, uint32_t* inputbytes){ErrorContext context; return GetProbePluginInstance()->InterfacePortWrite(interfaceID, input, maxinputbytes, inputbytes);}
OpenIPC_Error PPI_InterfaceTraceOpenWindow(OpenIPC_DeviceId interfaceID, PPI_Trace_PortAccessMode mode){ErrorContext context; return GetProbePluginInstance()->InterfacePortOpenWindow(interfaceID, mode);}
OpenIPC_Error PPI_InterfaceTraceCloseWindow(OpenIPC_DeviceId interfaceID){ErrorContext context; return GetProbePluginInstance()->InterfacePortCloseWindow(interfaceID);}
OpenIPC_Error PPI_InterfaceTraceIsReadDataAvailable(OpenIPC_DeviceId interfaceID, PPI_bool* isDataAvailable){
    ErrorContext context;
    bool isDataAvailableB = false;
    OpenIPC_Error result;
    result =  GetProbePluginInstance()->InterfacePortIsReadDataAvailable(interfaceID, &isDataAvailableB);
    *isDataAvailable = isDataAvailableB;
    return result;
}
OpenIPC_Error PPI_InterfaceTraceIsWindowOpen(OpenIPC_DeviceId interfaceID, PPI_bool* isWindowOpen){
    ErrorContext context;
    bool isWindowOpenB = false;
    OpenIPC_Error result;
    result =  GetProbePluginInstance()->InterfacePortIsWindowOpen(interfaceID, &isWindowOpenB);
    *isWindowOpen = isWindowOpenB;
    return result;
}
OpenIPC_Error PPI_PluginRegisterEventHandler(PluginEventCallbackHandler eventHandlerFunction){ErrorContext context; return GetProbePluginInstance()->PluginRegisterEventHandler(eventHandlerFunction);}
OpenIPC_Error PPI_PluginRegisterNotificationHandler(PluginNotificationCallbackHandler notificationHandlerFunction){ErrorContext context; return GetProbePluginInstance()->PluginRegisterNotificationHandler(notificationHandlerFunction);}
OpenIPC_Error PPI_PluginSetLogEventHandler(PluginLogCallbackHandler logHandlerFunction){ErrorContext context; return GetProbePluginInstance()->PluginSetLogEventHandler(logHandlerFunction);}

OpenIPC_Error PPI_Lock_Target_Interface(OpenIPC_DeviceId deviceInterface){
    ErrorContext context;
    return GetProbePluginInstance()->LockTargetInterface(deviceInterface);
}
OpenIPC_Error PPI_List_Locked_Interfaces(uint32_t maxNumberOfInterfaces, OpenIPC_DeviceId* interfaces, uint32_t* numberOfInterfaces){
    ErrorContext context;
    return GetProbePluginInstance()->ListLockedInterfaces(maxNumberOfInterfaces,interfaces,numberOfInterfaces);
}
OpenIPC_Error PPI_InterfaceListLockInterfacePeers( OpenIPC_DeviceId interfaceID, uint32_t peerInterfacesLength,OpenIPC_DeviceId * peerInterfaces, uint32_t* numberOfPeerInterfaces){
    ErrorContext context;
    return GetProbePluginInstance()->InterfaceListLockInterfacePeers(interfaceID,peerInterfacesLength,peerInterfaces,numberOfPeerInterfaces);
}

PPI_ProbeBundleHandle PPI_Bundle_Allocate()
{
    ErrorContext context;
    return GetProbePluginInstance()->BundleAllocate();
}
OpenIPC_Error PPI_Bundle_Free(PPI_ProbeBundleHandle* handle)
{
    ErrorContext context;
    return GetProbePluginInstance()->BundleFree(handle);
}
OpenIPC_Error PPI_Bundle_Execute(PPI_ProbeBundleHandle handle, OpenIPC_DeviceId deviceInterface, PPI_bool keepLock)
{
    ErrorContext context;
    return GetProbePluginInstance()->BundleExecute(handle,deviceInterface,keepLock);
}
OpenIPC_Error PPI_Bundle_Clear(PPI_ProbeBundleHandle handle)
{
    ErrorContext context;
    return GetProbePluginInstance()->BundleClear(handle);
}
OpenIPC_Error PPI_Bundle_Append(PPI_ProbeBundleHandle destHandle, PPI_ProbeBundleHandle sourceBundle, uint8_t* outputBuffer, uint32_t outputBufferSize)
{
    ErrorContext context;
    return GetProbePluginInstance()->BundleAppend(destHandle, sourceBundle, outputBuffer, outputBufferSize);
}

const char* PPI_PluginGetLastError(OpenIPC_Error error){
    return GetProbePluginInstance()->GetLastErrorMessage(error);
}

OpenIPC_Error PPI_PluginSetNotificationRequiredForLogCallBack(PPI_NotificationLevel level)
{
    ErrorContext context;
    return GetProbePluginInstance()->PluginSetNotificationRequiredForLogCallBack(level);
}

OpenIPC_Error PPI_PluginSetNotificationRequiredForNotificationCallBack(PPI_NotificationLevel level)
{
    ErrorContext context;
    return GetProbePluginInstance()->PluginSetNotificationRequiredForNotificationCallBack(level);
}

OpenIPC_Error PPI_TriggerGetTrigger(const PPI_InterfaceTriggerResponse* triggerResponse, PPI_Pins_TypeEncode* pin, PPI_bool* onAssert, uint32_t* identifier)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_TriggerGetTrigger(triggerResponse,pin,onAssert,identifier);
}
OpenIPC_Error PPI_TriggerGetResponse(const PPI_InterfaceTriggerResponse* triggerResponse, PPI_Pins_TypeEncode* pin, PPI_bool* respondAsAssert, uint32_t* identifier)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_TriggerGetResponse(triggerResponse,pin,respondAsAssert,identifier);
}
OpenIPC_Error PPI_InterfaceListTriggerResponses(OpenIPC_DeviceId device, uint32_t triggerResponseSupportedLength, PPI_InterfaceTriggerResponse** triggerResponseSupported, uint32_t* triggerResponsePopulated)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_InterfaceListTriggerResponses(device,triggerResponseSupportedLength,triggerResponseSupported,triggerResponsePopulated);
}
OpenIPC_Error PPI_InterfaceListObservableTriggerResponses(OpenIPC_DeviceId device, uint32_t triggerResponseSupportedLength, PPI_InterfaceTriggerResponse** triggerResponseSupported, uint32_t* triggerResponsePopulated)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_InterfaceListObservableTriggerResponses(device,triggerResponseSupportedLength,triggerResponseSupported,triggerResponsePopulated);
}
OpenIPC_Error PPI_InterfaceTriggerResponseIsSet(OpenIPC_DeviceId device, const PPI_InterfaceTriggerResponse* triggerResponse, PPI_bool* triggerResponseSet, PPI_TriggerResponseSetOptions* optionsSet)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_InterfaceTriggerResponseIsSet(device,triggerResponse,triggerResponseSet,optionsSet);
}
OpenIPC_Error PPI_InterfaceTriggerResponseSet(OpenIPC_DeviceId device, PPI_InterfaceTriggerResponse* triggerResponse, PPI_bool triggerResponseSet,const PPI_TriggerResponseSetOptions* optionsSet)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_InterfaceTriggerResponseSet(device,triggerResponse,triggerResponseSet,optionsSet);
}

OpenIPC_Error PPI_InterfacePinsQueryAliases(OpenIPC_DeviceId device, PPI_Pins_TypeEncode pin,
                                            uint32_t pinListLength, PPI_Pins_TypeEncode* pinList, uint32_t* numberOfPinsWritten)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_InterfacePinsQueryAliases(device,pin,pinListLength,pinList,numberOfPinsWritten);
}

OpenIPC_Error PPI_InterfacePinsListDrivablePins(
    OpenIPC_DeviceId device, uint32_t pinsListLength,
    PPI_Pins_TypeEncode* pinsList, uint32_t* numberOfPinsPopulated)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_InterfacePinsListDrivablePins(device,pinsListLength,pinsList,numberOfPinsPopulated);
}

OpenIPC_Error PPI_InterfacePinsListReadablePins(
        OpenIPC_DeviceId device, uint32_t pinsListLength,
        PPI_Pins_TypeEncode* pinsList, uint32_t* numberOfPinsPopulated)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_InterfacePinsListReadablePins(device,pinsListLength,pinsList,numberOfPinsPopulated);
}

OpenIPC_Error PPI_PluginSetStreamLogEventHandler(PluginStreamLogCallbackHandler logCallBackFunction)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_PluginSetStreamLogEventHandler(logCallBackFunction);
}

OpenIPC_Error PPI_PluginUsesLoggerStream( PPI_Stream stream, PPI_bool* usesLoggerStream)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_PluginUsesLoggerStream(stream, usesLoggerStream);
}

OpenIPC_Error PPI_PluginSetNotificationRequiredForStreamLogCallBack( PPI_Stream stream, PPI_NotificationLevel level)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_PluginSetNotificationRequiredForStreamLogCallBack(stream, level);
}

OpenIPC_Error PPI_PluginEventGenerated(OpenIPC_DeviceId probe, PPI_PluginEvent pluginEvent, PPI_bool* pluginGeneratesEvent)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_PluginEventGenerated(probe, pluginEvent,pluginGeneratesEvent);
}

OpenIPC_Error PPI_DeviceSetConfig(OpenIPC_DeviceId deviceID,const PPI_char* configType,const PPI_char value[PPI_MAX_INFO_LEN])
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_DeviceSetConfig(deviceID,configType,value);
}

OpenIPC_Error PPI_DeviceGetConfigDescription(OpenIPC_DeviceId deviceID, const PPI_char* configType, PPI_char description[PPI_MAX_INFO_LEN])
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_DeviceGetConfigDescription(deviceID, configType, description);
}

OpenIPC_Error PPI_DeviceListConfigValues(OpenIPC_DeviceId deviceID, const PPI_char* configType, uint32_t valuesLength, PPI_char const** values, uint32_t* numberOfValues)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_DeviceListConfigValues(deviceID, configType, valuesLength, values, numberOfValues);
}

OpenIPC_Error PPI_DeviceGetConfig(OpenIPC_DeviceId deviceID,const PPI_char* configType,PPI_char value[PPI_MAX_INFO_LEN])
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_DeviceGetConfig(deviceID,configType,value);
}
OpenIPC_Error PPI_DeviceListConfigString(OpenIPC_DeviceId deviceID,uint32_t configTypeLength,PPI_char const ** configType, uint32_t* numberOfConfigTypes)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_DeviceListConfigString(deviceID,configTypeLength,configType,numberOfConfigTypes);
}

OpenIPC_Error PPI_DeviceListActions(OpenIPC_DeviceId deviceID, uint32_t actionsLength, PPI_Action** actions, uint32_t* numberOfActions)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_DeviceListActions(deviceID,actionsLength,actions,numberOfActions);
}
OpenIPC_Error PPI_ActionGetName (const PPI_Action* action,uint32_t valueLength, PPI_char* value)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_ActionGetName(action,valueLength,value);
}
OpenIPC_Error PPI_ActionGetDescription(const PPI_Action* action, uint32_t valueLength, PPI_char* value)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_ActionGetDescription(action,valueLength,value);
}
OpenIPC_Error PPI_DeviceActionRun(OpenIPC_DeviceId deviceID,const PPI_Action* action, uint32_t messageLength, PPI_char* message)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_DeviceActionRun(deviceID,action,messageLength,message);
}
OpenIPC_Error PPI_ActionSetArgument(PPI_Action* action, const PPI_char* value)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_ActionSetArgument(action, value);
}

OpenIPC_Error PPI_ActionIsVisible(const PPI_Action* action, PPI_bool* isVisible)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_ActionIsVisible(action, isVisible);
}

OpenIPC_Error PPI_StatePortAllocateOperation(OpenIPC_DeviceId deviceId, const PPI_InterfaceStatePortDefinition* definition, PPI_StatePort_OPERATION_TYPE_ET operationType, PPI_InterfaceStatePortOperation** operation, const PPI_StatePortAllocateOperation_Options* options)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_StatePortAllocateOperation(deviceId, definition, operationType, operation, options);
}
OpenIPC_Error PPI_StatePortFreeOperation(OpenIPC_DeviceId deviceId, PPI_InterfaceStatePortOperation** operation)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_StatePortFreeOperation(deviceId, operation);
}
OpenIPC_Error PPI_StatePortAppendOperation(OpenIPC_DeviceId deviceId, PPI_ProbeBundleHandle bundle, PPI_InterfaceStatePortOperation** operationPointer, const PPI_StatePortAppendOperation_Options* options)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_StatePortAppendOperation(deviceId, bundle, operationPointer, options);
}
OpenIPC_Error PPI_StatePortOperationSetParameter(PPI_InterfaceStatePortOperation* operation, const PPI_InterfaceStatePortParameter* parameter, const uint8_t* value, uint32_t size)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_StatePortOperationSetParameter(operation, parameter, value, size);
}
OpenIPC_Error PPI_StatePortOperationSetWriteValue(PPI_InterfaceStatePortOperation* operation, const uint8_t* buffer, uint32_t size)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_StatePortOperationSetWriteValue(operation, buffer, size);
}
OpenIPC_Error PPI_StatePortOperationSetReadBuffer(PPI_InterfaceStatePortOperation* operation, uint8_t* buffer, uint32_t size)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_StatePortOperationSetReadBuffer(operation, buffer, size);
}
OpenIPC_Error PPI_StatePortOperationSetErrorBuffer(PPI_InterfaceStatePortOperation* operation, ProbeStatePortError* errorBuffer)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_StatePortOperationSetErrorBuffer(operation, errorBuffer);
}
OpenIPC_Error PPI_StatePortGetDefinitions(OpenIPC_DeviceId deviceId, const PPI_InterfaceStatePortDefinition** definitions, uint32_t definitionsSize, uint32_t* numberOfDefinitions)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_StatePortGetDefinitions(deviceId, definitions, definitionsSize, numberOfDefinitions);
}
OpenIPC_Error PPI_StatePortGetName(const PPI_InterfaceStatePortDefinition* definition, char* buffer, uint32_t bufferSize, uint32_t* realSize)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_StatePortGetName(definition, buffer, bufferSize, realSize);
}
OpenIPC_Error PPI_StatePortGetDescription(const PPI_InterfaceStatePortDefinition* definition, char* buffer, uint32_t bufferSize, uint32_t* realSize)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_StatePortGetDescription(definition, buffer, bufferSize, realSize);
}
OpenIPC_Error PPI_StatePortGetCapabilites(const PPI_InterfaceStatePortDefinition* definition, PPI_StatePort_OPERATION_TYPE_ET* capabilities)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_StatePortGetCapabilites(definition, capabilities);
}
OpenIPC_Error PPI_StatePortGetAccessSize(const PPI_InterfaceStatePortDefinition* definition, uint32_t* size)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_StatePortGetAccessSize(definition, size);
}
OpenIPC_Error PPI_StatePortGetParameters(const PPI_InterfaceStatePortDefinition* definition, const PPI_InterfaceStatePortParameter** parameters, uint32_t length, uint32_t* realLength)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_StatePortGetParameters(definition, parameters, length, realLength);
}
OpenIPC_Error PPI_StatePortParameterGetName(const PPI_InterfaceStatePortParameter* parameter, char* buffer, uint32_t bufferSize, uint32_t* realSize)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_StatePortParameterGetName(parameter, buffer, bufferSize, realSize);
}
OpenIPC_Error PPI_StatePortParameterGetDescription(const PPI_InterfaceStatePortParameter* parameter, char* buffer, uint32_t bufferSize, uint32_t* realSize)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_StatePortParameterGetDescription(parameter, buffer, bufferSize, realSize);
}
OpenIPC_Error PPI_StatePortParameterGetSize(const PPI_InterfaceStatePortParameter* parameter, uint32_t* size)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_StatePortParameterGetSize(parameter, size);
}
OpenIPC_Error PPI_StatePortParameterGetDefaultValue(const PPI_InterfaceStatePortParameter* parameter, uint8_t* buffer, uint32_t size)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_StatePortParameterGetDefaultValue(parameter, buffer, size);
}
OpenIPC_Error PPI_StatePortErrorGetName(const PPI_InterfaceStatePortDefinition* definition, ProbeStatePortError error, char* buffer, uint32_t bufferSize, uint32_t* realSize)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_StatePortErrorGetName(definition, error, buffer, bufferSize, realSize);
}
OpenIPC_Error PPI_StatePortErrorGetDescription(const PPI_InterfaceStatePortDefinition* definition, ProbeStatePortError error, char* buffer, uint32_t bufferSize, uint32_t* realSize)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_StatePortErrorGetDescription(definition, error, buffer, bufferSize, realSize);
}
OpenIPC_Error PPI_StatePortErrorGetOpenIpcError(const PPI_InterfaceStatePortDefinition* definition, ProbeStatePortError error, OpenIPC_Error* openIpcError)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_StatePortErrorGetOpenIpcError(definition, error, openIpcError);
}
OpenIPC_Error PPI_StatePortEventGetValueChangedEventDescription(OpenIPC_DeviceId deviceId, PPI_InterfaceStatePortValueChangedEventDescription const** eventDescriptionPtr)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_StatePortEventGetValueChangedEventDescription(deviceId, eventDescriptionPtr);
}


OpenIPC_Error PPI_JTAG_GetInterfacePadding(OpenIPC_DeviceId device,
                                           uint32_t* irPaddingNearTDI,
                                           uint32_t* irPaddingNearTDO,
                                           uint32_t* drPaddingNearTDI,
                                           uint32_t* drPaddingNearTDO,
                                           PPI_bool* drValueConstantOne
                                           )
{
    ErrorContext context;
    return GetProbePluginInstance()->GetInterfaceJTAGPadding(device,irPaddingNearTDI,irPaddingNearTDO,drPaddingNearTDI,drPaddingNearTDO,drValueConstantOne);
}
OpenIPC_Error PPI_JTAG_SetInterfacePadding( OpenIPC_DeviceId device,
                                           uint32_t irPaddingNearTDI,
                                           uint32_t irPaddingNearTDO,
                                           uint32_t drPaddingNearTDI,
                                           uint32_t drPaddingNearTDO,
                                           PPI_bool drValueConstantOne
                                           )
{
    ErrorContext context;
    return GetProbePluginInstance()->SetInterfaceJTAGPadding(device,irPaddingNearTDI,irPaddingNearTDO,drPaddingNearTDI,drPaddingNearTDO,drValueConstantOne);
}

OpenIPC_Error PPI_JTAG_GetCurrentBundlePadding(PPI_ProbeBundleHandle bundle, int32_t* irPaddingNearTDI, int32_t* irPaddingNearTDO, int32_t* drPaddingNearTDI,
    int32_t* drPaddingNearTDO, PPI_bool* drValueConstantOne)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_JTAG_GetCurrentBundlePadding(bundle, irPaddingNearTDI, irPaddingNearTDO, drPaddingNearTDI, drPaddingNearTDO, drValueConstantOne);
}

OpenIPC_Error PPI_JTAG_UpdateBundlePadding(PPI_ProbeBundleHandle bundle, int32_t irPaddingNearTDI, int32_t irPaddingNearTDO, int32_t drPaddingNearTDI,
    int32_t drPaddingNearTDO, PPI_bool drValueConstantOne)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_JTAG_UpdateBundlePadding(bundle, irPaddingNearTDI, irPaddingNearTDO, drPaddingNearTDI, drPaddingNearTDO, drValueConstantOne);
}

OpenIPC_Error PPI_JTAG_PinsShift(PPI_ProbeBundleHandle handle, uint32_t shiftLengthBits, const uint8_t* const tdi, const uint8_t* const tms, uint8_t* tdo, const PPI_JTAG_PinsOptions* const options)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_JTAG_PinsShift(handle, shiftLengthBits, tdi, tms, tdo, options);
}

OpenIPC_Error PPI_Bundle_ExecuteMultiChain(PPI_ProbeBundleHandle handle, OpenIPC_DeviceId* deviceInterfaces, uint32_t deviceInterfacesLength, PPI_bool keepLock)
{
    ErrorContext context;
    return GetProbePluginInstance()->BundleExecuteMultiChain(handle, deviceInterfaces, deviceInterfacesLength, keepLock);
}

OpenIPC_Error PPI_Bundle_InterfaceSetConfig(PPI_ProbeBundleHandle handle, const char* configType, const char* value)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_Bundle_InterfaceSetConfig(handle, configType, value);
}

OpenIPC_Error PPI_JTAG_GoToState(PPI_ProbeBundleHandle handle, JtagStateEncode gotoState, uint32_t numberOfClocksInState, const PPI_JTAG_StateGotoOptions* const options)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_JTAG_GoToState(handle, gotoState, numberOfClocksInState, options);
}

OpenIPC_Error PPI_JTAG_StateIRShift(PPI_ProbeBundleHandle handle, uint32_t shiftLengthBits, const uint8_t* const inBits, uint8_t* outBits, const PPI_JTAG_StateShiftOptions* const options)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_JTAG_StateIRShift(handle, shiftLengthBits, inBits, outBits, options);
}

OpenIPC_Error PPI_JTAG_StateDRShift(PPI_ProbeBundleHandle handle, uint32_t shiftLengthBits, const uint8_t* const inBits, uint8_t* outBits, const PPI_JTAG_StateShiftOptions* const options)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_JTAG_StateDRShift(handle, shiftLengthBits, inBits, outBits, options);
}

OpenIPC_Error PPI_PluginCreateStaticProbe(const PPI_char* probeType, PPI_RefId* refId)
{
    ErrorContext context;
    return GetProbePluginInstance()->PluginCreateStaticProbe(probeType, refId);
}

OpenIPC_Error PPI_PluginGetProbeTypes(uint32_t maxProbeTypes, PPI_char const** probeTypes, uint32_t* probeTypeCount)
{
    ErrorContext context;
    return GetProbePluginInstance()->PluginGetProbeTypes(maxProbeTypes, probeTypes, probeTypeCount);
}

OpenIPC_Error PPI_MemoryRead(PPI_ProbeBundleHandle bundle, uint64_t address, uint8_t* buffer, uint32_t* size)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_MemoryRead(bundle, address, buffer, size);
}

OpenIPC_Error PPI_MemoryWrite(PPI_ProbeBundleHandle bundle, uint64_t address, const uint8_t* buffer, uint32_t* size)
{
    ErrorContext context;
    return GetProbePluginInstance()->PPI_MemoryWrite(bundle, address, buffer, size);
}
