#pragma once
//////////////////////////////////////////////////////////////////////////////
//
//                      INTEL CONFIDENTIAL
//       Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written
// permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or
// delivery of the Materials, either expressly, by implication, inducement,
// estoppel or otherwise. Any license under such intellectual property rights
// must be express and approved by Intel in writing.
//
// Unless otherwise agreed by Intel in writing, you may not remove or alter
// this notice or any other notice embedded in Materials by Intel or Intel'ss
// suppliers or licensors in any way.
//
//////////////////////////////////////////////////////////////////////////////
///  @file
///
///  @brief Contains methods for interacting with bundles.
///
///  For additional information on obtaining and using instances, see @ref probeusage.
///
//////////////////////////////////////////////////////////////////////////////

#ifndef __BUNDLE_OPERATIONS_H__
#define __BUNDLE_OPERATIONS_H__
#include "ProbePlugin.h"

/** Model of the bundle handles:
*
*
* Slot is a mathematical type, where a slot is a type with two fields:
* 1. value: An integer >= 0
* 2. length: An integer >= 0.
* with constraint: value < 2^length
*
* SLOT_DEFAULT is a constant provided by the implementation that of type Slot where all items in the string have value (0,0).
*
* A non PPI_PROBE_LOCK_RELEASE nor PPI_PROBE_LOCK_HOLD bundle handle has the following fields:
* 1. slots: An string of Slot
* 2. commands: A string of commands; types commands are defined by the services
* 3. has_been_run: a boolean
* 4. If it is lock bundle, it includes the device to target.
*
* As an example, we denote a value ("(2,3)", "", False) to denote:
* a bundle handle with one slot with value 2,  length of 3 bits. No commands queued and it has never been run.
*
* In this document, a client is a software component that is using a Probe
* Abstraction API implementation, e.g., the OpenIPC.
*/

typedef void* PPI_ProbeBundleHandle;
/**
* @brief A released lock.
*/
const PPI_ProbeBundleHandle PPI_PROBE_LOCK_RELEASE = (PPI_ProbeBundleHandle)0;
/**
* @brief A held lock.
*/
const PPI_ProbeBundleHandle PPI_PROBE_LOCK_HOLD = (PPI_ProbeBundleHandle)1;

/**
* @brief Allocates a bundle handle for the probe = ("", "",False).
*
* Allocates a bundle handle for the probe; returns PROBE_LOCK_RELEASE if unable to.
* This function is not required to be provided (in that case, the client is responsible for emulating bundle behaviour).
* Note that for optimal performance (with a "full-featured" probe), it is *highly* recommended that one implement full bundles.
* @returns NULL iff unable to allocate a new bundle handle.
*/
PPI_FUNCTION(PPI_Bundle_Allocate, PPI_ProbeBundleHandle, void);

/**
* @brief Locks a given device id and sets the target of the PPI_PROBE_LOCK_HOLD to use the given device ID.
*
* This operation allows for a client to 'lock' the interface from additional peers to the Probe Plugin from using it. One example is if the probe plugin
* interacts with another debug tool that polls for run control events using JTAG. If we lock an interface, then the requirement is that the debug tool does not
* modify the JTAG chain while the lock is set.
* Moreover, the model is that we can have more that 1 lock (and use Bundle_Execute to release those locks), but only 1 distinguished locked interface that we can send scans to using PPI_PROBE_LOCK_HOLD.
* @param [in] deviceInterface The device that this bundle should target. This is required of all implementations.
* @returns OpenIPC_Error_No_Error iff it is able to lock this interface.
*/
PPI_FUNCTION(PPI_Lock_Target_Interface, OpenIPC_Error, OpenIPC_DeviceId deviceInterface);

/**
* @brief This operation returns the currently locked interfaces.
*
* This operation returns the currently locked interfaces and returns *all* interfaces locked due to peer locking (e.g., the set of interfaces returned is closed under the symetric, transitive, and reflexive peer relation).
* @param [in] maxNumberOfInterfaces The size of the interfaces parameter; the implementation will not write more than this number of interfaces to the interfaces parameter in any case.
* @param [out] interfaces The list of interfaces currently locked.
* @param [out] numberOfInterfaces The number of interfaces that are locked; if it is greater that the maxNumberOfInterfaces the client should recall with a larger interfaces parameter.
* @returns OpenIPC_Error_No_Error iff the operation completed successfully.
*/
PPI_FUNCTION(PPI_List_Locked_Interfaces, OpenIPC_Error, uint32_t maxNumberOfInterfaces, OpenIPC_DeviceId* interfaces, uint32_t* numberOfInterfaces);

/**
* @brief Executes the commands contained in a bundle handle.
*
* Executes the commands contained in a bundle handle, with an option to keep a API lock.
* When executing a locked or release handle, this operation simply keeps or releases the lock. This is required of all implementations.
* @param [in, out] handle handle containing the commands to be executed; handle.has_been_run = True after the call completes
* @param [in] keepLock Ignored if executing a locked or release handle. Otherwise, is true iff the interface should remain locked after the handle is executed.
* @param [in] deviceInterface the interface to run scans on
* @return (OpenIPC_Error) A code indicating success or failure.
* @retval OpenIPC_Error_No_Error on success
* @retval OpenIPC_Error_Probe_Bundle_Execute_Error on the inability to execute all or some of the bundle.
* @retval OpenIPC_Error_Probe_Bundle_Invalid when the handle is not a valid probe handle or if the deviceInterface is not appropriate for the scans in the bundle
*/
PPI_FUNCTION(PPI_Bundle_Execute, OpenIPC_Error, PPI_ProbeBundleHandle handle, OpenIPC_DeviceId deviceInterface, PPI_bool keepLock);

/**
* @brief Frees a bundle handle. Handle is no longer a valid handle.
*
* Clears and Frees a bundle handle.  This is required of all implementations (though it is almost a nop if the handle is PPI_PROBE_LOCK_RELEASE or PPI_PROBE_LOCK_HOLD).
* @param [in, out] handle The handle to free. It is set to PPI_PROBE_LOCK_RELEASE upon returning.
* @return (OpenIPC_Error) A code indicating success or failure.
* @retval OpenIPC_Error_No_Error on success
* @retval OpenIPC_Error_Probe_Bundle_Invalid when the handle is not a valid probe handle
*/
PPI_FUNCTION(PPI_Bundle_Free, OpenIPC_Error, PPI_ProbeBundleHandle* handle);

/**
* @brief Clears the commands queued in a bundle handle.
*
* Clears out all commands contained within a bundle handle, i.e., handle.commands = "" and handle.slots = #handle.slots and handle.has_been_run = False.
*  This is required of all implementations (though it is a nop if the handle is PPI_PROBE_LOCK_RELEASE or PPI_PROBE_LOCK_HOLD).
* @param [in, out] handle bundle to clear all commands.
* @return (OpenIPC_Error) A code indicating success or failure.
* @retval OpenIPC_Error_No_Error on success
* @retval OpenIPC_Error_Probe_Bundle_Invalid when the handle is not a valid probe handle
*/
PPI_FUNCTION( PPI_Bundle_Clear, OpenIPC_Error, PPI_ProbeBundleHandle handle);

/**
* @brief Logically appends a bundle on to the end of another.
*
* Appends an existing bundle (that cannot be modified anymore) to another bundle that is modifiable. The intended use case for this functionality is to
* build up repeated JTAG scans that are logically register access algorithms. This will enable better compression by potentaily allowing implementations to
* reuse previously declared bundles.
* @param [in, out] destHandle bundle that will be appended.
* @param [in, out] sourceBundle the bundle that is to be appended to destBundle; this is modified to not be modifiable after this call.
* @param [out] outputBuffer Buffer to place any writeback of this bundle (writeback will be byte aligned per operation)
* @param [in] outputBufferSize Length of the outputBuffer in bytes
* @return (OpenIPC_Error) A code indicating success or failure.
* @retval OpenIPC_Error_No_Error on success
* @retval OpenIPC_Error_Probe_Bundle_Invalid when the destHandle or sourceBundle is not a valid probe handle
*/
PPI_FUNCTION( PPI_Bundle_Append, OpenIPC_Error, PPI_ProbeBundleHandle destHandle, PPI_ProbeBundleHandle sourceBundle, uint8_t* outputBuffer, uint32_t outputBufferSize);

/**
* @brief Executes the commands contained in a bundle handle on multiple devices.
*
* Executes the commands contained in a bundle handle, with an option to keep a API lock.
* When executing a locked or release handle, this operation simply keeps or releases the lock. This is required of all implementations.
* @param [in, out] handle handle containing the commands to be executed; handle.has_been_run = True after the call completes
* @param [in] keepLock Ignored if executing a locked or release handle. Otherwise, is true iff the interfaces should remain locked after the handle is executed.
* @param [in] deviceInterfaces the interfaces to run scans on
* @param [in] deviceInterfacesLength the length of the deviceInterfaces array
* @return (OpenIPC_Error) A code indicating success or failure.
* @retval OpenIPC_Error_No_Error on success
* @retval OpenIPC_Error_Probe_Bundle_Execute_Error on the inability to execute all or some of the bundle.
* @retval OpenIPC_Error_Probe_Bundle_Invalid when the handle is not a valid probe handle or if the deviceInterface is not appropriate for the scans in the bundle
*/
PPI_FUNCTION(PPI_Bundle_ExecuteMultiChain, OpenIPC_Error, PPI_ProbeBundleHandle handle, OpenIPC_DeviceId* deviceInterfaces, uint32_t deviceInterfacesLength, PPI_bool keepLock);

/**
* @brief Executes the commands contained in a bundle handle on multiple devices.
*
* Executes the commands contained in a bundle handle, with an option to keep a API lock.
* When executing a locked or release handle, this operation simply keeps or releases the lock. This is required of all implementations.
* @param [in, out] handle handle containing the commands to be executed; handle.has_been_run = True after the call completes
* @param [in] configType the config to set the value for
* @param [in] value the value to set thit the specified config to
* @return (OpenIPC_Error) A code indicating success or failure.
* @retval OpenIPC_Error_No_Error on success
* @retval OpenIPC_Error_Operation_Not_Supported when the probe plugin does not support setting this config in a bundle
* @retval OpenIPC_Error_Probe_Bundle_Invalid when the handle is not a valid probe handle or if the deviceInterface is not appropriate for the scans in the bundle
*/
PPI_FUNCTION(PPI_Bundle_InterfaceSetConfig, OpenIPC_Error, PPI_ProbeBundleHandle handle, const char* configType, const char* value);

/**
* @brief Performs a memory read on the target.
*
* @param [in] handle handle to add this operation to
* @param [in] address the address to read memory from
* @param [in, out] buffer the location to store the memory that was read
* @param [in, out] size the size to read and set to the actual size read
* @return (OpenIPC_Error) A code indicating success or failure.
* @retval OpenIPC_Error_No_Error on success
* @retval OpenIPC_Error_Operation_Not_Supported when the probe plugin does not support this memory operation
* @retval OpenIPC_Error_Probe_Bundle_Invalid when the handle is not a valid probe handle or if the deviceInterface is not appropriate for the scans in the bundle
*/
PPI_FUNCTION(PPI_MemoryRead, OpenIPC_Error,
    PPI_ProbeBundleHandle bundle,
    uint64_t address,
    uint8_t* buffer,
    uint32_t* size);

/**
* @brief Performs a memory write on the target.
*
* @param [in] handle handle to add this operation to
* @param [in] address the address to write memory to
* @param [in, out] buffer the data to write to the target memory
* @param [in, out] size the size of the data to write and set to the actual size written
* @return (OpenIPC_Error) A code indicating success or failure.
* @retval OpenIPC_Error_No_Error on success
* @retval OpenIPC_Error_Operation_Not_Supported when the probe plugin does not support this memory operation
* @retval OpenIPC_Error_Probe_Bundle_Invalid when the handle is not a valid probe handle or if the deviceInterface is not appropriate for the scans in the bundle
*/
PPI_FUNCTION(PPI_MemoryWrite, OpenIPC_Error,
    PPI_ProbeBundleHandle bundle,
    uint64_t address,
    const uint8_t* buffer,
    uint32_t* size);


/*! \page probeusage

\section gen General

The API in general uses named, formal parameters for common item for a function, while it uses a option structure
for less common items. Option structures are generally optional, so NULL can be passed in. In this case, the defaults
for those options are taken. Also, a design consideration is for the API to provide functionality for optimization (e.g.,
the bundles, slots, and loops), but to leave that as optional. Simply put, a goal of the API is for it to be both easy for
new probe plugins to be added (e.g., minimize work as part of a new plugin) while still providing a high-performance
capable API. The idea is that the SDK will provide generic libraries
that emulate this functionality (as well as the translations between different JTAG APIs, for example) and the client
of the probe abstraction API is required to use them. These generic
libaries will be functionality correct but are unlikely to be optimal for a full featured probe. Those typically
have mechanisms to queue a sequence of operations and execute them on the physical probes, as well as probe local storage
and basic loop constructs. So, for those types of probes it is *highly* recommended that bundles, slots, and loop are
implemented by the probe plugin.

\section bun Bundle Operations

The bundle operations include the basics of how to interact with "bundle-able" portions
of the API. It includes mechanisms to create, execute, clear, and free a bundle as well
as an operation to return the number of slots that are available. A bundle is a tuple of
a mathematical string of bit-vectors (slots), a mathematical string of Commands, and a
boolean to indicate if the handle has been run. A slot is used to temporarily store data
from an interface in the probe plugin implementation during the bundle execution (for
a potential performance optimization).

A client would first call PPI_Bundle_Allocate to obtain a new bundle (or
PPI_Lock_Bundle_Target_Allocate to set the device to target for immediate calls).
Then the client would add operations to the bundle using the other operations.
After the client is done adding new operations to the bundle, she would call
PPI_Bundle_Execute to execute the operation (and may do so multiple times).
Once the client is done with the bundle as it is, she may call PPI_Bundle_Clear to
reset the bundle to an initial state and restart the whole flow again, or she may call
PPI_Bundle_Free to release the bundle to the plugin. Please examine the documentation for
the detailed mental model of a bundle. Note that a plugin implementation need
not implement bundles other than relase and lock.

For each of the different parts of the API (JTAG, I2C, and Pins) a lock implies
that the client of the Probe Abstraction API has *exclusive* access to the
interface (e.g., no other client or internal agent can change the state of that
interface). Each of the different parts of the API can have additional
assumptions/requirements.

Specifically for JTAG, when a lock is acquired or released, properties
that must be maintained in the abscence of an extraordinary event such as power
loss* are:
P1. Only the top-level TAPs are in the JTAG chain, and
P2. The JTAG state machine is in RTI.

* These events corrupt the state of the JTAG chain and so neither the client nor
implementer can ensure P1 nor P2 in that case.

In the case that the Probe plugin implementation has no additional JTAG masters
(e.g., another debugger), P1 and P2 must be setup by the client. Otherwise,
the other JTAG master may or may not ensure the P1 and P2 hold prior to a lock
being acquired (to be safe, a client should always ensure that P1 and P2 hold
if there is an indication that power loss or some other extraordinary event was
detected).

Neither I2C and Pins have additional assumptions/requirements currently.

In general, all operations that take bundle handles will enqueue a new Command to the
end of the bundle handle's Commands string. Exceptions to this rule will be noted
in this description.

\section slot Slot Operations

These operations deal with manipulating slots directly. In general, most operations that
send/capture data from the DUT can be given slots instead of passed in data. These
operations allow for creation, deletion, comparisons, modification of slot contents,
inspection of a slot's length, and comparison of a slot's value to a constant.
This does not currently allow for the comparison of two slot values as it was determined
that this is a low ROI activity.

In order to use a slot, the client request one using the PPI_Slot_Allocate function (passing
in the desired bitsize of the slot); the probe plugin is responsible for presenting a model
of as many slots as the client asks for (they may be emulated if needed).
A client might wish to explicitly set a slot to a particular value, or in a few bits, or
mask out a few bits; all of the modifications to a slot can be done using
PPI_Slot_Modification. Prior to this, however, the client might want to see how much data
is contained in a slot using PPI_Slot_Size; this is executed immediately and is in a bundle or
lock context. Comparing
a slot's value to a constant is via PPI_Slot_ComparisonToConstant; this allows for the
use of a mask to ignore certain bits. One can use slots in a lock bundle mostly for
debug purposes.

\section loop Loop Operations

These operations allow for a client to specify that a certain sequence of JTAG operations
should be repeated for a certain number of times, or until an event occurs. The desired,
and advanced, use case is to accelerate power and run control flows by allowing for those
flows to be executed in the probe hardware if possible.

A client can decide if they'd like a variable number of iterations to be executed
(depending on a condition given by comparing to a slot) with a maximum number by using
PPI_Loop_LoopBreakOnComparisonSuccess, or they can decide to simply loop a fixed number of
times using PPI_Loop_CaptureAll. The buffer to capture the data acts as a circular buffer
with PPI_Loop_CaptureAll. In both cases, any buffers used to capture output within the
loop will have the value from the last iteration.

\section jtagreset JTAG Reset Operations

The operations defined as part of JTAG reset allow for the definition of both *how* the
TAP network is reset. The options are via a pin (TRST) or via the TAP state machine;
also, the length of time spent in TLR (e.g., the number of TCKs) is specified. By
default, a plugin implementation will try to use the best option for the probe's
capabilities (though this can be over-ridden). Some plugin implementations are unable to
perform certain types (e.g., they lack a TRST pin), so the InterfaceInfo will have that
information. Also, this set of operations includes the ability to add a delay to a flow.
This is delay is either implemented by stopping TCK (and so it precise), or (if the probe
cannot stop TCK) is implemented by leaving TMS and TDI constant and toggling TCK.

A common flow would be for a client to allocate a bundle handle, then either query or set
the options using: PPI_JTAG_GetParameters/PPI_JTAG_SetParameters. Finally, when the client
wishes to reset the TAP network, they call PPI_JTAG_EnterTLR to do so. They then might
wish to add a delay, so they stop in a stable state (Pause*, Shift*, TLR or RTI) and then
issue the delay using PPI_JTAG_Delay.

\section jtagreg JTAG Register Based Operations

The operations defined in this file are part of the JTAG functionality that allows for
access to the "high level" IR and DR registers on each TAP. They do not allow for direct
control of the JTAG TAP state machine, however. These operations are designed to map to
JTAG probes that have these high-level operations as part of their interface (a probe
is allowed--and encouraged--to implement multiple JTAG operation types).

A common flow would be for a user to perform an IR shift using the
PPI_JTAG_IRRegisterShift function, followed by a DR shift using PPI_JTAG_DRRegisterShift.
Depending on the capabilities of the probe, these operations can be instructed to remain
in the corresponding Pause state, instead of returning to RTI.

\section jtagState JTAG State Based Operations

The operations defined in this file are part of JTAG functionality that allows for
access to control the JTAG state transitions. The operations allow for exact state
transitions as well as shift operations to perform the Shift -> Ext1 operation. These
operations are designed to map to JTAG probes that have a GoToState and Shift-based
interface (a probe is allowed--and encouraged--to implement multiple
JTAG operation types).

For these operations, a client would call PPI_JTAG_GoToState to cycle the JTAG state
machine to particular states (without capturing any TDI/TDO data). The client would call
PPI_JTAG_StateIRShift, and PPI_JTAG_StateDRShift to perform the corresponding IR and DR
scans (transitioning to Ext1*).

\section jtagPins JTAG Pins Based Operations

The operations defined in this file are part of JTAG functionality that allows for
access to control the JTAG pin values. The operations allow for exact control over the
JTAG state machine sequence. The operations allow for exact pin control. These
operations are designed to map to JTAG probes that expose a true pin-based interface
(generally for high-performing probes).

For these operations, a client would call PPI_JTAG_PinsShift with the TMS, TDI, and TDO
buffers needed (as well as providing number of TCK).

\section jtagPad JTAG Padding Operations

The operations defined in this file are part of the JTAG functionality and are applicable
only to the Register- and State-based operations. These operations allow for a client to
set the IR and DR shift padding bits (to account for TAPs to be in bypass). The theory
here is that a
client can have part of the software set the padding bits to talk to the correct TAP,
while another part actually talks to that TAP (and it is independent of the TAP location).

For these operations, a client would call PPI_JTAG_GetBundlePadding to get the current
padding on the interface, and PPI_JTAG_SetBundlePadding to set the padding
on that interface. Both of the operations have an ability to specify what
TDI should be held at in a DR shift (either 0 or 1).
*/

#endif /* __BUNDLE_SERVICE_H__ */
