/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////

#pragma once

#include <Foundation/Error/ErrorMacros.h>
#include <Foundation/Error/ErrorTypes.h>

#undef POST_ERROR
#undef POST_ERROR_MESSAGE

#define PPI_ERROR(DEVICE, ERROR) \
	ErrorContext::PostError(DEVICE, ERROR)

#define PPI_ERROR_WITH_MESSAGE(DEVICE, ERROR, FUNC) \
	ErrorContext::PostError(DEVICE, ERROR, [&](std::basic_ostream<char>& stream) { stream << FUNC; } )

#define PPI_EXCEPT(DEVICE, ERROR) \
    { \
        ErrorContext::PostError(DEVICE, ERROR); \
        throw OpenIPC_Error_Exception(ERROR); \
    }

#define PPI_EXCEPT_WITH_MESSAGE(DEVICE, ERROR, FUNC) \
    { \
        ErrorContext::PostError(DEVICE, ERROR, [&](std::basic_ostream<char>& stream) { stream << FUNC; }); \
        throw OpenIPC_Error_Exception(ERROR); \
    }

// Define the OpenIPC macros to point to the PPI macros for legacy support
#define POST_ERROR_MESSAGE(ERROR, FUNC) PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, ERROR, FUNC)
#define POST_ERROR(ERROR) PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, ERROR)

#include "PluginNotifications.h"

#include <vector>
#include <string>
#include <thread>

class ErrorContext
{
private:
	void _PostInternal(OpenIPC_Error errorCode, std::string message);

	std::vector<std::pair<OpenIPC_Error, std::string>> _errorStack;
public:
	ErrorContext();
	~ErrorContext();

	std::string GetErrorMessage() const;

	template <typename TFunc>
	static void PostError(OpenIPC_DeviceId deviceId, OpenIPC_Error errorCode, const TFunc& fn)
	{
		if (!OpenIPC_PASS(errorCode))
		{
			ErrorContext* _this = GetErrorContext();

			if (_this != nullptr)
			{
				std::stringstream ss;
				fn(ss);

				// Error context implies that a PPI function exists further up the stack
				_this->_PostInternal(errorCode, ss.str());
			}
			else
			{
				// No error context implies that the method is internal to the plugin and need to notify error async.
				PluginNotifications::GetInstance().Notify(deviceId, errorCode, PPI_errorNotification, fn);
			}
		}
	}

	static void PostError(OpenIPC_DeviceId deviceId, OpenIPC_Error errorCode)
	{
		ErrorContext* _this = GetErrorContext();

		if (_this != nullptr)
		{
			// Error context implies that a PPI function exists further up the stack
			_this->_PostInternal(errorCode, "");
		}
		else
		{
			// No error context implies that the method is internal to the plugin and need to notify error async.
			PluginNotifications::GetInstance().Notify(deviceId, errorCode, PPI_errorNotification, [&](std::basic_ostream<char>&) { } );
		}
	}

	static ErrorContext* GetErrorContext();
};
